/**
 * AI Base Theme JavaScript
 * Handles keyboard navigation, header scroll effects, and mobile navigation
 */

// Enhanced focus indicators for keyboard-only users
document.addEventListener('keydown', function(e) {
    if (e.key === 'Tab') {
        document.body.classList.add('keyboard-navigation');
    }
});

document.addEventListener('mousedown', function() {
    document.body.classList.remove('keyboard-navigation');
});

// Modern header scroll effects
(function() {
    const header = document.getElementById('site-header');
    if (!header) return;

    let ticking = false;

    function updateHeader() {
        const scrollY = window.scrollY;

        if (scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }

        ticking = false;
    }

    function requestTick() {
        if (!ticking) {
            requestAnimationFrame(updateHeader);
            ticking = true;
        }
    }

    window.addEventListener('scroll', requestTick, { passive: true });
})();

// Mobile navigation functionality
(function() {
    const mobileToggle = document.getElementById('mobile-menu-toggle');
    const navigation = document.getElementById('primary-navigation');

    if (!mobileToggle || !navigation) return;

    function toggleMobileMenu() {
        const isExpanded = mobileToggle.getAttribute('aria-expanded') === 'true';

        mobileToggle.setAttribute('aria-expanded', !isExpanded);
        mobileToggle.classList.toggle('active');
        navigation.classList.toggle('active');

        // Prevent body scroll when menu is open
        document.body.style.overflow = !isExpanded ? 'hidden' : '';
    }

    // Handle mobile submenu toggles
    function initMobileSubmenus() {
        const parentItems = navigation.querySelectorAll('.menu-item-has-children');

        parentItems.forEach(item => {
            const link = item.querySelector('a');

            if (link) {
                link.addEventListener('click', function(e) {
                    // Only prevent default on mobile
                    if (window.innerWidth <= 768) {
                        e.preventDefault();
                        item.classList.toggle('expanded');
                    }
                });
            }
        });
    }

    // Initialize mobile submenus
    initMobileSubmenus();

    // Toggle menu on button click
    mobileToggle.addEventListener('click', toggleMobileMenu);

    // Close menu when clicking outside
    document.addEventListener('click', function(e) {
        if (!navigation.contains(e.target) && !mobileToggle.contains(e.target)) {
            if (navigation.classList.contains('active')) {
                toggleMobileMenu();
            }
        }
    });

    // Close menu on escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && navigation.classList.contains('active')) {
            toggleMobileMenu();
            mobileToggle.focus();
        }
    });

    // Close menu when clicking nav sublinks (not parent items)
    const navLinks = navigation.querySelectorAll('.sub-menu a');
    navLinks.forEach(link => {
        link.addEventListener('click', function() {
            if (window.innerWidth <= 768 && navigation.classList.contains('active')) {
                setTimeout(() => toggleMobileMenu(), 150);
            }
        });
    });

    // Handle window resize
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            if (window.innerWidth > 768) {
                // Reset mobile submenu states on desktop
                const expandedItems = navigation.querySelectorAll('.expanded');
                expandedItems.forEach(item => item.classList.remove('expanded'));

                if (navigation.classList.contains('active')) {
                    toggleMobileMenu();
                }
            }
        }, 150);
    });

})();