<?php
/**
 * Dreamformer Base Theme functions and definitions
 *
 * @package Dreamformer_Base
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Theme version
define( 'AI_BASE_THEME_VERSION', '1.0.0' );

/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function ai_base_theme_setup() {
    // Make theme available for translation
    load_theme_textdomain( 'dreamformer-base', get_template_directory() . '/languages' );

    // Add default posts and comments RSS feed links to head
    add_theme_support( 'automatic-feed-links' );

    // Let WordPress manage the document title
    add_theme_support( 'title-tag' );

    // Enable support for Post Thumbnails on posts and pages
    add_theme_support( 'post-thumbnails' );

    // Register navigation menus
    register_nav_menus( array(
        'primary' => esc_html__( 'Primary Menu', 'dreamformer-base' ),
        'footer'  => esc_html__( 'Footer Menu', 'dreamformer-base' ),
    ) );

    // Switch default core markup to valid HTML5
    add_theme_support( 'html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script',
    ) );

    // Add theme support for selective refresh for widgets
    add_theme_support( 'customize-selective-refresh-widgets' );

    // Add support for core custom logo
    add_theme_support( 'custom-logo', array(
        'height'      => 250,
        'width'       => 250,
        'flex-width'  => true,
        'flex-height' => true,
    ) );

    // Add support for responsive embedded content
    add_theme_support( 'responsive-embeds' );

    // Add support for editor styles
    add_theme_support( 'editor-styles' );
    add_editor_style( 'style.css' );

    // Yoast SEO breadcrumbs support
    add_theme_support( 'yoast-seo-breadcrumbs' );

    // Set content width
    $GLOBALS['content_width'] = 800;
}
add_action( 'after_setup_theme', 'ai_base_theme_setup' );

/**
 * Enqueue scripts and styles.
 */
function ai_base_theme_scripts() {
    // Main theme stylesheet - use filemtime() for cache busting when AI edits files
    $style_version = file_exists( get_stylesheet_directory() . '/style.css' )
        ? filemtime( get_stylesheet_directory() . '/style.css' )
        : AI_BASE_THEME_VERSION;
    wp_enqueue_style( 'ai-base-theme-style', get_stylesheet_uri(), array(), $style_version );

    // Add RTL language support
    wp_style_add_data( 'ai-base-theme-style', 'rtl', 'replace' );

    // Theme JavaScript - use filemtime() for cache busting when AI edits files
    $js_file = get_template_directory() . '/js/theme.js';
    $js_version = file_exists( $js_file ) ? filemtime( $js_file ) : AI_BASE_THEME_VERSION;
    wp_enqueue_script( 'ai-base-theme-script', get_template_directory_uri() . '/js/theme.js', array(), $js_version, true );

    // Threaded comment reply styles
    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }
}
add_action( 'wp_enqueue_scripts', 'ai_base_theme_scripts' );

/**
 * Register widget areas.
 */
function ai_base_theme_widgets_init() {
    register_sidebar( array(
        'name'          => esc_html__( 'Footer', 'dreamformer-base' ),
        'id'            => 'footer-1',
        'description'   => esc_html__( 'Footer widget area.', 'dreamformer-base' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
}
add_action( 'widgets_init', 'ai_base_theme_widgets_init' );

/**
 * Customizer additions.
 */
function ai_base_theme_customize_register( $wp_customize ) {
    // Theme Options Section
    $wp_customize->add_section( 'ai_base_theme_options', array(
        'title'    => esc_html__( 'Theme Options', 'dreamformer-base' ),
        'priority' => 160,
    ) );

    // Primary Color Setting
    $wp_customize->add_setting( 'primary_color', array(
        'default'           => '#2563eb',
        'type'              => 'theme_mod',
        'capability'        => 'edit_theme_options',
        'transport'         => 'refresh',
        'sanitize_callback' => 'sanitize_hex_color',
    ) );

    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'primary_color', array(
        'label'    => esc_html__( 'Primary Color', 'dreamformer-base' ),
        'section'  => 'ai_base_theme_options',
        'settings' => 'primary_color',
    ) ) );

    // Footer Text Setting
    $wp_customize->add_setting( 'footer_text', array(
        'default'           => esc_html__( '© 2024 Your Website. All rights reserved.', 'dreamformer-base' ),
        'type'              => 'theme_mod',
        'capability'        => 'edit_theme_options',
        'transport'         => 'refresh',
        'sanitize_callback' => 'sanitize_text_field',
    ) );

    $wp_customize->add_control( 'footer_text', array(
        'label'    => esc_html__( 'Footer Text', 'dreamformer-base' ),
        'section'  => 'ai_base_theme_options',
        'settings' => 'footer_text',
        'type'     => 'text',
    ) );
}
add_action( 'customize_register', 'ai_base_theme_customize_register' );

/**
 * Output custom CSS for customizer settings.
 */
function ai_base_theme_customizer_css() {
    $primary_color = get_theme_mod( 'primary_color', '#2563eb' );
    
    if ( '#2563eb' !== $primary_color ) {
        ?>
        <style type="text/css">
            :root {
                --color-primary: <?php echo esc_html( $primary_color ); ?>;
            }
        </style>
        <?php
    }
}
add_action( 'wp_head', 'ai_base_theme_customizer_css' );

/**
 * Custom template tags for this theme.
 */
function ai_base_theme_posted_on() {
    $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
    if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
        $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
    }

    $time_string = sprintf( $time_string,
        esc_attr( get_the_date( DATE_W3C ) ),
        esc_html( get_the_date() ),
        esc_attr( get_the_modified_date( DATE_W3C ) ),
        esc_html( get_the_modified_date() )
    );

    $posted_on = sprintf(
        /* translators: %s: post date. */
        esc_html_x( 'Posted on %s', 'post date', 'dreamformer-base' ),
        '<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>'
    );

    echo '<span class="posted-on">' . $posted_on . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}

function ai_base_theme_posted_by() {
    $byline = sprintf(
        /* translators: %s: post author. */
        esc_html_x( 'by %s', 'post author', 'dreamformer-base' ),
        '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
    );

    echo '<span class="byline"> ' . $byline . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}


/**
 * Performance optimizations
 */

// Remove unnecessary WordPress features for performance
function ai_base_theme_remove_bloat() {
    // Remove emoji scripts
    remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
    remove_action( 'wp_print_styles', 'print_emoji_styles' );
    
    // Remove Windows Live Writer link
    remove_action( 'wp_head', 'wlwmanifest_link' );
    
    // Remove RSD link
    remove_action( 'wp_head', 'rsd_link' );
    
    // Remove shortlink
    remove_action( 'wp_head', 'wp_shortlink_wp_head' );
    
    // Remove feed links (can be re-enabled if needed)
    remove_action( 'wp_head', 'feed_links', 2 );
    remove_action( 'wp_head', 'feed_links_extra', 3 );
}
add_action( 'init', 'ai_base_theme_remove_bloat' );

// Security enhancements
function ai_base_theme_security_headers() {
    // Remove WordPress version from head
    remove_action( 'wp_head', 'wp_generator' );

    // Add filters to remove version strings from scripts and styles
    add_filter( 'script_loader_src', 'ai_base_theme_remove_version_strings', 15, 1 );
    add_filter( 'style_loader_src', 'ai_base_theme_remove_version_strings', 15, 1 );
}
add_action( 'init', 'ai_base_theme_security_headers' );

/**
 * Remove version strings from script and style URLs for security
 *
 * @param string $src The source URL
 * @return string Modified URL without version parameter
 */
function ai_base_theme_remove_version_strings( $src ) {
    if ( strpos( $src, 'ver=' ) ) {
        $src = remove_query_arg( 'ver', $src );
    }
    return $src;
}

/**
 * Helper function to get theme option
 */
function ai_base_theme_get_option( $option, $default = '' ) {
    return get_theme_mod( $option, $default );
}

/**
 * Body classes for AI targeting
 */
function ai_base_theme_body_classes( $classes ) {
    // Add theme identifier class
    $classes[] = 'dreamformer-base';

    // Add page template class
    if ( is_page_template() ) {
        $classes[] = 'page-template-' . sanitize_html_class( str_replace( '.', '-', get_page_template_slug() ) );
    }

    // Add AI-generated page class
    if ( is_page() && get_post_meta( get_the_ID(), 'ai_generated_page', true ) ) {
        $classes[] = 'ai-generated-page';
    }

    return $classes;
}
add_filter( 'body_class', 'ai_base_theme_body_classes' );

/**
 * Disable wpautop for AI-generated pages completely
 * This prevents WordPress from corrupting the HTML structure
 */
function ai_base_theme_protect_ai_pages_from_wpautop( $content ) {
    global $post;

    // Check if this is an AI-generated page
    if ( $post && $post->post_type === 'page' && get_post_meta( $post->ID, 'ai_generated_page', true ) ) {
        // Remove wpautop filter for this content
        remove_filter( 'the_content', 'wpautop' );
        remove_filter( 'the_excerpt', 'wpautop' );
    }

    return $content;
}
// Run BEFORE wpautop (priority 8, wpautop runs at 10)
add_filter( 'the_content', 'ai_base_theme_protect_ai_pages_from_wpautop', 8 );

/**
 * Prevent wpautop from running when saving AI pages
 */
function ai_base_theme_prevent_wpautop_on_save( $data, $postarr ) {
    // Check if this is an AI-generated page
    if ( $data['post_type'] === 'page' && isset( $postarr['ID'] ) ) {
        $is_ai_page = get_post_meta( $postarr['ID'], 'ai_generated_page', true );
        if ( $is_ai_page ) {
            // Remove wpautop from content_save_pre filter
            remove_filter( 'content_save_pre', 'wpautop' );
            remove_filter( 'content_save_pre', 'wp_filter_post_kses' );
        }
    }
    return $data;
}
// Hook into wp_insert_post_data BEFORE content is saved
add_filter( 'wp_insert_post_data', 'ai_base_theme_prevent_wpautop_on_save', 5, 2 );

/**
 * WCAG 2.1 AA Accessibility Features
 */

// Skip links for keyboard navigation
function ai_base_theme_skip_links() {
    ?>
    <a class="skip-link" href="#main-content"><?php esc_html_e( 'Skip to content', 'dreamformer-base' ); ?></a>
    <a class="skip-link" href="#primary-navigation"><?php esc_html_e( 'Skip to navigation', 'dreamformer-base' ); ?></a>
    <?php
}
add_action( 'wp_body_open', 'ai_base_theme_skip_links' );

// Core Web Vitals optimized image loading
function ai_base_theme_image_optimization() {
    // Smart image loading - lazy load by default, prioritize above-the-fold
    add_filter( 'wp_get_attachment_image_attributes', function( $attr, $attachment, $size ) {
        if ( is_admin() ) {
            return $attr;
        }
        
        // Above-the-fold images get high priority, no lazy loading
        $is_above_fold = false;
        
        // Hero images on front page
        if ( is_front_page() && in_array( $size, array( 'large', 'full', 'hero' ) ) ) {
            $is_above_fold = true;
        }
        
        // Featured image on single posts (likely above-the-fold)
        if ( is_single() && has_post_thumbnail() && get_post_thumbnail_id() === $attachment->ID ) {
            $is_above_fold = true;
        }
        
        if ( $is_above_fold ) {
            // High priority loading for above-the-fold images
            $attr['fetchpriority'] = 'high';
            $attr['decoding'] = 'sync'; // Decode immediately
            // Don't set loading="lazy" for above-fold content
        } else {
            // Lazy load below-the-fold images
            $attr['loading'] = 'lazy';
            $attr['decoding'] = 'async';
        }
        
        return $attr;
    }, 10, 3 );
    
    // Improve image alt text when missing
    add_filter( 'wp_get_attachment_image_attributes', function( $attr, $attachment ) {
        if ( empty( $attr['alt'] ) ) {
            $alt_text = get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true );
            if ( empty( $alt_text ) ) {
                $attr['alt'] = get_the_title( $attachment->ID );
            }
        }
        return $attr;
    }, 10, 2 );
    
    // Preload critical above-the-fold images
    add_action( 'wp_head', function() {
        if ( is_front_page() && has_custom_header() ) {
            $header_image = get_header_image();
            if ( $header_image ) {
                echo '<link rel="preload" as="image" href="' . esc_url( $header_image ) . '">' . "\n";
            }
        }
        
        if ( is_single() && has_post_thumbnail() ) {
            $thumbnail_url = get_the_post_thumbnail_url( null, 'large' );
            if ( $thumbnail_url ) {
                echo '<link rel="preload" as="image" href="' . esc_url( $thumbnail_url ) . '">' . "\n";
            }
        }
    }, 5 );
}
add_action( 'init', 'ai_base_theme_image_optimization' );

// Accessibility enhancements
function ai_base_theme_accessibility_enhancements() {
    // Improve search form accessibility
    add_filter( 'get_search_form', function( $form ) {
        // Add proper labels and ARIA attributes
        $form = str_replace( 
            'placeholder="',
            'aria-label="' . esc_attr__( 'Search', 'dreamformer-base' ) . '" placeholder="',
            $form 
        );
        return $form;
    });
    
    // Add ARIA current to menu items
    add_filter( 'nav_menu_link_attributes', function( $atts, $item, $args ) {
        // Add aria-current for current page
        if ( in_array( 'current-menu-item', $item->classes ) || in_array( 'current-page-ancestor', $item->classes ) ) {
            $atts['aria-current'] = 'page';
        }
        return $atts;
    }, 10, 3 );
}
add_action( 'init', 'ai_base_theme_accessibility_enhancements' );

/**
 * Custom walker for clean navigation HTML
 */
class Clean_Nav_Walker extends Walker_Nav_Menu {

    // Start the element output
    public function start_el( &$output, $item, $depth = 0, $args = null, $id = 0 ) {
        $indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';

        $classes = empty( $item->classes ) ? array() : (array) $item->classes;

        // Build class string
        $class_names = '';

        // Check for current page
        if ( in_array( 'current-menu-item', $classes ) || in_array( 'current_page_item', $classes ) ) {
            $class_names .= ' current';
        }

        // Check if has children
        if ( in_array( 'menu-item-has-children', $classes ) ) {
            $class_names .= ' menu-item-has-children';
        }

        $class_names = $class_names ? ' class="' . trim( $class_names ) . '"' : '';

        $output .= $indent . '<li' . $class_names . '>';

        $attributes = ! empty( $item->attr_title ) ? ' title="'  . esc_attr( $item->attr_title ) .'"' : '';
        $attributes .= ! empty( $item->target )     ? ' target="' . esc_attr( $item->target     ) .'"' : '';
        $attributes .= ! empty( $item->xfn )        ? ' rel="'    . esc_attr( $item->xfn        ) .'"' : '';
        $attributes .= ! empty( $item->url )        ? ' href="'   . esc_attr( $item->url        ) .'"' : '';

        $item_output = '<a' . $attributes . '>';
        $item_output .= apply_filters( 'the_title', $item->title, $item->ID );
        $item_output .= '</a>';

        $output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
    }

    // End the element output
    public function end_el( &$output, $item, $depth = 0, $args = null ) {
        $output .= "</li>\n";
    }
}