<?php
/**
 * AI Site Builder - WooCommerce Product Page Styler
 * Handles AI-generated styling for WooCommerce product pages
 */

if (!defined('ABSPATH')) {
    exit;
}

class AI_WooCommerce_Styler {

    private $logger;
    private $option_name = 'ai_woocommerce_product_styles';
    private $history_name = 'ai_woocommerce_product_styles_history';

    public function __construct() {
        $this->logger = AI_Logger::get_instance('AI_WooCommerce_Styler');
        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        add_action('after_switch_theme', [$this, 'check_theme_compatibility']);
    }

    /**
     * Register REST API routes for WooCommerce styling
     */
    public function register_rest_routes() {
        // Generate product page styles
        register_rest_route('ai-builder/v1', '/woocommerce/style-products', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_generate_product_styles'],
            'permission_callback' => function() {
                return current_user_can('edit_theme_options');
            },
            'args' => [
                'prompt' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'User prompt for product page styling'
                ],
                'brand_colors' => [
                    'required' => false,
                    'type' => 'array',
                    'description' => 'Array of brand colors'
                ],
                'style_mood' => [
                    'required' => false,
                    'type' => 'string',
                    'description' => 'Style mood (modern, luxury, minimal, etc.)'
                ],
                'business_info' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Business information for context'
                ]
            ]
        ]);

        // Reset to default styles
        register_rest_route('ai-builder/v1', '/woocommerce/reset-styles', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_reset_styles'],
            'permission_callback' => function() {
                return current_user_can('edit_theme_options');
            }
        ]);

        // Get current style status
        register_rest_route('ai-builder/v1', '/woocommerce/style-status', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_style_status'],
            'permission_callback' => function() {
                return current_user_can('edit_theme_options');
            }
        ]);

        // Get current styles (for manage styles modal)
        register_rest_route('ai-builder/v1', '/woocommerce/get-styles', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_current_styles'],
            'permission_callback' => function() {
                return current_user_can('edit_theme_options');
            }
        ]);

        // Update styles (for manual editing)
        register_rest_route('ai-builder/v1', '/woocommerce/update-styles', [
            'methods' => 'PUT',
            'callback' => [$this, 'rest_update_styles'],
            'permission_callback' => function() {
                return current_user_can('edit_theme_options');
            },
            'args' => [
                'css' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'Updated CSS code'
                ]
            ]
        ]);
    }

    /**
     * REST endpoint: Generate product page styles
     */
    public function rest_generate_product_styles($request) {
        // Validate WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'WooCommerce must be active to style product pages'
            ], 400);
        }

        // Validate theme compatibility
        if (!$this->is_theme_compatible()) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Product styling requires the AI WooCommerce Theme for full compatibility'
            ], 400);
        }

        $prompt = sanitize_textarea_field($request->get_param('prompt'));
        $brand_colors = $request->get_param('brand_colors') ?: [];
        $style_mood = sanitize_text_field($request->get_param('style_mood') ?: '');
        $business_info = $request->get_param('business_info') ?: [];

        $this->logger->info('Starting product style generation', [
            'prompt' => $prompt,
            'style_mood' => $style_mood,
            'colors_count' => count($brand_colors)
        ]);

        try {
            // Generate styles using Vercel API
            $css = $this->generate_styles_via_api($prompt, $brand_colors, $style_mood, $business_info);

            if (is_wp_error($css)) {
                throw new Exception($css->get_error_message());
            }

            // Validate and sanitize CSS
            $validated_css = $this->validate_and_sanitize_css($css);
            if (!$validated_css) {
                throw new Exception('Generated CSS failed validation');
            }

            // Save styles to database
            $this->save_styles($validated_css, $prompt, $style_mood);

            // Clear caches
            $this->clear_style_caches();

            $this->logger->info('Product styles generated successfully', [
                'css_length' => strlen($validated_css)
            ]);

            return new WP_REST_Response([
                'success' => true,
                'message' => 'Product page styles generated successfully',
                'sample_product_url' => $this->get_sample_product_url(),
                'css_preview' => substr($validated_css, 0, 200) . '...'
            ]);

        } catch (Exception $e) {
            $this->logger->error('Product style generation failed', [
                'error' => $e->getMessage(),
                'prompt' => $prompt
            ]);

            return new WP_REST_Response([
                'success' => false,
                'error' => 'Style generation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * REST endpoint: Reset to default styles
     */
    public function rest_reset_styles($request) {
        // Delete custom styles
        delete_option($this->option_name);

        // Clear caches
        $this->clear_style_caches();

        $this->logger->info('Product styles reset to defaults');

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Product page styles reset to defaults'
        ]);
    }

    /**
     * REST endpoint: Get current style status
     */
    public function rest_get_style_status($request) {
        $current_styles = get_option($this->option_name);

        return new WP_REST_Response([
            'has_custom_styles' => !empty($current_styles),
            'generated_at' => $current_styles['generated_at'] ?? null,
            'style_mood' => $current_styles['style_mood'] ?? null,
            'theme_compatible' => $this->is_theme_compatible(),
            'woocommerce_active' => class_exists('WooCommerce')
        ]);
    }

    /**
     * REST endpoint: Get current styles for editing
     */
    public function rest_get_current_styles($request) {
        $current_styles = get_option($this->option_name);

        if (empty($current_styles)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'No custom styles found. Generate some first!'
            ], 404);
        }

        return new WP_REST_Response([
            'success' => true,
            'css' => $current_styles['css'] ?? '',
            'generated_at' => $current_styles['generated_at'] ?? null,
            'prompt' => $current_styles['prompt'] ?? '',
            'style_mood' => $current_styles['style_mood'] ?? '',
            'ai_model' => $current_styles['ai_model'] ?? '',
            'active' => $current_styles['active'] ?? true
        ]);
    }

    /**
     * REST endpoint: Update styles manually
     */
    public function rest_update_styles($request) {
        $css = $request->get_param('css');

        // Validate CSS is not empty
        if (empty($css)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'CSS cannot be empty'
            ], 400);
        }

        // Validate and sanitize CSS
        $validated_css = $this->validate_and_sanitize_css($css);
        if (!$validated_css) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'CSS failed validation. Please check for invalid or unsafe CSS.'
            ], 400);
        }

        try {
            // Get current styles to preserve metadata
            $current_styles = get_option($this->option_name, []);

            // Update the CSS while preserving other data
            $updated_styles = array_merge($current_styles, [
                'css' => $validated_css,
                'updated_at' => current_time('mysql'),
                'manually_edited' => true,
                'active' => true
            ]);

            // Save to database
            update_option($this->option_name, $updated_styles);

            // Add to history
            $this->add_to_history($updated_styles);

            // Clear caches
            $this->clear_style_caches();

            $this->logger->info('Product styles updated manually', [
                'css_length' => strlen($validated_css),
                'user_id' => get_current_user_id()
            ]);

            return new WP_REST_Response([
                'success' => true,
                'message' => 'Styles updated successfully',
                'css_length' => strlen($validated_css)
            ]);

        } catch (Exception $e) {
            $this->logger->error('Manual style update failed', [
                'error' => $e->getMessage(),
                'user_id' => get_current_user_id()
            ]);

            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to update styles: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate styles using Vercel API
     */
    private function generate_styles_via_api($prompt, $brand_colors, $style_mood, $business_info) {
        $api_request = [
            'prompt' => $prompt,
            'brand_colors' => $brand_colors,
            'style_mood' => $style_mood,
            'business_info' => $business_info
        ];

        $this->logger->info('Calling Vercel API for product style generation');

        $response = wp_remote_post('https://ai-site-builder-api.vercel.app/api/generate-product-styles', [
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($api_request),
            'timeout' => 120
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('api_request_failed', 'Failed to connect to AI styling service');
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['success']) || !$data['success']) {
            return new WP_Error('generation_failed', 'Style generation failed: ' . ($data['error'] ?? 'Unknown error'));
        }

        return $data['css'];
    }

    /**
     * Validate and sanitize CSS
     */
    private function validate_and_sanitize_css($css) {
        // Remove dangerous content
        $dangerous_patterns = [
            'javascript:',
            'expression(',
            '<script',
            'data:',
            'vbscript:',
            '@import',
            'url(' // Be careful with external URLs
        ];

        foreach ($dangerous_patterns as $pattern) {
            if (stripos($css, $pattern) !== false) {
                $this->logger->warning('CSS validation failed: dangerous pattern detected', [
                    'pattern' => $pattern
                ]);
                return false;
            }
        }

        // Basic CSS validation - check for balanced braces
        $open_braces = substr_count($css, '{');
        $close_braces = substr_count($css, '}');

        if ($open_braces !== $close_braces) {
            $this->logger->warning('CSS validation failed: unbalanced braces');
            return false;
        }

        // Sanitize and return
        return wp_strip_all_tags($css);
    }

    /**
     * Save styles to database
     */
    private function save_styles($css, $prompt, $style_mood) {
        // Backup current styles to history
        $current = get_option($this->option_name);
        if ($current) {
            $this->add_to_history($current);
        }

        // Save new styles
        $data = [
            'css' => $css,
            'generated_at' => current_time('mysql'),
            'ai_model' => AI_Config_Manager::AI_MODEL,
            'prompt' => $prompt,
            'style_mood' => $style_mood,
            'version' => '1.0',
            'active' => true
        ];

        update_option($this->option_name, $data);

        $this->logger->info('Product styles saved to database', [
            'css_length' => strlen($css),
            'style_mood' => $style_mood
        ]);
    }

    /**
     * Add current styles to history
     */
    private function add_to_history($styles) {
        $history = get_option($this->history_name, []);

        // Keep only last 5 versions
        if (count($history) >= 5) {
            array_shift($history);
        }

        $history[] = $styles;
        update_option($this->history_name, $history);
    }

    /**
     * Clear various caches
     */
    private function clear_style_caches() {
        // Clear WordPress transients
        delete_transient('ai_woocommerce_styles_cache');

        // Clear object cache
        wp_cache_flush();

        // Clear popular cache plugins
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }

        // LiteSpeed Cache
        if (class_exists('LiteSpeed_Cache_API')) {
            do_action('litespeed_purge_all');
        }

        // WP Rocket
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain();
        }

        // W3 Total Cache
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all();
        }

        // Trigger action for other plugins
        do_action('ai_woocommerce_styles_updated');

        $this->logger->info('Cleared all style caches');
    }

    /**
     * Check if current theme is compatible
     */
    private function is_theme_compatible() {
        $current_theme = get_option('stylesheet');
        return $current_theme === 'dreamformer-woocommerce';
    }

    /**
     * Check theme compatibility on theme switch
     */
    public function check_theme_compatibility() {
        if (!$this->is_theme_compatible()) {
            // Deactivate custom styles if theme is not compatible
            $current = get_option($this->option_name);
            if ($current) {
                $current['active'] = false;
                update_option($this->option_name, $current);

                $this->logger->warning('Custom product styles deactivated due to theme change');
            }
        }
    }

    /**
     * Get sample product URL for testing
     */
    private function get_sample_product_url() {
        if (!class_exists('WooCommerce')) {
            return '';
        }

        // Get first published product
        $products = wc_get_products([
            'limit' => 1,
            'status' => 'publish'
        ]);

        if (!empty($products)) {
            return get_permalink($products[0]->get_id());
        }

        return wc_get_page_permalink('shop');
    }

    /**
     * Get current styles for frontend injection
     */
    public static function get_current_styles() {
        $styles = get_option('ai_woocommerce_product_styles');

        if ($styles && !empty($styles['css']) && $styles['active']) {
            return $styles['css'];
        }

        return false;
    }
}