<?php
/**
 * AI Site Builder - Unified Version Control System
 *
 * Handles version control for both themes and pages with undo/redo functionality
 * Replaces the old fragmented backup system with a reliable database approach
 *
 * @package AI_Site_Builder
 * @since 4.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Version_Control {

    private $logger;
    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'ai_version_control';
        $this->logger = AI_Logger::get_instance( 'AI_Version_Control' );

        $this->init_hooks();
    }

    private function init_hooks() {
        // Register REST API endpoints
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
    }

    /**
     * Register REST API routes for version control
     */
    public function register_rest_routes() {
        // Create version endpoint
        register_rest_route( 'ai-builder/v1', '/versions/create', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_create_version' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_themes' ) || current_user_can( 'edit_pages' );
            },
            'args' => array(
                'entity_type' => array(
                    'required' => true,
                    'type' => 'string',
                    'enum' => array( 'theme', 'page' ),
                    'description' => 'Type of entity (theme or page)'
                ),
                'entity_id' => array(
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'ID of the theme or page'
                ),
                'content_before' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => 'Content before changes'
                ),
                'content_after' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'Content after changes'
                ),
                'change_description' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => 'Description of the changes'
                ),
                'user_prompt' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => 'User prompt that triggered the change'
                ),
                'ai_model' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => 'AI model used for the change'
                ),
                'change_metadata' => array(
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Additional metadata about the change'
                )
            )
        ) );

        // Undo endpoint
        register_rest_route( 'ai-builder/v1', '/versions/(?P<entity_type>theme|page)/(?P<entity_id>\d+)/undo', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_undo_change' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_themes' ) || current_user_can( 'edit_pages' );
            }
        ) );

        // Redo endpoint
        register_rest_route( 'ai-builder/v1', '/versions/(?P<entity_type>theme|page)/(?P<entity_id>\d+)/redo', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_redo_change' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_themes' ) || current_user_can( 'edit_pages' );
            }
        ) );

        // Version history endpoint
        register_rest_route( 'ai-builder/v1', '/versions/(?P<entity_type>theme|page)/(?P<entity_id>\d+)/history', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_version_history' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_themes' ) || current_user_can( 'edit_pages' );
            }
        ) );

        // Restore to specific version endpoint
        register_rest_route( 'ai-builder/v1', '/versions/(?P<version_id>\d+)/restore', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_restore_version' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_themes' ) || current_user_can( 'edit_pages' );
            }
        ) );

        // Cleanup old versions endpoint
        register_rest_route( 'ai-builder/v1', '/versions/cleanup', array(
            'methods' => 'DELETE',
            'callback' => array( $this, 'rest_cleanup_versions' ),
            'permission_callback' => function() {
                return current_user_can( 'manage_options' );
            },
            'args' => array(
                'keep_count' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 50,
                    'description' => 'Number of versions to keep per entity'
                )
            )
        ) );
    }

    /**
     * Save a new version when content changes
     */
    public function save_version( $entity_type, $entity_id, $content_before, $content_after, $options = array() ) {
        global $wpdb;

        $defaults = array(
            'change_description' => '',
            'user_prompt' => '',
            'ai_model' => AI_Config_Manager::AI_MODEL,
            'change_metadata' => array()
        );

        $options = wp_parse_args( $options, $defaults );

        // Start transaction for atomicity
        $wpdb->query( 'START TRANSACTION' );

        try {
            // Get next version number with row lock to prevent race conditions
            $current_version = $wpdb->get_var( $wpdb->prepare(
                "SELECT MAX(version_number) FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d
                 FOR UPDATE",
                $entity_type, $entity_id
            ) );

            $new_version = ( $current_version ?? 0 ) + 1;

            // Mark all previous versions as not current
            $wpdb->update(
                $this->table_name,
                array( 'is_current' => 0 ),
                array(
                    'entity_type' => $entity_type,
                    'entity_id' => $entity_id
                ),
                array( '%d' ),
                array( '%s', '%d' )
            );

            // Get parent version ID (current version becomes parent)
            $parent_version_id = null;
            if ( $current_version > 0 ) {
                $parent_version_id = $wpdb->get_var( $wpdb->prepare(
                    "SELECT id FROM {$this->table_name}
                     WHERE entity_type = %s AND entity_id = %d AND version_number = %d",
                    $entity_type, $entity_id, $current_version
                ) );
            }

            // Insert new version
            $result = $wpdb->insert(
                $this->table_name,
                array(
                    'entity_type' => $entity_type,
                    'entity_id' => $entity_id,
                    'version_number' => $new_version,
                    'content_before' => $content_before,
                    'content_after' => $content_after,
                    'change_description' => $options['change_description'],
                    'user_prompt' => $options['user_prompt'],
                    'ai_model' => $options['ai_model'],
                    'user_id' => get_current_user_id(),
                    'is_current' => 1,
                    'change_metadata' => json_encode( $options['change_metadata'] ),
                    'parent_version_id' => $parent_version_id
                ),
                array(
                    '%s', '%d', '%d', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%d'
                )
            );

            if ( $result === false ) {
                throw new Exception( $wpdb->last_error ?: 'Unknown database error' );
            }

            $version_id = $wpdb->insert_id;

            // Commit transaction - all operations succeeded
            $wpdb->query( 'COMMIT' );

            $this->logger->info( 'Version saved successfully', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'version_id' => $version_id,
                'version_number' => $new_version
            ] );

            // Clean up old versions (after commit, outside transaction)
            $this->cleanup_old_versions( $entity_type, $entity_id );

            return $version_id;

        } catch ( Exception $e ) {
            // Rollback transaction - undo all changes
            $wpdb->query( 'ROLLBACK' );

            $this->logger->error( 'Failed to save version - rolled back transaction', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'error' => $e->getMessage()
            ] );

            return new WP_Error( 'save_failed', 'Failed to save version: ' . $e->getMessage() );
        }
    }

    /**
     * Undo the last change
     */
    public function undo_change( $entity_type, $entity_id ) {
        global $wpdb;

        // Start transaction for atomicity
        $wpdb->query( 'START TRANSACTION' );

        try {
            // Get current version
            $current_version = $wpdb->get_row( $wpdb->prepare(
                "SELECT * FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d AND is_current = 1",
                $entity_type, $entity_id
            ) );

            if ( ! $current_version ) {
                throw new Exception( 'No current version found' );
            }

            // Can't undo if we're at version 1
            if ( $current_version->version_number <= 1 ) {
                throw new Exception( 'Cannot undo the initial version' );
            }

            // Get previous version
            $previous_version = $wpdb->get_row( $wpdb->prepare(
                "SELECT * FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d AND version_number = %d",
                $entity_type, $entity_id, $current_version->version_number - 1
            ) );

            if ( ! $previous_version ) {
                throw new Exception( 'Previous version not found' );
            }

            // Apply the previous version content
            $result = $this->apply_version_content( $entity_type, $entity_id, $previous_version->content_after );

            if ( is_wp_error( $result ) ) {
                throw new Exception( $result->get_error_message() );
            }

            // Mark previous version as current
            $wpdb->update(
                $this->table_name,
                array( 'is_current' => 0 ),
                array( 'entity_type' => $entity_type, 'entity_id' => $entity_id ),
                array( '%d' ),
                array( '%s', '%d' )
            );

            $wpdb->update(
                $this->table_name,
                array( 'is_current' => 1 ),
                array( 'id' => $previous_version->id ),
                array( '%d' ),
                array( '%d' )
            );

            // Handle linked version (undo theme if page was undone, or vice versa)
            if ( ! empty( $current_version->linked_version_id ) ) {
                $linked_version = $wpdb->get_row( $wpdb->prepare(
                    "SELECT * FROM {$this->table_name} WHERE id = %d",
                    $current_version->linked_version_id
                ) );

                if ( $linked_version && $linked_version->is_current == 1 ) {
                    // Get previous version of linked entity
                    $linked_previous = $wpdb->get_row( $wpdb->prepare(
                        "SELECT * FROM {$this->table_name}
                         WHERE entity_type = %s AND entity_id = %d AND version_number = %d",
                        $linked_version->entity_type,
                        $linked_version->entity_id,
                        $linked_version->version_number - 1
                    ) );

                    if ( $linked_previous ) {
                        // Apply linked previous version
                        $linked_result = $this->apply_version_content(
                            $linked_version->entity_type,
                            $linked_version->entity_id,
                            $linked_previous->content_after
                        );

                        if ( ! is_wp_error( $linked_result ) ) {
                            // Mark linked previous as current
                            $wpdb->update(
                                $this->table_name,
                                array( 'is_current' => 0 ),
                                array( 'entity_type' => $linked_version->entity_type, 'entity_id' => $linked_version->entity_id ),
                                array( '%d' ),
                                array( '%s', '%d' )
                            );

                            $wpdb->update(
                                $this->table_name,
                                array( 'is_current' => 1 ),
                                array( 'id' => $linked_previous->id ),
                                array( '%d' ),
                                array( '%d' )
                            );

                            $this->logger->info( 'Linked version also undone', [
                                'linked_entity_type' => $linked_version->entity_type,
                                'linked_entity_id' => $linked_version->entity_id
                            ] );
                        }
                    }
                }
            }

            // Commit transaction - all operations succeeded
            $wpdb->query( 'COMMIT' );

            $this->logger->info( 'Change undone successfully', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'from_version' => $current_version->version_number,
                'to_version' => $previous_version->version_number
            ] );

            return $previous_version;

        } catch ( Exception $e ) {
            // Rollback transaction - undo all changes
            $wpdb->query( 'ROLLBACK' );

            $this->logger->error( 'Undo failed - rolled back transaction', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'error' => $e->getMessage()
            ] );

            return new WP_Error( 'undo_failed', $e->getMessage() );
        }
    }

    /**
     * Redo the next change
     */
    public function redo_change( $entity_type, $entity_id ) {
        global $wpdb;

        // Start transaction for atomicity
        $wpdb->query( 'START TRANSACTION' );

        try {
            // Get current version
            $current_version = $wpdb->get_row( $wpdb->prepare(
                "SELECT * FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d AND is_current = 1",
                $entity_type, $entity_id
            ) );

            if ( ! $current_version ) {
                throw new Exception( 'No current version found' );
            }

            // Get next version
            $next_version = $wpdb->get_row( $wpdb->prepare(
                "SELECT * FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d AND version_number = %d",
                $entity_type, $entity_id, $current_version->version_number + 1
            ) );

            if ( ! $next_version ) {
                throw new Exception( 'No next version to redo' );
            }

            // Apply the next version content
            $result = $this->apply_version_content( $entity_type, $entity_id, $next_version->content_after );

            if ( is_wp_error( $result ) ) {
                throw new Exception( $result->get_error_message() );
            }

            // Mark next version as current
            $wpdb->update(
                $this->table_name,
                array( 'is_current' => 0 ),
                array( 'entity_type' => $entity_type, 'entity_id' => $entity_id ),
                array( '%d' ),
                array( '%s', '%d' )
            );

            $wpdb->update(
                $this->table_name,
                array( 'is_current' => 1 ),
                array( 'id' => $next_version->id ),
                array( '%d' ),
                array( '%d' )
            );

            // Handle linked version (redo theme if page was redone, or vice versa)
            if ( ! empty( $next_version->linked_version_id ) ) {
                $linked_version = $wpdb->get_row( $wpdb->prepare(
                    "SELECT * FROM {$this->table_name} WHERE id = %d",
                    $next_version->linked_version_id
                ) );

                if ( $linked_version ) {
                    // Get current version of linked entity to check if redo is needed
                    $linked_current = $wpdb->get_row( $wpdb->prepare(
                        "SELECT * FROM {$this->table_name}
                         WHERE entity_type = %s AND entity_id = %d AND is_current = 1",
                        $linked_version->entity_type,
                        $linked_version->entity_id
                    ) );

                    // Only redo if linked version is NOT already current (avoid double-redo)
                    if ( $linked_current && $linked_current->id !== $linked_version->id ) {
                        // Apply linked version content
                        $linked_result = $this->apply_version_content(
                            $linked_version->entity_type,
                            $linked_version->entity_id,
                            $linked_version->content_after
                        );

                        if ( ! is_wp_error( $linked_result ) ) {
                            // Mark linked version as current
                            $wpdb->update(
                                $this->table_name,
                                array( 'is_current' => 0 ),
                                array( 'entity_type' => $linked_version->entity_type, 'entity_id' => $linked_version->entity_id ),
                                array( '%d' ),
                                array( '%s', '%d' )
                            );

                            $wpdb->update(
                                $this->table_name,
                                array( 'is_current' => 1 ),
                                array( 'id' => $linked_version->id ),
                                array( '%d' ),
                                array( '%d' )
                            );

                            $this->logger->info( 'Linked version also redone', [
                                'linked_entity_type' => $linked_version->entity_type,
                                'linked_entity_id' => $linked_version->entity_id,
                                'linked_version_number' => $linked_version->version_number
                            ] );
                        }
                    }
                }
            }

            // Commit transaction - all operations succeeded
            $wpdb->query( 'COMMIT' );

            $this->logger->info( 'Change redone successfully', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'from_version' => $current_version->version_number,
                'to_version' => $next_version->version_number
            ] );

            return $next_version;

        } catch ( Exception $e ) {
            // Rollback transaction - undo all changes
            $wpdb->query( 'ROLLBACK' );

            $this->logger->error( 'Redo failed - rolled back transaction', [
                'entity_type' => $entity_type,
                'entity_id' => $entity_id,
                'error' => $e->getMessage()
            ] );

            return new WP_Error( 'redo_failed', $e->getMessage() );
        }
    }

    /**
     * Apply version content to the actual entity
     */
    private function apply_version_content( $entity_type, $entity_id, $content ) {
        if ( $entity_type === 'theme' ) {
            return $this->apply_theme_content( $entity_id, $content );
        } elseif ( $entity_type === 'page' ) {
            return $this->apply_page_content( $entity_id, $content );
        }

        return new WP_Error( 'invalid_entity_type', 'Invalid entity type: ' . $entity_type );
    }

    /**
     * Apply content to theme files
     */
    private function apply_theme_content( $theme_id, $content ) {
        // For themes, content is JSON with file paths and contents
        $theme_data = json_decode( $content, true );

        if ( ! $theme_data ) {
            return new WP_Error( 'invalid_theme_data', 'Invalid theme content data' );
        }

        // Determine theme slug: check if it's an AI theme post ID or a hash of regular theme slug
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );

        if ( ! $theme_slug ) {
            // Not an AI theme - check if it's a hash ID for a regular theme
            $theme_slug = get_option( 'ai_theme_slug_' . $theme_id );

            if ( ! $theme_slug ) {
                return new WP_Error( 'theme_not_found', 'Theme directory slug not found for ID: ' . $theme_id );
            }
        }

        $theme = wp_get_theme( $theme_slug );
        if ( ! $theme->exists() ) {
            return new WP_Error( 'theme_not_found', 'Theme not found: ' . $theme_slug );
        }

        $theme_root = $theme->get_stylesheet_directory();
        $updated_files = array();

        foreach ( $theme_data as $file_path => $file_content ) {
            $full_path = $theme_root . '/' . $file_path;

            // Security check - ensure file is within theme directory
            if ( strpos( realpath( dirname( $full_path ) ), realpath( $theme_root ) ) !== 0 ) {
                continue;
            }

            $write_result = file_put_contents( $full_path, $file_content );
            if ( $write_result !== false ) {
                $updated_files[] = $file_path;
            }
        }

        return $updated_files;
    }

    /**
     * Apply content to page
     */
    private function apply_page_content( $page_id, $content ) {
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_Error( 'page_not_found', 'Page not found' );
        }

        // Detect content format: JSON (split files) or HTML (monolithic)
        $page_files = json_decode( $content, true );

        if ( json_last_error() === JSON_ERROR_NONE && is_array( $page_files ) ) {
            // New format: Split files as JSON
            return $this->apply_split_files_content( $page_id, $page_files );
        } else {
            // Old format: Combined HTML (monolithic pages)
            return $this->apply_monolithic_content( $page_id, $content );
        }
    }

    /**
     * Apply split files content to page directory
     */
    private function apply_split_files_content( $page_id, $page_files ) {
        // CRITICAL BUG FIX: Prevent data loss from empty versions
        // Without this check, empty versions delete all files and write nothing back
        if ( empty( $page_files ) ) {
            $this->logger->error( 'Cannot restore empty version - would delete all page files', [
                'page_id' => $page_id,
                'page_files' => $page_files,
                'reason' => 'Version contains no files - likely saved during theme-only edit or from empty draft directory'
            ] );
            return new WP_Error( 'empty_version', 'Cannot restore empty version - this would delete all page files. The version may be corrupted or was saved incorrectly.' );
        }

        $upload_dir = wp_upload_dir();

        // PHASE 2 FIX: Check if draft is active at restoration time
        // Undo/redo should write to draft directory if editing, live directory if published
        $is_draft = get_post_meta( $page_id, '_ai_draft_active', true ) === '1';
        $suffix = $is_draft ? '-draft' : '';
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id . $suffix;

        // Create page directory if it doesn't exist
        if ( ! is_dir( $page_dir ) ) {
            wp_mkdir_p( $page_dir );
        }

        // Clear existing files (except .htaccess)
        // SAFE: We validated above that $page_files is not empty
        $this->clear_page_directory( $page_dir );

        // Restore all files
        $updated_files = array();
        foreach ( $page_files as $filename => $file_content ) {
            $full_path = $page_dir . '/' . $filename;

            // Security check - ensure file is within page directory
            $real_page_dir = realpath( $page_dir );
            $real_file_dir = realpath( dirname( $full_path ) );

            // If directory doesn't exist yet, check the intended path
            if ( ! $real_file_dir ) {
                $intended_dir = dirname( $full_path );
                if ( strpos( $intended_dir, $page_dir ) !== 0 ) {
                    continue; // Path traversal attempt
                }
            } elseif ( strpos( $real_file_dir, $real_page_dir ) !== 0 ) {
                continue; // Path traversal attempt
            }

            // Create subdirectories if needed
            $dir = dirname( $full_path );
            if ( ! is_dir( $dir ) ) {
                wp_mkdir_p( $dir );
            }

            // Write file
            $write_result = file_put_contents( $full_path, $file_content );
            if ( $write_result !== false ) {
                $updated_files[] = $filename;
            }
        }

        // PHASE 2 FIX: Only regenerate published content if NOT in draft mode
        // During draft editing: Update _ai_draft_content (preview only)
        // After publishing: Update _ai_protected_content and post_content (live site)
        $is_draft = get_post_meta( $page_id, '_ai_draft_active', true ) === '1';
        if ( ! $is_draft ) {
            // Update live site - regenerate _ai_protected_content and post_content
            $this->regenerate_page_content_from_files( $page_id );
        } else {
            // Update draft preview only - regenerate _ai_draft_content
            require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/pages/class-ai-dreamformer-page-editor.php';
            $page_editor = new AI_Dreamformer_Page_Editor();
            $reflection = new ReflectionClass( $page_editor );
            if ( $reflection->hasMethod( 'regenerate_draft_preview' ) ) {
                $method = $reflection->getMethod( 'regenerate_draft_preview' );
                $method->setAccessible( true );
                $method->invoke( $page_editor, $page_id );
            }
        }

        return $updated_files;
    }

    /**
     * Apply monolithic content (old format compatibility)
     */
    private function apply_monolithic_content( $page_id, $content ) {
        // For AI-generated pages, update protected content
        if ( get_post_meta( $page_id, 'ai_generated_page', true ) ) {
            update_post_meta( $page_id, '_ai_protected_content', $content );
        } else {
            // Regular pages update post_content
            wp_update_post( array(
                'ID' => $page_id,
                'post_content' => $content
            ) );
        }

        return true;
    }

    /**
     * Clear page directory contents (except .htaccess)
     */
    private function clear_page_directory( $page_dir ) {
        if ( ! is_dir( $page_dir ) ) {
            return;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $page_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ( $iterator as $file ) {
            // Keep .htaccess for security
            if ( $file->getFilename() === '.htaccess' ) {
                continue;
            }

            if ( $file->isDir() ) {
                rmdir( $file->getPathname() );
            } else {
                unlink( $file->getPathname() );
            }
        }
    }

    /**
     * Regenerate combined page content from split files
     */
    private function regenerate_page_content_from_files( $page_id ) {
        // Call the page editor's regenerate method
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/pages/class-ai-dreamformer-page-editor.php';
        $page_editor = new AI_Dreamformer_Page_Editor();

        // Use reflection to call private method (or make it public)
        $reflection = new ReflectionClass( $page_editor );
        if ( $reflection->hasMethod( 'regenerate_page_content' ) ) {
            $method = $reflection->getMethod( 'regenerate_page_content' );
            $method->setAccessible( true );
            $method->invoke( $page_editor, $page_id );
        }
    }

    /**
     * Get current version number for entity
     */
    private function get_current_version_number( $entity_type, $entity_id ) {
        global $wpdb;

        $version = $wpdb->get_var( $wpdb->prepare(
            "SELECT MAX(version_number) FROM {$this->table_name}
             WHERE entity_type = %s AND entity_id = %d",
            $entity_type, $entity_id
        ) );

        return intval( $version );
    }

    /**
     * Get version history limit based on site's monthly credits
     *
     * Credit-based Limits:
     * - 25 credits (Free): 2 versions (1 undo)
     * - 100+ credits (Pro): 20 versions (19 undos)
     *
     * @return int Number of versions to keep
     */
    private function get_version_limit_for_plan() {
        // Default to free tier
        $default_limit = 2;

        // Use credit manager to get site-wide credits
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
        $credit_manager = new Dreamformer_Credit_Manager();
        $credits_info = $credit_manager->get_user_credits();

        if ( ! $credits_info || ! isset( $credits_info['monthly_credits'] ) ) {
            return $default_limit;
        }

        $credits = intval( $credits_info['monthly_credits'] );

        // Determine limit based on monthly credits
        if ( $credits >= 100 ) {
            return 20; // All Pro plans: 20 versions
        }

        return 2; // Free: 2 versions
    }

    /**
     * Clean up old versions (keep last N ACCESSIBLE versions based on plan)
     *
     * Walks the version chain backwards from current version to find accessible versions.
     * Deletes orphaned versions that can't be reached via undo.
     *
     * Example: v1 → v2 → v3 (undo to v2) → v4 (current)
     * Accessible: v4, v3, v2, v1 (can reach via undo)
     * Orphaned: v3 (unreachable after branching)
     */
    private function cleanup_old_versions( $entity_type, $entity_id, $keep_count = null ) {
        global $wpdb;

        // Get plan-based limit if not specified
        if ( $keep_count === null ) {
            $keep_count = $this->get_version_limit_for_plan();
        }

        // Step 1: Find current version
        $current = $wpdb->get_row( $wpdb->prepare(
            "SELECT version_number FROM {$this->table_name}
             WHERE entity_type = %s AND entity_id = %d AND is_current = 1",
            $entity_type, $entity_id
        ) );

        if ( ! $current ) {
            // No current version - nothing to clean up
            return;
        }

        // Step 2: Walk version chain backwards to find accessible versions
        $accessible_ids = array();
        $version_num = $current->version_number;

        while ( $version_num >= 1 && count( $accessible_ids ) < $keep_count ) {
            $version = $wpdb->get_row( $wpdb->prepare(
                "SELECT id FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d AND version_number = %d",
                $entity_type, $entity_id, $version_num
            ) );

            if ( $version ) {
                $accessible_ids[] = $version->id;
            }

            $version_num--;
        }

        // Step 3: Delete everything NOT in accessible list
        if ( ! empty( $accessible_ids ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $accessible_ids ), '%d' ) );

            $deleted = $wpdb->query( $wpdb->prepare(
                "DELETE FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d
                 AND id NOT IN ($placeholders)",
                $entity_type, $entity_id, ...$accessible_ids
            ) );

            if ( $deleted > 0 ) {
                $this->logger->info( 'Cleaned up orphaned versions', [
                    'entity_type' => $entity_type,
                    'entity_id' => $entity_id,
                    'deleted_count' => $deleted,
                    'kept_count' => count( $accessible_ids )
                ] );
            }
        }
    }

    /**
     * REST API: Create version
     */
    public function rest_create_version( WP_REST_Request $request ) {
        $entity_type = $request->get_param( 'entity_type' );
        $entity_id = intval( $request->get_param( 'entity_id' ) );
        $content_before = $request->get_param( 'content_before' );
        $content_after = $request->get_param( 'content_after' );

        $options = array(
            'change_description' => $request->get_param( 'change_description' ) ?: '',
            'user_prompt' => $request->get_param( 'user_prompt' ) ?: '',
            'ai_model' => $request->get_param( 'ai_model' ) ?: AI_Config_Manager::AI_MODEL,
            'change_metadata' => $request->get_param( 'change_metadata' ) ?: array()
        );

        $result = $this->save_version( $entity_type, $entity_id, $content_before, $content_after, $options );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Version created successfully',
            'version_id' => $result
        ) );
    }

    /**
     * REST API: Undo change
     */
    public function rest_undo_change( WP_REST_Request $request ) {
        $entity_type = $request->get_param( 'entity_type' );
        $entity_id = intval( $request->get_param( 'entity_id' ) );

        $result = $this->undo_change( $entity_type, $entity_id );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Change undone successfully',
            'version' => array(
                'id' => $result->id,
                'version_number' => $result->version_number,
                'change_description' => $result->change_description,
                'created_at' => $result->created_at
            )
        ) );
    }

    /**
     * REST API: Redo change
     */
    public function rest_redo_change( WP_REST_Request $request ) {
        $entity_type = $request->get_param( 'entity_type' );
        $entity_id = intval( $request->get_param( 'entity_id' ) );

        $result = $this->redo_change( $entity_type, $entity_id );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Change redone successfully',
            'version' => array(
                'id' => $result->id,
                'version_number' => $result->version_number,
                'change_description' => $result->change_description,
                'created_at' => $result->created_at
            )
        ) );
    }

    /**
     * REST API: Get version history
     */
    public function rest_get_version_history( WP_REST_Request $request ) {
        global $wpdb;

        $entity_type = $request->get_param( 'entity_type' );
        $entity_id = intval( $request->get_param( 'entity_id' ) );

        // Get plan-based limit
        $limit = $this->get_version_limit_for_plan();

        $versions = $wpdb->get_results( $wpdb->prepare(
            "SELECT id, version_number, change_description, user_prompt, ai_model,
                    user_id, created_at, is_current, parent_version_id
             FROM {$this->table_name}
             WHERE entity_type = %s AND entity_id = %d
             ORDER BY version_number DESC
             LIMIT %d",
            $entity_type, $entity_id, $limit
        ) );

        // Add user display names
        foreach ( $versions as $version ) {
            $user = get_userdata( $version->user_id );
            $version->user_name = $user ? $user->display_name : 'Unknown User';
        }

        return new WP_REST_Response( array(
            'success' => true,
            'versions' => $versions,
            'plan_limit' => $limit
        ) );
    }

    /**
     * REST API: Restore to specific version
     */
    public function rest_restore_version( WP_REST_Request $request ) {
        global $wpdb;

        $version_id = intval( $request->get_param( 'version_id' ) );

        // Get the specified version
        $version = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE id = %d",
            $version_id
        ) );

        if ( ! $version ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Version not found'
            ), 404 );
        }

        // Apply the version content
        $result = $this->apply_version_content( $version->entity_type, $version->entity_id, $version->content_after );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 500 );
        }

        // Mark this version as current
        $wpdb->update(
            $this->table_name,
            array( 'is_current' => 0 ),
            array( 'entity_type' => $version->entity_type, 'entity_id' => $version->entity_id ),
            array( '%d' ),
            array( '%s', '%d' )
        );

        $wpdb->update(
            $this->table_name,
            array( 'is_current' => 1 ),
            array( 'id' => $version_id ),
            array( '%d' ),
            array( '%d' )
        );

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Version restored successfully',
            'version' => array(
                'id' => $version->id,
                'version_number' => $version->version_number,
                'change_description' => $version->change_description,
                'created_at' => $version->created_at
            )
        ) );
    }

    /**
     * REST API: Cleanup old versions
     */
    public function rest_cleanup_versions( WP_REST_Request $request ) {
        global $wpdb;

        $keep_count = intval( $request->get_param( 'keep_count' ) ?: $this->get_version_limit_for_plan() );

        // Get all entities that have versions
        $entities = $wpdb->get_results(
            "SELECT DISTINCT entity_type, entity_id FROM {$this->table_name}"
        );

        $total_deleted = 0;

        foreach ( $entities as $entity ) {
            // Clean up old versions for this entity
            $versions_to_delete = $wpdb->get_results( $wpdb->prepare(
                "SELECT id FROM {$this->table_name}
                 WHERE entity_type = %s AND entity_id = %d
                 ORDER BY version_number DESC
                 LIMIT %d, 999999",
                $entity->entity_type, $entity->entity_id, $keep_count
            ) );

            if ( $versions_to_delete ) {
                $ids_to_delete = wp_list_pluck( $versions_to_delete, 'id' );
                $placeholders = implode( ',', array_fill( 0, count( $ids_to_delete ), '%d' ) );

                $deleted = $wpdb->query( $wpdb->prepare(
                    "DELETE FROM {$this->table_name} WHERE id IN ($placeholders)",
                    $ids_to_delete
                ) );

                $total_deleted += $deleted;
            }
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Cleanup completed successfully',
            'deleted_count' => $total_deleted,
            'entities_processed' => count( $entities )
        ) );
    }

    /**
     * Get version control instance
     */
    public static function get_instance() {
        static $instance = null;
        if ( $instance === null ) {
            $instance = new self();
        }
        return $instance;
    }
}