<?php
/**
 * AI Site Builder - Centralized Logger
 * 
 * Provides consistent logging throughout the plugin with proper formatting,
 * log levels, and context tracking.
 * 
 * @package AI_Site_Builder
 * @since 2.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Logger {
	
	/**
	 * Log levels
	 */
	const LEVEL_ERROR = 'ERROR';
	const LEVEL_WARNING = 'WARNING';
	const LEVEL_INFO = 'INFO';
	const LEVEL_DEBUG = 'DEBUG';
	
	/**
	 * Component identifier for grouping logs
	 */
	private $component;
	
	/**
	 * Whether debug logging is enabled
	 */
	private $debug_enabled;
	
	/**
	 * Log prefix for consistency
	 */
	const LOG_PREFIX = 'AI Site Builder';
	
	/**
	 * Constructor
	 * 
	 * @param string $component Component name for log grouping
	 */
	public function __construct( $component = 'General' ) {
		$this->component = $component;
		$this->debug_enabled = defined( 'WP_DEBUG' ) && WP_DEBUG;
	}
	
	/**
	 * Log an error message
	 * 
	 * @param string $message Error message
	 * @param mixed $data Additional data to log
	 */
	public function error( $message, $data = null ) {
		$this->log( self::LEVEL_ERROR, $message, $data );
	}
	
	/**
	 * Log a warning message
	 * 
	 * @param string $message Warning message
	 * @param mixed $data Additional data to log
	 */
	public function warning( $message, $data = null ) {
		$this->log( self::LEVEL_WARNING, $message, $data );
	}
	
	/**
	 * Log an info message
	 * 
	 * @param string $message Info message
	 * @param mixed $data Additional data to log
	 */
	public function info( $message, $data = null ) {
		$this->log( self::LEVEL_INFO, $message, $data );
	}
	
	/**
	 * Log a debug message (only if WP_DEBUG is enabled)
	 * 
	 * @param string $message Debug message
	 * @param mixed $data Additional data to log
	 */
	public function debug( $message, $data = null ) {
		if ( $this->debug_enabled ) {
			$this->log( self::LEVEL_DEBUG, $message, $data );
		}
	}
	
	/**
	 * Log a progress update for long operations
	 * 
	 * @param string $operation Operation name
	 * @param int $progress Progress percentage (0-100)
	 * @param string $message Status message
	 */
	public function progress( $operation, $progress, $message = '' ) {
		$log_message = sprintf( '[%s] Progress: %d%% - %s', $operation, $progress, $message );
		$this->log( self::LEVEL_INFO, $log_message );
	}
	
	/**
	 * Log API request details
	 * 
	 * @param string $service API service name
	 * @param string $endpoint Endpoint being called
	 * @param array $params Request parameters
	 * @param bool $sanitize Whether to sanitize sensitive data
	 */
	public function api_request( $service, $endpoint, $params = array(), $sanitize = true ) {
		$log_data = array(
			'service' => $service,
			'endpoint' => $endpoint
		);
		
		if ( $sanitize && isset( $params['api_key'] ) ) {
			$params['api_key'] = substr( $params['api_key'], 0, 8 ) . '...';
		}
		
		if ( ! empty( $params ) ) {
			$log_data['params'] = $params;
		}
		
		$this->info( sprintf( 'API Request to %s', $service ), $log_data );
	}
	
	/**
	 * Log API response details
	 * 
	 * @param string $service API service name
	 * @param int $status_code HTTP status code
	 * @param mixed $response Response data
	 * @param float $duration Request duration in seconds
	 */
	public function api_response( $service, $status_code, $response = null, $duration = 0 ) {
		$log_data = array(
			'service' => $service,
			'status_code' => $status_code,
			'duration' => sprintf( '%.2fs', $duration )
		);
		
		if ( $status_code !== 200 ) {
			$log_data['response'] = $response;
			$this->error( sprintf( 'API Response from %s failed', $service ), $log_data );
		} else {
			$this->info( sprintf( 'API Response from %s succeeded', $service ), $log_data );
		}
	}
	
	/**
	 * Start timing an operation
	 * 
	 * @param string $operation Operation identifier
	 * @return float Start time
	 */
	public function start_timing( $operation ) {
		$start = microtime( true );
		$this->debug( sprintf( 'Starting operation: %s', $operation ) );
		return $start;
	}
	
	/**
	 * End timing an operation and log duration
	 * 
	 * @param string $operation Operation identifier
	 * @param float $start Start time from start_timing()
	 * @param string $message Optional completion message
	 */
	public function end_timing( $operation, $start, $message = '' ) {
		$duration = microtime( true ) - $start;
		$log_message = sprintf( 
			'Completed operation: %s (Duration: %.2fs)%s', 
			$operation, 
			$duration,
			$message ? ' - ' . $message : ''
		);
		$this->info( $log_message );
	}
	
	/**
	 * Core logging method
	 * 
	 * @param string $level Log level
	 * @param string $message Log message
	 * @param mixed $data Additional data
	 */
	private function log( $level, $message, $data = null ) {
		// Format the log message
		$formatted_message = sprintf(
			'[%s] [%s] [%s] %s',
			self::LOG_PREFIX,
			$this->component,
			$level,
			$message
		);
		
		// Add data if provided
		if ( $data !== null ) {
			if ( is_array( $data ) || is_object( $data ) ) {
				$formatted_message .= ' | Data: ' . wp_json_encode( $data );
			} else {
				$formatted_message .= ' | Data: ' . $data;
			}
		}
		
		// Use WordPress error_log
		error_log( $formatted_message );
		
		// For errors, also log to WordPress debug.log if WP_DEBUG_LOG is enabled
		if ( $level === self::LEVEL_ERROR && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// This will write to wp-content/debug.log
			error_log( $formatted_message, 3, WP_CONTENT_DIR . '/debug.log' );
		}
	}
	
	/**
	 * Create a logger instance for a specific component
	 * 
	 * @param string $component Component name
	 * @return AI_Logger Logger instance
	 */
	public static function get_instance( $component ) {
		return new self( $component );
	}
	
	/**
	 * Log WordPress Error object
	 * 
	 * @param WP_Error $error WordPress error object
	 * @param string $context Additional context
	 */
	public function wp_error( $error, $context = '' ) {
		if ( ! is_wp_error( $error ) ) {
			return;
		}
		
		$error_data = array(
			'code' => $error->get_error_code(),
			'message' => $error->get_error_message(),
			'data' => $error->get_error_data()
		);
		
		$message = $context ? sprintf( '%s: WP_Error', $context ) : 'WP_Error occurred';
		$this->error( $message, $error_data );
	}
	
	/**
	 * Create a log entry for theme generation steps
	 * 
	 * @param string $step Step name
	 * @param string $status Status (started, completed, failed)
	 * @param mixed $details Additional details
	 */
	public function theme_generation_step( $step, $status, $details = null ) {
		$message = sprintf( 'Theme Generation [%s]: %s', $step, $status );
		
		switch ( $status ) {
			case 'started':
				$this->info( $message, $details );
				break;
			case 'completed':
				$this->info( $message, $details );
				break;
			case 'failed':
				$this->error( $message, $details );
				break;
			default:
				$this->debug( $message, $details );
		}
	}
}