<?php
/**
 * AI Site Builder - JSON Handler
 * 
 * Centralized JSON encoding/decoding with error handling,
 * large response support, and markdown code block cleaning.
 * 
 * @package AI_Site_Builder
 * @since 2.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_JSON_Handler {
	
	/**
	 * Logger instance
	 */
	private static $logger = null;
	
	/**
	 * Maximum JSON depth
	 */
	const MAX_DEPTH = 512;
	
	/**
	 * Get logger instance
	 */
	private static function get_logger() {
		if ( self::$logger === null ) {
			require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-logger.php';
			self::$logger = AI_Logger::get_instance( 'JSONHandler' );
		}
		return self::$logger;
	}
	
	/**
	 * Decode JSON string with error handling
	 * 
	 * @param string $json JSON string to decode
	 * @param bool $assoc Return associative array instead of object
	 * @param bool $clean_markdown Clean markdown code blocks
	 * @return mixed|WP_Error Decoded data or error
	 */
	public static function decode( $json, $assoc = true, $clean_markdown = false ) {
		if ( empty( $json ) ) {
			return new WP_Error( 'empty_json', 'Empty JSON string provided' );
		}
		
		// Clean markdown code blocks if requested
		if ( $clean_markdown ) {
			$json = self::clean_markdown_blocks( $json );
		}
		
		
		// Decode JSON
		$decoded = json_decode( $json, $assoc, self::MAX_DEPTH );
		
		// Check for errors
		$error = json_last_error();
		if ( $error !== JSON_ERROR_NONE ) {
			$error_message = self::get_json_error_message( $error );
			self::get_logger()->error( 'JSON decode failed', array(
				'error' => $error_message,
				'json_preview' => substr( $json, 0, 10000 ), // Increased from 200 to 10K for proper debugging
				'json_length' => strlen( $json )
			) );
			return new WP_Error( 'json_decode_error', $error_message );
		}
		
		self::get_logger()->debug( 'JSON decoded successfully', array(
			'size' => strlen( $json ),
			'type' => gettype( $decoded )
		) );
		
		return $decoded;
	}
	
	/**
	 * Encode data to JSON with error handling
	 * 
	 * @param mixed $data Data to encode
	 * @param int $options JSON encoding options
	 * @return string|WP_Error JSON string or error
	 */
	public static function encode( $data, $options = 0 ) {
		if ( ! defined( 'JSON_UNESCAPED_UNICODE' ) ) {
			define( 'JSON_UNESCAPED_UNICODE', 256 );
		}
		
		// Default options for better readability
		if ( $options === 0 ) {
			$options = JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES;
		}
		
		// Encode to JSON
		$json = json_encode( $data, $options, self::MAX_DEPTH );
		
		// Check for errors
		$error = json_last_error();
		if ( $error !== JSON_ERROR_NONE ) {
			$error_message = self::get_json_error_message( $error );
			self::get_logger()->error( 'JSON encode failed', array(
				'error' => $error_message,
				'data_type' => gettype( $data )
			) );
			return new WP_Error( 'json_encode_error', $error_message );
		}
		
		if ( $json === false ) {
			return new WP_Error( 'json_encode_failed', 'JSON encoding returned false' );
		}
		
		self::get_logger()->debug( 'JSON encoded successfully', array(
			'size' => strlen( $json )
		) );
		
		return $json;
	}
	
	/**
	 * Validate JSON string
	 * 
	 * @param string $json JSON string to validate
	 * @return bool True if valid JSON
	 */
	public static function validate( $json ) {
		if ( ! is_string( $json ) ) {
			return false;
		}
		
		json_decode( $json );
		return json_last_error() === JSON_ERROR_NONE;
	}
	
	/**
	 * Clean markdown code blocks from JSON
	 * 
	 * @param string $text Text containing JSON with possible markdown
	 * @return string Cleaned JSON string
	 */
	public static function clean_markdown_blocks( $text ) {
		// Remove ```json blocks
		$text = preg_replace( '/^```json\s*$/m', '', $text );
		$text = preg_replace( '/^```\s*$/m', '', $text );
		
		// Remove other code block markers
		$text = preg_replace( '/^```[a-z]*\s*$/m', '', $text );
		
		return trim( $text );
	}
	
	/**
	 * Sanitize AI response JSON by fixing smart quotes and common formatting issues
	 * 
	 * @param string $response Raw AI response
	 * @return string Sanitized JSON string
	 */
	public static function sanitize_ai_response( $response ) {
		// MINIMAL sanitization - only remove markdown blocks
		$sanitized = $response;
		
		// Remove markdown code blocks if present
		$sanitized = preg_replace('/```json\s*/', '', $sanitized);
		$sanitized = preg_replace('/```\s*$/', '', $sanitized);
		$sanitized = preg_replace('/^```[a-z]*\s*/m', '', $sanitized);
		
		
		return trim( $sanitized );
	}
	
	/**
	 * Extract JSON from text that may contain other content
	 * 
	 * @param string $text Text containing JSON
	 * @return string Extracted JSON or original text
	 */
	public static function extract_json( $text ) {
		// Try to find JSON object
		if ( preg_match( '/\{(?:[^{}]|(?R))*\}/s', $text, $matches ) ) {
			return $matches[0];
		}
		
		// Try to find JSON array
		if ( preg_match( '/\[(?:[^\[\]]|(?R))*\]/s', $text, $matches ) ) {
			return $matches[0];
		}
		
		// Return original if no JSON found
		return $text;
	}
	
	/**
	 * Fix common JSON syntax issues from AI responses
	 * 
	 * @param string $json Potentially malformed JSON
	 * @return string Fixed JSON
	 */
	public static function fix_common_json_issues( $json ) {
		
		// Only keep the trailing comma fix as it's genuinely helpful and safe
		$json = preg_replace( '/,(\s*[}\]])/', '$1', $json );
		
		return $json;
	}
	
	/**
	 * Parse AI response that may contain JSON
	 * 
	 * @param string $response AI response text
	 * @return array|WP_Error Parsed data or error
	 */
	public static function parse_ai_response( $response ) {
		// First try with sanitization (fixes smart quotes and formatting)
		$sanitized = self::sanitize_ai_response( $response );
		$result = self::decode( $sanitized );
		
		if ( ! is_wp_error( $result ) ) {
			self::get_logger()->debug( 'AI response parsed successfully after sanitization' );
			return $result;
		}
		
		// Try direct decode as fallback
		$result = self::decode( $response );
		if ( ! is_wp_error( $result ) ) {
			self::get_logger()->debug( 'AI response parsed successfully without sanitization' );
			return $result;
		}
		
		// Try extracting JSON from response
		$extracted = self::extract_json( $response );
		if ( $extracted !== $response ) {
			// Try sanitized extraction
			$sanitized_extracted = self::sanitize_ai_response( $extracted );
			$result = self::decode( $sanitized_extracted, true, false );
			if ( ! is_wp_error( $result ) ) {
				self::get_logger()->debug( 'AI response parsed successfully after extraction and sanitization' );
				return $result;
			}
			
			// Try raw extraction
			$result = self::decode( $extracted, true, false );
			if ( ! is_wp_error( $result ) ) {
				self::get_logger()->debug( 'AI response parsed successfully after extraction only' );
				return $result;
			}
		}
		
		// Log failure with more details
		self::get_logger()->error( 'Failed to parse AI response as JSON', array(
			'response_preview' => substr( $response, 0, 10000 ),
			'sanitized_preview' => substr( $sanitized, 0, 10000 )
		) );
		
		return new WP_Error( 'parse_failed', 'Could not parse AI response as JSON' );
	}
	
	/**
	 * Get human-readable JSON error message
	 * 
	 * @param int $error JSON error code
	 * @return string Error message
	 */
	private static function get_json_error_message( $error ) {
		switch ( $error ) {
			case JSON_ERROR_NONE:
				return 'No error';
			case JSON_ERROR_DEPTH:
				return 'Maximum stack depth exceeded';
			case JSON_ERROR_STATE_MISMATCH:
				return 'Invalid or malformed JSON';
			case JSON_ERROR_CTRL_CHAR:
				return 'Control character error';
			case JSON_ERROR_SYNTAX:
				return 'Syntax error in JSON';
			case JSON_ERROR_UTF8:
				return 'Malformed UTF-8 characters';
			default:
				if ( function_exists( 'json_last_error_msg' ) ) {
					return json_last_error_msg();
				}
				return 'Unknown JSON error: ' . $error;
		}
	}
	
	/**
	 * Safely merge JSON objects
	 * 
	 * @param array $base Base array
	 * @param array $additions Additions to merge
	 * @param bool $recursive Recursive merge
	 * @return array Merged array
	 */
	public static function merge( $base, $additions, $recursive = true ) {
		if ( ! is_array( $base ) ) {
			$base = array();
		}
		
		if ( ! is_array( $additions ) ) {
			return $base;
		}
		
		if ( $recursive ) {
			return array_merge_recursive( $base, $additions );
		}
		
		return array_merge( $base, $additions );
	}
	
	/**
	 * Format JSON for display
	 * 
	 * @param mixed $data Data to format
	 * @param bool $html_output Format for HTML display
	 * @return string Formatted JSON
	 */
	public static function format( $data, $html_output = false ) {
		$json = self::encode( $data );
		
		if ( is_wp_error( $json ) ) {
			return '{}';
		}
		
		if ( $html_output ) {
			// Convert to HTML-safe display
			$json = esc_html( $json );
			$json = '<pre>' . $json . '</pre>';
		}
		
		return $json;
	}
	
	/**
	 * Validate JSON schema
	 * 
	 * @param array $data Data to validate
	 * @param array $required_keys Required keys
	 * @return bool|WP_Error True if valid, error otherwise
	 */
	public static function validate_schema( $data, $required_keys ) {
		if ( ! is_array( $data ) ) {
			return new WP_Error( 'invalid_type', 'Data must be an array' );
		}
		
		$missing = array();
		foreach ( $required_keys as $key ) {
			if ( ! isset( $data[ $key ] ) ) {
				$missing[] = $key;
			}
		}
		
		if ( ! empty( $missing ) ) {
			return new WP_Error( 
				'missing_keys', 
				sprintf( 'Missing required keys: %s', implode( ', ', $missing ) )
			);
		}
		
		return true;
	}
	
	/**
	 * Handle large JSON responses by streaming
	 * 
	 * @param string $file_path Path to JSON file
	 * @param callable $callback Callback for each JSON object
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function stream_parse( $file_path, $callback ) {
		if ( ! file_exists( $file_path ) ) {
			return new WP_Error( 'file_not_found', 'JSON file not found' );
		}
		
		if ( ! is_callable( $callback ) ) {
			return new WP_Error( 'invalid_callback', 'Callback must be callable' );
		}
		
		// This is a simplified streaming parser
		// For production, consider using a proper streaming JSON parser library
		$content = file_get_contents( $file_path );
		if ( $content === false ) {
			return new WP_Error( 'read_failed', 'Failed to read JSON file' );
		}
		
		$data = self::decode( $content );
		if ( is_wp_error( $data ) ) {
			return $data;
		}
		
		if ( is_array( $data ) ) {
			foreach ( $data as $item ) {
				call_user_func( $callback, $item );
			}
		} else {
			call_user_func( $callback, $data );
		}
		
		return true;
	}
}