<?php
/**
 * AI Site Builder - File Manager
 * 
 * Centralized file operations with error handling, security checks,
 * and atomic operations to prevent data corruption.
 * 
 * @package AI_Site_Builder
 * @since 2.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_File_Manager {
	
	/**
	 * Logger instance
	 */
	private static $logger = null;
	
	/**
	 * Get logger instance
	 */
	private static function get_logger() {
		if ( self::$logger === null ) {
			require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-logger.php';
			self::$logger = AI_Logger::get_instance( 'FileManager' );
		}
		return self::$logger;
	}
	
	/**
	 * Read file contents safely
	 * 
	 * @param string $file_path Path to file
	 * @param bool $verify_path Whether to verify path safety
	 * @return string|WP_Error File contents or error
	 */
	public static function read( $file_path, $verify_path = true ) {
		// Verify path safety
		if ( $verify_path && ! self::is_safe_path( $file_path ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Read operation blocked' );
			return $error;
		}
		
		// Check if file exists
		if ( ! file_exists( $file_path ) ) {
			return new WP_Error( 'file_not_found', sprintf( 'File not found: %s', $file_path ) );
		}
		
		// Check if file is readable
		if ( ! is_readable( $file_path ) ) {
			return new WP_Error( 'file_not_readable', sprintf( 'File not readable: %s', $file_path ) );
		}
		
		// Read file contents
		$contents = file_get_contents( $file_path );
		
		if ( $contents === false ) {
			$error = new WP_Error( 'read_failed', sprintf( 'Failed to read file: %s', $file_path ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		self::get_logger()->debug( sprintf( 'File read successfully: %s (%d bytes)', $file_path, strlen( $contents ) ) );
		return $contents;
	}
	
	/**
	 * Write file contents safely with atomic operations
	 * 
	 * @param string $file_path Path to file
	 * @param string $content Content to write
	 * @param bool $atomic Use atomic write (write to temp file first)
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function write( $file_path, $content, $atomic = true ) {
		// Verify path safety
		if ( ! self::is_safe_path( $file_path ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Write operation blocked' );
			return $error;
		}
		
		// Ensure directory exists
		$dir = dirname( $file_path );
		$dir_created = self::ensure_directory( $dir );
		if ( is_wp_error( $dir_created ) ) {
			return $dir_created;
		}
		
		if ( $atomic ) {
			// Write to temporary file first
			$temp_file = $file_path . '.tmp.' . uniqid();
			$bytes_written = file_put_contents( $temp_file, $content );
			
			if ( $bytes_written === false ) {
				@unlink( $temp_file );
				$error = new WP_Error( 'write_failed', sprintf( 'Failed to write temporary file: %s', $temp_file ) );
				self::get_logger()->wp_error( $error );
				return $error;
			}
			
			// Rename temp file to target (atomic operation)
			if ( ! rename( $temp_file, $file_path ) ) {
				@unlink( $temp_file );
				$error = new WP_Error( 'rename_failed', sprintf( 'Failed to rename temp file to: %s', $file_path ) );
				self::get_logger()->wp_error( $error );
				return $error;
			}
		} else {
			// Direct write
			$bytes_written = file_put_contents( $file_path, $content );
			
			if ( $bytes_written === false ) {
				$error = new WP_Error( 'write_failed', sprintf( 'Failed to write file: %s', $file_path ) );
				self::get_logger()->wp_error( $error );
				return $error;
			}
		}
		
		self::get_logger()->debug( sprintf( 'File written successfully: %s (%d bytes)', $file_path, strlen( $content ) ) );

		// Auto-trigger cache bust for theme CSS/JS files
		self::maybe_trigger_cache_bust( $file_path );

		return true;
	}

	/**
	 * Trigger cache bust if file is a theme CSS or JS file
	 *
	 * @param string $file_path Path to the written file
	 */
	private static function maybe_trigger_cache_bust( $file_path ) {
		// Check if this is a theme file (CSS or JS)
		$themes_dir = get_theme_root();
		$extension = pathinfo( $file_path, PATHINFO_EXTENSION );

		if ( strpos( $file_path, $themes_dir ) !== false && in_array( $extension, array( 'css', 'js' ), true ) ) {
			// Extract theme slug from path
			$relative_path = str_replace( $themes_dir . '/', '', $file_path );
			$path_parts = explode( '/', $relative_path );
			$theme_slug = $path_parts[0] ?? '';

			if ( $theme_slug && function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
				ai_site_builder_trigger_cache_bust( 'theme', $theme_slug );
				self::get_logger()->debug( sprintf( 'Triggered cache bust for theme: %s', $theme_slug ) );
			}
		}
	}

	/**
	 * Append to file
	 * 
	 * @param string $file_path Path to file
	 * @param string $content Content to append
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function append( $file_path, $content ) {
		// Verify path safety
		if ( ! self::is_safe_path( $file_path ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Append operation blocked' );
			return $error;
		}
		
		// Ensure directory exists
		$dir = dirname( $file_path );
		$dir_created = self::ensure_directory( $dir );
		if ( is_wp_error( $dir_created ) ) {
			return $dir_created;
		}
		
		$result = file_put_contents( $file_path, $content, FILE_APPEND );
		
		if ( $result === false ) {
			$error = new WP_Error( 'append_failed', sprintf( 'Failed to append to file: %s', $file_path ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		self::get_logger()->debug( sprintf( 'Content appended to file: %s (%d bytes)', $file_path, strlen( $content ) ) );
		return true;
	}
	
	/**
	 * Delete file
	 * 
	 * @param string $file_path Path to file
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function delete( $file_path ) {
		// Verify path safety
		if ( ! self::is_safe_path( $file_path ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Delete operation blocked' );
			return $error;
		}
		
		if ( ! file_exists( $file_path ) ) {
			return true; // Already deleted
		}
		
		if ( ! unlink( $file_path ) ) {
			$error = new WP_Error( 'delete_failed', sprintf( 'Failed to delete file: %s', $file_path ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		self::get_logger()->debug( sprintf( 'File deleted: %s', $file_path ) );
		return true;
	}
	
	/**
	 * Ensure directory exists with proper permissions
	 * 
	 * @param string $dir_path Directory path
	 * @param int $permissions Directory permissions (octal)
	 * @return bool|WP_Error True if exists or created, error on failure
	 */
	public static function ensure_directory( $dir_path, $permissions = 0755 ) {
		// Check if already exists
		if ( is_dir( $dir_path ) ) {
			return true;
		}
		
		// Create directory
		if ( ! wp_mkdir_p( $dir_path ) ) {
			$error = new WP_Error( 'mkdir_failed', sprintf( 'Failed to create directory: %s', $dir_path ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		// Set permissions
		@chmod( $dir_path, $permissions );
		
		self::get_logger()->debug( sprintf( 'Directory created: %s', $dir_path ) );
		return true;
	}
	
	/**
	 * Copy file
	 * 
	 * @param string $source Source file path
	 * @param string $destination Destination file path
	 * @param bool $overwrite Whether to overwrite existing file
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function copy( $source, $destination, $overwrite = false ) {
		// Verify paths
		if ( ! self::is_safe_path( $source ) || ! self::is_safe_path( $destination ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Copy operation blocked' );
			return $error;
		}
		
		// Check source exists
		if ( ! file_exists( $source ) ) {
			return new WP_Error( 'source_not_found', sprintf( 'Source file not found: %s', $source ) );
		}
		
		// Check destination
		if ( file_exists( $destination ) && ! $overwrite ) {
			return new WP_Error( 'destination_exists', sprintf( 'Destination already exists: %s', $destination ) );
		}
		
		// Ensure destination directory exists
		$dest_dir = dirname( $destination );
		$dir_created = self::ensure_directory( $dest_dir );
		if ( is_wp_error( $dir_created ) ) {
			return $dir_created;
		}
		
		// Copy file
		if ( ! copy( $source, $destination ) ) {
			$error = new WP_Error( 'copy_failed', sprintf( 'Failed to copy %s to %s', $source, $destination ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		self::get_logger()->debug( sprintf( 'File copied: %s -> %s', $source, $destination ) );
		return true;
	}
	
	/**
	 * Move/rename file
	 * 
	 * @param string $source Source file path
	 * @param string $destination Destination file path
	 * @param bool $overwrite Whether to overwrite existing file
	 * @return bool|WP_Error True on success, error on failure
	 */
	public static function move( $source, $destination, $overwrite = false ) {
		// Verify paths
		if ( ! self::is_safe_path( $source ) || ! self::is_safe_path( $destination ) ) {
			$error = new WP_Error( 'unsafe_path', 'Unsafe file path detected' );
			self::get_logger()->wp_error( $error, 'Move operation blocked' );
			return $error;
		}
		
		// Check source exists
		if ( ! file_exists( $source ) ) {
			return new WP_Error( 'source_not_found', sprintf( 'Source file not found: %s', $source ) );
		}
		
		// Check destination
		if ( file_exists( $destination ) && ! $overwrite ) {
			return new WP_Error( 'destination_exists', sprintf( 'Destination already exists: %s', $destination ) );
		}
		
		// Ensure destination directory exists
		$dest_dir = dirname( $destination );
		$dir_created = self::ensure_directory( $dest_dir );
		if ( is_wp_error( $dir_created ) ) {
			return $dir_created;
		}
		
		// Move file
		if ( ! rename( $source, $destination ) ) {
			$error = new WP_Error( 'move_failed', sprintf( 'Failed to move %s to %s', $source, $destination ) );
			self::get_logger()->wp_error( $error );
			return $error;
		}
		
		self::get_logger()->debug( sprintf( 'File moved: %s -> %s', $source, $destination ) );
		return true;
	}
	
	/**
	 * Check if path is safe (no directory traversal)
	 * 
	 * @param string $path Path to check
	 * @return bool True if safe
	 */
	public static function is_safe_path( $path ) {
		// Normalize path
		$path = wp_normalize_path( $path );
		
		// Check for directory traversal
		if ( strpos( $path, '..' ) !== false ) {
			return false;
		}
		
		// Additional checks for absolute paths outside WordPress
		$abspath = wp_normalize_path( ABSPATH );
		$wp_content = wp_normalize_path( WP_CONTENT_DIR );
		
		// Allow paths within WordPress installation
		if ( strpos( $path, $abspath ) === 0 || strpos( $path, $wp_content ) === 0 ) {
			return true;
		}
		
		// Allow temp directory
		$temp_dir = wp_normalize_path( get_temp_dir() );
		if ( strpos( $path, $temp_dir ) === 0 ) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Get file info
	 * 
	 * @param string $file_path Path to file
	 * @return array|WP_Error File information or error
	 */
	public static function get_info( $file_path ) {
		if ( ! file_exists( $file_path ) ) {
			return new WP_Error( 'file_not_found', sprintf( 'File not found: %s', $file_path ) );
		}
		
		return array(
			'path' => $file_path,
			'name' => basename( $file_path ),
			'directory' => dirname( $file_path ),
			'extension' => pathinfo( $file_path, PATHINFO_EXTENSION ),
			'size' => filesize( $file_path ),
			'modified' => filemtime( $file_path ),
			'readable' => is_readable( $file_path ),
			'writable' => is_writable( $file_path ),
			'is_file' => is_file( $file_path ),
			'is_dir' => is_dir( $file_path )
		);
	}
	
	/**
	 * List files in directory
	 * 
	 * @param string $dir_path Directory path
	 * @param string $pattern Optional glob pattern
	 * @return array|WP_Error Array of file paths or error
	 */
	public static function list_files( $dir_path, $pattern = '*' ) {
		if ( ! is_dir( $dir_path ) ) {
			return new WP_Error( 'not_directory', sprintf( 'Not a directory: %s', $dir_path ) );
		}
		
		$files = glob( $dir_path . '/' . $pattern );
		
		if ( $files === false ) {
			return new WP_Error( 'glob_failed', sprintf( 'Failed to list files in: %s', $dir_path ) );
		}
		
		return $files;
	}
	
	/**
	 * Create a secure temporary file
	 * 
	 * @param string $prefix File prefix
	 * @param string $content Optional content to write
	 * @return string|WP_Error Temp file path or error
	 */
	public static function create_temp_file( $prefix = 'ai_temp_', $content = '' ) {
		$temp_dir = get_temp_dir();
		$temp_file = tempnam( $temp_dir, $prefix );
		
		if ( $temp_file === false ) {
			return new WP_Error( 'temp_file_failed', 'Failed to create temporary file' );
		}
		
		if ( ! empty( $content ) ) {
			$result = self::write( $temp_file, $content, false );
			if ( is_wp_error( $result ) ) {
				@unlink( $temp_file );
				return $result;
			}
		}
		
		self::get_logger()->debug( sprintf( 'Temporary file created: %s', $temp_file ) );
		return $temp_file;
	}
}