<?php
/**
 * AI Site Builder - Smart CSS Extractor
 *
 * Extracts relevant CSS rules from large files to avoid token limits
 * Uses string operations and regex - no external parser dependencies
 *
 * @package AI_Site_Builder
 * @since 3.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_CSS_Extractor {

	private $logger;

	/**
	 * Maximum size for extracted contiguous block (150KB = ~75k tokens)
	 * Claude has 200k token limit (~400KB), so 150KB leaves room for:
	 * - System prompts and instructions
	 * - Other context files (header.php, js, etc)
	 * - Claude's response
	 */
	const MAX_EXTRACTED_SIZE = 153600; // 150KB

	public function __construct() {
		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-logger.php';
		$this->logger = AI_Logger::get_instance( 'CSSExtractor' );
	}

	/**
	 * Extract relevant CSS from large file
	 *
	 * @param string $css_content Full CSS content
	 * @param array $context Context with selected_element, viewport, etc.
	 * @return string Extracted CSS
	 */
	public function extract( $css_content, $context = array() ) {

		$start_time = microtime( true );

		try {
			$this->logger->debug( 'Starting CSS extraction', array(
				'size' => $this->format_bytes( strlen( $css_content ) ),
				'has_selector' => ! empty( $context['selected_element'] ),
				'viewport' => $context['viewport'] ?? 'unknown'
			) );

			$extracted = array();

			// Skip variables extraction - they'll be included in contiguous element block
			// This reduces redundancy and size

			// Selected element rules (REQUIRED for large files)
			if ( ! empty( $context['selected_element'] ) ) {
				$element_css = $this->extract_element_rules( $css_content, $context['selected_element'] );

				if ( $element_css ) {

					$extracted[] = "/* Selected Element */\n" . $element_css;

					$this->logger->debug( 'Element rules extracted (contiguous block)', array(
						'selector' => $context['selected_element'],
						'size' => $this->format_bytes( strlen( $element_css ) )
					) );
				} else {
					// SMART FALLBACK: Try related selectors before giving up
					$this->logger->warning( 'No matching element rules found, trying fallback selectors', array(
						'selector' => $context['selected_element']
					) );

					$fallback_selectors = $this->generate_fallback_selectors( $context['selected_element'] );
					$found_fallback = false;

					foreach ( $fallback_selectors as $fallback_selector ) {
						$element_css = $this->extract_element_rules( $css_content, $fallback_selector );
						if ( $element_css ) {
							$extracted[] = "/* Fallback: Using related selector '{$fallback_selector}' */\n" . $element_css;
							$this->logger->info( 'Fallback selector successful', array(
								'original' => $context['selected_element'],
								'fallback' => $fallback_selector,
								'size' => $this->format_bytes( strlen( $element_css ) )
							) );
							$found_fallback = true;
							break;
						}
					}

					// Last resort: include full file with warning
					if ( ! $found_fallback ) {
						$this->logger->warning( 'All fallback selectors failed, including full CSS file', array(
							'selector' => $context['selected_element'],
							'file_size' => $this->format_bytes( strlen( $css_content ) )
						) );
						// Return full file - let AI handle it
						return $css_content;
					}
				}
			} else {
				// No selector provided - can't extract from large file
				$this->logger->warning( 'No selected_element provided - cannot extract CSS', array(
					'file_size' => $this->format_bytes( strlen( $css_content ) )
				) );
				throw new Exception( 'Element selection required for large files. Please Ctrl/⌘+Click on the element you want to edit.' );
			}

			// Media queries for viewport (usually smaller, so less risky)
			if ( ! empty( $context['viewport'] ) && $context['viewport'] !== 'desktop' ) {
				$media_css = $this->extract_media_queries( $css_content, $context['viewport'], $context['selected_element'] );
				if ( $media_css ) {
					$extracted[] = "/* Viewport Media Queries */\n" . $media_css;
				}
			}

			$result = implode( "\n\n", $extracted );

			$original_size = strlen( $css_content );
			$extracted_size = strlen( $result );
			$reduction = round( ( 1 - ( $extracted_size / $original_size ) ) * 100, 1 );

			$this->logger->info( 'CSS extraction completed', array(
				'original_size' => $this->format_bytes( $original_size ),
				'extracted_size' => $this->format_bytes( $extracted_size ),
				'reduction' => $reduction . '%',
				'time' => round( microtime( true ) - $start_time, 3 ) . 's'
			) );

			return $result;

		} catch ( Exception $e ) {
			$this->logger->error( 'CSS extraction failed', array(
				'error' => $e->getMessage()
			) );

			// Don't return original - throw error so user gets helpful message
			throw $e;
		}
	}

	/**
	 * Extract CSS variables from :root
	 * Returns ONE contiguous block from first :root to last :root
	 */
	private function extract_variables( $css_content ) {
		$matches = array();
		$i = 0;
		$length = strlen( $css_content );

		while ( $i < $length ) {
			// Find next opening brace
			$brace_pos = strpos( $css_content, '{', $i );
			if ( $brace_pos === false ) break;

			// Get selector (everything before {)
			$selector = substr( $css_content, $i, $brace_pos - $i );

			// Check if this is a :root or :host rule
			if ( preg_match( '/:root|:host/', $selector ) ) {
				// Track position instead of extracting immediately
				$matches[] = array(
					'start' => $i,
					'selector_end' => $brace_pos
				);
			}

			$i = $brace_pos + 1;
		}

		if ( empty( $matches ) ) {
			return '';
		}

		// Extract from FIRST :root to LAST :root (one continuous block)
		$first_pos = $matches[0]['start'];

		// Find end of last :root rule
		$last_brace = $matches[ count( $matches ) - 1 ]['selector_end'];
		$depth = 1;
		$j = $last_brace + 1;
		while ( $j < $length && $depth > 0 ) {
			if ( $css_content[$j] === '{' ) $depth++;
			if ( $css_content[$j] === '}' ) $depth--;
			$j++;
		}

		// Extract ONE continuous block - preserves ALL structure!
		return substr( $css_content, $first_pos, $j - $first_pos );
	}

	/**
	 * Extract rules for selected element
	 * Uses regex for accurate selector matching
	 * Returns ONE contiguous block from first match, limited to MAX_EXTRACTED_SIZE
	 */
	private function extract_element_rules( $css_content, $target_selector ) {
		$matches = array();
		$i = 0;
		$length = strlen( $css_content );

		while ( $i < $length ) {
			// Find next opening brace
			$brace_pos = strpos( $css_content, '{', $i );
			if ( $brace_pos === false ) break;

			// Get selector (everything before {)
			$selector = substr( $css_content, $i, $brace_pos - $i );

			// Check if selector matches our target
			if ( $this->selector_matches_target( $selector, $target_selector ) ) {
				// Find end of this rule
				$depth = 1;
				$j = $brace_pos + 1;
				while ( $j < $length && $depth > 0 ) {
					if ( $css_content[$j] === '{' ) $depth++;
					if ( $css_content[$j] === '}' ) $depth--;
					$j++;
				}

				// Track position with end position
				$matches[] = array(
					'start' => $i,
					'end' => $j
				);
			}

			$i = $brace_pos + 1;
		}

		if ( empty( $matches ) ) {
			return '';
		}

		// Start from first match
		$first_pos = $matches[0]['start'];
		$last_match_index = 0;

		// Find how many matches we can include within size limit
		for ( $idx = 0; $idx < count( $matches ); $idx++ ) {
			$potential_end = $matches[$idx]['end'];
			$potential_size = $potential_end - $first_pos;

			if ( $potential_size > self::MAX_EXTRACTED_SIZE ) {
				// Stop at previous match if we have one
				if ( $idx > 0 ) {
					$last_match_index = $idx - 1;
				} else {
					// Even first match is too large - extract just first rule
					$last_match_index = 0;
				}
				break;
			}

			$last_match_index = $idx;
		}

		$last_end = $matches[$last_match_index]['end'];
		$extracted_size = $last_end - $first_pos;

		$this->logger->debug( 'Element extraction size-limited', array(
			'total_matches' => count( $matches ),
			'included_matches' => $last_match_index + 1,
			'extracted_size' => $this->format_bytes( $extracted_size )
		) );

		// Extract contiguous block up to size limit
		return substr( $css_content, $first_pos, $extracted_size );
	}

	/**
	 * Extract media queries for viewport that contain the selected element
	 * Returns ONE contiguous block from first match to last match
	 */
	private function extract_media_queries( $css_content, $viewport, $target_selector ) {
		$matches = array();

		$breakpoints = array(
			'mobile' => 767,
			'tablet' => 1024
		);

		$max_width = $breakpoints[ $viewport ] ?? 0;
		if ( ! $max_width ) {
			return '';
		}

		$i = 0;
		$length = strlen( $css_content );

		while ( $i < $length ) {
			// Find @media
			$media_pos = stripos( $css_content, '@media', $i );
			if ( $media_pos === false ) break;

			// Find opening brace of media query
			$brace_pos = strpos( $css_content, '{', $media_pos );
			if ( $brace_pos === false ) break;

			// Get media query condition
			$media_query = substr( $css_content, $media_pos, $brace_pos - $media_pos );

			// Check if it matches our viewport
			if ( preg_match( '/max-width:\s*(\d+)px/', $media_query, $match_result ) ) {
				$mq_width = intval( $match_result[1] );

				if ( $mq_width <= $max_width + 200 ) {
					// Count braces to find closing }
					$depth = 1;
					$j = $brace_pos + 1;
					$media_content_start = $j;

					while ( $j < $length && $depth > 0 ) {
						if ( $css_content[$j] === '{' ) $depth++;
						if ( $css_content[$j] === '}' ) $depth--;
						$j++;
					}

					// Get content inside media query
					$media_content = substr( $css_content, $media_content_start, $j - $media_content_start - 1 );

					// Check if it contains our target selector
					if ( $this->selector_matches_target( $media_content, $target_selector ) ) {
						// Track position instead of extracting immediately
						$matches[] = array(
							'start' => $media_pos,
							'end' => $j
						);
					}
				}
			}

			$i = $brace_pos + 1;
		}

		if ( empty( $matches ) ) {
			return '';
		}

		// Extract from FIRST media query to LAST media query (one continuous block)
		$first_pos = $matches[0]['start'];
		$last_end = $matches[ count( $matches ) - 1 ]['end'];

		// Extract ONE continuous block - preserves ALL structure!
		return substr( $css_content, $first_pos, $last_end - $first_pos );
	}

	/**
	 * Check if a selector matches the target using regex
	 * Prevents matching .hero-section-title when target is .hero-section
	 */
	private function selector_matches_target( $selector, $target ) {
		$selector = strtolower( trim( $selector ) );
		$target = strtolower( trim( $target ) );

		// Escape special regex characters in target
		$escaped = preg_quote( $target, '/' );

		// Match target followed by valid CSS separator or end of string
		// Valid separators: space, :, ., ,, [, {, >, +, ~, ), or end
		return preg_match( '/' . $escaped . '(?=[\s:.,\[{>+~\)]|$)/', $selector ) === 1;
	}

	/**
	 * Format bytes
	 */
	private function format_bytes( $bytes ) {
		if ( $bytes < 1024 ) return $bytes . 'B';
		if ( $bytes < 1048576 ) return round( $bytes / 1024, 1 ) . 'KB';
		return round( $bytes / 1048576, 1 ) . 'MB';
	}

	/**
	 * Generate fallback selectors to try when exact selector fails
	 *
	 * @param string $selector Original selector
	 * @return array List of fallback selectors to try
	 */
	private function generate_fallback_selectors( $selector ) {
		$fallbacks = array();

		// Remove class/ID prefix and get base name
		$base = preg_replace( '/^[.#]/', '', $selector );

		// Strategy 1: Try parent/container selectors
		if ( strpos( $base, '-' ) !== false ) {
			$parts = explode( '-', $base );

			// .header-actions → .header
			if ( count( $parts ) > 1 ) {
				array_pop( $parts );
				$parent = implode( '-', $parts );
				$fallbacks[] = '.' . $parent;
			}
		}

		// Strategy 2: Try element selector (without class/ID)
		// .header-actions → header
		if ( preg_match( '/^\.?(header|footer|nav|main|aside|section|article)/', $base, $matches ) ) {
			$fallbacks[] = $matches[1];
		}

		// Strategy 3: Common WordPress/theme selectors
		if ( strpos( $base, 'header' ) !== false ) {
			$fallbacks[] = '.site-header';
			$fallbacks[] = '.main-header';
			$fallbacks[] = 'header';
			$fallbacks[] = '.header';
		}

		if ( strpos( $base, 'nav' ) !== false || strpos( $base, 'menu' ) !== false ) {
			$fallbacks[] = '.site-navigation';
			$fallbacks[] = '.main-navigation';
			$fallbacks[] = 'nav';
		}

		// Strategy 4: Try just the element name if it's a class
		if ( strpos( $selector, '.' ) === 0 ) {
			$fallbacks[] = str_replace( '.', '', $selector );
		}

		// Remove duplicates and empty values
		$fallbacks = array_unique( array_filter( $fallbacks ) );

		return $fallbacks;
	}
}
