<?php
/**
 * AI Site Builder - Configuration Manager
 * 
 * Centralized management of plugin configuration and options with caching
 * and type-safe access.
 * 
 * @package AI_Site_Builder
 * @since 2.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Config_Manager {
	
	/**
	 * Cache for configuration values
	 */
	private static $cache = array();
	
	/**
	 * Option prefix
	 */
	const OPTION_PREFIX = 'ai_site_builder_';
	
	/**
	 * AI Model Configuration
	 * Change this constant to switch AI providers globally (e.g., 'gemini-2.0-flash-exp')
	 */
	const AI_MODEL = 'claude-sonnet-4-5-20250929';

	/**
	 * Vercel API URL
	 * Configure this to point to your Vercel deployment
	 */
	const VERCEL_API_URL = 'https://ai-site-builder-api.vercel.app';

	/**
	 * Configuration defaults
	 */
	private static $defaults = array(
		'openai_api_key' => '',
		'claude_api_key' => '',
		'image_model' => 'dall-e-3',
		'max_execution_time' => 1800,
		'http_timeout' => 1500,
		'debug_mode' => false,
		'cache_enabled' => true,
		'cache_ttl' => 3600
	);
	
	/**
	 * Get a configuration value
	 * 
	 * @param string $key Configuration key
	 * @param mixed $default Default value if not found
	 * @return mixed Configuration value
	 */
	public static function get( $key, $default = null ) {
		// Check cache first
		if ( isset( self::$cache[ $key ] ) ) {
			return self::$cache[ $key ];
		}
		
		// Get from WordPress options
		$option_name = self::OPTION_PREFIX . $key;
		$value = get_option( $option_name );
		
		// If not found, check defaults
		if ( $value === false ) {
			if ( isset( self::$defaults[ $key ] ) ) {
				$value = self::$defaults[ $key ];
			} else {
				$value = $default;
			}
		}
		
		// Cache the value
		self::$cache[ $key ] = $value;
		
		return $value;
	}
	
	/**
	 * Set a configuration value
	 * 
	 * @param string $key Configuration key
	 * @param mixed $value Configuration value
	 * @return bool True if updated successfully
	 */
	public static function set( $key, $value ) {
		$option_name = self::OPTION_PREFIX . $key;
		$result = update_option( $option_name, $value );
		
		// Update cache
		if ( $result ) {
			self::$cache[ $key ] = $value;
		}
		
		return $result;
	}
	
	/**
	 * Get OpenAI API key
	 * 
	 * @return string API key or empty string
	 */
	public static function get_openai_api_key() {
		return self::get( 'openai_api_key', '' );
	}
	
	/**
	 * Get Claude API key
	 * 
	 * @return string API key or empty string
	 */
	public static function get_claude_api_key() {
		return self::get( 'claude_api_key', '' );
	}

	/**
	 * Get image model
	 * 
	 * @return string Model identifier
	 */
	public static function get_image_model() {
		return self::get( 'image_model', 'dall-e-3' );
	}
	
	/**
	 * Check if any AI service is configured
	 * 
	 * @return bool True if at least one API key is set
	 */
	public static function has_ai_service() {
		return ! empty( self::get_openai_api_key() ) || ! empty( self::get_claude_api_key() );
	}
	
	/**
	 * Check if OpenAI is configured
	 * 
	 * @return bool True if OpenAI API key is set
	 */
	public static function has_openai() {
		return ! empty( self::get_openai_api_key() );
	}
	
	/**
	 * Check if Claude is configured
	 * 
	 * @return bool True if Claude API key is set
	 */
	public static function has_claude() {
		return ! empty( self::get_claude_api_key() );
	}
	
	/**
	 * Get HTTP timeout for API requests
	 * 
	 * @return int Timeout in seconds
	 */
	public static function get_http_timeout() {
		return (int) self::get( 'http_timeout', 1500 );
	}
	
	/**
	 * Get max execution time for PHP scripts
	 * 
	 * @return int Time in seconds
	 */
	public static function get_max_execution_time() {
		return (int) self::get( 'max_execution_time', 1800 );
	}
	
	/**
	 * Check if debug mode is enabled
	 * 
	 * @return bool True if debug mode is on
	 */
	public static function is_debug_mode() {
		return (bool) self::get( 'debug_mode', false ) || ( defined( 'WP_DEBUG' ) && WP_DEBUG );
	}
	
	/**
	 * Get all configuration as array
	 * 
	 * @param bool $include_sensitive Include sensitive data like API keys
	 * @return array Configuration array
	 */
	public static function get_all( $include_sensitive = false ) {
		$config = array();
		
		foreach ( self::$defaults as $key => $default ) {
			// Skip sensitive data if requested
			if ( ! $include_sensitive && strpos( $key, 'api_key' ) !== false ) {
				$value = self::get( $key );
				$config[ $key ] = ! empty( $value ) ? 'configured' : 'not_configured';
			} else {
				$config[ $key ] = self::get( $key );
			}
		}
		
		return $config;
	}
	
	/**
	 * Clear configuration cache
	 */
	public static function clear_cache() {
		self::$cache = array();
	}
	
	/**
	 * Validate API key format
	 * 
	 * @param string $key API key to validate
	 * @param string $service Service name (openai or claude)
	 * @return bool True if valid format
	 */
	public static function validate_api_key( $key, $service ) {
		if ( empty( $key ) ) {
			return false;
		}
		
		switch ( $service ) {
			case 'openai':
				// OpenAI keys start with 'sk-'
				return strpos( $key, 'sk-' ) === 0 && strlen( $key ) > 20;
				
			case 'claude':
				// Claude keys typically have a specific format
				return strlen( $key ) > 20;
				
			default:
				return false;
		}
	}
	
	/**
	 * Get Vercel API URL
	 *
	 * @return string Vercel API URL
	 */
	public static function get_vercel_api_url() {
		return self::VERCEL_API_URL;
	}

	/**
	 * Get service status
	 *
	 * @return array Status information for each service
	 */
	public static function get_service_status() {
		return array(
			'openai' => array(
				'configured' => self::has_openai(),
				'primary' => false,
				'key_valid' => self::has_openai() ? self::validate_api_key( self::get_openai_api_key(), 'openai' ) : false
			),
			'claude' => array(
				'configured' => self::has_claude(),
				'primary' => true,
				'key_valid' => self::has_claude() ? self::validate_api_key( self::get_claude_api_key(), 'claude' ) : false
			),
			'has_service' => self::has_ai_service(),
			'default_model' => self::AI_MODEL,
			'vercel_api_url' => self::get_vercel_api_url()
		);
	}
	
	/**
	 * Initialize configuration with defaults
	 * Called during plugin activation
	 */
	public static function init_defaults() {
		foreach ( self::$defaults as $key => $value ) {
			$option_name = self::OPTION_PREFIX . $key;
			add_option( $option_name, $value );
		}
	}
}