<?php
/**
 * Dreamformer Freemius Webhook Handler
 *
 * Handles subscription lifecycle events from Freemius:
 * - New subscriptions (account connection)
 * - Plan changes (upgrades/downgrades)
 * - Renewals (monthly credit resets)
 * - Cancellations
 *
 * @package AI_Site_Builder
 * @since 4.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Dreamformer_Freemius_Handler {

	/**
	 * Credit allocation for free plan
	 */
	const FREE_PLAN_CREDITS = 25;

	/**
	 * Explicit plan name to credits mapping
	 *
	 * Maps Freemius plan names (lowercase) to monthly credit allocations.
	 * This is the authoritative source - price parsing is only a fallback.
	 *
	 * To add a new plan: Add entry here, no other code changes needed.
	 */
	const PLAN_CREDITS = array(
		'free'     => 25,
		'pro-100'  => 100,
		'pro-200'  => 200,
		'pro-500'  => 500,
		'pro-1000' => 1000,
		'pro'      => 100,  // Generic pro fallback
	);

	/**
	 * Constructor - Register Freemius hooks
	 */
	public function __construct() {
		// Only hook if Freemius is active and SDK is loaded
		if ( ! function_exists( 'dreamformer_fs' ) ) {
			return;
		}

		// Check if Freemius SDK actually initialized (returns null if SDK not downloaded)
		$freemius = dreamformer_fs();
		if ( ! $freemius ) {
			// SDK not downloaded yet or failed to initialize - skip hook registration
			return;
		}

		// Hook into Freemius events
		$freemius->add_action( 'after_account_connection', array( $this, 'on_account_connection' ) );
		$freemius->add_action( 'after_license_change', array( $this, 'on_license_change' ) );
		$freemius->add_action( 'after_account_details', array( $this, 'on_subscription_change' ) );
		$freemius->add_action( 'after_account_cancellation', array( $this, 'on_subscription_cancelled' ) );

		// Auto-initialize free plan for users without credit records (self-healing)
		add_action( 'admin_init', array( $this, 'ensure_user_has_credits' ) );
	}

	/**
	 * Get monthly credits from Freemius plan
	 *
	 * Priority order:
	 * 1. Explicit PLAN_CREDITS mapping (most reliable)
	 * 2. Price-based detection (fallback for edge cases)
	 * 3. Title parsing (legacy fallback)
	 * 4. Free tier default
	 *
	 * @return int Monthly credit allocation
	 */
	private function get_credits_from_pricing() {
		$license = dreamformer_fs()->_get_license();
		$plan    = dreamformer_fs()->get_plan();

		// Priority 1: Check explicit PLAN_CREDITS mapping (most reliable)
		if ( $plan && ! empty( $plan->name ) ) {
			$plan_name = strtolower( $plan->name );
			if ( isset( self::PLAN_CREDITS[ $plan_name ] ) ) {
				return self::PLAN_CREDITS[ $plan_name ];
			}
		}

		// Priority 2: Fall back to price-based detection
		if ( $license && isset( $license->pricing ) && isset( $license->pricing->monthly_price ) ) {
			$monthly_price = floatval( $license->pricing->monthly_price );

			if ( $monthly_price >= 140 ) {
				return 1000; // $149/month tier
			}
			if ( $monthly_price >= 80 ) {
				return 500;  // $89/month tier
			}
			if ( $monthly_price >= 35 ) {
				return 200;  // $39/month tier
			}
			if ( $monthly_price >= 15 ) {
				return 100;  // $19/month tier
			}
		}

		// Priority 3: Legacy fallback - parse plan title for numbers
		if ( $plan && isset( $plan->title ) ) {
			$plan_title = strtolower( $plan->title );

			if ( preg_match( '/1000/', $plan_title ) ) {
				return 1000;
			}
			if ( preg_match( '/500/', $plan_title ) ) {
				return 500;
			}
			if ( preg_match( '/200/', $plan_title ) ) {
				return 200;
			}
			if ( preg_match( '/100/', $plan_title ) ) {
				return 100;
			}
		}

		// Priority 4: Default to free tier
		return self::FREE_PLAN_CREDITS;
	}

	/**
	 * Get the site's license key for site-wide credit operations
	 *
	 * For premium: Gets license from Freemius
	 * For free: Generates site-based key from URL hash
	 *
	 * @return string|null License key or null if not available
	 */
	private function get_site_license_key() {
		// Check for Freemius premium license first
		if ( function_exists( 'dreamformer_fs' ) && dreamformer_fs() ) {
			$fs = dreamformer_fs();
			if ( $fs->is_registered() && $fs->is_premium() ) {
				$license = $fs->_get_license();
				if ( $license && ! empty( $license->secret_key ) ) {
					return $license->secret_key;
				}
			}
		}

		// Fall back to site-based free license key
		$site_identifier = substr( md5( get_site_url() ), 0, 16 );
		return 'free_' . $site_identifier;
	}

	/**
	 * User activates license for first time
	 *
	 * @param object $user Freemius user object
	 */
	public function on_account_connection( $user ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		$user_id = get_current_user_id();
		$license = dreamformer_fs()->_get_license();
		$plan    = dreamformer_fs()->get_plan();

		if ( ! $license || ! $plan ) {
			error_log( 'Dreamformer: License or plan not found on account connection' );
			return;
		}

		// Determine credits based on pricing tier
		$plan_id = strtolower( $plan->name ); // 'free' or 'pro'
		$monthly_credits = $this->get_credits_from_pricing();

		// Enhanced logging
		error_log( sprintf(
			'Dreamformer: Account connection - Plan: %s, Title: %s, Detected credits: %d',
			$plan->name,
			$plan->title ?? 'N/A',
			$monthly_credits
		) );

		// Get billing cycle (UTC for consistency with Vercel API)
		$billing_start = gmdate( 'Y-m-d' );
		$billing_end   = gmdate( 'Y-m-d', strtotime( '+1 month', time() ) );

		// REPLACE (not INSERT) - handles both new accounts and reactivations
		$result = $wpdb->replace(
			$table,
			array(
				'user_id'             => $user_id,
				'license_key'         => $license->secret_key,
				'plan_id'             => $plan_id,
				'monthly_credits'     => $monthly_credits,
				'used_credits'        => 0,
				'subscription_status' => dreamformer_fs()->is_trial() ? 'trial' : 'active',
				'billing_cycle_start' => $billing_start,
				'billing_cycle_end'   => $billing_end,
			),
			array( '%d', '%s', '%s', '%d', '%d', '%s', '%s', '%s' )
		);

		if ( false === $result ) {
			error_log( 'Dreamformer: Failed to create user credits on account connection: ' . $wpdb->last_error );
		} else {
			error_log( sprintf( 'Dreamformer: ✅ User %d connected with %s plan (%d credits/month)',
				$user_id, $plan_id, $monthly_credits ) );
		}

		// BUGFIX: Clear transient so ensure_user_has_credits() can re-sync on next admin load
		delete_transient( 'dreamformer_credits_checked_' . $user_id );
	}

	/**
	 * User upgrades/downgrades plan
	 *
	 * @param object $license Freemius license object
	 */
	public function on_license_change( $license ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		$user_id = get_current_user_id();
		$plan    = dreamformer_fs()->get_plan();

		if ( ! $plan ) {
			error_log( 'Dreamformer: Plan not found on license change' );
			return;
		}

		$plan_id = strtolower( $plan->name ); // 'free' or 'pro'
		$monthly_credits = $this->get_credits_from_pricing();

		// Enhanced logging
		error_log( sprintf(
			'Dreamformer: License change - Plan: %s, Title: %s, Detected credits: %d',
			$plan->name,
			$plan->title ?? 'N/A',
			$monthly_credits
		) );

		// Get current record to preserve some data (by license_key - site-wide)
		$current = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table} WHERE license_key = %s",
				$license->secret_key
			),
			ARRAY_A
		);

		// Determine used_credits based on upgrade/downgrade scenario
		$used_credits = 0; // Default for new record

		if ( $current ) {
			// Preserve used_credits if upgrading
			$used_credits = (int) $current['used_credits'];

			// If downgrading and used > new limit, cap to prevent negative balance
			if ( $used_credits > $monthly_credits ) {
				error_log( sprintf(
					'Dreamformer: User %d downgraded - capping used_credits from %d to %d (new limit)',
					$user_id,
					$used_credits,
					$monthly_credits
				) );
				$used_credits = $monthly_credits;
			}
		}

		// PRODUCTION FIX: Use REPLACE instead of UPDATE
		// This ensures the record is created if missing (handles edge cases)
		$billing_start = gmdate( 'Y-m-d' );
		$billing_end   = gmdate( 'Y-m-d', strtotime( '+1 month', time() ) );

		$result = $wpdb->replace(
			$table,
			array(
				'user_id'             => $user_id,
				'license_key'         => $license->secret_key,
				'plan_id'             => $plan_id,
				'monthly_credits'     => $monthly_credits,
				'used_credits'        => $used_credits,
				'subscription_status' => dreamformer_fs()->is_trial() ? 'trial' : 'active',
				'billing_cycle_start' => $billing_start,
				'billing_cycle_end'   => $billing_end,
			),
			array( '%d', '%s', '%s', '%d', '%d', '%s', '%s', '%s' )
		);

		if ( false === $result ) {
			error_log( 'Dreamformer: Failed to update plan on license change: ' . $wpdb->last_error );
		} else {
			error_log( sprintf(
				'Dreamformer: ✅ User %d changed to %s plan (%d credits/month, %d used)',
				$user_id,
				$plan_id,
				$monthly_credits,
				$used_credits
			) );
		}

		// BUGFIX: Clear transient so ensure_user_has_credits() can re-sync on next admin load
		delete_transient( 'dreamformer_credits_checked_' . $user_id );
	}

	/**
	 * Monthly billing cycle - reset credits
	 *
	 * @param object $user Freemius user object
	 */
	public function on_subscription_change( $user ) {
		// Called on renewal - reset credits
		$user_id = get_current_user_id();
		$this->reset_monthly_credits( $user_id );
	}

	/**
	 * User cancels subscription
	 *
	 * Updates site-wide subscription status (any admin can cancel).
	 *
	 * @param object $user Freemius user object
	 */
	public function on_subscription_cancelled( $user ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		// Get site's license key (site-wide operation)
		$license_key = $this->get_site_license_key();
		if ( ! $license_key ) {
			error_log( 'Dreamformer: Cannot cancel subscription - no license key found' );
			return;
		}

		// Mark as cancelled (access continues until billing end)
		// Uses license_key for site-wide update - any admin can cancel
		$result = $wpdb->update(
			$table,
			array( 'subscription_status' => 'cancelled' ),
			array( 'license_key' => $license_key ),
			array( '%s' ),
			array( '%s' )
		);

		if ( false === $result ) {
			error_log( 'Dreamformer: Failed to mark subscription as cancelled: ' . $wpdb->last_error );
		} else {
			error_log( sprintf( 'Dreamformer: Site subscription cancelled (license: %s...)', substr( $license_key, 0, 12 ) ) );
		}
	}

	/**
	 * Reset monthly credits (called via webhook or cron)
	 * Resets used credits to 0, starts new billing cycle
	 *
	 * Site-wide operation - uses license_key, not user_id.
	 *
	 * @param int $user_id WordPress user ID (unused, kept for backwards compatibility)
	 */
	private function reset_monthly_credits( $user_id ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		// Get site's license key (site-wide operation)
		$license_key = $this->get_site_license_key();
		if ( ! $license_key ) {
			error_log( 'Dreamformer: Cannot reset credits - no license key found' );
			return;
		}

		// CRITICAL: Start transaction to prevent race conditions with credit deduction
		$wpdb->query( 'START TRANSACTION' );

		try {
			// Get current state WITH ROW LOCK to prevent concurrent modifications
			// Uses license_key for site-wide lookup
			$current = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM {$table} WHERE license_key = %s FOR UPDATE",
					$license_key
				),
				ARRAY_A
			);

			if ( ! $current ) {
				$wpdb->query( 'ROLLBACK' );
				error_log( 'Dreamformer: Cannot reset credits - site record not found (license: ' . substr( $license_key, 0, 12 ) . '...)' );
				return;
			}

			// Calculate unused credits to rollover
			$unused = $current['monthly_credits'] - $current['used_credits'];
			$new_rollover_amount = max( 0, $unused );

			// Check if existing rollover has expired (use UTC for consistency)
			$existing_rollover = 0;
			if ( $current['rollover_expiry'] && strtotime( $current['rollover_expiry'] . ' UTC' ) >= time() ) {
				// Rollover still valid, keep it
				$existing_rollover = (int) $current['rollover_credits'];
			}
			// If expired, it's automatically lost (set to 0)

			// Free plan gets NO rollover
			$plan_id = $current['plan_id'];
			if ( $plan_id === 'free' ) {
				$new_rollover_amount = 0;
				$existing_rollover = 0;
			}

			// Cap rollover at monthly credit limit (prevents unlimited accumulation)
			// Matches Lovable model: monthly plans can only accumulate up to monthly limit
			$max_rollover = $current['monthly_credits'];
			$total_rollover = min( $max_rollover, $existing_rollover + $new_rollover_amount );
			$rollover_expiry = $total_rollover > 0 ? gmdate( 'Y-m-d', strtotime( '+1 month', time() ) ) : null;

			// Reset (UTC for consistency with Vercel API)
			// Uses license_key for site-wide update
			$result = $wpdb->update(
				$table,
				array(
					'used_credits'        => 0,
					'rollover_credits'    => $total_rollover,
					'rollover_expiry'     => $rollover_expiry,
					'billing_cycle_start' => gmdate( 'Y-m-d' ),
					'billing_cycle_end'   => gmdate( 'Y-m-d', strtotime( '+1 month', time() ) ),
				),
				array( 'license_key' => $license_key ),
				array( '%d', '%d', '%s', '%s', '%s' ),
				array( '%s' )
			);

			if ( false === $result ) {
				$wpdb->query( 'ROLLBACK' );
				error_log( 'Dreamformer: Failed to reset monthly credits: ' . $wpdb->last_error );
				return;
			}

			// COMMIT transaction
			$wpdb->query( 'COMMIT' );

			error_log( sprintf(
				'Dreamformer: Reset site credits (license: %s...). Rollover: %d (unused: %d + existing: %d, capped at: %d)',
				substr( $license_key, 0, 12 ),
				$total_rollover,
				$new_rollover_amount,
				$existing_rollover,
				$max_rollover
			) );

		} catch ( Exception $e ) {
			$wpdb->query( 'ROLLBACK' );
			error_log( 'Dreamformer: Reset credits transaction failed: ' . $e->getMessage() );
		}
	}

	/**
	 * Ensure current user has a credit record
	 *
	 * Self-healing: Creates free plan (25 credits) if missing
	 * Called on admin_init to auto-initialize free users
	 * Prevents issues when Freemius webhooks don't fire for free activations
	 *
	 * SECURITY: Uses site-based license key to prevent credit farming attacks:
	 * - Attack vector 1: User deletes DB record -> generates new random key -> fresh 25 credits
	 * - Attack vector 2: User creates multiple admin accounts -> each gets separate key -> N x 25 credits
	 * - Solution: Site-based key (hash of site URL) ensures same key is regenerated every time
	 * - Result: One free plan per WordPress site, shared by all users on that site
	 */
	public function ensure_user_has_credits() {
		// Only run in admin for logged-in users
		$user_id = get_current_user_id();
		if ( ! $user_id || ! is_admin() ) {
			return;
		}

		$transient_key = 'dreamformer_credits_checked_' . $user_id;

		// CRITICAL: Check premium status FIRST, before transient check
		// This ensures users who upgrade mid-session get synced immediately
		// Bug fix: Previously transient check blocked premium sync after upgrade
		if ( function_exists( 'dreamformer_fs' ) && dreamformer_fs() && dreamformer_fs()->is_registered() && dreamformer_fs()->is_premium() ) {
			// Always sync premium users (clears any stale free records)
			$this->sync_premium_user_credits( $user_id );
			set_transient( $transient_key, true, HOUR_IN_SECONDS );
			return; // Premium user credits synced
		}

		// For FREE users only: Check transient to avoid repeated DB queries
		if ( get_transient( $transient_key ) ) {
			return; // Already checked this session
		}

		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		// SECURITY FIX: Generate stable site-based license key FIRST
		// This ensures all admins on the same site share the same 25 free credits
		$site_identifier = substr( md5( get_site_url() ), 0, 16 );
		$license_key     = 'free_' . $site_identifier;

		// Check if THIS USER already has a record
		$user_has_record = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table} WHERE user_id = %d",
				$user_id
			)
		);

		if ( $user_has_record > 0 ) {
			set_transient( $transient_key, true, HOUR_IN_SECONDS );
			return;
		}

		// Check if SITE already has free credits (another admin created them)
		// All free users on the same site share the same license_key
		$site_has_credits = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table} WHERE license_key = %s",
				$license_key
			)
		);

		if ( $site_has_credits > 0 ) {
			// Site already has free credits - this user shares them
			// No need to create a duplicate record
			set_transient( $transient_key, true, HOUR_IN_SECONDS );
			return;
		}

		// First admin on this site creates the free plan record
		// License key already generated above (site-based, shared by all admins)
		$billing_start   = gmdate( 'Y-m-d' );
		$billing_end     = gmdate( 'Y-m-d', strtotime( '+1 month', time() ) );

		$result = $wpdb->insert(
			$table,
			array(
				'user_id'             => $user_id,
				'license_key'         => $license_key,
				'plan_id'             => 'free',
				'monthly_credits'     => self::FREE_PLAN_CREDITS,
				'used_credits'        => 0,
				'subscription_status' => 'free',
				'billing_cycle_start' => $billing_start,
				'billing_cycle_end'   => $billing_end,
			),
			array( '%d', '%s', '%s', '%d', '%d', '%s', '%s', '%s' )
		);

		if ( $result ) {
			error_log( sprintf( 'Dreamformer: Auto-initialized free plan for user %d (%d credits/month)',
				$user_id, self::FREE_PLAN_CREDITS ) );
			set_transient( $transient_key, true, HOUR_IN_SECONDS );
		} else {
			error_log( 'Dreamformer: Failed to auto-initialize free plan: ' . $wpdb->last_error );
		}
	}

	/**
	 * Sync premium user credits to match their Freemius plan
	 *
	 * Self-healing function that runs on every admin page load.
	 * Automatically fixes credit mismatches from:
	 * - Failed Freemius webhooks
	 * - Timing issues during license activation
	 * - Plan upgrades/downgrades that didn't sync
	 * - Database corruption or manual edits
	 *
	 * Uses UPDATE instead of REPLACE to preserve user_id and other data.
	 * Cleans up orphaned free records when user upgrades to premium.
	 *
	 * @param int $user_id WordPress user ID
	 */
	private function sync_premium_user_credits( $user_id ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		// Get what the user SHOULD have based on their Freemius plan
		$license = dreamformer_fs()->_get_license();
		$plan    = dreamformer_fs()->get_plan();

		if ( ! $license || ! $plan ) {
			error_log( 'Dreamformer: Cannot sync credits - license or plan not found' );
			return;
		}

		// Clean up orphaned free records when upgrading to premium
		// This ensures only one credit record exists per site
		$site_identifier  = substr( md5( get_site_url() ), 0, 16 );
		$free_license_key = 'free_' . $site_identifier;

		if ( $free_license_key !== $license->secret_key ) {
			$deleted = $wpdb->delete(
				$table,
				array( 'license_key' => $free_license_key ),
				array( '%s' )
			);

			if ( $deleted > 0 ) {
				error_log( sprintf(
					'Dreamformer: Cleaned up orphaned free record for site (was: %s, now: %s...)',
					$free_license_key,
					substr( $license->secret_key, 0, 12 )
				) );
			}
		}

		$expected_credits = $this->get_credits_from_pricing();
		$plan_id          = strtolower( $plan->name );

		// Get what the SITE actually has in the database (by license_key, not user_id)
		$current = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table} WHERE license_key = %s",
				$license->secret_key
			),
			ARRAY_A
		);

		// Check if credits need to be updated
		$needs_update = false;

		if ( ! $current ) {
			// No record exists - will create it
			$needs_update = true;
		} elseif ( (int) $current['monthly_credits'] !== $expected_credits ) {
			// Credits don't match plan
			$needs_update = true;
		} elseif ( $current['plan_id'] !== $plan_id ) {
			// Plan ID doesn't match
			$needs_update = true;
		}

		// Only update if needed
		if ( ! $needs_update ) {
			return; // Credits are already correct
		}

		// Prepare billing dates
		$billing_start = gmdate( 'Y-m-d' );
		$billing_end   = gmdate( 'Y-m-d', strtotime( '+1 month', time() ) );
		$status        = dreamformer_fs()->is_trial() ? 'trial' : 'active';

		if ( ! $current ) {
			// No record exists - INSERT new premium record
			$result = $wpdb->insert(
				$table,
				array(
					'user_id'             => $user_id,
					'license_key'         => $license->secret_key,
					'plan_id'             => $plan_id,
					'monthly_credits'     => $expected_credits,
					'used_credits'        => 0,
					'subscription_status' => $status,
					'billing_cycle_start' => $billing_start,
					'billing_cycle_end'   => $billing_end,
				),
				array( '%d', '%s', '%s', '%d', '%d', '%s', '%s', '%s' )
			);

			if ( false === $result ) {
				error_log( 'Dreamformer: Failed to create premium credit record: ' . $wpdb->last_error );
			} else {
				error_log( sprintf(
					'Dreamformer: Created premium credit record for user %d - Plan: %s (%d credits/month)',
					$user_id,
					$plan_id,
					$expected_credits
				) );
			}
		} else {
			// Record exists - UPDATE only the fields that need changing
			// Preserves: user_id, used_credits, rollover_credits, billing dates
			$result = $wpdb->update(
				$table,
				array(
					'plan_id'             => $plan_id,
					'monthly_credits'     => $expected_credits,
					'subscription_status' => $status,
				),
				array( 'license_key' => $license->secret_key ),
				array( '%s', '%d', '%s' ),
				array( '%s' )
			);

			if ( false === $result ) {
				error_log( 'Dreamformer: Failed to update premium credits: ' . $wpdb->last_error );
			} else {
				error_log( sprintf(
					'Dreamformer: Updated credits for user %d - Plan: %s (%d credits/month)',
					$user_id,
					$plan_id,
					$expected_credits
				) );
			}
		}
	}
}

// Initialize handler (only if Freemius SDK is loaded and initialized)
if ( function_exists( 'dreamformer_fs' ) && dreamformer_fs() ) {
	new Dreamformer_Freemius_Handler();
}
