<?php
/**
 * Dreamformer Credit Management System
 *
 * Handles all credit operations:
 * - Checking if user has sufficient credits
 * - Deducting credits after API calls
 * - Logging usage with token counts
 * - Rate limit enforcement
 * - Usage statistics
 *
 * @package AI_Site_Builder
 * @since 4.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Dreamformer_Credit_Manager {

	/**
	 * Credit costs by operation type
	 *
	 * PAID OPERATIONS:
	 * - Generate Page: 8 credits (complete AI-generated pages)
	 * - Edit Page: 1 credit (conversational modifications)
	 * - Conversational: 1 credit (Q&A and guidance)
	 * - Edit Theme: 3 credits (advanced WordPress customization)
	 *
	 * FREE OPERATIONS:
	 * - Everything else costs 0 credits
	 */
	const CREDIT_COSTS = array(
		// PAID OPERATIONS
		'page_generation' => 8,  // Generate Page: Complete AI-generated pages
		'page_edit'       => 1,  // Edit Page: Conversational modifications
		'conversational'  => 1,  // Conversational: Q&A responses
		'theme_edit'      => 3,  // Edit Theme: Advanced WordPress customization

		// FREE OPERATIONS
		'detect_files'    => 0,  // File detection (preview)
		'extract_snippet' => 0,  // Component extraction
		'preview_draft'   => 0,  // Preview changes
		'undo'            => 0,  // Version history
		'redo'            => 0,  // Version history
		'get_versions'    => 0,  // View history
		'restore_version' => 0,  // Restore from backup
	);

	/**
	 * Get daily rate limit based on monthly credits
	 *
	 * Prevents abuse and infinite loops
	 *
	 * @param int $monthly_credits Monthly credit allocation
	 * @return int Daily page generation limit
	 */
	private function get_daily_limit( $monthly_credits ) {
		if ( $monthly_credits >= 1000 ) {
			return 40; // Pro-1000: 40 pages per day
		}
		if ( $monthly_credits >= 500 ) {
			return 25; // Pro-500: 25 pages per day
		}
		if ( $monthly_credits >= 200 ) {
			return 15; // Pro-200: 15 pages per day
		}
		if ( $monthly_credits >= 100 ) {
			return 10; // Pro-100: 10 pages per day
		}
		return 5; // Free: 5 pages per day
	}

	/**
	 * Get site's current credit balance
	 *
	 * Credits are SITE-WIDE, not per-user. All admins share the same credits.
	 * Looks up by license_key (from Freemius for premium, site-based for free).
	 *
	 * @param int $user_id WordPress user ID (kept for backwards compatibility, not used for lookup)
	 * @return array|null Credit info or null if not found
	 */
	public function get_user_credits( $user_id = null ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_user_credits';

		// Get site's license key
		$license_key = $this->get_site_license_key();

		if ( ! $license_key ) {
			return null;
		}

		// Look up credits by license_key (site-wide)
		$credits = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table} WHERE license_key = %s",
				$license_key
			),
			ARRAY_A
		);

		return $credits;
	}

	/**
	 * Get the site's license key
	 *
	 * For premium: Gets license from Freemius
	 * For free: Generates site-based key from URL hash
	 *
	 * @return string|null License key or null if not available
	 */
	private function get_site_license_key() {
		// Check for Freemius premium license first
		if ( function_exists( 'dreamformer_fs' ) && dreamformer_fs() ) {
			$fs = dreamformer_fs();
			if ( $fs->is_registered() && $fs->is_premium() ) {
				$license = $fs->_get_license();
				if ( $license && ! empty( $license->secret_key ) ) {
					return $license->secret_key;
				}
			}
		}

		// Fall back to site-based free license key
		$site_identifier = substr( md5( get_site_url() ), 0, 16 );
		return 'free_' . $site_identifier;
	}

	/**
	 * Check if user has sufficient credits
	 *
	 * @param int $user_id WordPress user ID
	 * @param int $required_credits Credits needed for operation
	 * @return bool True if user has enough credits
	 */
	public function has_credits( $user_id, $required_credits ) {
		$credits = $this->get_user_credits( $user_id );

		if ( ! $credits ) {
			return false;
		}

		// Check subscription is active or free
		$valid_statuses = array( 'active', 'free', 'trial' );
		if ( ! in_array( $credits['subscription_status'], $valid_statuses, true ) ) {
			return false;
		}

		// Check billing cycle hasn't expired
		if ( strtotime( $credits['billing_cycle_end'] ) < time() ) {
			return false;
		}

		// Check credits available (computed column: (monthly_credits - used_credits) + rollover_credits)
		return $credits['available_credits'] >= $required_credits;
	}

	/**
	 * Deduct credits from site account
	 *
	 * Uses MySQL transaction for atomicity.
	 * Credits are SITE-WIDE, deducted by license_key not user_id.
	 *
	 * @param int    $user_id WordPress user ID (for logging only)
	 * @param int    $credits_to_deduct Credits to deduct
	 * @param string $endpoint API endpoint called
	 * @param array  $usage_data Token counts, costs, etc.
	 * @return bool|WP_Error True on success, WP_Error on failure
	 */
	public function deduct_credits( $user_id, $credits_to_deduct, $endpoint, $usage_data = array() ) {
		global $wpdb;

		// Get site's license key
		$license_key = $this->get_site_license_key();
		if ( ! $license_key ) {
			return new WP_Error( 'no_license', 'No license key found for this site' );
		}

		// Start transaction
		$wpdb->query( 'START TRANSACTION' );

		try {
			// Get current credits with row lock (prevents race conditions)
			// Uses license_key for site-wide credits
			$credits_table = $wpdb->prefix . 'dreamformer_user_credits';
			$current       = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM {$credits_table} WHERE license_key = %s FOR UPDATE",
					$license_key
				),
				ARRAY_A
			);

			if ( ! $current ) {
				$wpdb->query( 'ROLLBACK' );
				return new WP_Error( 'no_credits_record', 'Site credit record not found' );
			}

			if ( $current['available_credits'] < $credits_to_deduct ) {
				$wpdb->query( 'ROLLBACK' );
				return new WP_Error(
					'insufficient_credits',
					sprintf(
						'Insufficient credits. Required: %d, Available: %d',
						$credits_to_deduct,
						$current['available_credits']
					)
				);
			}

			// Deduct in order: rollover credits first, then monthly credits
			$monthly_available  = $current['monthly_credits'] - $current['used_credits'];
			$rollover_available = (int) ( $current['rollover_credits'] ?? 0 );

			$new_used     = $current['used_credits'];
			$new_rollover = $rollover_available;

			if ( $rollover_available >= $credits_to_deduct ) {
				// All from rollover
				$new_rollover = $rollover_available - $credits_to_deduct;
			} elseif ( $rollover_available > 0 ) {
				// Use all rollover + some monthly
				$from_rollover = $rollover_available;
				$from_monthly  = $credits_to_deduct - $from_rollover;
				$new_rollover  = 0;
				$new_used      = $current['used_credits'] + $from_monthly;
			} else {
				// All from monthly
				$new_used = $current['used_credits'] + $credits_to_deduct;
			}

			// Update both used_credits and rollover_credits atomically
			// Uses license_key for site-wide credits
			$update_result = $wpdb->update(
				$credits_table,
				array(
					'used_credits'     => $new_used,
					'rollover_credits' => $new_rollover,
				),
				array( 'license_key' => $license_key ),
				array( '%d', '%d' ),
				array( '%s' )
			);

			if ( false === $update_result ) {
				$wpdb->query( 'ROLLBACK' );
				return new WP_Error( 'update_failed', 'Failed to update credits: ' . $wpdb->last_error );
			}

			// Log usage
			$log_table = $wpdb->prefix . 'dreamformer_usage_log';
			$log_data  = array(
				'user_id'          => $user_id,
				'endpoint'         => $endpoint,
				'request_type'     => isset( $usage_data['request_type'] ) ? $usage_data['request_type'] : 'unknown',
				'input_tokens'     => isset( $usage_data['input_tokens'] ) ? $usage_data['input_tokens'] : 0,
				'output_tokens'    => isset( $usage_data['output_tokens'] ) ? $usage_data['output_tokens'] : 0,
				'thinking_tokens'  => isset( $usage_data['thinking_tokens'] ) ? $usage_data['thinking_tokens'] : 0,
				'credits_deducted' => $credits_to_deduct,
				'estimated_cost'   => isset( $usage_data['estimated_cost'] ) ? $usage_data['estimated_cost'] : 0,
				'success'          => true,
			);

			$log_result = $wpdb->insert(
				$log_table,
				$log_data,
				array( '%d', '%s', '%s', '%d', '%d', '%d', '%d', '%f', '%d' )
			);

			if ( false === $log_result ) {
				// Log failed but don't rollback credit deduction
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Dreamformer: Failed to log usage: ' . $wpdb->last_error );
			}
			}

			$wpdb->query( 'COMMIT' );
			return true;

		} catch ( Exception $e ) {
			$wpdb->query( 'ROLLBACK' );
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Dreamformer: Credit deduction failed: ' . $e->getMessage() );
		}
			return new WP_Error( 'transaction_failed', 'Credit deduction transaction failed' );
		}
	}

	/**
	 * Check rate limit for user
	 *
	 * @param int    $user_id WordPress user ID
	 * @param string $limit_type 'daily' or 'monthly'
	 * @return bool True if under limit, false if exceeded
	 */
	public function check_rate_limit( $user_id, $limit_type = 'daily' ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_rate_limits';

		// Get or create rate limit record
		$limit = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table} WHERE user_id = %d AND limit_type = %s",
				$user_id,
				$limit_type
			),
			ARRAY_A
		);

		if ( ! $limit ) {
			// No limit record exists yet - user is under limit
			return true;
		}

		// Check if reset time has passed
		if ( strtotime( $limit['reset_at'] ) < time() ) {
			// Reset the counter
			$wpdb->update(
				$table,
				array(
					'requests_made' => 0,
					'reset_at'      => $this->get_reset_time( $limit_type ),
				),
				array( 'id' => $limit['id'] ),
				array( '%d', '%s' ),
				array( '%d' )
			);
			return true;
		}

		// Check if under limit
		return $limit['requests_made'] < $limit['limit_threshold'];
	}

	/**
	 * Increment rate limit counter
	 *
	 * Called after successful API request
	 *
	 * @param int $user_id WordPress user ID
	 */
	public function increment_rate_limit( $user_id ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_rate_limits';

		// Get site's monthly credits to determine daily limit (site-wide credits)
		$credits_info = $this->get_user_credits();
		$monthly_credits = $credits_info ? intval( $credits_info['monthly_credits'] ) : 25;

		// Get daily limit based on monthly credits
		$daily_limit = $this->get_daily_limit( $monthly_credits );

		// Insert or increment daily limit
		$wpdb->query(
			$wpdb->prepare(
				"INSERT INTO {$table} (user_id, limit_type, limit_threshold, requests_made, reset_at)
				 VALUES (%d, 'daily', %d, 1, %s)
				 ON DUPLICATE KEY UPDATE requests_made = requests_made + 1",
				$user_id,
				$daily_limit,
				$this->get_reset_time( 'daily' )
			)
		);
	}

	/**
	 * Get next reset time
	 *
	 * @param string $limit_type 'daily' or 'monthly'
	 * @return string Timestamp in MySQL format
	 */
	private function get_reset_time( $limit_type ) {
		if ( 'daily' === $limit_type ) {
			// Reset at midnight
			return date( 'Y-m-d 00:00:00', strtotime( 'tomorrow' ) );
		} else {
			// Reset on first of next month
			return date( 'Y-m-d 00:00:00', strtotime( 'first day of next month' ) );
		}
	}

	/**
	 * Get usage statistics for user
	 *
	 * @param int    $user_id WordPress user ID
	 * @param string $period 'week', 'month', or 'all'
	 * @return array Usage statistics breakdown
	 */
	public function get_usage_stats( $user_id, $period = 'month' ) {
		global $wpdb;
		$table = $wpdb->prefix . 'dreamformer_usage_log';

		// Date filter
		$date_filter = '';
		if ( 'week' === $period ) {
			$date_filter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
		} elseif ( 'month' === $period ) {
			$date_filter = "AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
		}

		// Get breakdown by request type
		$breakdown = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT request_type,
						COUNT(*) as count,
						SUM(credits_deducted) as total_credits,
						SUM(input_tokens) as total_input_tokens,
						SUM(output_tokens) as total_output_tokens
				 FROM {$table}
				 WHERE user_id = %d {$date_filter}
				 GROUP BY request_type",
				$user_id
			),
			ARRAY_A
		);

		// Calculate totals
		$total_requests     = 0;
		$total_credits_used = 0;

		foreach ( $breakdown as $row ) {
			$total_requests     += $row['count'];
			$total_credits_used += $row['total_credits'];
		}

		return array(
			'breakdown'          => $breakdown,
			'total_requests'     => $total_requests,
			'total_credits_used' => $total_credits_used,
		);
	}

	/**
	 * Get credit cost for an operation type
	 *
	 * @param string $operation_type Operation type key
	 * @return int Credit cost
	 */
	public function get_credit_cost( $operation_type ) {
		return isset( self::CREDIT_COSTS[ $operation_type ] ) ? self::CREDIT_COSTS[ $operation_type ] : 1;
	}
}
