<?php
/**
 * AI Site Builder - Vercel API Client
 * 
 * Handles all communication with Vercel-hosted AI algorithms
 * Replaces local AI processing with secure API calls
 * 
 * @package AI_Site_Builder
 * @since 3.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Vercel_Client {
	
	/**
	 * Base URL for Vercel APIs
	 */
	private $base_url;
	
	/**
	 * Logger instance
	 */
	private $logger;
	
	/**
	 * Request timeout in seconds
	 */
	const TIMEOUT = 300; // 5 minutes for large AI operations
	
	/**
	 * API Secret Key
	 */
	private $api_key;

	/**
	 * Constructor
	 */
	public function __construct() {
		// Initialize logger FIRST (needed by other methods)
		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-logger.php';
		$this->logger = AI_Logger::get_instance( 'VercelClient' );

		// Get endpoint from settings or use default
		$this->base_url = get_option(
			'ai_site_builder_vercel_endpoint',
			'https://ai-site-builder-api.vercel.app'
		);

		// Ensure trailing slash is removed
		$this->base_url = rtrim( $this->base_url, '/' );

		// Get per-user license key from database
		$this->api_key = $this->get_user_license_key();
	}

	/**
	 * Get site's license key for API authentication
	 *
	 * Uses Credit Manager for site-wide license key lookup.
	 * All admins on the same site share the same license key.
	 *
	 * @return string|null License key or null if not found
	 */
	private function get_user_license_key() {
		// Use Credit Manager for consistent site-wide lookup
		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
		$credit_manager = new Dreamformer_Credit_Manager();
		$credits_info = $credit_manager->get_user_credits();

		if ( ! $credits_info || empty( $credits_info['license_key'] ) ) {
			$this->logger->warning( 'No license key found for site' );
			return null;
		}

		// Verify subscription is valid
		$valid_statuses = array( 'active', 'free', 'trial' );
		if ( ! in_array( $credits_info['subscription_status'], $valid_statuses, true ) ) {
			$this->logger->warning( 'Site subscription not active', array(
				'status' => $credits_info['subscription_status']
			) );
			return null;
		}

		return $credits_info['license_key'];
	}

	/**
	 * Get site's monthly credit limit
	 *
	 * Uses Credit Manager for site-wide credit lookup.
	 * All admins on the same site share the same credits.
	 *
	 * @return int Monthly credit limit (25, 100, 200, 500, 1000)
	 */
	private function get_user_monthly_credits() {
		// Use Credit Manager for consistent site-wide lookup
		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
		$credit_manager = new Dreamformer_Credit_Manager();
		$credits_info = $credit_manager->get_user_credits();

		if ( ! $credits_info || ! isset( $credits_info['monthly_credits'] ) ) {
			return 25; // Free tier default
		}

		return intval( $credits_info['monthly_credits'] );
	}

	/**
	 * Get shared secret key for HMAC signatures
	 *
	 * SECURITY NOTE: This secret is hardcoded and visible in plugin code, which is INTENTIONAL.
	 * Knowing this secret does NOT grant users extra credits or privileges.
	 *
	 * Real security comes from:
	 * 1. Redis credit tracking on Vercel (users cannot access)
	 * 2. License key validation with Freemius
	 * 3. Credit limits enforced server-side
	 *
	 * The signature only prevents request tampering in transit,
	 * similar to how JWT tokens work in WordPress authentication.
	 *
	 * @return string Secret key (same for all installations)
	 */
	private function get_secret_key() {
		// Hardcoded secret shared between ALL WordPress installations and Vercel
		// This MUST match DREAMFORMER_PLAN_SECRET_KEY in Vercel environment variables
		return 'c2523081066e83eb9bf22d418f53495057cfc54b6a899c7686e380b8284059ab';
	}

	/**
	 * Generate HMAC signature for plan tier validation
	 *
	 * @param string $plan_tier Plan tier to sign
	 * @return string HMAC signature
	 */
	private function generate_plan_signature( $plan_tier ) {
		$secret = $this->get_secret_key();
		return hash_hmac( 'sha256', $plan_tier, $secret );
	}

	/**
	 * Get current user's credit information
	 *
	 * @return array|null Credit info or null if not found
	 */
	public function get_user_credit_info() {
		$user_id = get_current_user_id();

		if ( ! $user_id ) {
			return null;
		}

		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
		$credit_manager = new Dreamformer_Credit_Manager();

		return $credit_manager->get_user_credits( $user_id );
	}

	/**
	 * Check if user has sufficient credits for an operation
	 *
	 * @param string $operation_type Operation type (page_generation, page_edit, etc.)
	 * @return bool|WP_Error True if has credits, WP_Error if insufficient
	 */
	public function check_user_credits( $operation_type ) {
		$user_id = get_current_user_id();

		if ( ! $user_id ) {
			return new WP_Error( 'no_user', 'No user logged in' );
		}

		require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
		$credit_manager = new Dreamformer_Credit_Manager();

		$required_credits = $credit_manager->get_credit_cost( $operation_type );

		if ( ! $credit_manager->has_credits( $user_id, $required_credits ) ) {
			$credits_info = $credit_manager->get_user_credits( $user_id );
			$available = $credits_info ? $credits_info['available_credits'] : 0;

			return new WP_Error(
				'insufficient_credits',
				sprintf(
					'Insufficient credits. Required: %d, Available: %d',
					$required_credits,
					$available
				),
				array(
					'required' => $required_credits,
					'available' => $available
				)
			);
		}

		return true;
	}
	
	/**
	 * Detect required files using AI with metadata for intelligent selection
	 *
	 * @param string $user_prompt User's request
	 * @param array $available_files List of available file names
	 * @param string $theme_path Path to theme directory
	 * @param array|null $file_metadata Optional file metadata for smart selection
	 * @return array|WP_Error Detection result or error
	 */
	public function detect_files( $user_prompt, $available_files, $theme_path, $file_metadata = null ) {
		$this->logger->info( 'Calling Vercel API for file detection', array(
			'prompt_preview' => substr( $user_prompt, 0, 100 ) . '...',
			'files_count' => count( $available_files ),
			'has_metadata' => ! empty( $file_metadata )
		) );

		$payload = array(
			'user_prompt' => $user_prompt,
			'available_files' => $available_files,
			'theme_path' => $theme_path
		);

		// Add metadata if provided (enables intelligent file selection)
		if ( $file_metadata ) {
			$payload['file_metadata'] = $file_metadata;
		}

		$response = $this->make_request( '/api/detect-files', $payload );
		
		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'File detection API failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}
		
		// Validate response structure
		if ( ! isset( $response['required_files'] ) || ! is_array( $response['required_files'] ) ) {
			return new WP_Error( 'invalid_response', 'Invalid response from file detection API' );
		}
		
		$this->logger->info( 'File detection completed', array(
			'detected_files' => $response['required_files'],
			'estimated_changes' => $response['estimated_changes'] ?? 'unknown'
		) );
		
		return $response;
	}
	
	/**
	 * Generate diff using AI
	 *
	 * @param string $user_prompt User's request
	 * @param array $required_files Files to modify (name => content)
	 * @param array $context Optional element context
	 * @param string $complexity 'simple' for CSS-only, 'complex' for multi-file
	 * @return string|WP_Error AI response or error
	 */
	public function generate_diff( $user_prompt, $required_files, $context = array(), $complexity = 'complex' ) {
		$this->logger->info( 'Calling Vercel API for diff generation', array(
			'complexity' => $complexity,
			'files_count' => count( $required_files ),
			'has_context' => ! empty( $context )
		) );

		$response = $this->make_request( '/api/generate-diff', array(
			'user_prompt' => $user_prompt,
			'required_files' => $required_files,
			'context' => $context,
			'complexity' => $complexity
		) );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Diff generation API failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		// Extract the diff JSON from response
		if ( isset( $response['diff_json'] ) ) {
			$this->logger->info( 'Diff generation completed', array(
				'model_used' => $response['model_used'] ?? 'unknown',
				'has_user_message' => ! empty( $response['user_message'] )
			) );

			// Store user message if provided
			if ( ! empty( $response['user_message'] ) ) {
				set_transient( 'ai_site_builder_last_message', $response['user_message'], 60 );
			}

			return $response['diff_json'];
		}

		return new WP_Error( 'invalid_response', 'Invalid response from diff generation API' );
	}

	/**
	 * Generate page diff using AI with HTML/CSS/JS constraints
	 *
	 * @param string $user_prompt User's request
	 * @param array $required_files Files to modify (name => content)
	 * @param array $context Optional element context
	 * @return string|WP_Error AI response or error
	 */
	public function generate_page_diff( $user_prompt, $required_files, $context = array(), $attempt_history = '' ) {
		$this->logger->info( 'Calling Vercel API for page diff generation', array(
			'files_count' => count( $required_files ),
			'has_context' => ! empty( $context ),
			'has_attempt_history' => ! empty( $attempt_history )
		) );

		$payload = array(
			'user_prompt' => $user_prompt,
			'required_files' => $required_files,
			'context' => $context,
			'complexity' => 'complex' // Always complex for pages
		);

		// Add attempt history if provided (Memory System)
		if ( ! empty( $attempt_history ) ) {
			$payload['attempt_history'] = $attempt_history;
		}

		$response = $this->make_request( '/api/generate-page-diff', $payload );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Page diff generation API failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		// Let the LLM choose: conversational response OR diff changes
		if ( isset( $response['conversational_response'] ) ) {
			$this->logger->info( 'Page diff API returned conversational response', array(
				'model_used' => $response['model_used'] ?? 'unknown'
			) );
			return $response['conversational_response'];
		}

		// Extract the diff JSON from response
		if ( isset( $response['diff_json'] ) ) {
			$this->logger->info( 'Page diff generation completed', array(
				'model_used' => $response['model_used'] ?? 'unknown',
				'has_user_message' => ! empty( $response['user_message'] )
			) );

			// Store user message if provided
			if ( ! empty( $response['user_message'] ) ) {
				set_transient( 'ai_site_builder_last_message', $response['user_message'], 60 );
			}

			return $response['diff_json'];
		}

		return new WP_Error( 'invalid_response', 'Invalid response from page diff generation API' );
	}

	/**
	 * Generate page diff using AI with streaming progress updates
	 *
	 * @param string $user_prompt User's request
	 * @param array $required_files Files to modify (name => content)
	 * @param array $context Optional element context
	 * @param callable $callback Callback function for SSE events (event_type, data)
	 * @param string $attempt_history Optional attempt history for memory system
	 * @return string|WP_Error AI response or error
	 */
	public function generate_page_diff_stream( $user_prompt, $required_files, $context = array(), $callback = null, $attempt_history = '' ) {
		$this->logger->info( 'Calling Vercel streaming API for page diff generation', array(
			'files_count' => count( $required_files ),
			'has_context' => ! empty( $context ),
			'has_callback' => ! empty( $callback ),
			'has_attempt_history' => ! empty( $attempt_history )
		) );

		$url = $this->base_url . '/api/generate-page-diff-stream';

		$payload = array(
			'user_prompt' => $user_prompt,
			'required_files' => $required_files,
			'context' => $context,
		);

		// Add attempt history if provided (Memory System)
		if ( ! empty( $attempt_history ) ) {
			$payload['attempt_history'] = $attempt_history;
		}

		// Build headers
		$headers = array(
			'Content-Type: application/json',
			'Accept: text/event-stream',
		);

		// Add API key authorization if available
		if ( $this->api_key ) {
			// Get user's monthly credits and generate signature
			$monthly_credits = $this->get_user_monthly_credits();
			$credits_signature = $this->generate_plan_signature( (string) $monthly_credits );

			$headers[] = 'Authorization: Bearer ' . $this->api_key;
			$headers[] = 'X-WordPress-URL: ' . home_url(); // Tell Vercel which WordPress to call back to
			$headers[] = 'X-Monthly-Credits: ' . $monthly_credits; // User's monthly credit limit
			$headers[] = 'X-Plan-Signature: ' . $credits_signature; // HMAC signature to prevent tampering

			$this->logger->debug( '✅ API key, monthly credits, and signature added to streaming request', array(
				'key_length' => strlen( $this->api_key ),
				'wordpress_url' => home_url(),
				'monthly_credits' => $monthly_credits,
				'signature_length' => strlen( $credits_signature )
			) );
		} else {
			$this->logger->warning( '⚠️ No API key for streaming request - will be rejected', array(
				'endpoint' => $endpoint
			) );
		}

		// Use cURL for streaming support
		$ch = curl_init( $url );

		curl_setopt_array( $ch, array(
			CURLOPT_POST => true,
			CURLOPT_POSTFIELDS => wp_json_encode( $payload ),
			CURLOPT_HTTPHEADER => $headers,
			CURLOPT_RETURNTRANSFER => false,
			CURLOPT_TIMEOUT => self::TIMEOUT,
			CURLOPT_SSL_VERIFYPEER => true,
		) );

		// Accumulate SSE data
		$buffer = '';
		$final_result = null;
		$has_error = false;

		// Set write function to process streaming data
		curl_setopt( $ch, CURLOPT_WRITEFUNCTION, function( $ch, $data ) use ( &$buffer, $callback, &$final_result, &$has_error ) {
			$buffer .= $data;

			// Process complete SSE events (separated by \n\n)
			while ( strpos( $buffer, "\n\n" ) !== false ) {
				$pos = strpos( $buffer, "\n\n" );
				$event_data = substr( $buffer, 0, $pos );
				$buffer = substr( $buffer, $pos + 2 );

				if ( empty( trim( $event_data ) ) ) {
					continue;
				}

				// Parse SSE format
				$lines = explode( "\n", $event_data );
				$event_type = 'message';
				$event_content = '';

				foreach ( $lines as $line ) {
					if ( strpos( $line, 'event:' ) === 0 ) {
						$event_type = trim( substr( $line, 6 ) );
					} elseif ( strpos( $line, 'data:' ) === 0 ) {
						$event_content = trim( substr( $line, 5 ) );
					}
				}

				// Handle different event types
				if ( $event_type === 'stage' ) {
					// Progress update - call callback if provided
					if ( $callback ) {
						call_user_func( $callback, 'stage', $event_content );
					}
				} elseif ( $event_type === 'complete' ) {
					// Final result
					$final_result = json_decode( $event_content, true );
					if ( $callback ) {
						call_user_func( $callback, 'complete', $event_content );
					}
				} elseif ( $event_type === 'error' ) {
					// Error occurred
					$has_error = true;
					if ( $callback ) {
						call_user_func( $callback, 'error', $event_content );
					}
				}
			}

			return strlen( $data );
		} );

		// Execute request
		$exec_result = curl_exec( $ch );
		$curl_error = curl_error( $ch );
		$http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
		curl_close( $ch );

		// Check for cURL errors
		if ( $exec_result === false && ! empty( $curl_error ) ) {
			$this->logger->error( 'cURL streaming request failed', array(
				'error' => $curl_error
			) );
			return new WP_Error( 'curl_error', 'Failed to connect to streaming API: ' . $curl_error );
		}

		// Check HTTP status
		if ( $http_code !== 200 ) {
			$this->logger->error( 'Streaming API returned error status', array(
				'http_code' => $http_code
			) );
			return new WP_Error( 'http_error', sprintf( 'Streaming API error (HTTP %d)', $http_code ) );
		}

		// Check if error occurred during streaming
		if ( $has_error ) {
			return new WP_Error(
				'streaming_error',
				'Connection interrupted while generating. Please try again. If this keeps happening, try a shorter prompt.',
				array( 'retry' => true )
			);
		}

		// Check if we received final result
		if ( empty( $final_result ) ) {
			$this->logger->error( 'No final result received from streaming API' );
			return new WP_Error( 'no_result', 'No result received from streaming API' );
		}

		// Handle conversational responses
		if ( isset( $final_result['conversational_response'] ) ) {
			$this->logger->info( 'Streaming page diff API returned conversational response', array(
				'model_used' => $final_result['model_used'] ?? 'unknown'
			) );
			return $final_result['conversational_response'];
		}

		// Handle diff responses
		if ( isset( $final_result['diff_json'] ) ) {
			$this->logger->info( 'Streaming page diff generation completed', array(
				'model_used' => $final_result['model_used'] ?? 'unknown',
				'has_user_message' => ! empty( $final_result['user_message'] )
			) );

			// Store user message if provided
			if ( ! empty( $final_result['user_message'] ) ) {
				set_transient( 'ai_site_builder_last_message', $final_result['user_message'], 60 );
			}

			return $final_result['diff_json'];
		}

		return new WP_Error( 'invalid_response', 'Invalid response from streaming page diff generation API' );
	}

	/**
	 * Generate conversational response using Vercel API
	 * 
	 * @param string $user_prompt User's question
	 * @param array $theme_files Theme file contents (name => content)
	 * @return array|WP_Error Response array or error
	 */
	public function conversational_response( $user_prompt, $theme_files ) {
		$this->logger->info( 'Calling Vercel API for conversational response', array(
			'prompt_preview' => substr( $user_prompt, 0, 100 ) . '...',
			'files_count' => count( $theme_files )
		) );
		
		$response = $this->make_request( '/api/conversational-response', array(
			'user_prompt' => $user_prompt,
			'theme_files' => $theme_files
		) );
		
		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Conversational response API failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}
		
		$this->logger->info( 'Conversational response completed', array(
			'success' => $response['success'] ?? false,
			'has_message' => ! empty( $response['message'] )
		) );
		
		return $response;
	}
	
	/**
	 * Extract snippet component code using AI
	 *
	 * @param string $selector CSS selector for the component
	 * @param array $required_files Files containing the component (name => content)
	 * @return array|WP_Error Extracted code (html, css, js) or error
	 */
	public function extract_snippet( $selector, $required_files ) {
		$this->logger->info( 'Calling Vercel API for snippet extraction', array(
			'selector' => $selector,
			'files_count' => count( $required_files )
		) );

		$response = $this->make_request( '/api/extract-snippet', array(
			'selector' => $selector,
			'required_files' => $required_files
		) );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Snippet extraction API failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		// Validate response structure
		if ( ! isset( $response['html'] ) ) {
			return new WP_Error( 'invalid_response', 'Invalid response from snippet extraction API' );
		}

		$this->logger->info( 'Snippet extraction completed', array(
			'has_html' => ! empty( $response['html'] ),
			'has_css' => ! empty( $response['css'] ),
			'has_js' => ! empty( $response['js'] )
		) );

		return $response;
	}

	/**
	 * Apply a section from one page to another using AI
	 *
	 * Sends source and target page files to AI, which extracts the section
	 * and integrates it into the target page. Returns updated file contents.
	 *
	 * @param string $selector CSS selector for the section to clone
	 * @param array $source_files Source page files (index.html, styles.css, app.js)
	 * @param array $target_files Target page files (index.html, styles.css, app.js)
	 * @return array|WP_Error Updated target files or error
	 */
	public function apply_section( $selector, $source_files, $target_files ) {
		$this->logger->info( 'Starting apply section job', array(
			'selector' => $selector,
			'source_files_count' => count( $source_files ),
			'target_files_count' => count( $target_files )
		) );

		// Start the job (returns 202 Accepted with job_id)
		$response = $this->call_api_json( '/api/apply-section', array(
			'selector' => $selector,
			'source_files' => $source_files,
			'target_files' => $target_files
		) );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Section apply job creation failed', array(
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		// Check for job_id
		if ( ! isset( $response['job_id'] ) ) {
			return new WP_Error( 'invalid_response', 'No job_id in apply section response' );
		}

		$this->logger->info( 'Apply section job created', array(
			'job_id' => $response['job_id']
		) );

		// Return immediately - browser will poll for status
		return $response;
	}

	/**
	 * Poll apply section job status
	 *
	 * @param string $job_id Job ID
	 * @return array|WP_Error Status data or error
	 */
	public function poll_apply_section_status( $job_id ) {
		$url = $this->base_url . '/api/apply-section/' . $job_id . '/status';

		$response = wp_remote_get( $url, array(
			'timeout' => 10,
			'headers' => array( 'Accept' => 'application/json' ),
			'sslverify' => true
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$http_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		if ( $http_code !== 200 ) {
			return new WP_Error( 'poll_error', sprintf( 'Poll failed (HTTP %d)', $http_code ) );
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return new WP_Error( 'json_parse_error', 'Failed to parse poll response' );
		}

		return $data;
	}

	/**
	 * Make HTTP request to Vercel API
	 *
	 * @param string $endpoint API endpoint path
	 * @param array $data Request data
	 * @param int $timeout Optional custom timeout
	 * @return array|WP_Error Response data or error
	 */
	private function make_request( $endpoint, $data, $timeout = null ) {
		// Block request if no valid license key
		if ( empty( $this->api_key ) ) {
			$this->logger->error( 'No license key found - request blocked', array(
				'endpoint' => $endpoint,
				'user_id' => get_current_user_id()
			) );
			return new WP_Error(
				'no_license',
				'No valid license key found. Please activate a plan at Dreamformer → Pricing to use AI features.',
				array( 'required_action' => 'activate_license' )
			);
		}

		$url = $this->base_url . $endpoint;
		$timeout = $timeout ?? self::TIMEOUT;

		$this->logger->debug( 'Making API request', array(
			'url' => $url,
			'timeout' => $timeout
		) );

		// Get user's monthly credits and generate signature
		$monthly_credits = $this->get_user_monthly_credits();
		$credits_signature = $this->generate_plan_signature( (string) $monthly_credits );

		$headers = array(
			'Content-Type' => 'application/json',
			'Accept' => 'application/json',
			'Authorization' => 'Bearer ' . $this->api_key,
			'X-WordPress-URL' => home_url(), // Tell Vercel which WordPress site to call back to
			'X-Monthly-Credits' => $monthly_credits, // User's monthly credit limit
			'X-Plan-Signature' => $credits_signature // HMAC signature to prevent tampering
		);

		$this->logger->debug( '✅ API key, monthly credits, and signature added to request headers', array(
			'key_length' => strlen( $this->api_key ),
			'wordpress_url' => home_url(),
			'monthly_credits' => $monthly_credits,
			'signature_length' => strlen( $credits_signature )
		) );

		$args = array(
			'method' => 'POST',
			'timeout' => $timeout,
			'headers' => $headers,
			'body' => wp_json_encode( $data ),
			'sslverify' => true
		);
		
		$response = wp_remote_post( $url, $args );
		
		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'HTTP request failed', array(
				'endpoint' => $endpoint,
				'error' => $response->get_error_message()
			) );
			return new WP_Error( 
				'http_error', 
				sprintf( 'Failed to connect to Vercel API: %s', $response->get_error_message() )
			);
		}
		
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		
		$this->logger->debug( 'API response received', array(
			'status_code' => $response_code,
			'body_length' => strlen( $response_body )
		) );
		
		// Handle non-200 responses
		if ( $response_code !== 200 ) {
			$error_message = 'API request failed';
			
			// Try to parse error from response
			$decoded = json_decode( $response_body, true );
			if ( isset( $decoded['error'] ) ) {
				$error_message = $decoded['error'];
			} elseif ( isset( $decoded['message'] ) ) {
				$error_message = $decoded['message'];
			}
			
			$this->logger->error( 'API returned error', array(
				'status_code' => $response_code,
				'error' => $error_message
			) );
			
			return new WP_Error( 
				'api_error', 
				sprintf( 'Vercel API error (HTTP %d): %s', $response_code, $error_message )
			);
		}
		
		// Parse JSON response
		$decoded = json_decode( $response_body, true );
		
		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Failed to parse API response', array(
				'json_error' => json_last_error_msg(),
				'response_preview' => substr( $response_body, 0, 200 )
			) );
			return new WP_Error( 'json_error', 'Failed to parse API response' );
		}
		
		return $decoded;
	}

	/**
	 * Proxy streaming request from Vercel API to browser
	 *
	 * Streams Server-Sent Events (SSE) from Vercel API directly to the browser.
	 * This acts as a secure proxy that:
	 * - Adds API key authentication automatically
	 * - Allows future license/payment validation hooks
	 * - Keeps API key secure on server (never exposed to browser)
	 * - Enables usage tracking and rate limiting
	 *
	 * @param string $endpoint Vercel API endpoint (e.g., '/api/generate-page-content-stream')
	 * @param array $payload Request payload to send to Vercel
	 * @param callable|null $callback Optional callback function for processing events
	 * @return bool True on success, false on error
	 */
	public function proxy_stream_to_browser( $endpoint, $payload, $callback = null ) {
		// Block streaming if no valid license key
		if ( empty( $this->api_key ) ) {
			$this->logger->error( 'No license key for streaming - blocked', array(
				'endpoint' => $endpoint,
				'user_id' => get_current_user_id()
			) );

			// Return proper HTTP error if streaming hasn't started
			if ( ! headers_sent() ) {
				http_response_code( 401 );
				header( 'Content-Type: application/json' );
				echo wp_json_encode( array(
					'success' => false,
					'error' => 'unauthorized',
					'message' => 'No valid license key found. Please activate a plan at Dreamformer → Pricing.'
				) );
			} else {
				// If streaming already started, send SSE error event
				echo "event: error\n";
				echo "data: No valid license key found. Please activate a plan at Dreamformer → Pricing.\n\n";
				flush();
			}
			return false;
		}

		$url = $this->base_url . $endpoint;

		$this->logger->info( 'Starting streaming proxy to browser', array(
			'endpoint' => $endpoint,
			'url' => $url
		) );

		// Set SSE headers for browser
		if ( ! headers_sent() ) {
			header( 'Content-Type: text/event-stream' );
			header( 'Cache-Control: no-cache' );
			header( 'Connection: keep-alive' );
			header( 'X-Accel-Buffering: no' ); // Disable nginx buffering
		}

		// Build headers with API key
		$headers = array(
			'Content-Type: application/json',
			'Accept: text/event-stream',
		);

		// Add API key authorization
		if ( $this->api_key ) {
			// Get user's monthly credits and generate signature
			$monthly_credits = $this->get_user_monthly_credits();
			$credits_signature = $this->generate_plan_signature( (string) $monthly_credits );

			$headers[] = 'Authorization: Bearer ' . $this->api_key;
			$headers[] = 'X-WordPress-URL: ' . home_url(); // Tell Vercel which WordPress to call back to
			$headers[] = 'X-Monthly-Credits: ' . $monthly_credits; // User's monthly credit limit
			$headers[] = 'X-Plan-Signature: ' . $credits_signature; // HMAC signature to prevent tampering

			$this->logger->debug( '✅ API key, monthly credits, and signature added to streaming proxy request', array(
				'wordpress_url' => home_url(),
				'monthly_credits' => $monthly_credits,
				'signature_length' => strlen( $credits_signature )
			) );
		} else {
			$this->logger->error( '⚠️ No API key available for streaming proxy' );
			return false;
		}

		// Allow plugins to hook before streaming (license/payment validation)
		do_action( 'ai_site_builder_before_stream', $endpoint, $payload );

		// Initialize cURL
		$ch = curl_init( $url );

		curl_setopt_array( $ch, array(
			CURLOPT_POST => true,
			CURLOPT_POSTFIELDS => wp_json_encode( $payload ),
			CURLOPT_HTTPHEADER => $headers,
			CURLOPT_RETURNTRANSFER => false,
			CURLOPT_TIMEOUT => self::TIMEOUT,
			CURLOPT_SSL_VERIFYPEER => true,
		) );

		// Track streaming status
		$had_error = false;
		$event_count = 0;

		// Set write function to forward SSE data to browser
		curl_setopt( $ch, CURLOPT_WRITEFUNCTION, function( $ch, $data ) use ( &$had_error, &$event_count, $callback ) {
			$len = strlen( $data );

			// Forward raw SSE data directly to browser
			echo $data;

			// Count events for logging
			if ( strpos( $data, 'event:' ) !== false ) {
				$event_count++;
			}

			// Check for error events
			if ( strpos( $data, 'event: error' ) !== false ) {
				$had_error = true;
			}

			// Call optional callback with data
			if ( $callback && is_callable( $callback ) ) {
				call_user_func( $callback, $data );
			}

			// Flush output to browser immediately
			if ( ob_get_level() > 0 ) {
				ob_flush();
			}
			flush();

			return $len;
		} );

		// Execute request
		$result = curl_exec( $ch );
		$http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
		$error = curl_error( $ch );

		curl_close( $ch );

		// Log results
		if ( $result === false || $http_code !== 200 ) {
			$this->logger->error( 'Streaming proxy failed', array(
				'http_code' => $http_code,
				'error' => $error,
				'endpoint' => $endpoint
			) );

			// Send error event to browser if not already sent
			if ( ! $had_error ) {
				echo "event: error\n";
				echo "data: " . wp_json_encode( array(
					'message' => 'Streaming failed: ' . ( $error ?: 'HTTP ' . $http_code )
				) ) . "\n\n";
				flush();
			}

			return false;
		}

		$this->logger->info( 'Streaming proxy completed', array(
			'endpoint' => $endpoint,
			'event_count' => $event_count,
			'had_error' => $had_error
		) );

		// Allow plugins to hook after streaming (usage tracking)
		do_action( 'ai_site_builder_after_stream', $endpoint, array(
			'success' => ! $had_error,
			'event_count' => $event_count
		) );

		return ! $had_error;
	}

	/**
	 * Call Vercel API with JSON request/response (non-streaming)
	 *
	 * @param string $endpoint API endpoint path
	 * @param array $payload Request payload
	 * @return array|WP_Error Response data or error
	 */
	public function call_api_json( $endpoint, $payload ) {
		// Block request if no valid license key
		if ( empty( $this->api_key ) ) {
			$this->logger->error( 'No license key found - request blocked', array(
				'endpoint' => $endpoint,
				'user_id' => get_current_user_id()
			) );
			return new WP_Error(
				'no_license',
				'No valid license key found. Please activate a plan at Dreamformer → Pricing to use AI features.',
				array( 'required_action' => 'activate_license' )
			);
		}

		$url = $this->base_url . $endpoint;

		$this->logger->debug( 'Making JSON API request', array(
			'url' => $url,
			'endpoint' => $endpoint
		) );

		// Get user's monthly credits and generate signature
		$monthly_credits = $this->get_user_monthly_credits();
		$credits_signature = $this->generate_plan_signature( (string) $monthly_credits );

		$headers = array(
			'Content-Type' => 'application/json',
			'Accept' => 'application/json',
			'Authorization' => 'Bearer ' . $this->api_key,
			'X-WordPress-URL' => home_url(),
			'X-Monthly-Credits' => $monthly_credits,
			'X-Plan-Signature' => $credits_signature
		);

		$this->logger->debug( '✅ API key, monthly credits, and signature added to request headers', array(
			'key_length' => strlen( $this->api_key ),
			'wordpress_url' => home_url(),
			'monthly_credits' => $monthly_credits,
			'signature_length' => strlen( $credits_signature )
		) );

		$args = array(
			'method' => 'POST',
			'timeout' => 30, // Short timeout (endpoint returns immediately for polling)
			'headers' => $headers,
			'body' => wp_json_encode( $payload ),
			'sslverify' => true
		);

		$response = wp_remote_post( $url, $args );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'HTTP request failed', array(
				'endpoint' => $endpoint,
				'error' => $response->get_error_message()
			) );
			return new WP_Error(
				'http_error',
				sprintf( 'Failed to connect to Vercel API: %s', $response->get_error_message() )
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		$this->logger->debug( 'API response received', array(
			'status_code' => $response_code,
			'body_length' => strlen( $response_body )
		) );

		// Handle non-202 responses (202 Accepted is expected for polling job creation)
		if ( $response_code !== 200 && $response_code !== 202 ) {
			$error_message = 'API request failed';

			// Try to parse error from response
			$decoded = json_decode( $response_body, true );
			if ( isset( $decoded['error'] ) ) {
				$error_message = $decoded['error'];
			} elseif ( isset( $decoded['message'] ) ) {
				$error_message = $decoded['message'];
			}

			$this->logger->error( 'API returned error', array(
				'status_code' => $response_code,
				'error' => $error_message
			) );

			return new WP_Error(
				'api_error',
				sprintf( 'Vercel API error (HTTP %d): %s', $response_code, $error_message )
			);
		}

		// Parse JSON response
		$decoded = json_decode( $response_body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Failed to parse API response', array(
				'json_error' => json_last_error_msg(),
				'response_preview' => substr( $response_body, 0, 200 )
			) );
			return new WP_Error( 'json_error', 'Failed to parse API response' );
		}

		return $decoded;
	}

	/**
	 * Poll job status from Vercel API
	 *
	 * @param string $job_id Job ID
	 * @return array|WP_Error Status data or error
	 */
	public function poll_job_status( $job_id ) {
		$url = $this->base_url . '/api/generate-page-content-polling/' . $job_id . '/status';

		$headers = array(
			'Accept' => 'application/json'
		);

		$this->logger->debug( 'Polling job status', array(
			'job_id' => $job_id,
			'url' => $url
		) );

		$response = wp_remote_get( $url, array(
			'timeout' => 10,
			'headers' => $headers,
			'sslverify' => true
		) );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Poll request failed', array(
				'job_id' => $job_id,
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		$http_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		if ( $http_code !== 200 ) {
			$this->logger->error( 'Poll returned error', array(
				'job_id' => $job_id,
				'http_code' => $http_code
			) );
			return new WP_Error(
				'poll_error',
				sprintf( 'Poll failed (HTTP %d)', $http_code )
			);
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Failed to parse poll response', array(
				'job_id' => $job_id,
				'json_error' => json_last_error_msg()
			) );
			return new WP_Error(
				'json_parse_error',
				'Failed to parse poll response'
			);
		}

		return $data;
	}

	/**
	 * Poll job status from Vercel API (generic for page gen and edit)
	 *
	 * @param string $job_id Job ID
	 * @param string $type Job type: 'page' or 'edit'
	 * @return array|WP_Error Status data or error
	 */
	public function poll_job_status_generic( $job_id, $type = 'page' ) {
		$endpoint = $type === 'edit'
			? '/api/generate-page-diff-polling/' . $job_id . '/status'
			: '/api/generate-page-content-polling/' . $job_id . '/status';

		$url = $this->base_url . $endpoint;

		$this->logger->debug( 'Polling job status', array(
			'job_id' => $job_id,
			'type' => $type,
			'url' => $url
		) );

		$response = wp_remote_get( $url, array(
			'timeout' => 10,
			'headers' => array( 'Accept' => 'application/json' ),
			'sslverify' => true
		) );

		if ( is_wp_error( $response ) ) {
			$this->logger->error( 'Poll request failed', array(
				'job_id' => $job_id,
				'type' => $type,
				'error' => $response->get_error_message()
			) );
			return $response;
		}

		$http_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		if ( $http_code !== 200 ) {
			$this->logger->error( 'Poll returned error', array(
				'job_id' => $job_id,
				'type' => $type,
				'http_code' => $http_code
			) );
			return new WP_Error(
				'poll_error',
				sprintf( 'Poll failed (HTTP %d)', $http_code )
			);
		}

		$data = json_decode( $body, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->error( 'Failed to parse poll response', array(
				'job_id' => $job_id,
				'type' => $type,
				'json_error' => json_last_error_msg()
			) );
			return new WP_Error(
				'json_parse_error',
				'Failed to parse poll response'
			);
		}

		return $data;
	}

}