<?php
/**
 * AI Site Builder - Page Post Types
 * Handles WordPress page integration for AI-generated content
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Page_Post_Types {

    public function __construct() {
        add_action( 'init', [ $this, 'register_meta_fields' ] );
        add_action( 'add_meta_boxes', [ $this, 'add_page_meta_boxes' ] );
        add_action( 'save_post', [ $this, 'save_page_meta' ] );
    }

    /**
     * Register meta fields for AI-generated pages
     */
    public function register_meta_fields() {
        // AI generation prompt
        register_meta( 'post', 'ai_generation_prompt', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // AI generated content blocks
        register_meta( 'post', 'ai_content_blocks', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // AI generation metadata
        register_meta( 'post', 'ai_generation_metadata', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // Page type (landing, about, contact, etc.)
        register_meta( 'post', 'ai_page_type', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // Business information used for generation
        register_meta( 'post', 'ai_business_info', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // AI model used for generation
        register_meta( 'post', 'ai_model_used', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // Generation timestamp
        register_meta( 'post', 'ai_generated_at', [
            'object_subtype' => 'page',
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // Flag to identify AI-generated pages
        register_meta( 'post', 'ai_generated_page', [
            'object_subtype' => 'page',
            'type' => 'boolean',
            'single' => true,
            'show_in_rest' => true,
            'auth_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);
    }

    /**
     * Add meta box to page edit screen
     */
    public function add_page_meta_boxes() {
        add_meta_box(
            'ai-page-info',
            __( 'AI Generation Info', 'ai-site-builder' ),
            [ $this, 'render_page_meta_box' ],
            'page',
            'side',
            'high'
        );
    }

    /**
     * Render AI page information meta box
     */
    public function render_page_meta_box( $post ) {
        $is_ai_generated = get_post_meta( $post->ID, 'ai_generated_page', true );

        if ( ! $is_ai_generated ) {
            echo '<p>' . __( 'This page was not generated by AI.', 'ai-site-builder' ) . '</p>';
            echo '<p><a href="' . admin_url( 'admin.php?page=dreamformer' ) . '" class="button">' . __( 'Generate with AI', 'ai-site-builder' ) . '</a></p>';
            return;
        }

        $generation_prompt = get_post_meta( $post->ID, 'ai_generation_prompt', true );
        $page_type = get_post_meta( $post->ID, 'ai_page_type', true );
        $model_used = get_post_meta( $post->ID, 'ai_model_used', true );
        $generated_at = get_post_meta( $post->ID, 'ai_generated_at', true );

        echo '<div class="ai-page-meta">';

        if ( $generation_prompt ) {
            echo '<p><strong>' . __( 'Original Prompt:', 'ai-site-builder' ) . '</strong></p>';
            echo '<p class="description">' . esc_html( $generation_prompt ) . '</p>';
        }

        if ( $page_type ) {
            echo '<p><strong>' . __( 'Page Type:', 'ai-site-builder' ) . '</strong> ' . esc_html( ucfirst( $page_type ) ) . '</p>';
        }

        if ( $model_used ) {
            echo '<p><strong>' . __( 'AI Model:', 'ai-site-builder' ) . '</strong> ' . esc_html( $model_used ) . '</p>';
        }

        if ( $generated_at ) {
            echo '<p><strong>' . __( 'Generated:', 'ai-site-builder' ) . '</strong> ' . esc_html( date( 'M j, Y g:i A', strtotime( $generated_at ) ) ) . '</p>';
        }

        echo '<p style="margin-top: 15px;">';
        echo '<a href="' . admin_url( 'admin.php?page=dreamformer&edit=page&id=' . $post->ID ) . '" class="button button-primary">' . __( 'Edit with AI', 'ai-site-builder' ) . '</a>';
        echo '</p>';

        echo '</div>';

        // Add some basic styling
        echo '<style>
            .ai-page-meta .description {
                font-style: italic;
                background: #f9f9f9;
                padding: 8px;
                border-radius: 3px;
                max-height: 60px;
                overflow-y: auto;
            }
        </style>';
    }

    /**
     * Save page meta (placeholder for future functionality)
     */
    public function save_page_meta( $post_id ) {
        // Meta is handled via REST API and direct updates
        // This is here for WordPress compatibility
    }

    /**
     * Get AI-generated pages
     */
    public static function get_ai_pages( $args = [] ) {
        $default_args = [
            'post_type' => 'page',
            'post_status' => [ 'draft', 'publish', 'pending', 'future', 'private' ],
            'meta_query' => [
                [
                    'key' => 'ai_generated_page',
                    'value' => true,
                    'compare' => '='
                ]
            ],
            'posts_per_page' => -1,
            'orderby' => 'date',
            'order' => 'DESC',
            'suppress_filters' => false,
            'no_found_rows' => false,
            'cache_results' => false, // Disable caching to ensure fresh results
            'update_post_meta_cache' => true,
            'update_post_term_cache' => false
        ];

        $args = wp_parse_args( $args, $default_args );

        return get_posts( $args );
    }

    /**
     * Check if a page is AI-generated
     */
    public static function is_ai_generated_page( $page_id ) {
        return (bool) get_post_meta( $page_id, 'ai_generated_page', true );
    }
}