<?php
/**
 * AI Site Builder - Page Generator
 * Handles full page content generation using AI
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Page_Generator {

    private $logger;
    private $vercel_client;
    private $file_editor;

    public function __construct() {
        $this->logger = AI_Logger::get_instance( 'AI_Page_Generator' );

        // Reuse existing Vercel client if available
        if ( class_exists( 'AI_Vercel_Client' ) ) {
            $this->vercel_client = new AI_Vercel_Client();
        }

        // Initialize file editor to reuse diff application logic
        if ( class_exists( 'AI_Simple_File_Editor' ) ) {
            $this->file_editor = new AI_Simple_File_Editor();
        }

        $this->init_hooks();
    }

    private function init_hooks() {
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
        add_filter( 'wp_insert_post_data', [ $this, 'protect_ai_content_on_save' ], 10, 2 );
        add_action( 'admin_notices', [ $this, 'show_ai_page_protection_notice' ] );
        add_action( 'edit_form_after_title', [ $this, 'replace_ai_page_editor' ] );
        add_action( 'admin_head', [ $this, 'hide_ai_page_editor' ] );
        add_action( 'add_meta_boxes', [ $this, 'hide_ai_page_meta_boxes' ] );

        // PRODUCTION FIX: Enqueue CSS/JS from files (bypasses all host sanitization)
        // Industry-standard approach used by Elementor, Divi, Beaver Builder
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_ai_page_assets' ] );

        // Clear caches when page is restored from trash
        add_action( 'untrashed_post', [ $this, 'handle_page_restored' ] );

        // Cleanup split files on permanent deletion (not trash)
        add_action( 'before_delete_post', [ $this, 'cleanup_split_files_on_permanent_delete' ] );

        // Also cleanup when WordPress automatically deletes old trash items
        add_action( 'wp_scheduled_delete', [ $this, 'cleanup_split_files_for_scheduled_delete' ] );

    }

    /**
     * Register REST API routes for page generation
     */
    public function register_rest_routes() {
        // Delete page
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)', [
            'methods' => 'DELETE',
            'callback' => [ $this, 'rest_delete_page' ],
            'permission_callback' => function() {
                return current_user_can( 'delete_posts' );
            },
            'args' => [
                'page_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'Page ID to delete'
                ]
            ]
        ]);

        // Duplicate page
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/duplicate', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_duplicate_page' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'page_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'Page ID to duplicate'
                ]
            ]
        ]);

        // Get page content blocks
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/blocks', [
            'methods' => 'GET',
            'callback' => [ $this, 'rest_get_page_blocks' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // List AI pages
        register_rest_route( 'ai-builder/v1', '/pages', [
            'methods' => 'GET',
            'callback' => [ $this, 'rest_list_pages' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);

        // Smart edit endpoint - can modify page + theme when needed
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/smart-edit', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_smart_edit_page' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'prompt' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'AI prompt for smart page editing'
                ],
                'context' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Additional context for the request'
                ]
            ]
        ]);

        // Streaming page generation - prepare
        register_rest_route( 'ai-builder/v1', '/pages/generate-stream', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_generate_page_stream' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'prompt' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'User prompt for page generation'
                ],
                'page_type' => [
                    'required' => false,
                    'type' => 'string',
                    'enum' => [ 'landing', 'about', 'contact', 'app', 'cart', 'shop', 'checkout', 'blog', 'services', 'custom' ],
                    'default' => 'custom'
                ],
                'reference_page_id' => [
                    'required' => false,
                    'type' => ['integer', 'string'],
                    'description' => 'Reference page ID for brand consistency',
                    'sanitize_callback' => function( $value ) {
                        if ( empty( $value ) || $value === '' ) {
                            return null;
                        }
                        return is_numeric( $value ) ? (int) $value : null;
                    }
                ],
                'business_info' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Business information for context'
                ],
                'attachment' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'File attachment (PDF or image)',
                    'validate_callback' => [ $this, 'validate_attachment' ]
                ]
            ]
        ]);

        // Create page from stream result
        register_rest_route( 'ai-builder/v1', '/pages/create-from-stream', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_create_page_from_stream' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'result' => [
                    'required' => true,
                    'type' => 'object',
                    'description' => 'Generation result from Vercel stream'
                ],
                'prompt' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'Original user prompt'
                ],
                'page_type' => [
                    'required' => false,
                    'type' => 'string',
                    'default' => 'custom'
                ],
                'business_info' => [
                    'required' => false,
                    'type' => 'object'
                ]
            ]
        ]);

        // Export page as JSON
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/export', [
            'methods' => 'GET',
            'callback' => [ $this, 'rest_export_page' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'page_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'Page ID to export'
                ]
            ]
        ]);

        // Import page from JSON
        register_rest_route( 'ai-builder/v1', '/pages/import', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_import_page' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'data' => [
                    'required' => true,
                    'type' => 'object',
                    'description' => 'Exported page data'
                ]
            ]
        ]);

        // Polling-based page generation - create job
        register_rest_route( 'ai-builder/v1', '/pages/generate-polling', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_generate_page_polling' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'prompt' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'User prompt for page generation'
                ],
                'page_type' => [
                    'required' => false,
                    'type' => 'string',
                    'enum' => [ 'landing', 'about', 'contact', 'app', 'cart', 'shop', 'checkout', 'blog', 'services', 'custom' ],
                    'default' => 'custom'
                ],
                'reference_page_id' => [
                    'required' => false,
                    'type' => ['integer', 'string'],
                    'description' => 'Reference page ID for brand consistency',
                    'sanitize_callback' => function( $value ) {
                        if ( empty( $value ) || $value === '' ) {
                            return null;
                        }
                        return is_numeric( $value ) ? (int) $value : null;
                    }
                ],
                'business_info' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Business information for context'
                ],
                'attachment' => [
                    'required' => false,
                    'type' => 'object',
                    'description' => 'File attachment (PDF or image)'
                ]
            ]
        ]);

        // Polling status proxy - WordPress proxies Vercel status checks
        register_rest_route( 'ai-builder/v1', '/pages/poll-status/(?P<job_id>[a-f0-9\-]+)', [
            'methods' => 'GET',
            'callback' => [ $this, 'rest_poll_job_status' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            },
            'args' => [
                'job_id' => [
                    'required' => true,
                    'type' => 'string',
                    'description' => 'Polling job ID (UUID)',
                    'validate_callback' => function( $param ) {
                        // Validate UUID format
                        return preg_match('/^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i', $param);
                    }
                ]
            ]
        ]);

        // Recover split files from database
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/recover-split-files', [
            'methods' => 'POST',
            'callback' => [ $this, 'rest_recover_split_files' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => [
                'page_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'Page ID to recover split files for'
                ]
            ]
        ]);

        // Setup wizard: Activate theme
        register_rest_route( 'ai-builder/v1', '/setup/activate-theme', [
            'methods' => 'POST',
            'callback' => [ $this, 'setup_activate_theme' ],
            'permission_callback' => function() {
                return current_user_can( 'manage_options' );
            }
        ]);

        // Setup wizard: Dismiss Quick Start box
        register_rest_route( 'ai-builder/v1', '/setup/dismiss-quick-start', [
            'methods' => 'POST',
            'callback' => [ $this, 'setup_dismiss_quick_start' ],
            'permission_callback' => function() {
                return current_user_can( 'edit_posts' );
            }
        ]);
    }


    /**
     * REST endpoint: Get page content blocks
     */
    public function rest_get_page_blocks( $request ) {
        $page_id = (int) $request->get_param( 'page_id' );

        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Page not found'
            ], 404 );
        }

        $content_blocks = get_post_meta( $page_id, 'ai_content_blocks', true );

        // Handle legacy JSON string format for backward compatibility
        if ( is_string( $content_blocks ) ) {
            $content_blocks = json_decode( $content_blocks, true );
        }

        // Ensure we always have an array
        $content_blocks = is_array( $content_blocks ) ? $content_blocks : [];

        return new WP_REST_Response( [
            'success' => true,
            'content_blocks' => $content_blocks,
            'full_content' => $page->post_content
        ] );
    }

    /**
     * REST endpoint: List AI pages
     */
    public function rest_list_pages( $request ) {
        $pages = AI_Page_Post_Types::get_ai_pages();

        $formatted_pages = array_map( function( $page ) {
            return [
                'id' => $page->ID,
                'title' => $page->post_title,
                'status' => $page->post_status,
                'date' => $page->post_date,
                'page_type' => get_post_meta( $page->ID, 'ai_page_type', true ),
                'edit_url' => admin_url( 'post.php?post=' . $page->ID . '&action=edit' ),
                'preview_url' => get_permalink( $page->ID )
            ];
        }, $pages );

        return new WP_REST_Response( [
            'success' => true,
            'pages' => $formatted_pages
        ] );
    }

    /**
     * REST endpoint: Smart edit page (can modify page + theme when needed)
     */
    public function rest_smart_edit_page( $request ) {
        $page_id = (int) $request->get_param( 'page_id' );
        $prompt = $request->get_param( 'prompt' );
        $context = $request->get_param( 'context' ) ?: [];

        $this->logger->info( 'Smart edit request received', [
            'page_id' => $page_id,
            'prompt' => substr( $prompt, 0, 100 ) . '...'
        ] );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Page not found'
            ], 404 );
        }

        // Use Dreamformer page editor directly (AI_Smart_Page_Editor removed - was redundant)
        $dreamformer_editor = new AI_Dreamformer_Page_Editor();

        // Execute page edit directly
        $result = $dreamformer_editor->process_edit_request( $page_id, $prompt, $context );

        if ( is_wp_error( $result ) ) {
            $this->logger->error( 'Smart edit failed', [
                'error' => $result->get_error_message(),
                'page_id' => $page_id
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'error' => $result->get_error_message()
            ], 500 );
        }

        $this->logger->info( 'Smart edit completed successfully', [
            'page_id' => $page_id,
            'changes_applied' => $result['changes_count'],
            'systems_modified' => $result['systems'],
            'edit_type' => $result['edit_type']
        ] );

        $response_data = [
            'success' => true,
            'changes_applied' => $result['changes_count'],
            'systems_modified' => $result['systems'],
            'edit_type' => $result['edit_type']
        ];

        // Include conversational message if available
        if ( ! empty( $result['user_message'] ) ) {
            $response_data['message'] = $result['user_message'];
        }

        return new WP_REST_Response( $response_data );
    }

    /**
     * REST endpoint: Generate page with streaming (prepare)
     */
    public function rest_generate_page_stream( $request ) {
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $page_type = $request->get_param( 'page_type' );
        $business_info = $request->get_param( 'business_info' );
        $reference_page_id = $request->get_param( 'reference_page_id' );
        $attachment = $request->get_param( 'attachment' );

        $log_data = [
            'prompt' => substr( $prompt, 0, 100 ),
            'page_type' => $page_type,
            'reference_page_id' => $reference_page_id
        ];

        if ( $attachment ) {
            $log_data['attachment'] = [
                'filename' => $attachment['filename'],
                'mime_type' => $attachment['mime_type'],
                'size' => $attachment['size']
            ];
        }

        $this->logger->info( 'Preparing streaming page generation', $log_data );

        // CRITICAL: Check user has credits BEFORE calling Vercel API
        $credits_check = $this->vercel_client->check_user_credits( 'page_generation' );
        if ( is_wp_error( $credits_check ) ) {
            // Distinguish between auth failures (401) and insufficient credits (402)
            $error_code = $credits_check->get_error_code();
            $http_status = ( $error_code === 'insufficient_credits' ) ? 402 : 401;

            return new WP_REST_Response( [
                'success' => false,
                'message' => $credits_check->get_error_message(),
                'code' => $error_code
            ], $http_status );
        }

        try {
            // Get theme context files
            $theme_files = $this->get_theme_context_files( $prompt );

            if ( is_wp_error( $theme_files ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => $theme_files->get_error_message()
                ], 500 );
            }

            // Build brand reference if provided (includes HTML, CSS, and JS)
            $brand_reference = null;
            if ( $reference_page_id ) {
                $reference_page = get_post( $reference_page_id );
                if ( $reference_page && $reference_page->post_type === 'page' ) {
                    $upload_dir = wp_upload_dir();
                    $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $reference_page_id;
                    $has_split_files = get_post_meta( $reference_page_id, '_ai_split_files', true );

                    // Build complete reference content with all files
                    $reference_parts = ["=== HTML ===\n" . $reference_page->post_content];

                    // Include CSS and JS if split files exist
                    if ( $has_split_files && file_exists( $page_dir ) ) {
                        $css_path = $page_dir . '/styles.css';
                        $js_path = $page_dir . '/app.js';

                        if ( file_exists( $css_path ) ) {
                            $css_content = file_get_contents( $css_path );
                            $reference_parts[] = "=== CSS ===\n" . $css_content;
                        }

                        if ( file_exists( $js_path ) ) {
                            $js_content = file_get_contents( $js_path );
                            $reference_parts[] = "=== JS ===\n" . $js_content;
                        }
                    }

                    $brand_reference = [
                        'content' => implode( "\n\n", $reference_parts ),
                        'instruction' => "Use the provided page as a brand reference. The reference includes complete HTML, CSS, and JavaScript. CRITICAL: Match the CSS styling EXACTLY - same colors, typography, spacing, animations, and design patterns. Create NEW content and structure appropriate for a {$page_type} page, but maintain the EXACT same visual aesthetic."
                    ];

                    $this->logger->info( 'Added complete reference page to generation', [
                        'reference_page_id' => $reference_page_id,
                        'reference_title' => $reference_page->post_title,
                        'has_css' => strpos( $brand_reference['content'], '=== CSS ===' ) !== false,
                        'has_js' => strpos( $brand_reference['content'], '=== JS ===' ) !== false,
                        'total_size' => strlen( $brand_reference['content'] )
                    ]);
                }
            }

            // Build payload for Vercel
            $payload = [
                'user_prompt' => $prompt,
                'page_type' => $page_type,
                'business_info' => $business_info ?: [],
                'theme_files' => $theme_files,
                'brand_reference' => $brand_reference
            ];

            // Add attachment if provided
            if ( $attachment ) {
                $payload['attachment'] = $attachment;
            }

            // Stream response from Vercel through WordPress (secure proxy with API key)
            $this->logger->info( 'Starting streaming page generation proxy' );

            // Proxy the streaming request
            $success = $this->vercel_client->proxy_stream_to_browser(
                '/api/generate-page-content-stream',
                $payload
            );

            // If streaming failed, log it (error already sent to browser)
            if ( ! $success ) {
                $this->logger->error( 'Page generation streaming failed' );
            }

            // Exit to prevent WordPress from adding extra output
            exit;

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to prepare stream', [
                'error' => $e->getMessage()
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to prepare: ' . $e->getMessage()
            ], 500 );
        }
    }

    /**
     * REST endpoint: Generate page with polling
     * Returns job_id immediately, client polls for progress
     */
    public function rest_generate_page_polling( $request ) {
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $page_type = $request->get_param( 'page_type' );
        $business_info = $request->get_param( 'business_info' );
        $reference_page_id = $request->get_param( 'reference_page_id' );
        $attachment = $request->get_param( 'attachment' );

        $log_data = [
            'prompt' => substr( $prompt, 0, 100 ),
            'page_type' => $page_type,
            'reference_page_id' => $reference_page_id
        ];

        if ( $attachment ) {
            $log_data['attachment'] = [
                'filename' => $attachment['filename'],
                'mime_type' => $attachment['mime_type'],
                'size' => $attachment['size']
            ];
        }

        $this->logger->info( 'Starting polling-based page generation', $log_data );

        // CRITICAL: Check user has credits BEFORE calling Vercel API
        $credits_check = $this->vercel_client->check_user_credits( 'page_generation' );
        if ( is_wp_error( $credits_check ) ) {
            // Distinguish between auth failures and insufficient credits
            // Possible error codes from check_user_credits():
            //   - 'no_user' → 401 Unauthorized (no valid license)
            //   - 'insufficient_credits' → 402 Payment Required (credits exhausted)
            $error_code = $credits_check->get_error_code();
            $http_status = ( $error_code === 'insufficient_credits' ) ? 402 : 401;

            return new WP_REST_Response( [
                'success' => false,
                'message' => $credits_check->get_error_message(),
                'code' => $error_code
            ], $http_status );
        }

        try {
            // Get theme context files (SAME as streaming)
            $theme_files = $this->get_theme_context_files( $prompt );

            if ( is_wp_error( $theme_files ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => $theme_files->get_error_message()
                ], 500 );
            }

            // Build brand reference if provided (SAME as streaming)
            $brand_reference = null;
            if ( $reference_page_id ) {
                $reference_page = get_post( $reference_page_id );
                if ( $reference_page && $reference_page->post_type === 'page' ) {
                    $upload_dir = wp_upload_dir();
                    $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $reference_page_id;
                    $has_split_files = get_post_meta( $reference_page_id, '_ai_split_files', true );

                    // Build complete reference content with all files
                    $reference_parts = ["=== HTML ===\n" . $reference_page->post_content];

                    // Include CSS and JS if split files exist
                    if ( $has_split_files && file_exists( $page_dir ) ) {
                        $css_path = $page_dir . '/styles.css';
                        $js_path = $page_dir . '/app.js';

                        if ( file_exists( $css_path ) ) {
                            $css_content = file_get_contents( $css_path );
                            $reference_parts[] = "=== CSS ===\n" . $css_content;
                        }

                        if ( file_exists( $js_path ) ) {
                            $js_content = file_get_contents( $js_path );
                            $reference_parts[] = "=== JS ===\n" . $js_content;
                        }
                    }

                    $brand_reference = [
                        'content' => implode( "\n\n", $reference_parts ),
                        'instruction' => "Use the provided page as a brand reference. The reference includes complete HTML, CSS, and JavaScript. CRITICAL: Match the CSS styling EXACTLY - same colors, typography, spacing, animations, and design patterns. Create NEW content and structure appropriate for a {$page_type} page, but maintain the EXACT same visual aesthetic."
                    ];

                    $this->logger->info( 'Added complete reference page to generation', [
                        'reference_page_id' => $reference_page_id,
                        'reference_title' => $reference_page->post_title,
                        'has_css' => strpos( $brand_reference['content'], '=== CSS ===' ) !== false,
                        'has_js' => strpos( $brand_reference['content'], '=== JS ===' ) !== false,
                        'total_size' => strlen( $brand_reference['content'] )
                    ]);
                }
            }

            // Build payload
            $payload = [
                'user_prompt' => $prompt,
                'page_type' => $page_type,
                'business_info' => $business_info ?: [],
                'theme_files' => $theme_files,
                'brand_reference' => $brand_reference
            ];

            // Add attachment if provided
            if ( $attachment ) {
                $payload['attachment'] = $attachment;
            }

            // Call polling endpoint (returns 202 with job_id)
            $response = $this->vercel_client->call_api_json(
                '/api/generate-page-content-polling',
                $payload
            );

            if ( is_wp_error( $response ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => $response->get_error_message()
                ], 500 );
            }

            $this->logger->info( 'Polling job created successfully', [
                'job_id' => $response['job_id']
            ] );

            // Return job_id to client
            return new WP_REST_Response( [
                'success' => true,
                'job_id' => $response['job_id'],
                'status' => $response['status'],
                'poll_url' => $response['poll_url'],
                'poll_interval_ms' => 2000
            ], 202 );

        } catch ( Exception $e ) {
            $this->logger->error( 'Polling generation preparation failed', [
                'error' => $e->getMessage()
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to prepare: ' . $e->getMessage()
            ], 500 );
        }
    }

    /**
     * REST endpoint: Poll job status (WordPress proxy)
     * Proxies status checks to Vercel API to avoid direct Vercel calls from JavaScript
     */
    public function rest_poll_job_status( $request ) {
        // CRITICAL: Disable ALL caching plugins for this polling endpoint
        // LiteSpeed Cache - official API to disable caching
        do_action( 'litespeed_control_set_nocache', 'Dreamformer polling endpoint - must return fresh data' );

        // WP Super Cache, W3 Total Cache, and others
        if ( ! defined( 'DONOTCACHEPAGE' ) ) {
            define( 'DONOTCACHEPAGE', true );
        }

        $job_id = $request->get_param( 'job_id' );

        // Use Vercel client to poll status
        $status_data = $this->vercel_client->poll_job_status( $job_id );

        if ( is_wp_error( $status_data ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $status_data->get_error_message()
            ], 500 );
        }

        // Pass through the status data from Vercel with no-cache headers
        $response = new WP_REST_Response( $status_data, 200 );

        // Prevent caching by ALL proxies and browsers
        $response->header( 'Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0, private' );
        $response->header( 'Pragma', 'no-cache' );
        $response->header( 'Expires', '0' );
        $response->header( 'X-LiteSpeed-Cache-Control', 'no-cache' ); // LiteSpeed server-level

        return $response;
    }

    /**
     * REST endpoint: Create page from streaming result
     */
    public function rest_create_page_from_stream( $request ) {
        $result = $request->get_param( 'result' );
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $page_type = $request->get_param( 'page_type' );
        $business_info = $request->get_param( 'business_info' );

        $this->logger->info( 'Creating page from stream result', [
            'has_files' => isset( $result['files'] ),
            'page_type' => $page_type
        ]);

        try {
            // Check if new split files format or old monolithic format
            if ( isset( $result['files'] ) ) {
                // New split files format
                $page_id = $this->create_wordpress_page_with_files( $result, $prompt, $page_type, $business_info );
            } else {
                // Old monolithic format (backward compatibility)
                $page_id = $this->create_wordpress_page( $result, $prompt, $page_type, $business_info );
            }

            if ( is_wp_error( $page_id ) ) {
                $this->logger->error( 'Page creation failed', [
                    'error' => $page_id->get_error_message()
                ] );

                return new WP_REST_Response( [
                    'success' => false,
                    'message' => $page_id->get_error_message()
                ], 500 );
            }

            $this->logger->info( 'Page created successfully from stream', [
                'page_id' => $page_id
            ] );

            // Deduct credits after successful page generation (8 credits)
            $credit_manager = new Dreamformer_Credit_Manager();
            $user_id = get_current_user_id();
            $deduct_result = $credit_manager->deduct_credits( $user_id, 8, 'page_generation' );

            if ( is_wp_error( $deduct_result ) ) {
                $this->logger->warning( 'Credit deduction failed after successful page generation', [
                    'page_id' => $page_id,
                    'error' => $deduct_result->get_error_message()
                ] );
                // Continue anyway - page was created successfully, credit sync issue won't block user
            }

            return new WP_REST_Response( [
                'success' => true,
                'page_id' => $page_id,
                'edit_url' => admin_url( 'post.php?post=' . $page_id . '&action=edit' ),
                'preview_url' => get_permalink( $page_id ),
                'dreamformer_url' => admin_url( 'admin.php?page=dreamformer&edit=page&id=' . $page_id )
            ] );

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to create page from stream', [
                'error' => $e->getMessage()
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to create page: ' . $e->getMessage()
            ], 500 );
        }
    }

    /**
     * REST endpoint: Delete page
     */
    public function rest_delete_page( $request ) {
        $page_id = (int) $request->get_param( 'page_id' );

        $this->logger->info( 'Page deletion request', [
            'page_id' => $page_id
        ]);

        // Verify page exists and is a page
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Page not found'
            ], 404 );
        }

        // Move page to trash (standard WordPress behavior)
        $deleted = wp_trash_post( $page_id );

        if ( ! $deleted ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Failed to move page to trash'
            ], 500 );
        }

        $this->logger->info( 'Page moved to trash successfully', [
            'page_id' => $page_id,
            'title' => $page->post_title
        ]);

        return new WP_REST_Response( [
            'success' => true,
            'message' => 'Page moved to trash successfully'
        ] );
    }

    /**
     * REST endpoint: Duplicate page
     */
    public function rest_duplicate_page( $request ) {
        $page_id = (int) $request->get_param( 'page_id' );

        $this->logger->info( 'Page duplication request', [
            'page_id' => $page_id
        ]);

        // Verify page exists and is a page
        $original_page = get_post( $page_id );
        if ( ! $original_page || $original_page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Page not found'
            ], 404 );
        }

        // Get all post meta
        $meta_data = get_post_meta( $page_id );

        // Create duplicate page data
        $duplicate_data = [
            'post_title' => $original_page->post_title . ' (Copy)',
            'post_content' => $original_page->post_content,
            'post_status' => 'draft', // Always create as draft
            'post_type' => 'page',
            'post_excerpt' => $original_page->post_excerpt,
            'post_author' => get_current_user_id()
        ];

        // Insert the duplicate page
        $duplicate_id = wp_insert_post( $duplicate_data );

        if ( is_wp_error( $duplicate_id ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Failed to create duplicate page'
            ], 500 );
        }

        // Copy all meta data
        foreach ( $meta_data as $meta_key => $meta_values ) {
            foreach ( $meta_values as $meta_value ) {
                add_post_meta( $duplicate_id, $meta_key, maybe_unserialize( $meta_value ) );
            }
        }

        // COMPREHENSIVE FIX: Copy physical files if page uses split files
        $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
        if ( $has_split_files ) {
            $this->logger->info( 'Cloning split files for page', [
                'original_page_id' => $page_id,
                'duplicate_page_id' => $duplicate_id
            ]);

            $files_copied = $this->copy_page_files( $page_id, $duplicate_id );

            if ( is_wp_error( $files_copied ) ) {
                $this->logger->error( 'Failed to copy page files', [
                    'error' => $files_copied->get_error_message()
                ]);

                // Delete the duplicate page since files didn't copy
                wp_delete_post( $duplicate_id, true );

                return new WP_REST_Response( [
                    'success' => false,
                    'error' => 'Failed to copy page files: ' . $files_copied->get_error_message()
                ], 500 );
            }

            $this->logger->info( 'Page files copied successfully', [
                'files_copied' => $files_copied
            ]);
        }

        $this->logger->info( 'Page duplicated successfully', [
            'original_page_id' => $page_id,
            'duplicate_page_id' => $duplicate_id,
            'original_title' => $original_page->post_title,
            'has_split_files' => $has_split_files,
            'files_copied' => $has_split_files ? $files_copied : 0
        ]);

        return new WP_REST_Response( [
            'success' => true,
            'message' => 'Page duplicated successfully',
            'duplicate_id' => $duplicate_id,
            'edit_url' => admin_url( 'post.php?post=' . $duplicate_id . '&action=edit' ),
            'preview_url' => get_permalink( $duplicate_id ),
            'files_copied' => $has_split_files ? $files_copied : 0
        ] );
    }

    /**
     * Copy all physical files from one page to another
     *
     * Copies both live and draft directories with all contents
     *
     * @param int $source_page_id Original page ID
     * @param int $target_page_id Duplicate page ID
     * @return int|WP_Error Number of files copied or error
     */
    private function copy_page_files( $source_page_id, $target_page_id ) {
        $upload_dir = wp_upload_dir();
        $base_dir = $upload_dir['basedir'] . '/ai-pages';

        $source_live = $base_dir . '/page-' . $source_page_id;
        $source_draft = $base_dir . '/page-' . $source_page_id . '-draft';
        $target_live = $base_dir . '/page-' . $target_page_id;
        $target_draft = $base_dir . '/page-' . $target_page_id . '-draft';

        $files_copied = 0;

        // Copy live directory (required)
        if ( ! is_dir( $source_live ) ) {
            return new WP_Error( 'missing_source', 'Source page directory not found: ' . $source_live );
        }

        $copy_result = $this->recursive_copy( $source_live, $target_live );
        if ( is_wp_error( $copy_result ) ) {
            return $copy_result;
        }
        $files_copied += $copy_result;

        // Copy draft directory (optional - only if exists)
        if ( is_dir( $source_draft ) ) {
            $draft_copy_result = $this->recursive_copy( $source_draft, $target_draft );
            if ( is_wp_error( $draft_copy_result ) ) {
                // Log warning but don't fail - draft is optional
                $this->logger->warning( 'Failed to copy draft directory', [
                    'error' => $draft_copy_result->get_error_message()
                ]);
            } else {
                $files_copied += $draft_copy_result;
            }
        }

        return $files_copied;
    }

    /**
     * Recursively copy directory and all contents
     *
     * @param string $source Source directory path
     * @param string $target Target directory path
     * @return int|WP_Error Number of files copied or error
     */
    private function recursive_copy( $source, $target ) {
        if ( ! is_dir( $source ) ) {
            return new WP_Error( 'invalid_source', 'Source is not a directory: ' . $source );
        }

        // Create target directory with same permissions as source
        if ( ! is_dir( $target ) ) {
            $created = wp_mkdir_p( $target );
            if ( ! $created ) {
                return new WP_Error( 'mkdir_failed', 'Failed to create directory: ' . $target );
            }
        }

        $files_copied = 0;
        $dir_iterator = new RecursiveDirectoryIterator( $source, RecursiveDirectoryIterator::SKIP_DOTS );
        $iterator = new RecursiveIteratorIterator( $dir_iterator, RecursiveIteratorIterator::SELF_FIRST );

        foreach ( $iterator as $item ) {
            $target_path = $target . DIRECTORY_SEPARATOR . $iterator->getSubPathName();

            if ( $item->isDir() ) {
                // Create subdirectory
                if ( ! is_dir( $target_path ) ) {
                    $created = wp_mkdir_p( $target_path );
                    if ( ! $created ) {
                        return new WP_Error( 'mkdir_failed', 'Failed to create subdirectory: ' . $target_path );
                    }
                }
            } else {
                // Copy file
                $copied = copy( $item, $target_path );
                if ( ! $copied ) {
                    return new WP_Error( 'copy_failed', 'Failed to copy file: ' . $item->getPathname() );
                }

                // Preserve permissions
                chmod( $target_path, fileperms( $item ) );
                $files_copied++;
            }
        }

        return $files_copied;
    }


    /**
     * Create WordPress page from generated content
     */
    private function create_wordpress_page( $generation_result, $prompt, $page_type, $business_info ) {
        // Check if new split files format or old monolithic format
        if ( isset( $generation_result['files'] ) ) {
            // New split files format
            return $this->create_wordpress_page_with_files( $generation_result, $prompt, $page_type, $business_info );
        }

        // Old monolithic format (backward compatibility)
        $content_blocks = $generation_result['content_blocks'];
        $metadata = $generation_result['metadata'];

        // Combine content blocks into full page content
        $full_content = '';
        foreach ( $content_blocks as $block_name => $block_content ) {
            $full_content .= "<!-- AI Block: {$block_name} -->\n";
            $full_content .= $block_content . "\n\n";
        }

        // Add CSS if provided
        if ( ! empty( $metadata['styles'] ) ) {
            $full_content .= "<style>\n" . $metadata['styles'] . "\n</style>\n";
        }

        // Create WordPress page with real content for Yoast SEO
        $page_data = [
            'post_title' => $metadata['title'] ?: 'AI Generated Page',
            'post_content' => $full_content,
            'post_status' => 'draft',
            'post_type' => 'page',
            'post_excerpt' => $metadata['description'] ?: '',
            'meta_input' => [
                'ai_generated_page' => true,
                '_ai_protected_content' => $full_content, // Store real content in protected meta
                'ai_generation_prompt' => $prompt,
                'ai_content_blocks' => $content_blocks,
                'ai_generation_metadata' => json_encode( $metadata ),
                'ai_page_type' => $page_type,
                'ai_business_info' => json_encode( $business_info ?: [] ),
                'ai_model_used' => $generation_result['model_used'] ?? AI_Config_Manager::AI_MODEL,
                'ai_generated_at' => current_time( 'mysql' )
            ]
        ];

        $page_id = wp_insert_post( $page_data );

        if ( is_wp_error( $page_id ) ) {
            return new WP_Error( 'page_creation_failed', 'Failed to create WordPress page' );
        }

        $this->logger->info( 'Created AI-generated page', [
            'page_id' => $page_id,
            'title' => $metadata['title']
        ]);

        // Clear various caches after page creation
        $this->clear_page_caches( $page_id );

        return $page_id;
    }

    /**
     * Clear various page caches after creation/modification
     */
    private function clear_page_caches( $page_id ) {
        // Clear WordPress object cache
        wp_cache_delete( $page_id, 'posts' );

        // Clear get_posts cache (this is what the dashboard uses!)
        wp_cache_flush_group( 'posts' );

        // Clear WordPress transients that might cache page queries
        delete_transient( 'ai_pages_list' );

        // Clear WordPress page cache
        if ( function_exists( 'wp_cache_flush' ) ) {
            wp_cache_flush();
        }

        // LiteSpeed Cache
        if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
            do_action( 'litespeed_purge_post', $page_id );
            do_action( 'litespeed_purge_all' );
        }

        // WP Rocket
        if ( function_exists( 'rocket_clean_post' ) ) {
            rocket_clean_post( $page_id );
        }

        // W3 Total Cache
        if ( function_exists( 'w3tc_flush_post' ) ) {
            w3tc_flush_post( $page_id );
        }

        // WP Super Cache
        if ( function_exists( 'wp_cache_post_change' ) ) {
            wp_cache_post_change( $page_id );
        }

        // Generic action for other cache plugins
        do_action( 'ai_site_builder_page_cache_clear', $page_id );

        $this->logger->info( 'Cleared caches for page', [ 'page_id' => $page_id ] );
    }

    /**
     * Extract content blocks from updated content for metadata storage
     */
    private function extract_content_blocks( $content ) {
        $blocks = [];

        // Look for AI Block comments to identify sections
        if ( preg_match_all( '/<!-- AI Block: ([^>]+) -->(.*?)(?=<!-- AI Block:|$)/s', $content, $matches, PREG_SET_ORDER ) ) {
            foreach ( $matches as $match ) {
                $block_name = trim( $match[1] );
                $block_content = trim( $match[2] );
                $blocks[$block_name] = $block_content;
            }
        } else {
            // If no AI Block comments, treat entire content as single block
            $blocks['main_content'] = $content;
        }

        return $blocks;
    }

    /**
     * Get relevant theme files for page generation context
     * Reuses the existing file detection system from theme editing
     */
    private function get_theme_context_files( $prompt ) {
        // Get current active theme directory
        $theme_path = get_template_directory();

        // Get available theme files using existing scanning logic
        $available_files = [];
        $theme_files = scandir( $theme_path );

        foreach ( $theme_files as $file ) {
            if ( $file === '.' || $file === '..' ) {
                continue;
            }

            $file_path = $theme_path . '/' . $file;
            if ( is_file( $file_path ) && pathinfo( $file, PATHINFO_EXTENSION ) === 'php' || pathinfo( $file, PATHINFO_EXTENSION ) === 'css' ) {
                $available_files[] = $file;
            }
        }

        $this->logger->info( 'Scanning theme files for page context', [
            'theme_path' => $theme_path,
            'available_files' => $available_files
        ]);

        // Use existing Vercel client (includes API key authentication)
        $data = $this->vercel_client->detect_files(
            $prompt,
            $available_files,
            $theme_path,
            null // No metadata for page generation
        );

        if ( is_wp_error( $data ) ) {
            $this->logger->error( 'File detection failed', [
                'error' => $data->get_error_message()
            ]);
            return new WP_Error( 'file_detection_failed', 'Failed to detect relevant theme files: ' . $data->get_error_message() );
        }

        if ( ! $data || ! isset( $data['required_files'] ) ) {
            return new WP_Error( 'file_detection_failed', 'Invalid file detection response' );
        }

        // Read the detected files and return their content
        $theme_files = [];
        foreach ( $data['required_files'] as $file_name ) {
            $file_path = $theme_path . '/' . $file_name;
            if ( file_exists( $file_path ) ) {
                $content = file_get_contents( $file_path );
                if ( $content !== false ) {
                    // Truncate large files to avoid token limits (matches Vercel API behavior)
                    if ( strlen( $content ) > 25000 ) {
                        $original_size = strlen( $content );
                        $content = substr( $content, 0, 25000 ) . "\n/* [TRUNCATED - Original file was " . $original_size . " characters] */";
                    }
                    $theme_files[ $file_name ] = $content;
                }
            }
        }

        $this->logger->info( 'Retrieved theme files for context', [
            'detected_files' => array_keys( $theme_files ),
            'file_sizes' => array_map( 'strlen', $theme_files )
        ]);

        return $theme_files;
    }

    /**
     * Protect AI-generated content from being modified by WordPress editor
     */
    public function protect_ai_content_on_save( $data, $postarr ) {
        // Only protect existing posts (not new posts)
        if ( empty( $postarr['ID'] ) ) {
            return $data;
        }

        $post_id = $postarr['ID'];

        // Check if this is an AI-generated page
        $is_ai_page = get_post_meta( $post_id, 'ai_generated_page', true );

        if ( ! $is_ai_page ) {
            return $data;
        }

        // Get the original AI content
        $original_ai_content = get_post_meta( $post_id, '_ai_protected_content', true );

        if ( $original_ai_content ) {
            // Restore original AI content to prevent WordPress from mangling it
            $data['post_content'] = $original_ai_content;

            $this->logger->info( 'Protected AI content from WordPress editor modifications', [
                'post_id' => $post_id,
                'content_length' => strlen( $original_ai_content )
            ]);
        }

        return $data;
    }

    /**
     * PRODUCTION FIX: Enqueue CSS/JS from files (bypasses WordPress sanitization)
     *
     * This is the industry-standard approach used by Elementor, Divi, and Beaver Builder.
     * Works reliably across ALL hosting providers (Hostinger, Bluehost, WP Engine, etc.)
     * because files in wp-content/uploads are never sanitized by WordPress.
     *
     * Supports draft/publish workflow:
     * - Preview mode (?preview=true) loads from page-{id}-draft directory
     * - Published mode loads from page-{id} directory
     * - Automatic cache busting using filemtime()
     */
    public function enqueue_ai_page_assets() {
        // Only run on singular page view
        if ( ! is_singular( 'page' ) ) {
            return;
        }

        $page_id = get_the_ID();
        if ( ! $page_id ) {
            return;
        }

        // Only enqueue for AI pages with split files
        if ( ! get_post_meta( $page_id, '_ai_split_files', true ) ) {
            return;
        }

        $upload_dir = wp_upload_dir();

        // SMART: Detect preview/draft mode
        $is_preview = isset( $_GET['preview'] ) && $_GET['preview'] === 'true';
        $has_draft = get_post_meta( $page_id, '_ai_draft_active', true );

        // Choose directory based on context
        if ( $is_preview && $has_draft ) {
            // PREVIEW MODE: Load from draft directory
            $dir_name = 'page-' . $page_id . '-draft';
        } else {
            // PUBLISHED MODE: Load from live directory
            $dir_name = 'page-' . $page_id;
        }

        $base_url = $upload_dir['baseurl'] . '/ai-pages/' . $dir_name;
        $base_path = $upload_dir['basedir'] . '/ai-pages/' . $dir_name;

        // =======================================================
        // INLINE CSS/JS INJECTION (CDN-PROOF)
        // =======================================================
        // Instead of loading CSS/JS as external files (which CDNs cache),
        // we inject them directly into the HTML. This bypasses ALL caching:
        // - Hostinger CDN (HCDN)
        // - LiteSpeed cache
        // - Browser cache
        // - Any proxy/edge cache
        //
        // The files still exist on disk for the editor to read/write.
        // We're just changing HOW they're delivered to the browser.
        // =======================================================

        $style_handle = 'ai-page-' . $page_id;
        $script_handle = 'ai-page-' . $page_id . '-js';

        // Enqueue CSS inline if exists
        $css_path = $base_path . '/styles.css';
        if ( file_exists( $css_path ) ) {
            // Clear PHP stat cache to ensure we read the latest file
            clearstatcache( true, $css_path );

            // Read CSS content from disk
            $css_content = file_get_contents( $css_path );

            if ( $css_content !== false && ! empty( trim( $css_content ) ) ) {
                // Register empty style handle, then add inline CSS
                wp_register_style( $style_handle, false );
                wp_enqueue_style( $style_handle );
                wp_add_inline_style( $style_handle, $css_content );

                $this->logger->info( 'Injected AI page CSS inline (CDN-proof)', [
                    'page_id' => $page_id,
                    'mode' => $is_preview && $has_draft ? 'draft' : 'published',
                    'file' => $dir_name . '/styles.css',
                    'size' => strlen( $css_content )
                ] );
            }
        }

        // Enqueue JS inline if exists
        $js_path = $base_path . '/app.js';
        if ( file_exists( $js_path ) ) {
            // Clear PHP stat cache to ensure we read the latest file
            clearstatcache( true, $js_path );

            // Read JS content from disk
            $js_content = file_get_contents( $js_path );

            if ( $js_content !== false && ! empty( trim( $js_content ) ) ) {
                // Register empty script handle, then add inline JS
                wp_register_script( $script_handle, false, [], false, true ); // Load in footer
                wp_enqueue_script( $script_handle );
                wp_add_inline_script( $script_handle, $js_content );

                $this->logger->info( 'Injected AI page JS inline (CDN-proof)', [
                    'page_id' => $page_id,
                    'mode' => $is_preview && $has_draft ? 'draft' : 'published',
                    'file' => $dir_name . '/app.js',
                    'size' => strlen( $js_content )
                ] );
            }
        } else {
            // If no app.js, create a dummy handle for selection handler attachment
            wp_register_script( $script_handle, false, [], false, true );
            wp_enqueue_script( $script_handle );
        }

        // EDITOR ONLY: Add Dreamformer selection handler for Ctrl+click element selection
        // Only loads in preview/editor iframe, NOT on public pages
        $is_editor_context = $is_preview || isset( $_GET['dreamformer_editor'] ) || isset( $_GET['iframe'] );

        if ( $is_editor_context ) {
            // Get selection handler code (same as before, just loaded differently)
            $selection_handler = $this->get_dreamformer_selection_handler();

            // Enqueue as inline script (bypasses WordPress post_content sanitization)
            // This loads AFTER the page JS, so it has access to all elements
            // Works on ALL hosts because it's added via WordPress API, not post_content
            wp_add_inline_script(
                $script_handle,
                $selection_handler,
                'after'
            );

            $this->logger->info( 'Added Dreamformer selection handler for editor', [
                'page_id' => $page_id,
                'context' => 'editor_preview',
                'method' => 'wp_add_inline_script'
            ] );
        }
    }

    /**
     * Show admin notice on AI pages to warn about content protection
     */
    public function show_ai_page_protection_notice() {
        global $post;

        // Only show on post edit screen
        if ( ! is_admin() || ! isset( $post->ID ) ) {
            return;
        }

        // Check if this is an AI-generated page
        $is_ai_page = get_post_meta( $post->ID, 'ai_generated_page', true );

        if ( ! $is_ai_page ) {
            return;
        }

        ?>
        <div class="notice notice-info">
            <p>
                <strong>🤖 AI-Generated Page Protection:</strong>
                This page was created by AI Site Builder. Your AI formatting and styling are automatically protected when you save.
                Use the <a href="<?php echo admin_url( 'admin.php?page=dreamformer&edit=page&id=' . $post->ID ); ?>">AI Editor</a> for content changes.
            </p>
        </div>
        <?php
    }

    /**
     * Replace WordPress editor with custom readonly interface for AI pages
     */
    public function replace_ai_page_editor() {
        global $post;

        if ( ! $post || ! get_post_meta( $post->ID, 'ai_generated_page', true ) ) {
            return;
        }

        $ai_content = get_post_meta( $post->ID, '_ai_protected_content', true );
        $content_preview = wp_trim_words( wp_strip_all_tags( $ai_content ), 50, '...' );
        ?>
        <div id="ai-content-interface" style="margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 4px; background: #f9f9f9;">
            <h3 style="margin-top: 0;">🤖 AI-Generated Content</h3>
            <p><strong>Content Preview:</strong></p>
            <div style="background: white; padding: 10px; border: 1px solid #ddd; border-radius: 3px; margin: 10px 0; font-family: monospace; color: #666;">
                <?php echo esc_html( $content_preview ); ?>
            </div>
            <p>
                <strong>⚠️ This content is AI-protected.</strong>
                Use the AI Editor to make changes while preserving formatting and styling.
            </p>
            <p>
                <a href="<?php echo admin_url( 'admin.php?page=dreamformer&edit=page&id=' . $post->ID ); ?>" class="button button-primary">
                    ✏️ Edit with AI Editor
                </a>
                <a href="<?php echo get_permalink( $post->ID ); ?>" class="button" target="_blank">
                    👁️ Preview Page
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * Hide the default WordPress editor for AI pages
     */
    public function hide_ai_page_editor() {
        global $post;

        if ( ! is_admin() || ! $post || ! get_post_meta( $post->ID, 'ai_generated_page', true ) ) {
            return;
        }

        ?>
        <style>
            #postdivrich { display: none !important; }
            #wp-content-wrap { display: none !important; }
            .wp-editor-container { display: none !important; }
        </style>
        <?php
    }

    /**
     * Hide custom fields meta box for AI pages to keep interface clean
     */
    public function hide_ai_page_meta_boxes() {
        global $post;

        if ( ! $post || ! get_post_meta( $post->ID, 'ai_generated_page', true ) ) {
            return;
        }

        // Hide the custom fields meta box
        remove_meta_box( 'postcustom', 'page', 'normal' );
        remove_meta_box( 'postcustom', 'page', 'advanced' );
    }

    /**
     * Ensure correct .htaccess for ai-pages directory
     *
     * CRITICAL FIX: The .htaccess must ALLOW CSS/JS files but BLOCK HTML/JSON source
     * This fixes 403 Forbidden errors on CSS/JS that break page styling
     *
     * Works on ALL hosts (Hostinger, Bluehost, WP Engine, SiteGround, etc.)
     * because .htaccess is universally supported on Apache-based hosting
     */
    private function ensure_htaccess_security() {
        $upload_dir = wp_upload_dir();
        $ai_pages_dir = $upload_dir['basedir'] . '/ai-pages';
        $htaccess_path = $ai_pages_dir . '/.htaccess';

        $htaccess_content = <<<'HTACCESS'
# Dreamformer AI Pages - Security Rules
# =======================================================
# CSS/JS are now injected inline into HTML (CDN-proof)
# These files are only accessed by PHP, not browsers
# =======================================================

# BLOCK: All source files (CSS, JS, HTML, JSON)
# These contain source code and should not be directly accessible
<FilesMatch "\.(css|js|html?|json)$">
  Order Allow,Deny
  Deny from all
</FilesMatch>

# ALLOW: Images (may be referenced in pages)
<FilesMatch "\.(jpg|jpeg|png|gif|svg|webp|ico)$">
  Order Allow,Deny
  Allow from all
</FilesMatch>

# Prevent directory listing
Options -Indexes

# Block PHP execution (security)
<FilesMatch "\.php$">
  Order Allow,Deny
  Deny from all
</FilesMatch>
HTACCESS;

        // Create directory if doesn't exist
        if ( ! file_exists( $ai_pages_dir ) ) {
            wp_mkdir_p( $ai_pages_dir );
        }

        // Write/update .htaccess (always update to ensure correctness)
        file_put_contents( $htaccess_path, $htaccess_content );

        $this->logger->info( 'Updated ai-pages .htaccess for security', [
            'path' => $htaccess_path,
            'allows' => 'CSS, JS, images',
            'blocks' => 'HTML, JSON, directory listing'
        ] );
    }


    /**
     * Create WordPress page with split files
     */
    private function create_wordpress_page_with_files( $generation_result, $prompt, $page_type, $business_info ) {
        $files = $generation_result['files'];
        $metadata = $generation_result['metadata'] ?? [];

        // Create WordPress page first
        $page_data = [
            'post_title' => $metadata['title'] ?: 'AI Generated Page',
            'post_content' => '', // Will be filled by combined content
            'post_status' => 'publish',
            'post_type' => 'page'
        ];

        $page_id = wp_insert_post( $page_data );

        if ( is_wp_error( $page_id ) ) {
            $this->logger->error( 'Failed to create WordPress page', [
                'error' => $page_id->get_error_message()
            ] );
            return $page_id;
        }

        // Create page directory and save split files
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id;

        // CRITICAL: Ensure parent .htaccess allows CSS/JS access (fixes 403 errors)
        $this->ensure_htaccess_security();

        // Create page directory
        if ( ! is_dir( $page_dir ) ) {
            wp_mkdir_p( $page_dir );

            // Add security .htaccess
            $htaccess_content = "Options -Indexes\n<FilesMatch '\\.php$'>\n    Order Allow,Deny\n    Deny from all\n</FilesMatch>";
            file_put_contents( $page_dir . '/.htaccess', $htaccess_content );
        }

        // Save each file
        foreach ( $files as $filename => $content ) {
            // Sanitize filename
            $filename = str_replace( ['../', '..\\', '..'], '', $filename );
            $file_path = $page_dir . '/' . $filename;

            // Create subdirectories if needed
            $dir = dirname( $file_path );
            if ( ! is_dir( $dir ) ) {
                wp_mkdir_p( $dir );
            }

            // Write file
            file_put_contents( $file_path, $content );

            $this->logger->info( 'Created page file', [
                'page_id' => $page_id,
                'filename' => $filename,
                'size' => strlen( $content )
            ] );
        }

        // Combine files for display (render_split_files logic)
        $this->logger->info( 'Starting file combination', [
            'page_id' => $page_id,
            'files_to_combine' => array_keys( $files ),
            'file_sizes' => array_map( 'strlen', $files )
        ] );

        $combined_content = $this->render_split_files( $files );

        $this->logger->info( 'File combination completed', [
            'page_id' => $page_id,
            'combined_length' => strlen( $combined_content ),
            'preview' => substr( $combined_content, 0, 200 )
        ] );

        // CRITICAL FIX: Set metadata BEFORE updating post content
        // This ensures the protection filter can detect this is an AI page
        update_post_meta( $page_id, 'ai_generated_page', true );
        update_post_meta( $page_id, '_ai_protected_content', $combined_content );
        update_post_meta( $page_id, 'ai_generation_prompt', $prompt );
        update_post_meta( $page_id, 'ai_generation_metadata', json_encode( $metadata ) );
        update_post_meta( $page_id, 'ai_page_type', $page_type );
        update_post_meta( $page_id, 'ai_business_info', json_encode( $business_info ?: [] ) );
        update_post_meta( $page_id, 'ai_model_used', AI_Config_Manager::AI_MODEL );
        update_post_meta( $page_id, 'ai_generated_at', current_time( 'mysql' ) );
        update_post_meta( $page_id, '_ai_split_files', true );

         // UPDATE: Set the actual post_content with combined content
        // Now the protection filter will preserve <script> and <style> tags
        $update_result = wp_update_post( [
            'ID' => $page_id,
            'post_content' => $combined_content
        ] );

        if ( is_wp_error( $update_result ) ) {
            $this->logger->error( 'Failed to update post content', [
                'page_id' => $page_id,
                'error' => $update_result->get_error_message()
            ] );
            return $update_result;
        }

        $this->logger->info( 'Post content updated successfully', [
            'page_id' => $page_id,
            'update_result' => $update_result
        ] );

        // Set SEO meta
        if ( ! empty( $metadata['description'] ) ) {
            update_post_meta( $page_id, '_yoast_wpseo_metadesc', $metadata['description'] );
        }

        $this->logger->info( 'Page created with split files', [
            'page_id' => $page_id,
            'file_count' => count( $files ),
            'total_size' => array_sum( array_map( 'strlen', $files ) )
        ] );

        // Clear caches so page appears in admin immediately
        $this->clear_page_caches( $page_id );

        return $page_id;
    }

    /**
     * Render split files into combined content
     *
     * PRODUCTION FIX: Returns HTML only (no CSS/JS tags)
     * CSS and JS are loaded via enqueue_ai_page_assets() instead
     * This bypasses WordPress content sanitization completely
     *
     * Selection handler removed from inline injection because WordPress
     * strips <script> tags on Hostinger/Bluehost and displays code as text.
     */
    private function render_split_files( $files ) {
        $html = '';

        // Extract HTML only - no scripts, no styles
        foreach ( $files as $filename => $content ) {
            $extension = pathinfo( $filename, PATHINFO_EXTENSION );

            if ( $extension === 'html' || $extension === 'htm' ) {
                $html .= $content . "\n";
            }
        }

        // Return pure HTML only - all assets enqueued via wp_enqueue_scripts
        return $html;
    }

    /**
     * Recover split files from database (Emergency Recovery)
     *
     * Rebuilds split files directory from _ai_protected_content
     * Works only for published versions (not unpublished drafts)
     *
     * @param int $page_id Page ID to recover
     * @return array|WP_Error Success info or error
     */
    public function recover_split_files_from_db( $page_id ) {
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_Error( 'invalid_page', 'Invalid page ID' );
        }

        // Validate this is an AI page with split files
        if ( ! get_post_meta( $page_id, '_ai_split_files', true ) ) {
            return new WP_Error( 'not_split_files_page', 'This page does not use split files format' );
        }

        // Get combined content from database
        $combined_content = get_post_meta( $page_id, '_ai_protected_content', true );
        if ( empty( $combined_content ) ) {
            return new WP_Error( 'no_content', 'No published content found in database to recover from' );
        }

        $this->logger->info( 'Starting split files recovery', [
            'page_id' => $page_id,
            'content_size' => strlen( $combined_content )
        ] );

        // Parse HTML using DOMDocument (reliable)
        $dom = new DOMDocument();
        libxml_use_internal_errors( true );
        @$dom->loadHTML( $combined_content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
        libxml_clear_errors();

        // Extract CSS from <style> tags
        $css = '';
        $style_tags = $dom->getElementsByTagName( 'style' );
        $styles_to_remove = [];
        foreach ( $style_tags as $style ) {
            $css .= $style->textContent . "\n";
            $styles_to_remove[] = $style;
        }
        foreach ( $styles_to_remove as $style ) {
            $style->parentNode->removeChild( $style );
        }

        // Extract user JS from <script> tags (skip Dreamformer handler)
        $js = '';
        $script_tags = $dom->getElementsByTagName( 'script' );
        $scripts_to_remove = [];
        foreach ( $script_tags as $script ) {
            if ( empty( $script->getAttribute( 'src' ) ) ) {
                $script_content = $script->textContent;

                // Extract user JavaScript only
                if ( strpos( $script_content, '// === USER JAVASCRIPT ===' ) !== false ) {
                    $parts = explode( '// === USER JAVASCRIPT ===', $script_content, 2 );
                    $js .= trim( $parts[1] ) . "\n";
                } elseif ( strpos( $script_content, 'DREAMFORMER EDITOR INTEGRATION' ) === false ) {
                    $js .= $script_content . "\n";
                }

                $scripts_to_remove[] = $script;
            }
        }
        foreach ( $scripts_to_remove as $script ) {
            $script->parentNode->removeChild( $script );
        }

        // Remaining HTML
        $html = $dom->saveHTML();

        // Clean up
        $css = trim( $css );
        $js = trim( $js );
        $html = trim( $html );

        $this->logger->info( 'Extracted split files from combined content', [
            'page_id' => $page_id,
            'html_size' => strlen( $html ),
            'css_size' => strlen( $css ),
            'js_size' => strlen( $js )
        ] );

        // Get page directory
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id;

        // Backup existing directory if it exists
        if ( is_dir( $page_dir ) ) {
            $backup_dir = $page_dir . '-backup-recovery-' . time();
            if ( ! rename( $page_dir, $backup_dir ) ) {
                return new WP_Error( 'backup_failed', 'Failed to backup existing directory' );
            }
            $this->logger->info( 'Backed up existing directory', [
                'page_id' => $page_id,
                'backup_dir' => $backup_dir
            ] );
        }

        // Create fresh directory
        wp_mkdir_p( $page_dir );

        // Add security .htaccess
        $htaccess_content = "Options -Indexes\n<FilesMatch '\\.php$'>\n    Order Allow,Deny\n    Deny from all\n</FilesMatch>";
        file_put_contents( $page_dir . '/.htaccess', $htaccess_content );

        // Write files
        $files = [
            'index.html' => $html,
            'styles.css' => $css,
            'app.js' => $js
        ];

        $written_files = [];
        foreach ( $files as $filename => $content ) {
            if ( empty( $content ) ) {
                continue;
            }

            $filename = str_replace( ['../', '..\\', '..'], '', $filename );
            $file_path = $page_dir . '/' . $filename;

            if ( file_put_contents( $file_path, $content ) !== false ) {
                $written_files[] = $filename;
            }
        }

        if ( empty( $written_files ) ) {
            return new WP_Error( 'no_files_written', 'Failed to write any files' );
        }

        $this->logger->info( 'Split files recovery completed', [
            'page_id' => $page_id,
            'files_written' => $written_files
        ] );

        return [
            'success' => true,
            'files_recovered' => $written_files,
            'page_dir' => $page_dir,
            'message' => sprintf( 'Successfully recovered %d files from database', count( $written_files ) )
        ];
    }

    /**
     * Get Dreamformer selection handler JavaScript
     * This code is injected into every page to enable element selection via postMessage
     * even when iframe is in isolated process (no allow-same-origin)
     *
     * @return string Selection handler JavaScript code
     */
    private function get_dreamformer_selection_handler() {
        return <<<'JAVASCRIPT'
// ========================================
// DREAMFORMER EDITOR INTEGRATION v2.0
// Selection handler for isolated iframe
// ========================================
(function() {
    'use strict';

    // Skip if not in iframe
    if (window.self === window.top) {
        return;
    }

    let selectedElement = null;

    /**
     * Generate CSS selector for element
     */
    function getElementSelector(element) {
        if (element.id) {
            return '#' + element.id;
        } else if (element.className && element.className.trim()) {
            // Filter out the editor's selection highlight class
            const classes = element.className.trim().split(/\s+/)
                .filter(c => c !== 'ai-editor-selected')
                .join('.');
            if (classes) {
                return '.' + classes;
            }
        }
        return element.tagName.toLowerCase();
    }

    /**
     * Get computed styles for element
     */
    function getComputedStyles(element) {
        const styles = window.getComputedStyle(element);
        return {
            // Visual styles
            background: styles.backgroundColor || 'transparent',
            color: styles.color || 'inherit',
            fontSize: styles.fontSize || '16px',

            // TIER 1: Critical for debugging positioning/layout issues
            position: styles.position,
            top: styles.top,
            left: styles.left,
            right: styles.right,
            bottom: styles.bottom,

            // Spacing (often the issue!)
            marginTop: styles.marginTop,
            marginBottom: styles.marginBottom,
            marginLeft: styles.marginLeft,
            marginRight: styles.marginRight,
            paddingTop: styles.paddingTop,
            paddingBottom: styles.paddingBottom,
            paddingLeft: styles.paddingLeft,
            paddingRight: styles.paddingRight,

            // Dimensions
            width: styles.width,
            maxWidth: styles.maxWidth,
            height: styles.height,

            // Display & layout
            display: styles.display,
            textAlign: styles.textAlign,
            overflow: styles.overflow,

            // Visibility & layering
            zIndex: styles.zIndex,
            opacity: styles.opacity,
            visibility: styles.visibility,

            // Transforms
            transform: styles.transform
        };
    }

    /**
     * Find parent section
     */
    function getParentSection(element) {
        const section = element.closest('section, .wp-block-group, .entry-content, header, footer, main, article');

        if (!section) {
            return {
                selector: 'body',
                index: 0
            };
        }

        // Calculate section index among similar sections
        const sectionTag = section.tagName.toLowerCase();
        const sectionClasses = section.className || '';
        const sectionId = section.id || '';

        // Find all similar sections
        let selectorPattern = sectionTag;
        if (sectionClasses) {
            // Filter out empty strings to avoid invalid selectors like '.class1..class2'
            selectorPattern += '.' + sectionClasses.trim().split(/\s+/).filter(c => c).join('.');
        }

        const allSimilar = document.querySelectorAll(selectorPattern);
        const sectionIndex = Array.from(allSimilar).indexOf(section) + 1;

        // Build selector with nth-of-type
        let sectionSelector = sectionTag;
        if (sectionId) {
            sectionSelector = '#' + sectionId;
        } else if (sectionClasses) {
            // Filter out empty strings to avoid invalid selectors
            sectionSelector += '.' + sectionClasses.trim().split(/\s+/).filter(c => c).join('.');
            if (sectionIndex > 1) {
                sectionSelector += ':nth-of-type(' + sectionIndex + ')';
            }
        }

        return {
            selector: sectionSelector,
            index: sectionIndex,
            tag: sectionTag,
            classes: sectionClasses,
            id: sectionId
        };
    }

    /**
     * Deselect current element
     */
    function deselectElement() {
        if (selectedElement) {
            selectedElement.classList.remove('ai-editor-selected');
            selectedElement = null;
            return true;
        }
        return false;
    }

    /**
     * Send message to parent editor
     */
    function notifyParent(data) {
        try {
            window.parent.postMessage(data, '*');
        } catch (error) {
            console.error('[Dreamformer] Failed to send message to parent:', error);
        }
    }

    /**
     * Element selection handler (Ctrl+Click)
     */
    document.addEventListener('click', function(e) {
        // Only intercept Ctrl+click (or Cmd+click on Mac)
        if (!e.ctrlKey && !e.metaKey) {
            return; // Let normal clicks through
        }

        e.preventDefault();
        e.stopPropagation();

        const element = e.target;

        // TOGGLE: If clicking already selected element, deselect
        if (element === selectedElement) {
            deselectElement();
            notifyParent({ type: 'element-deselected' });
            return;
        }

        // Deselect previous
        deselectElement();

        // Select new element
        selectedElement = element;
        element.classList.add('ai-editor-selected');

        // Get element info
        const selector = getElementSelector(element);
        const section = getParentSection(element);
        const computed = getComputedStyles(element);

        // Send to parent editor
        notifyParent({
            type: 'element-selected',
            selector: selector,
            element_type: element.tagName.toLowerCase(),
            element_classes: element.className || '',
            parent_section: section.selector,
            section_index: section.index,
            computed_styles: computed,
            html: element.outerHTML.substring(0, 200),
            text: (element.textContent || '').substring(0, 50).trim()
        });

    }, true); // Use capture phase

    /**
     * ESC key handler
     */
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' || e.keyCode === 27) {
            if (deselectElement()) {
                e.preventDefault();
                e.stopPropagation();
                notifyParent({ type: 'element-deselected' });
            }
        }
    });

    /**
     * Click empty space handler
     */
    document.addEventListener('click', function(e) {
        if (e.ctrlKey || e.metaKey) return;

        if (e.target === document.body || e.target === document.documentElement) {
            if (deselectElement()) {
                notifyParent({ type: 'element-deselected' });
            }
        }
    });

    /**
     * Listen for manual style updates from parent
     */
    window.addEventListener('message', function(e) {
        if (e.data.type === 'update-manual-styles') {
            let styleEl = document.getElementById('manual-editor-styles');
            if (!styleEl) {
                styleEl = document.createElement('style');
                styleEl.id = 'manual-editor-styles';
                document.head.appendChild(styleEl);
                // Initialize rule tracking object
                styleEl.__manualRules = {};
            }

            // Initialize tracking object if missing (for pages that already have the element)
            if (!styleEl.__manualRules) {
                styleEl.__manualRules = {};
            }

            // Store rule keyed by selector
            const selector = e.data.selector;
            if (!styleEl.__manualRules[selector]) {
                styleEl.__manualRules[selector] = {};
            }

            // Update specific property (replaces old value if exists)
            styleEl.__manualRules[selector][e.data.property] = e.data.value;

            // Rebuild all rules with !important for proper specificity
            let cssText = '';
            for (const [sel, properties] of Object.entries(styleEl.__manualRules)) {
                cssText += sel + ' {\n';
                for (const [prop, val] of Object.entries(properties)) {
                    cssText += '  ' + prop + ': ' + val + ' !important;\n';
                }
                cssText += '}\n\n';
            }

            // Replace entire stylesheet content (not append)
            styleEl.textContent = cssText;

        }
    });

    /**
     * Inject selection styles
     */
    const style = document.createElement('style');
    style.id = 'dreamformer-selection-styles';
    style.textContent = `
        .ai-editor-selected {
            position: relative !important;
            outline: 3px solid #667eea !important;
            outline-offset: 4px !important;
            box-shadow: 0 0 0 1px rgba(102, 126, 234, 0.3),
                        0 0 20px rgba(102, 126, 234, 0.4),
                        inset 0 0 0 2px rgba(102, 126, 234, 0.1) !important;
            background: rgba(102, 126, 234, 0.05) !important;
            transition: all 0.3s ease !important;
            z-index: 9999 !important;
        }

        .ai-editor-selected::before {
            content: "★ SELECTED • ESC or Ctrl/⌘+click to deselect" !important;
            position: absolute !important;
            top: -32px !important;
            left: -3px !important;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
            color: white !important;
            padding: 4px 12px !important;
            font-size: 11px !important;
            font-weight: 600 !important;
            border-radius: 4px !important;
            letter-spacing: 0.5px !important;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.4) !important;
            z-index: 10000 !important;
            animation: ai-pulse 2s infinite !important;
            pointer-events: none !important;
            white-space: nowrap !important;
        }

        @keyframes ai-pulse {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.9; transform: scale(1.05); }
        }

        body.ai-selection-mode * {
            cursor: crosshair !important;
        }

        body.ai-selection-mode *:hover:not(.ai-editor-selected) {
            outline: 2px dashed rgba(102, 126, 234, 0.5) !important;
            outline-offset: 2px !important;
            transition: outline 0.2s ease !important;
        }
    `;
    document.head.appendChild(style);

    /**
     * Ctrl key cursor feedback
     */
    document.addEventListener('keydown', function(e) {
        if ((e.ctrlKey || e.metaKey) && !document.body.classList.contains('ai-selection-mode')) {
            document.body.classList.add('ai-selection-mode');
        }
    });

    document.addEventListener('keyup', function(e) {
        if (!e.ctrlKey && !e.metaKey && document.body.classList.contains('ai-selection-mode')) {
            document.body.classList.remove('ai-selection-mode');
        }
    });

    // Send heartbeat for freeze detection (extends existing heartbeat)
    setInterval(function() {
        notifyParent({
            type: 'dreamformer_heartbeat',
            timestamp: Date.now()
        });
    }, 1000);


    // Notify parent that handler is ready
    notifyParent({
        type: 'handler-loaded',
        version: '2.0'
    });
})();
JAVASCRIPT;
    }

    /**
     * Cleanup split files when page is permanently deleted (not just trashed)
     * This preserves files when pages are in trash so they can be restored
     */
    public function cleanup_split_files_on_permanent_delete( $post_id ) {
        $post = get_post( $post_id );

        // Only handle pages
        if ( ! $post || $post->post_type !== 'page' ) {
            return;
        }

        // Only handle AI-generated pages with split files
        $is_ai_page = get_post_meta( $post_id, 'ai_generated_page', true );
        $has_split_files = get_post_meta( $post_id, '_ai_split_files', true );

        if ( ! $is_ai_page || ! $has_split_files ) {
            return;
        }

        // Get the split files directory
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $post_id;

        if ( ! is_dir( $page_dir ) ) {
            return; // Directory doesn't exist, nothing to clean
        }

        // Recursively delete the page directory and all files
        $this->delete_directory_recursive( $page_dir );

        $this->logger->info( 'Cleaned up split files on permanent page deletion', [
            'page_id' => $post_id,
            'page_title' => $post->post_title,
            'directory_removed' => $page_dir
        ] );
    }

    /**
     * Recursively delete a directory and all its contents
     */
    private function delete_directory_recursive( $dir ) {
        if ( ! is_dir( $dir ) ) {
            return false;
        }

        $files = array_diff( scandir( $dir ), [ '.', '..' ] );

        foreach ( $files as $file ) {
            $path = $dir . '/' . $file;
            if ( is_dir( $path ) ) {
                $this->delete_directory_recursive( $path );
            } else {
                unlink( $path );
            }
        }

        return rmdir( $dir );
    }

    /**
     * Cleanup split files for pages that WordPress automatically deletes from trash
     * This runs daily and cleans up files for pages that were auto-deleted after EMPTY_TRASH_DAYS
     */
    public function cleanup_split_files_for_scheduled_delete() {
        $upload_dir = wp_upload_dir();
        $ai_pages_dir = $upload_dir['basedir'] . '/ai-pages';

        if ( ! is_dir( $ai_pages_dir ) ) {
            return; // No ai-pages directory exists
        }

        // Get all page directories
        $page_dirs = glob( $ai_pages_dir . '/page-*', GLOB_ONLYDIR );

        if ( empty( $page_dirs ) ) {
            return; // No page directories found
        }

        $cleaned_count = 0;

        foreach ( $page_dirs as $page_dir ) {
            // Extract page ID from directory name
            if ( preg_match( '/page-(\d+)$/', $page_dir, $matches ) ) {
                $page_id = (int) $matches[1];

                // Check if the page still exists in WordPress (including trash)
                $page = get_post( $page_id );

                if ( ! $page ) {
                    // Page no longer exists in WordPress, safe to delete files
                    $this->delete_directory_recursive( $page_dir );
                    $cleaned_count++;

                    $this->logger->info( 'Cleaned up orphaned split files directory', [
                        'page_id' => $page_id,
                        'directory' => $page_dir,
                        'reason' => 'Page no longer exists in WordPress'
                    ] );
                }
            }
        }

        if ( $cleaned_count > 0 ) {
            $this->logger->info( 'Scheduled cleanup completed', [
                'directories_cleaned' => $cleaned_count,
                'cleanup_type' => 'automatic_trash_cleanup'
            ] );
        }
    }

    /**
     * Handle page restored from trash
     * Clears caches so the page immediately shows up in Dreamformer
     */
    public function handle_page_restored( $post_id ) {
        $post = get_post( $post_id );

        // Only handle AI-generated pages
        if ( ! $post || $post->post_type !== 'page' ) {
            return;
        }

        $is_ai_page = get_post_meta( $post_id, 'ai_generated_page', true );
        if ( ! $is_ai_page ) {
            return;
        }

        // Clear caches so page shows up immediately in Dreamformer
        $this->clear_page_caches( $post_id );

        $this->logger->info( 'AI page restored from trash, caches cleared', [
            'page_id' => $post_id,
            'title' => $post->post_title
        ] );
    }

    /**
     * REST endpoint: Export page as portable JSON
     */
    public function rest_export_page( $request ) {
        $page_id = (int) $request->get_param( 'page_id' );

        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Page not found'
            ], 404 );
        }

        // Check if this is a Dreamformer page
        $is_ai_page = get_post_meta( $page_id, 'ai_generated_page', true );
        if ( ! $is_ai_page ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Not a Dreamformer page'
            ], 400 );
        }

        // Get split files
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id;

        $files = [];
        if ( is_dir( $page_dir ) ) {
            $file_names = ['index.html', 'styles.css', 'app.js'];
            foreach ( $file_names as $file_name ) {
                $file_path = $page_dir . '/' . $file_name;
                if ( file_exists( $file_path ) ) {
                    $content = file_get_contents( $file_path );
                    // Remove injected Dreamformer selection handler from export
                    if ( $file_name === 'app.js' ) {
                        $content = preg_replace( '/\/\/ ========================================\s*\/\/ DREAMFORMER EDITOR INTEGRATION.*?}\)\(\);/s', '', $content );
                        $content = trim( $content );
                    }
                    $files[$file_name] = $content;
                }
            }
        }

        if ( empty( $files ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'No page files found'
            ], 404 );
        }

        // Gather all metadata
        $page_type = get_post_meta( $page_id, 'ai_page_type', true );
        $business_info_json = get_post_meta( $page_id, 'ai_business_info', true );

        $metadata = [
            'ai_generation_prompt' => get_post_meta( $page_id, 'ai_generation_prompt', true ),
            'ai_page_type' => $page_type ?: 'custom',
            'ai_business_info' => $business_info_json ? json_decode( $business_info_json, true ) : [],
            'ai_model_used' => get_post_meta( $page_id, 'ai_model_used', true ),
            'ai_generated_at' => get_post_meta( $page_id, 'ai_generated_at', true ),
        ];

        // Check for dependencies
        $dependencies = [
            'wordpress' => '>=6.0',
            'dreamformer' => '>=1.0.0',
            'plugins' => []
        ];

        // Check if page uses WooCommerce shortcodes
        $combined_content = implode( "\n", $files );
        if ( strpos( $combined_content, '[woocommerce_' ) !== false || strpos( $combined_content, '[products' ) !== false ) {
            $dependencies['plugins']['woocommerce'] = [
                'required' => false,
                'reason' => 'Page uses WooCommerce shortcodes'
            ];
        }

        // Get SEO description
        $seo = [];
        $yoast_desc = get_post_meta( $page_id, '_yoast_wpseo_metadesc', true );
        if ( $yoast_desc ) {
            $seo['description'] = $yoast_desc;
        }

        // Build export package
        $export_data = [
            'format_version' => '1.0',
            'exported_at' => current_time( 'c' ), // ISO 8601
            'dreamformer_version' => defined( 'AI_SITE_BUILDER_VERSION' ) ? AI_SITE_BUILDER_VERSION : '1.0.0',
            'source_site' => get_site_url(),
            'dependencies' => $dependencies,
            'page' => [
                'title' => $page->post_title,
                'excerpt' => $page->post_excerpt,
                'status' => 'draft', // Always import as draft
                'files' => $files,
                'metadata' => $metadata,
                'seo' => $seo
            ]
        ];

        $this->logger->info( 'Page exported', [
            'page_id' => $page_id,
            'title' => $page->post_title,
            'file_count' => count( $files )
        ] );

        // Return JSON
        return new WP_REST_Response( $export_data, 200 );
    }

    /**
     * REST endpoint: Import page from exported JSON
     */
    public function rest_import_page( $request ) {
        // Get JSON body parameters
        $json_params = $request->get_json_params();

        // Extract data - handle both wrapped and unwrapped formats
        if ( isset( $json_params['data'] ) && is_array( $json_params['data'] ) ) {
            $data = $json_params['data'];
        } else {
            $data = $json_params;
        }

        // Validate data exists
        if ( empty( $data ) || ! is_array( $data ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Missing or invalid import data'
            ], 400 );
        }

        // Validate export format
        if ( ! isset( $data['format_version'] ) || ! isset( $data['page'] ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Invalid export format'
            ], 400 );
        }

        // Check format version
        if ( $data['format_version'] !== '1.0' ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Unsupported export format version: ' . $data['format_version']
            ], 400 );
        }

        $page_data = $data['page'];

        // Validate required fields
        if ( ! isset( $page_data['files'] ) || ! is_array( $page_data['files'] ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Invalid page data: missing files'
            ], 400 );
        }

        // Security validation
        $warnings = [];
        foreach ( $page_data['files'] as $filename => $content ) {
            // Check for PHP code
            if ( strpos( $content, '<?php' ) !== false ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'error' => 'Security risk: PHP code detected in ' . $filename
                ], 400 );
            }

            // Check for XSS vectors in HTML files only (not .js files which are source code)
            if ( pathinfo( $filename, PATHINFO_EXTENSION ) === 'html' ) {
                // Check for dangerous JavaScript in HTML context (inline scripts)
                if ( preg_match( '/eval\s*\(/i', $content ) || preg_match( '/Function\s*\(/i', $content ) ) {
                    $warnings[] = 'Potentially dangerous JavaScript patterns (eval/Function) detected in ' . $filename . '. Please review before publishing.';
                }
                // Check for inline event handlers (onclick, onerror, onload, etc.)
                if ( preg_match( '/\son\w+\s*=\s*["\']?/i', $content ) ) {
                    $warnings[] = 'Potentially unsafe inline event handlers detected in ' . $filename . '. Please review before publishing.';
                }

                // Check for javascript: protocol
                if ( preg_match( '/javascript\s*:/i', $content ) ) {
                    return new WP_REST_Response( [
                        'success' => false,
                        'error' => 'Security risk: javascript: protocol detected in ' . $filename
                    ], 400 );
                }

                // Check for data: URIs with scripts
                if ( preg_match( '/data:[^,]*;base64,/i', $content ) && preg_match( '/script/i', $content ) ) {
                    $warnings[] = 'Base64-encoded content detected in ' . $filename . '. Please verify it does not contain malicious scripts.';
                }
            }

            // Check file size (5MB limit per file)
            if ( strlen( $content ) > 5 * 1024 * 1024 ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'error' => 'File too large: ' . $filename . ' exceeds 5MB'
                ], 400 );
            }
        }

        // Check dependencies
        if ( isset( $data['dependencies']['plugins'] ) ) {
            foreach ( $data['dependencies']['plugins'] as $plugin => $info ) {
                if ( $plugin === 'woocommerce' && ! is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
                    $warnings[] = 'WooCommerce is required but not active. Page may not work correctly.';
                }
            }
        }

        // Create WordPress page
        $new_page_id = wp_insert_post( [
            'post_title' => $page_data['title'] . ' (Imported)',
            'post_content' => '', // Will be set from files
            'post_excerpt' => $page_data['excerpt'] ?? '',
            'post_status' => 'draft', // Always import as draft
            'post_type' => 'page'
        ] );

        if ( is_wp_error( $new_page_id ) ) {
            $this->logger->error( 'Failed to create page on import', [
                'error' => $new_page_id->get_error_message()
            ] );
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Failed to create page: ' . $new_page_id->get_error_message()
            ], 500 );
        }

        // Create page directory with rollback on failure
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $new_page_id;
        $rollback_needed = false;

        try {
            if ( ! wp_mkdir_p( $page_dir ) ) {
                throw new Exception( 'Failed to create page directory' );
            }

            // Write split files with error checking
            $combined_content = '';
            foreach ( $page_data['files'] as $file_name => $content ) {
                // Sanitize filename to prevent directory traversal
                $safe_filename = basename( $file_name );  // Only filename, no paths

                // Validate filename against allowed patterns
                if ( ! preg_match( '/^[a-zA-Z0-9_-]+\.(html|css|js)$/', $safe_filename ) ) {
                    throw new Exception( 'Invalid filename: ' . $file_name );
                }

                // Construct file path
                $file_path = $page_dir . '/' . $safe_filename;

                // Verify the resolved path is still within page_dir (prevent directory traversal)
                $real_page_dir = realpath( $page_dir );
                $real_file_dir = realpath( dirname( $file_path ) );

                if ( $real_file_dir !== $real_page_dir ) {
                    throw new Exception( 'Security error: Invalid file path detected for ' . $file_name );
                }

                // Attempt to write file
                $bytes_written = file_put_contents( $file_path, $content );

                if ( $bytes_written === false ) {
                    throw new Exception( 'Failed to write file: ' . $file_name );
                }

                // Verify file was written correctly
                if ( ! file_exists( $file_path ) || filesize( $file_path ) !== strlen( $content ) ) {
                    throw new Exception( 'File write verification failed: ' . $safe_filename );
                }

                // Build combined content for SEO
                if ( $safe_filename === 'index.html' ) {
                    $combined_content .= $content;
                } elseif ( $safe_filename === 'styles.css' ) {
                    $combined_content .= "\n\n<style>\n" . $content . "\n</style>";
                } elseif ( $safe_filename === 'app.js' ) {
                    $combined_content .= "\n\n<script>\n" . $content . "\n</script>";
                }
            }

            // Add security .htaccess
            $htaccess_written = file_put_contents( $page_dir . '/.htaccess',
                "Options -Indexes\n<FilesMatch '\\.php$'>\nOrder Allow,Deny\nDeny from all\n</FilesMatch>"
            );

            if ( $htaccess_written === false ) {
                $this->logger->warning( 'Failed to write .htaccess file', [ 'page_id' => $new_page_id ] );
            }

        } catch ( Exception $e ) {
            // Rollback: Delete the created page and directory
            $this->logger->error( 'Import failed, rolling back', [
                'page_id' => $new_page_id,
                'error' => $e->getMessage()
            ] );

            // Delete page directory and all files
            if ( is_dir( $page_dir ) ) {
                $this->delete_directory_recursive( $page_dir );
            }

            // Delete WordPress page
            wp_delete_post( $new_page_id, true );

            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Import failed: ' . $e->getMessage()
            ], 500 );
        }

        // Update post content for SEO
        wp_update_post( [
            'ID' => $new_page_id,
            'post_content' => $combined_content
        ] );

        // Save metadata
        update_post_meta( $new_page_id, 'ai_generated_page', true );
        update_post_meta( $new_page_id, '_ai_protected_content', $combined_content );
        update_post_meta( $new_page_id, '_ai_split_files', true );

        if ( isset( $page_data['metadata'] ) ) {
            $meta = $page_data['metadata'];
            update_post_meta( $new_page_id, 'ai_generation_prompt', $meta['ai_generation_prompt'] ?? '' );
            update_post_meta( $new_page_id, 'ai_page_type', $meta['ai_page_type'] ?? 'custom' );
            update_post_meta( $new_page_id, 'ai_business_info', json_encode( $meta['ai_business_info'] ?? [] ) );
            update_post_meta( $new_page_id, 'ai_model_used', $meta['ai_model_used'] ?? '' );
        }

        // Mark as imported
        update_post_meta( $new_page_id, '_ai_imported', true );
        update_post_meta( $new_page_id, '_ai_imported_from', $data['source_site'] ?? 'unknown' );
        update_post_meta( $new_page_id, '_ai_imported_at', current_time( 'mysql' ) );

        // Set SEO metadata
        if ( isset( $page_data['seo']['description'] ) ) {
            update_post_meta( $new_page_id, '_yoast_wpseo_metadesc', $page_data['seo']['description'] );
        }

        // Clear caches
        $this->clear_page_caches( $new_page_id );

        $this->logger->info( 'Page imported successfully', [
            'page_id' => $new_page_id,
            'source_site' => $data['source_site'] ?? 'unknown',
            'warnings' => $warnings
        ] );

        return new WP_REST_Response( [
            'success' => true,
            'page_id' => $new_page_id,
            'edit_url' => admin_url( 'post.php?post=' . $new_page_id . '&action=edit' ),
            'preview_url' => get_permalink( $new_page_id ),
            'dreamformer_url' => admin_url( 'admin.php?page=dreamformer&edit=page&id=' . $new_page_id ),
            'warnings' => $warnings
        ] );
    }

    /**
     * REST handler: Recover split files from database
     *
     * Emergency recovery function to rebuild split files from the combined
     * content stored in _ai_protected_content meta field
     *
     * @param WP_REST_Request $request REST request with page_id
     * @return WP_REST_Response Recovery result with success status and files recovered
     */
    public function rest_recover_split_files( $request ) {
        $page_id = $request->get_param( 'page_id' );

        if ( ! $page_id ) {
            return new WP_REST_Response( [
                'success' => false,
                'error' => 'Missing page_id parameter'
            ], 400 );
        }

        $this->logger->info( 'REST: Recovering split files from database', [
            'page_id' => $page_id,
            'user_id' => get_current_user_id()
        ] );

        // Call the recovery function
        $result = $this->recover_split_files_from_db( $page_id );

        // Handle WP_Error returns
        if ( is_wp_error( $result ) ) {
            $this->logger->error( 'REST: Split files recovery failed', [
                'page_id' => $page_id,
                'error_code' => $result->get_error_code(),
                'error_message' => $result->get_error_message()
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'error' => $result->get_error_message(),
                'error_code' => $result->get_error_code()
            ], 400 );
        }

        // Success
        $this->logger->info( 'REST: Split files recovered successfully', [
            'page_id' => $page_id,
            'files_recovered' => $result['files_recovered']
        ] );

        return new WP_REST_Response( $result, 200 );
    }

    /**
     * Validate attachment parameter (WordPress standard validation)
     *
     * @param mixed $param The attachment data
     * @param WP_REST_Request $request The REST request
     * @param string $key The parameter key
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_attachment( $param, $request, $key ) {
        // Optional parameter - null is valid
        if ( empty( $param ) || ! is_array( $param ) ) {
            return true;
        }

        // Check required fields
        if ( ! isset( $param['filename'] ) || ! isset( $param['mime_type'] ) || ! isset( $param['size'] ) || ! isset( $param['content'] ) ) {
            return new WP_Error(
                'invalid_attachment',
                'Attachment must include filename, mime_type, size, and content fields',
                [ 'status' => 400 ]
            );
        }

        // Sanitize filename
        $filename = sanitize_file_name( $param['filename'] );

        // Validate MIME type using WordPress function
        $filetype = wp_check_filetype( $filename );
        $allowed_mimes = [
            'application/pdf',
            'image/jpeg',
            'image/jpg',
            'image/png',
            'text/plain',
            'text/markdown',
            'text/csv',
            'text/html',
            'text/xml',
            'application/json',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];

        // Also allow by file extension (browsers report different MIME types)
        $allowed_extensions = [ 'pdf', 'png', 'jpg', 'jpeg', 'md', 'txt', 'doc', 'docx', 'csv', 'json', 'xml', 'html' ];
        $file_extension = strtolower( pathinfo( $filename, PATHINFO_EXTENSION ) );

        if ( ! in_array( $param['mime_type'], $allowed_mimes, true ) && ! in_array( $file_extension, $allowed_extensions, true ) ) {
            return new WP_Error(
                'invalid_file_type',
                'Invalid file type. Allowed: PDF, images (PNG/JPG), text files (MD, TXT, DOC, DOCX, CSV, JSON, XML, HTML).',
                [ 'status' => 400 ]
            );
        }

        // Validate file size (5MB = 5242880 bytes)
        $max_size = 5 * 1024 * 1024;
        if ( ! is_numeric( $param['size'] ) || $param['size'] > $max_size ) {
            return new WP_Error(
                'file_too_large',
                'File size exceeds maximum limit of 5MB.',
                [ 'status' => 400 ]
            );
        }

        // Validate base64 content
        if ( ! is_string( $param['content'] ) || empty( $param['content'] ) ) {
            return new WP_Error(
                'invalid_content',
                'Attachment content must be a non-empty base64 string.',
                [ 'status' => 400 ]
            );
        }

        // Verify base64 format (basic check)
        if ( ! preg_match( '/^[a-zA-Z0-9\/\r\n+]*={0,2}$/', $param['content'] ) ) {
            return new WP_Error(
                'invalid_base64',
                'Attachment content must be valid base64.',
                [ 'status' => 400 ]
            );
        }

        // Validate decoded size matches reported size (prevent manipulation)
        $decoded_size = strlen( base64_decode( $param['content'], true ) );
        if ( abs( $decoded_size - $param['size'] ) > 100 ) { // Allow small discrepancy
            return new WP_Error(
                'size_mismatch',
                'Attachment size mismatch. File may be corrupted.',
                [ 'status' => 400 ]
            );
        }

        return true;
    }

    /**
     * Setup wizard: Activate theme and mark setup as complete
     * Used by first-run setup wizard
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function setup_activate_theme( $request ) {
        $theme_slug = $request->get_param( 'theme' );

        // Handle "keep current theme" option - just complete setup without switching
        if ( $theme_slug === 'keep_current' ) {
            update_option( 'dreamformer_setup_complete', true );
            $this->logger->info( 'Setup completed - User chose to keep current theme' );

            return rest_ensure_response( [
                'success' => true,
                'message' => __( 'Setup completed! Using your current theme.', 'ai-site-builder' ),
                'theme'   => get_stylesheet()
            ] );
        }

        // Validate theme slug against constants
        $allowed_themes = [
            AI_SITE_BUILDER_BASE_THEME,
            AI_SITE_BUILDER_SHOP_THEME
        ];

        if ( ! in_array( $theme_slug, $allowed_themes, true ) ) {
            return new WP_Error(
                'invalid_theme',
                __( 'Invalid theme selected. Please choose from available options.', 'ai-site-builder' ),
                [ 'status' => 400 ]
            );
        }

        // Check if theme exists
        $theme = wp_get_theme( $theme_slug );
        if ( ! $theme->exists() ) {
            // Log error
            $this->logger->error( "Theme not found: {$theme_slug}" );

            // Fallback to first available theme
            $fallback = AI_SITE_BUILDER_BASE_THEME;
            if ( wp_get_theme( $fallback )->exists() ) {
                switch_theme( $fallback );
                update_option( 'dreamformer_setup_complete', true );

                return rest_ensure_response( [
                    'success' => true,
                    'message' => __( 'Selected theme not found. Activated fallback theme.', 'ai-site-builder' ),
                    'theme'   => $fallback
                ] );
            }

            return new WP_Error(
                'theme_not_found',
                __( 'Theme not found. Please ensure bundled themes are installed correctly.', 'ai-site-builder' ),
                [ 'status' => 500 ]
            );
        }

        // Activate theme
        switch_theme( $theme_slug );

        // Mark setup as complete
        update_option( 'dreamformer_setup_complete', true );

        // Log successful activation
        $this->logger->info( "Setup completed successfully - Theme activated: {$theme_slug}" );

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Theme activated successfully!', 'ai-site-builder' ),
            'theme'   => $theme_slug
        ] );
    }

    /**
     * Setup wizard: Dismiss Quick Start box
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function setup_dismiss_quick_start( $request ) {
        update_option( 'dreamformer_quick_start_dismissed', true );

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Quick Start dismissed', 'ai-site-builder' )
        ] );
    }
}