<?php
/**
 * AI Dreamformer Page Editor
 *
 * Integrates page editing into the Dreamformer editor interface
 * Provides same beautiful UI for page editing as theme editing
 *
 * @package AI_Site_Builder
 * @since 3.2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Dreamformer_Page_Editor {

    private $logger;
    private $vercel_client;

    public function __construct() {
        $this->logger = AI_Logger::get_instance( 'AI_Dreamformer_Page_Editor' );
        // Note: vercel_client is lazy-loaded via get_vercel_client() to ensure user is authenticated
        $this->init_hooks();
    }

    /**
     * Get Vercel client instance (lazy initialization)
     *
     * Creates the client only when first needed, ensuring WordPress has
     * authenticated the user (important for REST API calls).
     *
     * @return AI_Vercel_Client|null
     */
    private function get_vercel_client() {
        if ( $this->vercel_client === null && class_exists( 'AI_Vercel_Client' ) ) {
            $this->vercel_client = new AI_Vercel_Client();
        }
        return $this->vercel_client;
    }

    private function init_hooks() {
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
        // Filter preview content to show draft
        add_filter( 'the_content', array( $this, 'filter_preview_content' ), 999, 1 );
        // Intercept _ai_protected_content reads to show draft in preview mode
        add_filter( 'get_post_metadata', array( $this, 'filter_protected_content_for_preview' ), 10, 4 );
        // Sync draft files when publishing from WordPress admin (not Dreamformer)
        add_action( 'transition_post_status', array( $this, 'sync_draft_on_wp_publish' ), 10, 3 );
    }

    /**
     * SECURITY FIX: Get WordPress-based session ID
     *
     * WordPress doesn't start PHP sessions by default, so session_id() returns empty string.
     * This creates a user-based session using WordPress transients instead.
     *
     * @return string Session identifier
     */
    private function get_session_id() {
        $user_id = get_current_user_id();

        // For logged-in users, use user-based session
        if ( $user_id ) {
            $session_key = 'dreamformer_session_' . $user_id;
            $session_id = get_transient( $session_key );

            if ( ! $session_id ) {
                // Generate new session ID
                $session_id = wp_generate_password( 32, false );
                // Session expires after 1 hour of inactivity
                set_transient( $session_key, $session_id, HOUR_IN_SECONDS );
            }

            return $session_id;
        }

        // For non-logged-in users (shouldn't happen for editor, but handle gracefully)
        // Use IP-based session as fallback
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        return 'guest_' . md5( $ip . NONCE_SALT );
    }

    /**
     * Memory System - Save attempt for AI learning
     */
    private function save_attempt( $page_id, $prompt, $diff_data ) {
        global $wpdb;

        $wpdb->insert( $wpdb->prefix . 'ai_edit_attempts', array(
            'page_id' => $page_id,
            'session_id' => $this->get_session_id(),
            'user_prompt' => $prompt,
            'applied_diff' => json_encode( $diff_data ),
            'timestamp' => current_time( 'mysql' )
        ) );

        return $wpdb->insert_id;
    }

    /**
     * Memory System - Mark last attempt with user feedback
     */
    private function mark_last_attempt_feedback( $page_id, $feedback ) {
        global $wpdb;

        $last_attempt = $wpdb->get_row( $wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}ai_edit_attempts
             WHERE page_id = %d AND session_id = %s
             ORDER BY timestamp DESC LIMIT 1",
            $page_id,
            $this->get_session_id()
        ) );

        if ( $last_attempt ) {
            $wpdb->update(
                $wpdb->prefix . 'ai_edit_attempts',
                array( 'user_feedback' => $feedback ),
                array( 'id' => $last_attempt->id )
            );
        }
    }

    /**
     * Memory System - Get recent attempts for context
     */
    private function get_recent_attempts( $page_id ) {
        global $wpdb;

        return $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}ai_edit_attempts
             WHERE page_id = %d
             AND session_id = %s
             AND timestamp > DATE_SUB(NOW(), INTERVAL 30 MINUTE)
             ORDER BY timestamp ASC
             LIMIT 5",
            $page_id,
            $this->get_session_id()
        ) );
    }

    /**
     * Memory System - Build attempt context for AI
     */
    private function build_attempt_context( $attempts ) {
        if ( empty( $attempts ) ) {
            return '';
        }

        $context = "\n\n═══ PREVIOUS ATTEMPTS ═══\n\n";

        foreach ( $attempts as $i => $attempt ) {
            $num = $i + 1;
            $context .= "ATTEMPT {$num}:\n";
            $context .= "User said: \"{$attempt->user_prompt}\"\n\n";

            $diff = json_decode( $attempt->applied_diff, true );
            if ( $diff && isset( $diff['files'] ) ) {
                $context .= "You made these changes:\n";

                foreach ( $diff['files'] as $filename => $filedata ) {
                    $context .= "  File: {$filename}\n";

                    if ( isset( $filedata['changes'] ) ) {
                        foreach ( $filedata['changes'] as $change ) {
                            $old = substr( $change['old'], 0, 100 );
                            $new = substr( $change['new'], 0, 100 );
                            $context .= "    • Changed: {$old}... → {$new}...\n";
                        }
                    }
                }
            }

            if ( ! empty( $attempt->user_feedback ) ) {
                $context .= "\nUser's response: \"{$attempt->user_feedback}\"\n";
            }

            $context .= "\n" . str_repeat( '-', 50 ) . "\n\n";
        }

        $context .= "Based on these previous attempts and user feedback, determine what to try differently.\n";
        $context .= "Don't repeat the same changes. Try a different approach.\n\n";

        return $context;
    }

    /**
     * Register REST API routes for page editing in Dreamformer
     */
    public function register_rest_routes() {
        // Prepare data for direct Vercel streaming (browser calls Vercel directly)
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/prepare-stream', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_prepare_stream' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'prompt' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'AI prompt for page modification'
                ),
                'context' => array(
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Element context from frontend selection'
                )
            )
        ) );

        // Apply diff after direct Vercel streaming completes
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/apply-stream-diff', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_apply_stream_diff' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'prompt' => array(
                    'required' => true,
                    'type' => 'string'
                ),
                'diff_result' => array(
                    'required' => true,
                    'type' => 'object'
                ),
                'content_before' => array(
                    'required' => true,
                    'type' => 'string'
                )
            )
        ) );

        // Page preview session endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/preview-session', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_create_preview_session' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Publish draft endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/publish-draft', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_publish_draft' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Page edit history endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/history', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_edit_history' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Page draft status endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/draft-status', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_draft_status' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Page CSS rules endpoint (for element inspector)
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/css-rules', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_get_css_rules' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'selector' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS selector to analyze'
                )
            )
        ) );

        // Manual property update endpoint for pages
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/update-property', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_update_page_property' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'selector' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS selector to update'
                ),
                'property' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS property name'
                ),
                'value' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'New CSS value or text content'
                ),
                'match_index' => array(
                    'required' => false,
                    'type' => 'integer',
                    'description' => 'Index of element to update (for text content)'
                ),
                'element_type' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => 'Type of element being updated (for text content)'
                )
            )
        ) );

        // VERSION CONTROL ENDPOINTS FOR PAGES

        // Page undo endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/undo', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_undo_page_change' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Page redo endpoint
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/redo', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_redo_page_change' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Page version history endpoint (enhanced)
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/versions', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_page_versions' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // CLONE SECTION ENDPOINTS

        // Save section reference (no AI extraction - just saves reference)
        register_rest_route( 'ai-builder/v1', '/snippets/save', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_save_snippet' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // List all saved sections
        register_rest_route( 'ai-builder/v1', '/snippets', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_list_snippets' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Get/Delete single section
        register_rest_route( 'ai-builder/v1', '/snippets/(?P<id>\d+)', array(
            array(
                'methods' => 'GET',
                'callback' => array( $this, 'rest_get_snippet' ),
                'permission_callback' => function() {
                    return current_user_can( 'edit_pages' );
                }
            ),
            array(
                'methods' => 'DELETE',
                'callback' => array( $this, 'rest_delete_snippet' ),
                'permission_callback' => function() {
                    return current_user_can( 'edit_pages' );
                }
            )
        ) );

        // Apply section to another page (6 credits) - returns job_id for polling
        register_rest_route( 'ai-builder/v1', '/snippets/(?P<id>\d+)/apply', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_apply_snippet' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'target_page_id' => array(
                    'required' => true,
                    'type' => 'integer',
                    'description' => 'Page ID to apply the section to'
                )
            )
        ) );

        // Poll apply section status
        register_rest_route( 'ai-builder/v1', '/snippets/apply-status/(?P<job_id>[a-f0-9\-]+)', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_poll_apply_section_status' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // Apply completed section result
        register_rest_route( 'ai-builder/v1', '/snippets/apply-result', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_apply_section_result' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            }
        ) );

        // CODE EDITOR ENDPOINTS

        // Get all code files for editing (Premium feature - Pro plans only)
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/code-files', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_code_files' ),
            'permission_callback' => array( $this, 'check_code_editor_permission' )
        ) );

        // Save all code files (Premium feature - Pro plans only)
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/code-files', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_save_code_files' ),
            'permission_callback' => array( $this, 'check_code_editor_permission' ),
            'args' => array(
                'files' => array(
                    'required' => true,
                    'type' => 'object',
                    'description' => 'Files object with filenames as keys and content as values'
                )
            )
        ) );

        // POLLING-BASED PAGE EDITING ENDPOINTS

        // Create polling job for page edit
        register_rest_route( 'ai-builder/v1', '/pages/(?P<page_id>\d+)/edit-polling', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_edit_page_polling' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'prompt' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'AI prompt for page modification'
                ),
                'context' => array(
                    'required' => false,
                    'type' => 'object',
                    'description' => 'Element context from frontend selection'
                )
            )
        ) );

        // Poll edit job status
        register_rest_route( 'ai-builder/v1', '/pages/poll-edit-status/(?P<job_id>[a-f0-9\-]+)', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_poll_edit_status' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_pages' );
            },
            'args' => array(
                'job_id' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => function( $param ) {
                        return preg_match( '/^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i', $param );
                    }
                )
            )
        ) );
    }

    /**
     * Check if user has permission to use code editor
     *
     * Code editor is a premium feature available only to Pro plan users.
     * Free plan users must upgrade to access manual code editing.
     *
     * @return bool|WP_Error True if allowed, WP_Error if denied
     */
    public function check_code_editor_permission() {
        // Basic WordPress permission check
        if ( ! current_user_can( 'edit_pages' ) ) {
            return new WP_Error(
                'unauthorized',
                'You do not have permission to edit pages.',
                array( 'status' => 403 )
            );
        }

        // Check if Freemius is loaded and initialized
        if ( ! function_exists( 'dreamformer_fs' ) || ! dreamformer_fs() ) {
            // Freemius not initialized - allow access for development/testing
            // In production, this should not happen as Freemius is required
            return true;
        }

        // Check if user is registered with Freemius
        if ( ! dreamformer_fs()->is_registered() ) {
            return new WP_Error(
                'no_subscription',
                'Code editor requires an active subscription. Please activate your license or upgrade to Pro.',
                array(
                    'status' => 403,
                    'upgrade_url' => admin_url( 'admin.php?page=dreamformer-pricing' )
                )
            );
        }

        // Check if user has premium/pro plan (not free)
        if ( ! dreamformer_fs()->is_premium() ) {
            return new WP_Error(
                'premium_required',
                'Code editor is a premium feature. Please upgrade to Pro plan to access manual code editing.',
                array(
                    'status' => 403,
                    'upgrade_url' => admin_url( 'admin.php?page=dreamformer-pricing' )
                )
            );
        }

        // User has premium access
        return true;
    }

    /**
     * Streaming version of page editor with real-time progress updates
     */
    public function rest_edit_page_stream( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $context = $request->get_param( 'context' ) ?: array();

        $this->logger->info( 'Processing Dreamformer page edit request (STREAMING)', [
            'page_id' => $page_id,
            'prompt_preview' => substr( $prompt, 0, 100 ) . '...',
            'has_context' => ! empty( $context )
        ] );

        // CRITICAL: Check user has credits BEFORE calling Vercel API
        $credits_check = $this->get_vercel_client()->check_user_credits( 'page_edit' );
        if ( is_wp_error( $credits_check ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $credits_check->get_error_message(),
                'code' => $credits_check->get_error_code()
            ], 402 ); // 402 Payment Required
        }

        // Verify page exists and is editable
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found or invalid'
            ], 404 );
        }

        // Set SSE headers
        header( 'Content-Type: text/event-stream' );
        header( 'Cache-Control: no-cache' );
        header( 'Connection: keep-alive' );
        header( 'X-Accel-Buffering: no' );

        // Disable ALL levels of output buffering for streaming
        while ( ob_get_level() > 0 ) {
            ob_end_clean();
        }

        // Disable implicit flushing
        @ini_set( 'output_buffering', 'off' );
        @ini_set( 'zlib.output_compression', false );

        // Helper to send SSE events with aggressive flushing
        $send_event = function( $event_type, $data ) {
            echo "event: {$event_type}\n";
            echo "data: {$data}\n\n";

            // Flush all levels of output buffering
            while ( ob_get_level() > 0 ) {
                ob_flush();
            }

            // Send data to client immediately
            flush();
        };

        try {
            $send_event( 'stage', 'Preparing page content...' );

            // MEMORY SYSTEM: Get recent attempts and mark previous with feedback
            $attempts = $this->get_recent_attempts( $page_id );
            if ( ! empty( $attempts ) ) {
                $this->mark_last_attempt_feedback( $page_id, $prompt );
            }

            // MEMORY SYSTEM: Build context from previous attempts
            $attempt_context = $this->build_attempt_context( $attempts );

            // Get current page content from protected meta if AI page
            $protected_content = get_post_meta( $page_id, '_ai_protected_content', true );
            $current_content = $protected_content ?: $page->post_content;

            // VERSION CONTROL: Capture content before changes
            $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
            if ( $has_split_files ) {
                $page_files_before = $this->scan_page_files( $page_id );
                $content_before = json_encode( $page_files_before );
            } else {
                $content_before = $current_content;
            }

            $send_event( 'stage', 'Analyzing page structure...' );

            // Get AI-enhanced context with page content + theme files
            $required_files = $this->get_ai_enhanced_context( $prompt, $current_content, $page_id );

            $send_event( 'stage', 'Sending to AI...' );

            // Use streaming Vercel endpoint
            $diff_response = $this->get_vercel_client()->generate_page_diff_stream(
                $prompt,
                $required_files,
                $context,
                function( $event_type, $data ) use ( $send_event ) {
                    // Forward progress events from Vercel to browser
                    $send_event( $event_type, $data );
                },
                $attempt_context
            );

            if ( is_wp_error( $diff_response ) ) {
                $send_event( 'error', $diff_response->get_error_message() );
                exit;
            }

            // Handle conversational responses
            $is_conversational = false;
            $conversational_message = '';

            if ( is_array( $diff_response ) && isset( $diff_response['conversational_response'] ) ) {
                $conv_data = $diff_response['conversational_response'];
                $is_conversational = true;
                $conversational_message = $conv_data['message'] ?? 'I understand your question. What specific changes would you like me to make?';
            } elseif ( is_array( $diff_response ) && isset( $diff_response['type'] ) && $diff_response['type'] === 'conversational_response' ) {
                $is_conversational = true;
                $conversational_message = $diff_response['message'] ?? 'I understand your question. What specific changes would you like me to make?';
            }

            if ( $is_conversational ) {
                $send_event( 'complete', json_encode( [
                    'success' => true,
                    'message' => $conversational_message,
                    'is_conversational' => true,
                    'changes' => []
                ] ) );
                exit;
            }

            $send_event( 'stage', 'Applying changes...' );

            // Parse and apply diff changes
            if ( is_array( $diff_response ) && isset( $diff_response['diff_json'] ) ) {
                $diff_json = $diff_response['diff_json'];
            } else {
                $diff_json = $diff_response;
            }

            $diff_data = json_decode( $diff_json, true );

            if ( ! $diff_data ) {
                $send_event( 'error', 'Invalid AI response format: ' . json_last_error_msg() );
                exit;
            }

            // Apply changes to page content
            $result = $this->apply_page_changes( $page_id, $diff_data, $content_before, $prompt );

            if ( is_wp_error( $result ) ) {
                $send_event( 'error', 'Failed to apply changes: ' . $result->get_error_message() );
                exit;
            }

            // MEMORY SYSTEM: Save this attempt for future learning
            $this->save_attempt( $page_id, $prompt, $diff_data );

            // Trigger cache bust so browsers fetch fresh content
            if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
                ai_site_builder_trigger_cache_bust( 'page', $page_id );
            }

            // Send completion event
            $send_event( 'complete', json_encode( [
                'success' => true,
                'message' => 'Page updated successfully!',
                'user_message' => $diff_data['user_message'] ?? null,
                'changes' => $this->format_changes_for_frontend( $result ),
                'is_conversational' => false
            ] ) );

        } catch ( Exception $e ) {
            $this->logger->error( 'Page edit streaming failed', [
                'error' => $e->getMessage(),
                'page_id' => $page_id
            ] );

            $send_event( 'error', 'Internal error occurred' );
        }

        exit;
    }

    /**
     * Prepare data for direct Vercel streaming (browser connects directly to Vercel)
     */
    public function rest_prepare_stream( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $context = $request->get_param( 'context' ) ?: array();

        $this->logger->info( 'Preparing page edit for direct Vercel streaming', [
            'page_id' => $page_id,
            'prompt_preview' => substr( $prompt, 0, 100 ) . '...'
        ] );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found or invalid'
            ], 404 );
        }

        try {
            // Get recent attempts for memory system
            $attempts = $this->get_recent_attempts( $page_id );
            if ( ! empty( $attempts ) ) {
                $this->mark_last_attempt_feedback( $page_id, $prompt );
            }

            // Build context from previous attempts
            $attempt_context = $this->build_attempt_context( $attempts );

            // Get current page content
            $protected_content = get_post_meta( $page_id, '_ai_protected_content', true );
            $current_content = $protected_content ?: $page->post_content;

            // Capture content before changes for version control
            $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
            if ( $has_split_files ) {
                $page_files_before = $this->scan_page_files( $page_id );
                $content_before = json_encode( $page_files_before );
            } else {
                $content_before = $current_content;
            }

            // Capture theme files before changes (for linked versioning)
            $theme_files_before = null;
            $active_theme_id = $this->get_active_ai_theme_id();
            if ( $active_theme_id ) {
                $theme_path = get_template_directory();
                $theme_files_snapshot = $this->scan_theme_files( $theme_path );
                // Load file contents for snapshot
                $theme_files_before = array();
                foreach ( $theme_files_snapshot as $relative_path => $full_path ) {
                    if ( file_exists( $full_path ) && is_readable( $full_path ) ) {
                        $theme_files_before[$relative_path] = file_get_contents( $full_path );
                    }
                }

                // BUG FIX: Create baseline theme version if none exists (for streaming mode)
                $version_control = AI_Version_Control::get_instance();
                global $wpdb;
                $table_name = $wpdb->prefix . 'ai_version_control';
                $existing_baseline = $wpdb->get_var( $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE entity_type = 'theme' AND entity_id = %d",
                    $active_theme_id
                ) );

                if ( $existing_baseline == 0 && ! empty( $theme_files_before ) ) {
                    // No baseline exists - create one NOW
                    $theme_files_json = json_encode( $theme_files_before );
                    $version_control->save_version(
                        'theme',
                        $active_theme_id,
                        '', // No before content (this is the baseline)
                        $theme_files_json,
                        array(
                            'change_description' => 'Editing started - theme baseline version',
                            'user_prompt' => '',
                            'change_metadata' => array( 'is_baseline' => true )
                        )
                    );

                    $this->logger->info( 'Created baseline theme version', array(
                        'theme_id' => $active_theme_id,
                        'file_count' => count( $theme_files_before )
                    ) );
                }
            }

            // Get AI-enhanced context with page content + theme files
            $required_files = $this->get_ai_enhanced_context( $prompt, $current_content, $page_id );

            // Build payload for Vercel
            $payload = [
                'user_prompt' => $prompt,
                'required_files' => $required_files,
                'context' => $context,
                'attempt_history' => $attempt_context
            ];

            // Stream response from Vercel through WordPress (secure proxy with API key)
            $this->logger->info( 'Starting streaming page edit proxy', [
                'has_context' => ! empty( $context ),
                'has_selected_element' => isset( $context['selected_element'] ),
                'selected_element' => $context['selected_element'] ?? null,
                'element_type' => $context['element_type'] ?? null
            ] );

            // Proxy the streaming request - just forward the stream to browser
            $success = $this->get_vercel_client()->proxy_stream_to_browser(
                '/api/generate-page-diff-stream',
                $payload
            );

            // After streaming, send metadata for apply step
            if ( $success ) {
                // Send metadata event with info needed for apply endpoint
                echo "event: metadata\n";
                echo "data: " . wp_json_encode( [
                    'content_before' => $content_before,
                    'theme_files_before' => $theme_files_before ? json_encode( $theme_files_before ) : null,
                    'active_theme_id' => $active_theme_id,
                    'page_id' => $page_id
                ] ) . "\n\n";
                flush();
            } else {
                $this->logger->error( 'Page edit streaming failed' );
            }

            // Exit to prevent WordPress from adding extra output
            exit;

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to prepare stream', [
                'error' => $e->getMessage(),
                'page_id' => $page_id
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to prepare: ' . $e->getMessage()
            ], 500 );
        }
    }

    /**
     * REST endpoint: Edit page with polling
     * Returns job_id immediately, client polls for progress
     */
    public function rest_edit_page_polling( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $context = $request->get_param( 'context' ) ?: array();

        $this->logger->info( 'Starting polling-based page edit', array(
            'page_id' => $page_id,
            'prompt_preview' => substr( $prompt, 0, 100 ) . '...'
        ) );

        // Check credits
        $credits_check = $this->get_vercel_client()->check_user_credits( 'page_edit' );
        if ( is_wp_error( $credits_check ) ) {
            $error_code = $credits_check->get_error_code();
            $http_status = ( $error_code === 'insufficient_credits' ) ? 402 : 401;
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $credits_check->get_error_message(),
                'code' => $error_code
            ), $http_status );
        }

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Page not found or invalid'
            ), 404 );
        }

        try {
            // Memory system: Get recent attempts
            $attempts = $this->get_recent_attempts( $page_id );
            if ( ! empty( $attempts ) ) {
                $this->mark_last_attempt_feedback( $page_id, $prompt );
            }
            $attempt_context = $this->build_attempt_context( $attempts );

            // Get current page content
            $protected_content = get_post_meta( $page_id, '_ai_protected_content', true );
            $current_content = $protected_content ?: $page->post_content;

            // Capture content before changes
            $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
            if ( $has_split_files ) {
                $page_files_before = $this->scan_page_files( $page_id );
                $content_before = json_encode( $page_files_before );
            } else {
                $content_before = $current_content;
            }

            // Capture theme files before changes
            $theme_files_before = null;
            $active_theme_id = $this->get_active_ai_theme_id();
            if ( $active_theme_id ) {
                $theme_path = get_template_directory();
                $theme_files_snapshot = $this->scan_theme_files( $theme_path );
                $theme_files_before = array();
                foreach ( $theme_files_snapshot as $relative_path => $full_path ) {
                    if ( file_exists( $full_path ) && is_readable( $full_path ) ) {
                        $theme_files_before[ $relative_path ] = file_get_contents( $full_path );
                    }
                }

                // BUG FIX: Create baseline theme version if none exists (for polling mode)
                $version_control = AI_Version_Control::get_instance();
                global $wpdb;
                $table_name = $wpdb->prefix . 'ai_version_control';
                $existing_baseline = $wpdb->get_var( $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE entity_type = 'theme' AND entity_id = %d",
                    $active_theme_id
                ) );

                if ( $existing_baseline == 0 && ! empty( $theme_files_before ) ) {
                    // No baseline exists - create one NOW
                    $theme_files_json = json_encode( $theme_files_before );
                    $version_control->save_version(
                        'theme',
                        $active_theme_id,
                        '', // No before content (this is the baseline)
                        $theme_files_json,
                        array(
                            'change_description' => 'Editing started - theme baseline version',
                            'user_prompt' => '',
                            'change_metadata' => array( 'is_baseline' => true )
                        )
                    );

                    $this->logger->info( 'Created baseline theme version', array(
                        'theme_id' => $active_theme_id,
                        'file_count' => count( $theme_files_before )
                    ) );
                }
            }

            // Get AI-enhanced context
            $required_files = $this->get_ai_enhanced_context( $prompt, $current_content, $page_id );

            // Build payload
            $payload = array(
                'user_prompt' => $prompt,
                'required_files' => $required_files,
                'context' => $context,
                'attempt_history' => $attempt_context,
                'page_id' => $page_id
            );

            // Call polling endpoint
            $response = $this->get_vercel_client()->call_api_json(
                '/api/generate-page-diff-polling',
                $payload
            );

            if ( is_wp_error( $response ) ) {
                return new WP_REST_Response( array(
                    'success' => false,
                    'message' => $response->get_error_message()
                ), 500 );
            }

            $this->logger->info( 'Edit polling job created', array(
                'job_id' => $response['job_id'],
                'page_id' => $page_id
            ) );

            // Return job_id + metadata for apply step
            return new WP_REST_Response( array(
                'success' => true,
                'job_id' => $response['job_id'],
                'status' => $response['status'],
                'poll_interval_ms' => 2000,
                // Metadata for apply step (same as streaming)
                'content_before' => $content_before,
                'theme_files_before' => $theme_files_before ? json_encode( $theme_files_before ) : null,
                'active_theme_id' => $active_theme_id,
                'page_id' => $page_id
            ), 202 );

        } catch ( Exception $e ) {
            $this->logger->error( 'Edit polling preparation failed', array(
                'error' => $e->getMessage(),
                'page_id' => $page_id
            ) );

            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Failed to prepare: ' . $e->getMessage()
            ), 500 );
        }
    }

    /**
     * REST endpoint: Poll edit job status
     */
    public function rest_poll_edit_status( WP_REST_Request $request ) {
        // CRITICAL: Disable ALL caching plugins for this polling endpoint
        do_action( 'litespeed_control_set_nocache', 'Dreamformer polling endpoint' );
        if ( ! defined( 'DONOTCACHEPAGE' ) ) {
            define( 'DONOTCACHEPAGE', true );
        }

        $job_id = $request->get_param( 'job_id' );

        $status_data = $this->get_vercel_client()->poll_job_status_generic( $job_id, 'edit' );

        if ( is_wp_error( $status_data ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $status_data->get_error_message()
            ), 500 );
        }

        // No-cache headers (critical for caching plugins like LiteSpeed)
        $response = new WP_REST_Response( $status_data, 200 );
        $response->header( 'Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0, private' );
        $response->header( 'Pragma', 'no-cache' );
        $response->header( 'Expires', '0' );
        $response->header( 'X-LiteSpeed-Cache-Control', 'no-cache' );

        return $response;
    }

    /**
     * Apply diff after browser receives it from direct Vercel streaming
     */
    public function rest_apply_stream_diff( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $prompt = sanitize_textarea_field( $request->get_param( 'prompt' ) );
        $diff_result = $request->get_param( 'diff_result' );
        $content_before = $request->get_param( 'content_before' );
        $theme_files_before = $request->get_param( 'theme_files_before' );
        $active_theme_id = intval( $request->get_param( 'active_theme_id' ) ?: 0 );

        $this->logger->info( 'Applying diff from direct Vercel streaming', [
            'page_id' => $page_id,
            'has_theme_snapshot' => ! empty( $theme_files_before ),
            'active_theme_id' => $active_theme_id
        ] );

        try {
            // Deduct credits after successful response (conversational or diff - both cost 1 credit)
            $credit_manager = new Dreamformer_Credit_Manager();
            $user_id = get_current_user_id();
            $deduct_result = $credit_manager->deduct_credits( $user_id, 1, 'page_edit' );

            if ( is_wp_error( $deduct_result ) ) {
                $this->logger->warning( 'Credit deduction failed after successful page edit response', [
                    'page_id' => $page_id,
                    'error' => $deduct_result->get_error_message()
                ] );
                // Continue anyway - response was successful, credit sync issue won't block user
            }

            // Handle conversational responses
            if ( isset( $diff_result['conversational_response'] ) || isset( $diff_result['is_conversational'] ) ) {
                return new WP_REST_Response( [
                    'success' => true,
                    'message' => $diff_result['message'] ?? 'Understood.',
                    'is_conversational' => true,
                    'changes' => []
                ], 200 );
            }

            // Parse diff JSON
            if ( isset( $diff_result['diff_json'] ) ) {
                $diff_json = $diff_result['diff_json'];
            } else {
                $diff_json = json_encode( $diff_result );
            }

            $diff_data = json_decode( $diff_json, true );

            if ( ! $diff_data ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Invalid diff format: ' . json_last_error_msg()
                ], 400 );
            }

            // THEME WARNING SYSTEM: Check for theme file changes BEFORE applying
            // Get confirmation flag from request
            $theme_changes_confirmed = $request->get_param( 'theme_changes_confirmed' );

            // If not confirmed yet, analyze diff for theme files
            if ( ! $theme_changes_confirmed ) {
                $analysis = $this->analyze_diff_for_theme_files( $page_id, $diff_data );

                if ( $analysis['has_theme_changes'] ) {
                    // STOP - Don't apply yet! Ask user for confirmation first
                    $this->logger->info( 'Theme changes detected - requiring user confirmation', [
                        'page_id' => $page_id,
                        'theme_files' => $analysis['theme_files'],
                        'page_files' => $analysis['page_files']
                    ] );

                    return new WP_REST_Response( [
                        'needs_confirmation' => true,
                        'theme_files' => $analysis['theme_files'],
                        'page_files' => $analysis['page_files'],
                        'message' => 'Theme changes detected - require explicit confirmation',
                        'warning' => 'These theme changes will go LIVE immediately and affect ALL pages on your site.'
                    ], 200 );
                }
            } else {
                $this->logger->info( 'Theme changes confirmed by user - proceeding with apply', [
                    'page_id' => $page_id
                ] );
            }

            // Initialize draft mode on first edit
            $live_dir = $this->get_page_directory( $page_id, false );
            $draft_dir = $this->get_page_directory( $page_id, true );

            if ( ! $this->is_draft_mode( $page_id ) ) {
                // Copy live to draft on first edit (if live directory exists)
                if ( is_dir( $live_dir ) && ! is_dir( $draft_dir ) ) {
                    if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                        return new WP_REST_Response( [
                            'success' => false,
                            'message' => 'Failed to initialize draft mode. Please check file permissions.'
                        ], 500 );
                    }

                    $this->logger->info( 'Initialized draft mode - copied live to draft', [
                        'page_id' => $page_id,
                        'live_dir' => $live_dir,
                        'draft_dir' => $draft_dir
                    ] );
                }

                // Set draft flag only after successful copy
                update_post_meta( $page_id, '_ai_draft_active', '1' );

                // Initialize _ai_draft_content with current published content
                // So preview shows draft immediately, even before first AI edit
                $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
                if ( $published_content ) {
                    update_post_meta( $page_id, '_ai_draft_content', $published_content );
                }
            } else {
                // BUG FIX: Draft mode active but directory missing - recreate from live
                // This can happen if directory was deleted but flag remained set
                if ( is_dir( $live_dir ) && ! is_dir( $draft_dir ) ) {
                    $this->logger->warning( 'Draft directory missing but flag set - recreating from live', [
                        'page_id' => $page_id
                    ] );

                    if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                        return new WP_REST_Response( [
                            'success' => false,
                            'message' => 'Failed to recreate draft directory. Please check file permissions.'
                        ], 500 );
                    }

                    // Refresh _ai_draft_content from published content
                    $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
                    if ( $published_content ) {
                        update_post_meta( $page_id, '_ai_draft_content', $published_content );
                    }
                }
            }

            // Apply changes to page content
            $result = $this->apply_page_changes( $page_id, $diff_data, $content_before, $prompt );

            if ( is_wp_error( $result ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to apply changes: ' . $result->get_error_message()
                ], 500 );
            }

            // Save linked theme version if theme files were modified
            $theme_version_id = null;
            if ( ! empty( $result['theme_files_modified'] ) && ! empty( $theme_files_before ) && $active_theme_id ) {
                $theme_version_id = $this->save_linked_theme_version( $active_theme_id, $theme_files_before, $page_id, $prompt );
            }

            // Link the versions bidirectionally
            if ( $result['page_version_id'] && $theme_version_id ) {
                $this->link_versions( $result['page_version_id'], $theme_version_id );
            }

            // Save attempt for memory system
            $this->save_attempt( $page_id, $prompt, $diff_data );

            // Trigger cache bust so browsers fetch fresh content
            if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
                ai_site_builder_trigger_cache_bust( 'page', $page_id );
            }

            // Return success
            return new WP_REST_Response( [
                'success' => true,
                'message' => 'Page updated successfully!',
                'user_message' => $diff_data['user_message'] ?? null,
                'changes' => $this->format_changes_for_frontend( $result ),
                'is_conversational' => false
            ], 200 );

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to apply stream diff', [
                'error' => $e->getMessage(),
                'page_id' => $page_id
            ] );

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to apply changes: ' . $e->getMessage()
            ], 500 );
        }
    }

    /**
     * Create preview session for page (similar to theme preview)
     */
    public function rest_create_preview_session( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        $page = get_post( $page_id );
        if ( ! $page ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found'
            ], 404 );
        }

        // Initialize draft mode when preview session starts
        $live_dir = $this->get_page_directory( $page_id, false );
        $draft_dir = $this->get_page_directory( $page_id, true );

        // BUG FIX: If draft mode is active but draft directory is missing, recreate it
        if ( $this->is_draft_mode( $page_id ) && ! is_dir( $draft_dir ) && is_dir( $live_dir ) ) {
            $this->logger->info( 'Draft directory missing but flag set - recreating from live', [
                'page_id' => $page_id
            ] );

            if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                $this->logger->error( 'Failed to recreate draft directory from live', [
                    'page_id' => $page_id
                ] );
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to recreate draft directory. Please check file permissions.'
                ], 500 );
            }

            // Also refresh _ai_draft_content from published content
            $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
            if ( $published_content ) {
                update_post_meta( $page_id, '_ai_draft_content', $published_content );
            }
        }

        if ( ! $this->is_draft_mode( $page_id ) ) {
            // Copy live to draft on first edit (if live directory exists)
            if ( is_dir( $live_dir ) && ! is_dir( $draft_dir ) ) {
                if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                    $this->logger->error( 'Failed to initialize draft directory from live', [
                        'page_id' => $page_id
                    ] );
                    return new WP_REST_Response( [
                        'success' => false,
                        'message' => 'Failed to initialize draft mode. Please check file permissions.'
                    ], 500 );
                }
            }

            // Set draft flag (only after successful copy)
            update_post_meta( $page_id, '_ai_draft_active', '1' );

            // Initialize _ai_draft_content with current published content
            // So preview shows draft immediately
            $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
            if ( $published_content ) {
                update_post_meta( $page_id, '_ai_draft_content', $published_content );
            }

            // BUG FIX: Create initial version when draft mode starts
            // This gives the first edit something to undo to
            $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
            if ( $has_split_files && is_dir( $draft_dir ) ) {
                // Capture published state as initial version
                $published_files = $this->scan_page_files( $page_id, false ); // Scan LIVE directory
                $published_files_json = json_encode( $published_files );

                // Create initial version marking the published state
                $version_control = AI_Version_Control::get_instance();
                $version_control->save_version(
                    'page',
                    $page_id,
                    '', // No before content (this is the baseline)
                    $published_files_json,
                    array(
                        'change_description' => 'Draft mode started - baseline version',
                        'user_prompt' => '',
                        'change_metadata' => array( 'is_baseline' => true )
                    )
                );

                $this->logger->info( 'Created baseline version for draft mode', array(
                    'page_id' => $page_id,
                    'file_count' => count( $published_files )
                ) );
            }

            // BUG FIX: Also create baseline theme version
            // This gives theme edits something to undo to
            $active_theme_id = $this->get_active_ai_theme_id();
            if ( $active_theme_id ) {
                // Check if baseline already exists for this theme
                $version_control = AI_Version_Control::get_instance();
                global $wpdb;
                $table_name = $wpdb->prefix . 'ai_version_control';
                $existing_baseline = $wpdb->get_var( $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE entity_type = 'theme' AND entity_id = %d",
                    $active_theme_id
                ) );

                if ( $existing_baseline == 0 ) {
                    // No baseline exists - create one
                    $theme_path = get_template_directory();
                    $theme_files_snapshot = $this->scan_theme_files( $theme_path );
                    $theme_files_baseline = array();
                    foreach ( $theme_files_snapshot as $relative_path => $full_path ) {
                        if ( file_exists( $full_path ) && is_readable( $full_path ) ) {
                            $theme_files_baseline[$relative_path] = file_get_contents( $full_path );
                        }
                    }

                    if ( ! empty( $theme_files_baseline ) ) {
                        $theme_files_json = json_encode( $theme_files_baseline );
                        $version_control->save_version(
                            'theme',
                            $active_theme_id,
                            '', // No before content (this is the baseline)
                            $theme_files_json,
                            array(
                                'change_description' => 'Editing started - theme baseline version',
                                'user_prompt' => '',
                                'change_metadata' => array( 'is_baseline' => true )
                            )
                        );

                        $this->logger->info( 'Created baseline theme version for editing', array(
                            'theme_id' => $active_theme_id,
                            'file_count' => count( $theme_files_baseline )
                        ) );
                    }
                }
            }
        }

        // Generate preview URL using page's permalink to avoid canonical redirect
        // FIX: Using home_url('/?page_id=X') causes WordPress to 301 redirect to
        // the pretty permalink, which STRIPS all query params including preview=true!
        // Solution: Use get_permalink() as base + add dreamformer_editor for explicit detection
        $permalink = get_permalink( $page_id );
        $preview_url = add_query_arg( [
            'preview' => 'true',
            'dreamformer_editor' => '1',
            'preview_nonce' => wp_create_nonce( 'post_preview_' . $page_id ),
            'cb' => time()
        ], $permalink );

        return new WP_REST_Response( [
            'success' => true,
            'preview_url' => $preview_url,
            'session_id' => 'page_' . $page_id . '_' . time()
        ] );
    }

    /**
     * Publish draft changes to live page
     */
    public function rest_publish_draft( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        $page = get_post( $page_id );
        if ( ! $page ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found'
            ], 404 );
        }

        $draft_dir = $this->get_page_directory( $page_id, true );
        $live_dir = $this->get_page_directory( $page_id, false );

        if ( ! is_dir( $draft_dir ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'No draft to publish'
            ], 400 );
        }

        $backup_dir = null;

        // Backup live version (critical safety measure)
        if ( is_dir( $live_dir ) ) {
            $backup_dir = $live_dir . '-backup-' . time();
            if ( ! rename( $live_dir, $backup_dir ) ) {
                $this->logger->error( 'Failed to backup live directory', [
                    'page_id' => $page_id,
                    'live_dir' => $live_dir
                ] );
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to backup current version'
                ], 500 );
            }

            $this->logger->info( 'Backed up live directory', [
                'page_id' => $page_id,
                'backup_dir' => $backup_dir
            ] );
        }

        // Publish: Rename draft to live (atomic operation)
        if ( ! rename( $draft_dir, $live_dir ) ) {
            // CRITICAL: Restore backup if publish fails
            if ( $backup_dir && is_dir( $backup_dir ) ) {
                rename( $backup_dir, $live_dir );
                $this->logger->error( 'Publish failed, restored backup', [
                    'page_id' => $page_id
                ] );
            }

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to publish draft'
            ], 500 );
        }

        // === BEGIN TRANSACTIONAL DB OPERATIONS ===
        // If any of these fail, we need to rollback the file operation
        $db_operations_failed = false;
        $failure_reason = '';

        // Clear cache BEFORE db operations to prevent stale reads during the update window
        wp_cache_delete( $page_id, 'post_meta' );

        // Clear draft flag and draft preview content
        $meta_deleted_1 = delete_post_meta( $page_id, '_ai_draft_active' );
        $meta_deleted_2 = delete_post_meta( $page_id, '_ai_draft_content' );

        // Note: delete_post_meta returns false if meta doesn't exist, which is OK
        // We only care if it exists and fails to delete - check if it still exists
        if ( get_post_meta( $page_id, '_ai_draft_active', true ) ) {
            $db_operations_failed = true;
            $failure_reason = 'Failed to clear draft flag';
        }

        // Ensure page is published in WordPress (if still a draft)
        if ( ! $db_operations_failed && $page->post_status !== 'publish' ) {
            $update_result = wp_update_post( [
                'ID' => $page_id,
                'post_status' => 'publish'
            ], true );

            if ( is_wp_error( $update_result ) ) {
                $db_operations_failed = true;
                $failure_reason = 'Failed to update post status: ' . $update_result->get_error_message();
            }
        }

        // Regenerate published content from new live files
        if ( ! $db_operations_failed ) {
            $regenerate_result = $this->regenerate_page_content( $page_id );
            if ( is_wp_error( $regenerate_result ) ) {
                $db_operations_failed = true;
                $failure_reason = 'Failed to regenerate content: ' . $regenerate_result->get_error_message();
            }
        }

        // === ROLLBACK IF DB OPERATIONS FAILED ===
        if ( $db_operations_failed ) {
            $this->logger->error( 'DB operations failed after file publish, attempting rollback', [
                'page_id' => $page_id,
                'reason' => $failure_reason
            ] );

            // Rollback: Move live back to draft, restore backup to live
            $rollback_success = false;

            if ( rename( $live_dir, $draft_dir ) ) {
                if ( $backup_dir && is_dir( $backup_dir ) ) {
                    if ( rename( $backup_dir, $live_dir ) ) {
                        $rollback_success = true;
                        // Restore draft flag since we rolled back
                        update_post_meta( $page_id, '_ai_draft_active', '1' );
                        wp_cache_delete( $page_id, 'post_meta' );
                    }
                } else {
                    // No backup to restore (was a new page), rollback is complete
                    $rollback_success = true;
                    update_post_meta( $page_id, '_ai_draft_active', '1' );
                    wp_cache_delete( $page_id, 'post_meta' );
                }
            }

            if ( $rollback_success ) {
                $this->logger->info( 'Successfully rolled back publish operation', [
                    'page_id' => $page_id
                ] );
            } else {
                $this->logger->critical( 'CRITICAL: Failed to rollback publish - page may be in inconsistent state', [
                    'page_id' => $page_id,
                    'live_dir' => $live_dir,
                    'draft_dir' => $draft_dir,
                    'backup_dir' => $backup_dir
                ] );
            }

            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Publish failed: ' . $failure_reason . ( $rollback_success ? ' (changes rolled back)' : ' (WARNING: rollback failed)' )
            ], 500 );
        }
        // === END TRANSACTIONAL DB OPERATIONS ===

        // Clear cache again after all operations complete
        wp_cache_delete( $page_id, 'post_meta' );

        // Cleanup old backups (keep only last 3)
        $this->cleanup_old_backups( $page_id, 3 );

        // Trigger cache bust so browsers fetch fresh published content
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $page_id );
        }

        $this->logger->info( 'Published draft to live', [
            'page_id' => $page_id,
            'live_dir' => $live_dir
        ] );

        return new WP_REST_Response( [
            'success' => true,
            'message' => 'Page published successfully!'
        ] );
    }

    /**
     * Sync draft files when publishing from WordPress admin
     *
     * When a user publishes an AI page from WordPress Pages admin (not Dreamformer),
     * this ensures draft files are copied to live and flags are cleared.
     *
     * @param string  $new_status New post status
     * @param string  $old_status Old post status
     * @param WP_Post $post       Post object
     */
    public function sync_draft_on_wp_publish( $new_status, $old_status, $post ) {
        // Only handle transitions TO publish status
        if ( $new_status !== 'publish' ) {
            return;
        }

        // Only handle pages
        if ( $post->post_type !== 'page' ) {
            return;
        }

        // Only handle AI-generated pages
        if ( ! get_post_meta( $post->ID, 'ai_generated_page', true ) ) {
            return;
        }

        // Only if there's an active draft to sync
        if ( get_post_meta( $post->ID, '_ai_draft_active', true ) !== '1' ) {
            return;
        }

        $page_id = $post->ID;
        $draft_dir = $this->get_page_directory( $page_id, true );
        $live_dir = $this->get_page_directory( $page_id, false );

        // Check draft directory exists
        if ( ! is_dir( $draft_dir ) ) {
            $this->logger->warning( 'Draft flag set but no draft directory', [
                'page_id' => $page_id,
                'draft_dir' => $draft_dir
            ] );
            // Clear stale flag
            delete_post_meta( $page_id, '_ai_draft_active' );
            delete_post_meta( $page_id, '_ai_draft_content' );
            return;
        }

        $this->logger->info( 'Syncing draft on WordPress admin publish', [
            'page_id' => $page_id,
            'old_status' => $old_status,
            'new_status' => $new_status
        ] );

        // Backup live version if it exists
        if ( is_dir( $live_dir ) ) {
            $backup_dir = $live_dir . '-backup-' . time();
            if ( ! rename( $live_dir, $backup_dir ) ) {
                $this->logger->error( 'Failed to backup live directory during WP publish sync', [
                    'page_id' => $page_id
                ] );
                return;
            }
        }

        // Copy draft to live (rename for atomic operation)
        if ( ! rename( $draft_dir, $live_dir ) ) {
            $this->logger->error( 'Failed to sync draft to live during WP publish', [
                'page_id' => $page_id
            ] );
            // Restore backup if it exists
            if ( isset( $backup_dir ) && is_dir( $backup_dir ) ) {
                rename( $backup_dir, $live_dir );
            }
            return;
        }

        // Clear draft flags
        delete_post_meta( $page_id, '_ai_draft_active' );
        delete_post_meta( $page_id, '_ai_draft_content' );

        // Clear cache to ensure fresh data
        wp_cache_delete( $page_id, 'post_meta' );

        // Regenerate page content from live files
        $regenerate_result = $this->regenerate_page_content( $page_id );
        if ( is_wp_error( $regenerate_result ) ) {
            $this->logger->error( 'Failed to regenerate content during WP publish sync', [
                'page_id' => $page_id,
                'error' => $regenerate_result->get_error_message()
            ] );
            // Note: Can't rollback in a hook callback, but page files are still published
            // The content may be stale but at least files are in place
        }

        // Cleanup old backups
        $this->cleanup_old_backups( $page_id, 3 );

        $this->logger->info( 'Successfully synced draft on WordPress admin publish', [
            'page_id' => $page_id
        ] );
    }

    /**
     * Intercept _ai_protected_content reads in preview mode to show draft
     */
    public function filter_protected_content_for_preview( $value, $object_id, $meta_key, $single ) {
        // Only intercept _ai_protected_content reads
        if ( $meta_key !== '_ai_protected_content' ) {
            return $value;
        }

        // Only in preview mode
        if ( ! isset( $_GET['preview'] ) || $_GET['preview'] !== 'true' ) {
            return $value;
        }

        // Only if draft is active
        if ( get_post_meta( $object_id, '_ai_draft_active', true ) !== '1' ) {
            return $value;
        }

        // Return draft content instead
        $draft_content = get_post_meta( $object_id, '_ai_draft_content', true );
        if ( ! empty( $draft_content ) ) {
            // Return as single value (not array) - filter expects this format
            return $draft_content;
        }

        return $value;
    }

    /**
     * Filter content to show draft in preview mode and inject heartbeat script
     */
    public function filter_preview_content( $content ) {
        // Only filter in preview mode
        if ( ! isset( $_GET['preview'] ) || $_GET['preview'] !== 'true' ) {
            return $content;
        }

        $page_id = get_the_ID();
        if ( ! $page_id ) {
            return $content;
        }

        // Inject heartbeat script for freeze detection
        $heartbeat_script = "
            <script>
                (function() {
                    if (window.self === window.top) return; // Don't run outside of iframe
                    setInterval(function() {
                        window.parent.postMessage({ type: 'dreamformer_heartbeat' }, '*');
                    }, 100);
                })();
            </script>
        ";

        // Check if page has active draft
        if ( ! $this->is_draft_mode( $page_id ) ) {
            return $content . $heartbeat_script;
        }

        // NEW: Check for pre-generated draft content first (faster, more reliable)
        $draft_content = get_post_meta( $page_id, '_ai_draft_content', true );
        if ( ! empty( $draft_content ) ) {
            $this->logger->info( 'Showing draft content in preview (from meta)', [
                'page_id' => $page_id,
                'content_size' => strlen( $draft_content )
            ] );
            return $draft_content . $heartbeat_script;
        }

        // FALLBACK: Scan and render draft files if meta doesn't exist
        $draft_files = $this->scan_page_files( $page_id, true );
        if ( ! empty( $draft_files ) ) {
            $draft_content = $this->render_split_files( $draft_files );
            $this->logger->info( 'Showing draft content in preview (from files)', [
                'page_id' => $page_id,
                'file_count' => count( $draft_files )
            ] );
            return $draft_content . $heartbeat_script;
        }

        return $content . $heartbeat_script;
    }

    /**
     * Get edit history for page
     */
    public function rest_get_edit_history( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        // Old post meta history removed - using unified version control
        $history = [];

        return new WP_REST_Response( [
            'success' => true,
            'history' => $history,
            'message' => 'History will be available after unified version control implementation'
        ] );
    }

    /**
     * Get draft status for page
     */
    public function rest_get_draft_status( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        $is_draft = $this->is_draft_mode( $page_id );
        $draft_dir = $this->get_page_directory( $page_id, true );
        $has_draft_files = is_dir( $draft_dir );

        return new WP_REST_Response( [
            'success' => true,
            'is_draft' => $is_draft,
            'has_draft_files' => $has_draft_files
        ] );
    }

    /**
     * Get CSS rules for page element - Enhanced for Manual tab support
     */
    public function rest_get_css_rules( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $selector = sanitize_text_field( $request->get_param( 'selector' ) );

        // Get page content
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found'
            ], 404 );
        }

        // Read from split files (same as AI Editor does)
        // scan_page_files() automatically detects draft mode via is_draft_mode()
        $page_files = $this->scan_page_files( $page_id );

        if ( ! empty( $page_files ) ) {
            // Render split files into combined content (same as AI Editor)
            $page_content = $this->render_split_files( $page_files );
        } else {
            // Fallback for pages without split files (shouldn't happen, but keep as safety)
            $protected_content = get_post_meta( $page_id, '_ai_protected_content', true );
            $page_content = $protected_content ?: $page->post_content;
        }

        // Analyze page content for CSS rules and text content
        $css_analysis = $this->analyze_page_css_rules( $page_content, $selector );
        $text_content = $this->extract_text_content( $page_content, $selector );

        return new WP_REST_Response( [
            'success' => true,
            'css_rules' => $css_analysis['rules'],
            'text_content' => $text_content,
            'debug_info' => [
                'selector' => $selector,
                'page_id' => $page_id,
                'has_split_files' => ! empty( $page_files ),
                'file_count' => count( $page_files ),
                'rules_found' => count( $css_analysis['rules'] ),
                'style_blocks_found' => $css_analysis['style_blocks_found'],
                'inline_styles_found' => $css_analysis['inline_styles_found'],
                'text_elements_found' => count( $text_content ),
                'analysis_method' => ! empty( $page_files ) ? 'split_files' : 'database_fallback'
            ]
        ] );
    }

    /**
     * Analyze page content for CSS rules matching a selector using DOMDocument
     */
    private function analyze_page_css_rules( $page_content, $target_selector ) {
        $css_rules = [];
        $style_blocks_found = 0;
        $inline_styles_found = 0;

        // Create DOMDocument for HTML parsing
        $dom = $this->load_html_content( $page_content );
        if ( ! $dom ) {
            return [
                'rules' => [],
                'style_blocks_found' => 0,
                'inline_styles_found' => 0
            ];
        }

        $xpath = new DOMXPath( $dom );

        // 1. Extract and parse <style> blocks from page content
        $style_tags = $dom->getElementsByTagName( 'style' );
        $style_blocks_found = $style_tags->length;

        foreach ( $style_tags as $index => $style_tag ) {
            $css_content = $style_tag->nodeValue;
            $block_rules = $this->parse_css_for_selector( $css_content, $target_selector, "style-block-{$index}" );
            if ( ! empty( $block_rules ) ) {
                $css_rules = array_merge( $css_rules, $block_rules );
            }
        }

        // 2. Find inline styles on matching elements using XPath
        $nodes = $this->find_nodes_by_selector( $xpath, $target_selector );
        foreach ( $nodes as $node ) {
            $style_attr = $node->getAttribute( 'style' );
            if ( ! empty( $style_attr ) ) {
                $inline_rule = $this->parse_inline_style_to_rule( $style_attr, $target_selector, 'inline-style-' . $inline_styles_found );
                if ( ! empty( $inline_rule ) ) {
                    $css_rules[] = $inline_rule;
                    $inline_styles_found++;
                }
            }
        }

        // 3. Generate editable properties for common CSS properties
        if ( empty( $css_rules ) ) {
            $css_rules = $this->generate_default_editable_properties( $target_selector );
        }

        return [
            'rules' => $css_rules,
            'style_blocks_found' => $style_blocks_found,
            'inline_styles_found' => $inline_styles_found
        ];
    }

    /**
     * Parse CSS content for rules matching a selector (adapted from theme editor)
     */
    private function parse_css_for_selector( $css_content, $target_selector, $source_name ) {
        $rules = [];
        $lines = explode( "\n", $css_content );

        $current_selector = '';
        $in_rule = false;
        $rule_start_line = 0;
        $properties = [];
        $brace_count = 0;

        foreach ( $lines as $line_num => $line ) {
            $line = trim( $line );

            if ( empty( $line ) || strpos( $line, '/*' ) === 0 ) {
                continue;
            }

            if ( strpos( $line, '{' ) !== false ) {
                $brace_count++;
                if ( ! $in_rule && $brace_count == 1 ) {
                    $current_selector = trim( str_replace( '{', '', $line ) );
                    $in_rule = true;
                    $rule_start_line = $line_num + 1;
                    $properties = [];
                }
                continue;
            }

            if ( strpos( $line, '}' ) !== false ) {
                $brace_count--;
                if ( $in_rule && $brace_count == 0 ) {
                    if ( $this->selector_matches_page( $current_selector, $target_selector ) ) {
                        $css_text = $current_selector . " {\n";
                        foreach ( $properties as $property => $value ) {
                            $css_text .= "    " . $property . ": " . $value['value'] . ";\n";
                        }
                        $css_text .= "}";

                        $rules[] = [
                            'file' => $source_name,
                            'line' => $rule_start_line,
                            'selector' => $current_selector,
                            'css' => $css_text,
                            'properties' => $properties,
                            'specificity' => $this->calculate_css_specificity( $current_selector ),
                            'editable' => true,
                            'source_type' => 'page_style_block'
                        ];
                    }

                    $in_rule = false;
                    $current_selector = '';
                    $properties = [];
                }
                continue;
            }

            if ( $in_rule && strpos( $line, ':' ) !== false && $brace_count == 1 ) {
                $parts = explode( ':', $line, 2 );
                if ( count( $parts ) === 2 ) {
                    $property = trim( $parts[0] );
                    $value = trim( rtrim( $parts[1], ';' ) );
                    if ( ! empty( $property ) && ! empty( $value ) ) {
                        $properties[ $property ] = [
                            'value' => $value,
                            'type' => $this->determine_property_type( $property ),
                            'parsed_value' => $value,
                            'important' => false
                        ];
                    }
                }
            }
        }

        return $rules;
    }


    /**
     * Parse inline style attribute into properties array
     */
    private function parse_inline_style_attribute( $style_string ) {
        $properties = [];
        $declarations = explode( ';', $style_string );

        foreach ( $declarations as $declaration ) {
            $declaration = trim( $declaration );
            if ( empty( $declaration ) ) continue;

            $parts = explode( ':', $declaration, 2 );
            if ( count( $parts ) === 2 ) {
                $property = trim( $parts[0] );
                $value = trim( $parts[1] );
                if ( ! empty( $property ) && ! empty( $value ) ) {
                    $properties[ $property ] = [
                        'value' => $value,
                        'type' => $this->determine_property_type( $property ),
                        'parsed_value' => $value,
                        'important' => false
                    ];
                }
            }
        }

        return $properties;
    }

    /**
     * Generate default editable properties for a selector when no existing styles found
     */
    private function generate_default_editable_properties( $selector ) {
        // Provide common editable properties that users can modify
        return [
            [
                'file' => 'new-styles',
                'line' => 0,
                'selector' => $selector,
                'css' => $selector . ' {' . "\n" . '    /* Add your styles here */' . "\n" . '}',
                'properties' => [
                    'color' => [
                        'value' => '#333333',
                        'type' => 'color',
                        'parsed_value' => '#333333',
                        'important' => false
                    ],
                    'background-color' => [
                        'value' => 'transparent',
                        'type' => 'color',
                        'parsed_value' => 'transparent',
                        'important' => false
                    ],
                    'font-size' => [
                        'value' => '16px',
                        'type' => 'size',
                        'parsed_value' => '16px',
                        'important' => false
                    ],
                    'padding' => [
                        'value' => '0px',
                        'type' => 'spacing',
                        'parsed_value' => '0px',
                        'important' => false
                    ],
                    'margin' => [
                        'value' => '0px',
                        'type' => 'spacing',
                        'parsed_value' => '0px',
                        'important' => false
                    ],
                    'border' => [
                        'value' => 'none',
                        'type' => 'text',
                        'parsed_value' => 'none',
                        'important' => false
                    ]
                ],
                'specificity' => 10,
                'editable' => true,
                'source_type' => 'new_rule',
                'placeholder' => true
            ]
        ];
    }

    /**
     * Check if CSS selector matches target for pages (simplified matching)
     */
    private function selector_matches_page( $css_selector, $target_selector ) {
        $css_selector = trim( $css_selector );
        $target_selector = trim( $target_selector );

        // Exact match
        if ( $css_selector === $target_selector ) {
            return true;
        }

        // Class matching
        if ( strpos( $target_selector, '.' ) !== false ) {
            preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $target_selector, $target_matches );
            preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $css_selector, $css_matches );

            $target_classes = $target_matches[1] ?? [];
            $css_classes = $css_matches[1] ?? [];

            foreach ( $target_classes as $target_class ) {
                if ( in_array( $target_class, $css_classes ) ) {
                    return true;
                }
            }
        }

        // ID matching
        if ( strpos( $target_selector, '#' ) !== false ) {
            $target_id = str_replace( '#', '', $target_selector );
            if ( strpos( $css_selector, $target_id ) !== false ) {
                return true;
            }
        }

        // Element matching
        if ( ! strpos( $target_selector, '.' ) && ! strpos( $target_selector, '#' ) ) {
            if ( strpos( $css_selector, $target_selector ) !== false ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Calculate CSS specificity (simplified)
     */
    private function calculate_css_specificity( $selector ) {
        $specificity = 0;
        $specificity += substr_count( $selector, '#' ) * 100;
        $specificity += ( substr_count( $selector, '.' ) + substr_count( $selector, '[' ) ) * 10;
        $specificity += preg_match_all( '/\b[a-zA-Z][\w-]*\b/', $selector );
        return $specificity;
    }

    /**
     * Determine the type of CSS property for proper control rendering
     */
    private function determine_property_type( $property ) {
        $color_properties = [ 'color', 'background-color', 'border-color', 'outline-color' ];
        $size_properties = [ 'font-size', 'width', 'height', 'line-height' ];
        $spacing_properties = [ 'margin', 'padding', 'margin-top', 'margin-bottom', 'margin-left', 'margin-right', 'padding-top', 'padding-bottom', 'padding-left', 'padding-right' ];

        if ( in_array( $property, $color_properties ) ) {
            return 'color';
        } elseif ( in_array( $property, $size_properties ) ) {
            return 'size';
        } elseif ( in_array( $property, $spacing_properties ) ) {
            return 'spacing';
        } else {
            return 'text';
        }
    }

    /**
     * Extract text content from elements matching a selector using DOMDocument
     */
    private function extract_text_content( $page_content, $target_selector ) {
        $text_elements = [];

        // Create DOMDocument for HTML parsing
        $dom = $this->load_html_content( $page_content );
        if ( ! $dom ) {
            return [];
        }

        $xpath = new DOMXPath( $dom );
        $nodes = $this->find_nodes_by_selector( $xpath, $target_selector );

        $match_index = 0;
        foreach ( $nodes as $node ) {
            // Get ALL text content (including from nested elements like <h1><span>Text</span></h1>)
            $text_content = $node->textContent;

            if ( ! empty( trim( $text_content ) ) ) {
                // Clean up text content (remove extra whitespace, decode entities)
                $text_content = trim( preg_replace( '/\s+/', ' ', $text_content ) );
                $text_content = html_entity_decode( $text_content, ENT_QUOTES, 'UTF-8' );

                // Determine element type
                $element_type = strtolower( $node->nodeName );

                $text_elements[] = [
                    'selector' => $target_selector,
                    'content' => $text_content,
                    'element_type' => $element_type,
                    'editable' => $this->is_text_editable_element( $element_type ),
                    'match_index' => $match_index,
                    'pattern_type' => 'dom'
                ];

                $match_index++;
            }
        }

        return $text_elements;
    }


    /**
     * Determine element type from HTML content
     */
    private function determine_element_type_from_content( $html_content ) {
        if ( preg_match( '/<(h[1-6])[^>]*>/', $html_content, $matches ) ) {
            return $matches[1]; // h1, h2, h3, etc.
        } elseif ( preg_match( '/<(p|div|span|button|a|li)[^>]*>/', $html_content, $matches ) ) {
            return $matches[1];
        }

        return 'text';
    }

    /**
     * Check if element type supports text editing
     */
    private function is_text_editable_element( $element_type ) {
        $editable_elements = [ 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'span', 'button', 'a', 'li', 'div' ];
        return in_array( $element_type, $editable_elements );
    }

    /**
     * REST endpoint: Update CSS property in page content
     */
    public function rest_update_page_property( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );
        $selector = sanitize_text_field( $request->get_param( 'selector' ) );
        $property = sanitize_text_field( $request->get_param( 'property' ) );
        $value = sanitize_textarea_field( $request->get_param( 'value' ) );

        // Additional parameters for text content updates
        $match_index = intval( $request->get_param( 'match_index' ) );
        $element_type = sanitize_text_field( $request->get_param( 'element_type' ) );

        $this->logger->debug( 'Property update request', [
            'page_id' => $page_id,
            'selector' => $selector,
            'property' => $property,
            'value' => $value,
            'is_position_adjustment' => ( strpos( $property, 'nudge-' ) === 0 || strpos( $property, 'position-' ) === 0 )
        ] );


        // Get page
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page not found'
            ], 404 );
        }

        // Initialize draft mode on first manual edit (same logic as AI editing endpoint)
        $live_dir = $this->get_page_directory( $page_id, false );
        $draft_dir = $this->get_page_directory( $page_id, true );

        if ( ! $this->is_draft_mode( $page_id ) ) {
            // Copy live to draft on first edit (if live directory exists)
            if ( is_dir( $live_dir ) && ! is_dir( $draft_dir ) ) {
                if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                    return new WP_REST_Response( [
                        'success' => false,
                        'message' => 'Failed to initialize draft mode. Please check file permissions.'
                    ], 500 );
                }

                $this->logger->info( 'Manual editor initialized draft mode - copied live to draft', [
                    'page_id' => $page_id
                ] );
            } elseif ( ! is_dir( $live_dir ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'No page files found. Please generate content first.'
                ], 404 );
            }

            // Set draft flag only after successful copy
            update_post_meta( $page_id, '_ai_draft_active', '1' );

            // Initialize _ai_draft_content with current published content
            $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
            if ( $published_content ) {
                update_post_meta( $page_id, '_ai_draft_content', $published_content );
            }
        } else {
            // BUG FIX: Draft mode active but directory missing - recreate from live
            if ( is_dir( $live_dir ) && ! is_dir( $draft_dir ) ) {
                $this->logger->warning( 'Manual editor: Draft directory missing but flag set - recreating from live', [
                    'page_id' => $page_id
                ] );

                if ( ! $this->copy_directory( $live_dir, $draft_dir ) ) {
                    return new WP_REST_Response( [
                        'success' => false,
                        'message' => 'Failed to recreate draft directory. Please check file permissions.'
                    ], 500 );
                }

                // Refresh _ai_draft_content from published content
                $published_content = get_post_meta( $page_id, '_ai_protected_content', true );
                if ( $published_content ) {
                    update_post_meta( $page_id, '_ai_draft_content', $published_content );
                }
            }
        }

        // VERSION CONTROL: Capture BEFORE state (must do this BEFORE updating file)
        $files_before = $this->scan_page_files( $page_id, true );

        // Update the file directly (all pages use split files)
        if ( $property === 'text-content' ) {
            // TEXT CONTENT: Update index.html file
            $update_result = $this->update_draft_html_content( $page_id, $selector, $value, $match_index, $element_type );
        } else {
            // CSS PROPERTY: Update styles.css file
            $update_result = $this->update_draft_css_property( $page_id, $selector, $property, $value );
        }

        if ( is_wp_error( $update_result ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to update file: ' . $update_result->get_error_message()
            ], 500 );
        }

        // Regenerate _ai_draft_content cache from updated files (same as AI edits)
        $regenerate_result = $this->regenerate_draft_preview( $page_id );
        if ( is_wp_error( $regenerate_result ) ) {
            $this->logger->error( 'Failed to regenerate draft preview after manual edit', [
                'page_id' => $page_id,
                'error' => $regenerate_result->get_error_message()
            ] );
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'File updated but failed to regenerate preview: ' . $regenerate_result->get_error_message()
            ], 500 );
        }

        // VERSION CONTROL: Capture AFTER state and save version
        $files_after = $this->scan_page_files( $page_id, true );
        $this->save_page_version(
            $page_id,
            json_encode( $files_before ),
            json_encode( $files_after ),
            array(),
            "Manual edit: {$selector} → {$property}: {$value}"
        );

        $this->logger->debug( 'Manual edit saved to draft file', [
            'page_id' => $page_id,
            'selector' => $selector,
            'property' => $property,
            'file_type' => $property === 'text-content' ? 'HTML' : 'CSS'
        ] );

        // Trigger cache bust so browsers fetch fresh content
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $page_id );
        }

        $message = $property === 'text-content' ? 'Text content updated successfully' : 'Property updated successfully';

        return new WP_REST_Response( [
            'success' => true,
            'message' => $message,
            'selector' => $selector,
            'property' => $property,
            'value' => $value
        ] );
    }

    /**
     * Update text content in page elements using DOMDocument
     */
    private function update_page_text_content( $content, $selector, $new_text, $match_index = 0, $element_type = '' ) {
        // Use DOMDocument for reliable HTML parsing
        $dom = $this->load_html_content( $content );
        if ( ! $dom ) {
            return $content;
        }

        $xpath = new DOMXPath( $dom );
        $nodes = $this->find_nodes_by_selector( $xpath, $selector );

        // Update the specific match by index
        $current_index = 0;
        foreach ( $nodes as $node ) {
            if ( $current_index === $match_index ) {
                // Update the direct text content of this node
                foreach ( $node->childNodes as $child ) {
                    if ( $child->nodeType === XML_TEXT_NODE ) {
                        $child->nodeValue = $new_text;
                        break; // Only update the first text node
                    }
                }
                break;
            }
            $current_index++;
        }

        // Extract the body content
        $body = $dom->getElementsByTagName( 'body' )->item( 0 );
        $result = '';
        foreach ( $body->childNodes as $child ) {
            $result .= $dom->saveHTML( $child );
        }

        return $result;
    }

    /**
     * Update CSS property in page content
     */
    private function update_page_css_property( $content, $selector, $property, $value ) {
        // Use DOMDocument for reliable HTML parsing (handles malformed HTML)
        $dom = $this->load_html_content( $content );
        if ( ! $dom ) {
            return $content;
        }

        $xpath = new DOMXPath( $dom );
        $nodes = $this->find_nodes_by_selector( $xpath, $selector );

        $updated = false;
        foreach ( $nodes as $node ) {
            $current_style = $node->getAttribute( 'style' );
            $updated_style = $this->update_inline_style_property( $current_style, $property, $value );
            $node->setAttribute( 'style', $updated_style );
            $updated = true;
        }

        // Also update CSS in <style> blocks
        $style_tags = $dom->getElementsByTagName( 'style' );
        foreach ( $style_tags as $style_tag ) {
            $css_content = $style_tag->nodeValue;
            $updated_css = $this->update_css_in_style_block( $css_content, $selector, $property, $value );
            if ( $updated_css !== $css_content ) {
                $style_tag->nodeValue = $updated_css;
                $updated = true;
            }
        }

        if ( $updated ) {
            // Extract the body content
            $body = $dom->getElementsByTagName( 'body' )->item( 0 );
            $result = '';
            foreach ( $body->childNodes as $child ) {
                $result .= $dom->saveHTML( $child );
            }
            return $result;
        }

        return $content;
    }

    /**
     * Update CSS property in a <style> block
     */
    private function update_css_in_style_block( $css_content, $selector, $property, $value ) {
        // Simple CSS update - find the selector and update its properties
        $pattern = '/' . preg_quote( $selector, '/' ) . '\s*\{([^}]*)\}/';

        if ( preg_match( $pattern, $css_content, $matches ) ) {
            $rules = $matches[1];
            $updated_rules = $this->update_css_rule_property( $rules, $property, $value );
            $updated_css = str_replace( $matches[0], $selector . ' {' . $updated_rules . '}', $css_content );
            return $updated_css;
        }

        return $css_content;
    }

    /**
     * Update a property in CSS rules string
     */
    private function update_css_rule_property( $rules, $property, $value ) {
        $properties = [];
        $declarations = preg_split( '/;(?![^(]*\))/', $rules );
        $found = false;

        foreach ( $declarations as $declaration ) {
            $declaration = trim( $declaration );
            if ( empty( $declaration ) ) continue;

            $parts = explode( ':', $declaration, 2 );
            if ( count( $parts ) === 2 ) {
                $prop = trim( $parts[0] );
                $val = trim( $parts[1] );

                if ( $prop === $property ) {
                    $properties[] = $property . ': ' . $value;
                    $found = true;
                } else {
                    $properties[] = $prop . ': ' . $val;
                }
            }
        }

        if ( ! $found ) {
            $properties[] = $property . ': ' . $value;
        }

        return "\n    " . implode( ";\n    ", $properties ) . ";\n";
    }

    /**
     * Update a specific property in an inline style string
     */
    private function update_inline_style_property( $style_string, $property, $value ) {
        // Special handling for transform property to preserve existing transforms
        if ( $property === 'transform' ) {
            return $this->merge_transform_property( $style_string, $value );
        }

        $styles = [];
        $declarations = explode( ';', $style_string );

        $property_found = false;

        foreach ( $declarations as $declaration ) {
            $declaration = trim( $declaration );
            if ( empty( $declaration ) ) continue;

            $parts = explode( ':', $declaration, 2 );
            if ( count( $parts ) === 2 ) {
                $existing_property = trim( $parts[0] );
                $existing_value = trim( $parts[1] );

                if ( $existing_property === $property ) {
                    // Update existing property
                    $styles[] = $property . ': ' . $value;
                    $property_found = true;
                } else {
                    // Keep other properties
                    $styles[] = $existing_property . ': ' . $existing_value;
                }
            }
        }

        // Add new property if it wasn't found
        if ( ! $property_found ) {
            $styles[] = $property . ': ' . $value;
        }

        return implode( '; ', $styles ) . ';';
    }

    /**
     * Merge transform functions intelligently
     * Preserves existing transforms while adding/updating specific functions
     */
    private function merge_transform_property( $style_string, $new_transform ) {
        $styles = [];
        $declarations = explode( ';', $style_string );
        $existing_transforms = [];
        $transform_found = false;

        // First pass - extract all styles and existing transform
        foreach ( $declarations as $declaration ) {
            $declaration = trim( $declaration );
            if ( empty( $declaration ) ) continue;

            $parts = explode( ':', $declaration, 2 );
            if ( count( $parts ) === 2 ) {
                $prop = trim( $parts[0] );
                $val = trim( $parts[1] );

                if ( $prop === 'transform' ) {
                    $transform_found = true;
                    // Parse existing transform functions
                    $existing_transforms = $this->parse_transform_functions( $val );
                } else {
                    $styles[] = $prop . ': ' . $val;
                }
            }
        }

        // Parse new transform functions
        $new_transforms = $this->parse_transform_functions( $new_transform );

        // Merge transforms - new values override existing for same function
        foreach ( $new_transforms as $func => $val ) {
            $existing_transforms[$func] = $val;
        }

        // Rebuild transform string
        if ( ! empty( $existing_transforms ) ) {
            $transform_parts = [];
            // Maintain order: translate, rotate, scale, skew, others
            $order = ['translateX', 'translateY', 'translate', 'rotate', 'rotateX', 'rotateY', 'rotateZ', 'scale', 'scaleX', 'scaleY', 'skewX', 'skewY', 'skew'];

            // Add ordered functions first
            foreach ( $order as $func ) {
                if ( isset( $existing_transforms[$func] ) ) {
                    $transform_parts[] = $func . '(' . $existing_transforms[$func] . ')';
                    unset( $existing_transforms[$func] );
                }
            }

            // Add remaining functions
            foreach ( $existing_transforms as $func => $val ) {
                $transform_parts[] = $func . '(' . $val . ')';
            }

            $styles[] = 'transform: ' . implode( ' ', $transform_parts );
        }

        return implode( '; ', $styles ) . ';';
    }

    /**
     * Parse transform string into individual functions
     * e.g., "translateX(10px) rotate(45deg)" -> ['translateX' => '10px', 'rotate' => '45deg']
     */
    private function parse_transform_functions( $transform_string ) {
        $functions = [];

        // Match function_name(value) pattern
        preg_match_all( '/(\w+)\(([^)]+)\)/', $transform_string, $matches, PREG_SET_ORDER );

        foreach ( $matches as $match ) {
            $function_name = $match[1];
            $function_value = $match[2];

            // Handle translate(x, y) -> translateX and translateY
            if ( $function_name === 'translate' && strpos( $function_value, ',' ) !== false ) {
                $values = array_map( 'trim', explode( ',', $function_value ) );
                if ( count( $values ) >= 2 ) {
                    $functions['translateX'] = $values[0];
                    $functions['translateY'] = $values[1];
                } else {
                    $functions[$function_name] = $function_value;
                }
            } else {
                $functions[$function_name] = $function_value;
            }
        }

        return $functions;
    }

    /**
     * Load HTML content into DOMDocument with proper error handling
     */
    private function load_html_content( $content ) {
        $dom = new DOMDocument();

        // Suppress errors for malformed HTML
        libxml_use_internal_errors( true );

        // Load with UTF-8 support and HTML5 compatibility
        $loaded = $dom->loadHTML(
            '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>' . $content . '</body></html>',
            LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
        );

        libxml_clear_errors();

        return $loaded ? $dom : null;
    }

    /**
     * Find DOM nodes by CSS selector using XPath
     */
    private function find_nodes_by_selector( $xpath, $selector ) {
        if ( strpos( $selector, '.' ) === 0 ) {
            // Class selector - handles malformed class attributes
            $class_name = substr( $selector, 1 );
            return $xpath->query( "//*[contains(concat(' ', normalize-space(@class), ' '), ' {$class_name} ')]" );
        } elseif ( strpos( $selector, '#' ) === 0 ) {
            // ID selector
            $id_name = substr( $selector, 1 );
            return $xpath->query( "//*[@id='{$id_name}']" );
        } else {
            // Element selector
            return $xpath->query( "//{$selector}" );
        }
    }

    /**
     * Get direct text content of a node (excluding child elements)
     */
    private function get_direct_text_content( $node ) {
        $text = '';
        foreach ( $node->childNodes as $child ) {
            if ( $child->nodeType === XML_TEXT_NODE ) {
                $text .= $child->nodeValue;
            }
        }
        return $text;
    }

    /**
     * Parse inline style attribute into a rule format
     */
    private function parse_inline_style_to_rule( $style_string, $selector, $source ) {
        $properties = $this->parse_inline_style_attribute( $style_string );

        if ( empty( $properties ) ) {
            return null;
        }

        $css_text = $selector . " {\n";
        foreach ( $properties as $prop => $data ) {
            $css_text .= "    " . $prop . ": " . $data['value'] . ";\n";
        }
        $css_text .= "}";

        return [
            'selector' => $selector,
            'source' => $source,
            'line' => 0,
            'properties' => $properties,
            'css_text' => $css_text,
            'specificity' => 1000 // Inline styles have highest specificity
        ];
    }



    /**
     * Apply diff changes to page content
     */
    private function apply_page_changes( $page_id, $diff_data, $content_before = '', $user_prompt = '' ) {
        if ( empty( $diff_data['files'] ) ) {
            return new WP_Error( 'no_changes', 'No file changes found' );
        }

        $total_changes_applied = 0;
        $theme_files_modified = [];
        $page_content_modified = false;
        $page_files_modified = false;

        // Process ALL files in the diff, not just page_content.html
        foreach ( $diff_data['files'] as $filename => $file_data ) {
            $changes = $file_data['changes'] ?? [];

            if ( empty( $changes ) ) {
                continue;
            }

            if ( $filename === 'page_content.html' ) {
                // ERROR: page_content.html should NEVER be edited directly
                // It's a build artifact auto-generated by render_split_files()
                // Claude should only edit split files (styles.css, app.js, index.html, etc.)
                $this->logger->error( 'INVALID OPERATION: Attempted to edit page_content.html directly', [
                    'page_id' => $page_id,
                    'filename' => $filename,
                    'reason' => 'page_content.html is a build artifact and should not be edited',
                    'solution' => 'Edit split files instead (styles.css, app.js, index.html)'
                ] );

                // Skip this change - it should never happen
                continue;
            } elseif ( strpos( $filename, 'NEW:' ) === 0 ) {
                // Create new page file
                $new_filename = substr( $filename, 4 );
                $result = $this->create_page_file( $page_id, $new_filename, $file_data );
                if ( ! is_wp_error( $result ) ) {
                    $total_changes_applied += count( $changes );
                    $page_files_modified = true;
                    $this->logger->info( "Created new page file", [
                        'page_id' => $page_id,
                        'filename' => $new_filename,
                        'changes' => count( $changes )
                    ] );
                } else {
                    $this->logger->error( "Failed to create page file", [
                        'filename' => $new_filename,
                        'error' => $result->get_error_message()
                    ] );
                }
            } elseif ( $this->is_page_file( $page_id, $filename ) ) {
                // Update existing page file
                $result = $this->update_page_file( $page_id, $filename, $file_data );
                if ( ! is_wp_error( $result ) ) {
                    $total_changes_applied += count( $changes );
                    $page_files_modified = true;
                    $this->logger->info( "Updated existing page file", [
                        'page_id' => $page_id,
                        'filename' => $filename,
                        'changes' => count( $changes )
                    ] );
                } else {
                    $this->logger->error( "Failed to update page file", [
                        'filename' => $filename,
                        'error' => $result->get_error_message()
                    ] );
                }
            } else {
                // Handle theme file changes (style.css, functions.php, etc.)
                // Strip "theme:" prefix if present (from API response format)
                $actual_filename = $filename;
                if ( strpos( $filename, 'theme:' ) === 0 ) {
                    $actual_filename = substr( $filename, 6 ); // Strip "theme:" prefix
                    $this->logger->debug( "Stripped theme: prefix from filename", [
                        'original' => $filename,
                        'actual' => $actual_filename
                    ] );
                }

                $theme_path = get_template_directory();
                $file_path = $theme_path . '/' . $actual_filename;

                // Security check - ensure file is within theme directory
                $real_theme_path = realpath( $theme_path );
                $real_file_path = realpath( dirname( $file_path ) );

                if ( ! $real_file_path || strpos( $real_file_path, $real_theme_path ) !== 0 ) {
                    $this->logger->error( "Security: Attempted to modify file outside theme directory", [
                        'file' => $filename,
                        'theme_path' => $theme_path
                    ] );
                    continue;
                }

                // Read current file content
                if ( ! file_exists( $file_path ) ) {
                    $this->logger->warning( "Theme file not found", [ 'file' => $file_path ] );
                    continue;
                }

                $current_content = AI_File_Manager::read( $file_path );
                if ( is_wp_error( $current_content ) ) {
                    $this->logger->error( "Failed to read theme file", [
                        'file' => $file_path,
                        'error' => $current_content->get_error_message()
                    ] );
                    continue;
                }

                $lines = explode( "\n", $current_content );
                $changes_applied = 0;

                // Apply changes to theme file
                foreach ( $changes as $index => $change ) {
                    $this->logger->debug( "Applying theme change {$index} to {$actual_filename}", [
                        'change_data' => $change
                    ] );

                    $change_result = $this->apply_single_change( $lines, $change );
                    if ( ! is_wp_error( $change_result ) ) {
                        $changes_applied++;
                        $this->logger->debug( "Theme change {$index} applied to {$actual_filename}" );
                    } else {
                        $this->logger->debug( "Theme change {$index} failed for {$actual_filename}", [
                            'error' => $change_result->get_error_message()
                        ] );
                    }
                }

                if ( $changes_applied > 0 ) {
                    // Write updated theme file
                    $updated_content = implode( "\n", $lines );
                    $write_result = AI_File_Manager::write( $file_path, $updated_content );

                    if ( ! is_wp_error( $write_result ) ) {
                        $theme_files_modified[] = $actual_filename;
                        $total_changes_applied += $changes_applied;
                        $this->logger->info( "Theme file updated", [
                            'file' => $actual_filename,
                            'changes' => $changes_applied
                        ] );
                    } else {
                        $this->logger->error( "Failed to write theme file", [
                            'file' => $file_path,
                            'error' => $write_result->get_error_message()
                        ] );
                    }
                }
            }
        }

        // CRITICAL FIX: Clear all caches after theme file modifications
        // This prevents the next edit from reading stale cached content
        if ( ! empty( $theme_files_modified ) ) {
            $theme_path = get_template_directory();

            // 1. Clear PHP OpCache for each modified file
            if ( function_exists( 'opcache_invalidate' ) ) {
                foreach ( $theme_files_modified as $filename ) {
                    $full_path = $theme_path . '/' . $filename;
                    if ( file_exists( $full_path ) ) {
                        $opcache_cleared = opcache_invalidate( $full_path, true );
                        $this->logger->info( "Cleared OpCache for theme file", [
                            'file' => $filename,
                            'success' => $opcache_cleared
                        ] );
                    }
                }
            }

            // 2. Clear WordPress transient cache for theme file list
            // This forces next scan to see the updated file list
            $dir_mtime = filemtime( $theme_path );
            $cache_key = 'theme_scan_' . md5( $theme_path . '_' . $dir_mtime );
            delete_transient( $cache_key );
            delete_transient( $cache_key . '_time' );

            // 3. Also clear any cache with older mtime (file changes update mtime)
            // Delete all theme_scan_* transients to be safe
            global $wpdb;
            $wpdb->query(
                "DELETE FROM {$wpdb->options}
                WHERE option_name LIKE '_transient_theme_scan_%'
                OR option_name LIKE '_transient_timeout_theme_scan_%'"
            );

            $this->logger->info( "Cleared all caches after theme modifications", [
                'modified_files' => $theme_files_modified,
                'opcache_available' => function_exists( 'opcache_invalidate' )
            ] );
        }

        if ( $total_changes_applied === 0 ) {
            return new WP_Error( 'no_changes_applied', 'No changes could be applied' );
        }

        // Build comprehensive result info
        $result_info = [
            'total_changes' => $total_changes_applied,
            'page_modified' => $page_content_modified,
            'theme_files_modified' => $theme_files_modified
        ];

        $this->logger->info( "Page editor changes applied", $result_info );

        // BUG FIX: Regenerate DRAFT content for preview, but don't update post_content (live)
        // During editing: Update _ai_draft_content meta for preview filter
        // When publishing: Line 1032 regenerates post_content from live files
        if ( $page_files_modified ) {
            $regenerate_result = $this->regenerate_draft_preview( $page_id );
            if ( is_wp_error( $regenerate_result ) ) {
                $this->logger->error( 'Failed to regenerate draft preview after applying changes', [
                    'page_id' => $page_id,
                    'error' => $regenerate_result->get_error_message()
                ] );
                // Return error - files were modified but preview sync failed
                return new WP_Error(
                    'preview_sync_failed',
                    'Changes applied to files but failed to sync preview: ' . $regenerate_result->get_error_message()
                );
            }
        }

        // VERSION CONTROL: Save version after successful content update
        // BUG FIX: Always create page version if theme files were modified (even if page unchanged)
        // This creates a "marker" version that theme versions can link to for undo/redo
        $page_version_id = null;
        if ( $page_content_modified || $page_files_modified || ! empty( $theme_files_modified ) ) {
            // Capture content after changes (same format as before)
            // For split files: JSON with all file contents
            // For monolithic pages: combined HTML

            // BUG FIX: Check if page uses split files (this was missing!)
            $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );

            if ( $has_split_files ) {
                $page_files_after = $this->scan_page_files( $page_id );

                // CRITICAL BUG FIX: Don't save empty versions - would cause data loss on restore
                // This happens when only theme is modified but page directory is empty/missing
                if ( empty( $page_files_after ) && ! $page_files_modified ) {
                    $this->logger->warning( 'Skipping page version - no page files to save', [
                        'page_id' => $page_id,
                        'page_modified' => $page_content_modified,
                        'theme_modified' => ! empty( $theme_files_modified ),
                        'reason' => 'Page directory is empty and page was not modified'
                    ] );
                    $page_version_id = null;  // Don't save version
                } else {
                    $content_after = json_encode( $page_files_after );
                    $page_version_id = $this->save_page_version( $page_id, $content_before, $content_after, $diff_data, $user_prompt );
                }
            } else {
                $protected_content = get_post_meta( $page_id, '_ai_protected_content', true );
                $content_after = $protected_content ?: get_post_field( 'post_content', $page_id );
                $page_version_id = $this->save_page_version( $page_id, $content_before, $content_after, $diff_data, $user_prompt );
            }
        }

        return [
            'changes_applied' => $total_changes_applied,
            'page_modified' => $page_content_modified,
            'theme_files_modified' => $theme_files_modified,
            'content_length' => $page_content_modified ? strlen( get_post_field( 'post_content', $page_id ) ) : 0,
            'page_version_id' => $page_version_id
        ];
    }

    /**
     * Apply single change to content lines
     */
    private function apply_single_change( &$lines, $change ) {
        $old_text = $change['old'] ?? null;
        $new_text = $change['new'] ?? '';  // Allow empty string for deletions

        // Log content change attempt for debugging
        $this->logger->debug( 'Processing content diff change', [
            'old_text_preview' => substr( $old_text, 0, 100 ),
            'new_text_preview' => substr( $new_text, 0, 100 ),
            'target_line' => $change['target_line'] ?? 'none',
            'context_before' => $change['context_before'] ?? 'none',
            'context_after' => $change['context_after'] ?? 'none'
        ] );

        // Handle insertions (empty old_text means we're adding new content)
        if ( $old_text === '' || $old_text === null ) {
            // For insertions, we need context_before or context_after to know where to insert
            if ( ! empty( $change['context_after'] ) && is_array( $change['context_after'] ) ) {
                // Find the context and insert before it
                $context_after = implode( "\n", $change['context_after'] );
                $content = implode( "\n", $lines );

                if ( strpos( $content, $context_after ) !== false ) {
                    // Use safe replacement with uniqueness check
                    $updated_content = $this->str_replace_once(
                        $content,
                        $context_after,
                        $new_text . "\n" . $context_after
                    );

                    // Check if replacement failed due to non-unique pattern
                    if ( is_wp_error( $updated_content ) ) {
                        $this->logger->error( 'SAFETY: Insertion before context_after failed', [
                            'error' => $updated_content->get_error_message(),
                            'context_after_preview' => substr( $context_after, 0, 100 )
                        ] );
                        return $updated_content; // Return the error
                    }

                    $lines = explode( "\n", $updated_content );
                    return true;
                }
            }

            if ( ! empty( $change['context_before'] ) && is_array( $change['context_before'] ) ) {
                // Find the context and insert after it
                $context_before = implode( "\n", $change['context_before'] );
                $content = implode( "\n", $lines );

                if ( strpos( $content, $context_before ) !== false ) {
                    // Use safe replacement with uniqueness check
                    $updated_content = $this->str_replace_once(
                        $content,
                        $context_before,
                        $context_before . "\n" . $new_text
                    );

                    // Check if replacement failed due to non-unique pattern
                    if ( is_wp_error( $updated_content ) ) {
                        $this->logger->error( 'SAFETY: Insertion after context_before failed', [
                            'error' => $updated_content->get_error_message(),
                            'context_before_preview' => substr( $context_before, 0, 100 )
                        ] );
                        return $updated_content; // Return the error
                    }

                    $lines = explode( "\n", $updated_content );
                    return true;
                }
            }

            // CRITICAL FIX: Never silently append when insertion context fails
            // This was causing CSS duplication by appending on every failed match
            $this->logger->error( 'DIFF DEBUG: Insertion failed - no valid context found', [
                'new_text_preview' => substr( $new_text, 0, 200 ),
                'has_context_before' => ! empty( $change['context_before'] ),
                'has_context_after' => ! empty( $change['context_after'] )
            ] );
            return new WP_Error( 'insertion_context_failed', 'Cannot insert without valid context - prevents accidental duplication' );
        }

        // Handle replacements with safe, context-aware matching
        $content = implode( "\n", $lines );

        // Use target_line for precise replacement if provided
        if ( isset( $change['target_line'] ) && is_numeric( $change['target_line'] ) ) {
            $target_line_num = intval( $change['target_line'] ) - 1; // Convert to 0-based index

            $this->logger->info( 'DIFF DEBUG: Trying target_line match', [
                'target_line' => $change['target_line'],
                'target_line_num' => $target_line_num,
                'line_exists' => isset( $lines[$target_line_num] ) ? 'yes' : 'no',
                'actual_line' => isset( $lines[$target_line_num] ) ? $lines[$target_line_num] : 'N/A',
                'expected_old' => $old_text,
                'exact_match' => isset( $lines[$target_line_num] ) && $lines[$target_line_num] === $old_text ? 'yes' : 'no'
            ] );

            if ( isset( $lines[$target_line_num] ) && $lines[$target_line_num] === $old_text ) {
                // Exact line match - safe to replace
                $lines[$target_line_num] = $new_text;
                $this->logger->info( 'DIFF DEBUG: target_line replacement SUCCESS' );
                return true;
            }
        }

        // Fallback: Use context-aware replacement (safer than global replace)
        if ( !empty( $change['context_before'] ) && is_array( $change['context_before'] ) ) {
            $context_before = implode( "\n", $change['context_before'] );
            $search_pattern = $context_before . "\n" . $old_text;
            $replace_with = $context_before . "\n" . $new_text;

            if ( strpos( $content, $search_pattern ) !== false ) {
                // Use safe replacement with uniqueness check
                $updated_content = $this->str_replace_once( $content, $search_pattern, $replace_with );

                // Check if replacement failed due to non-unique pattern
                if ( is_wp_error( $updated_content ) ) {
                    $this->logger->error( 'SAFETY: Context before replacement failed', [
                        'error' => $updated_content->get_error_message(),
                        'search_pattern_preview' => substr( $search_pattern, 0, 100 )
                    ] );
                    return $updated_content; // Return the error
                }

                $lines = explode( "\n", $updated_content );
                return true;
            }
        }

        if ( !empty( $change['context_after'] ) && is_array( $change['context_after'] ) ) {
            $context_after = implode( "\n", $change['context_after'] );
            $search_pattern = $old_text . "\n" . $context_after;
            $replace_with = $new_text . "\n" . $context_after;

            if ( strpos( $content, $search_pattern ) !== false ) {
                // Use safe replacement with uniqueness check
                $updated_content = $this->str_replace_once( $content, $search_pattern, $replace_with );

                // Check if replacement failed due to non-unique pattern
                if ( is_wp_error( $updated_content ) ) {
                    $this->logger->error( 'SAFETY: Context after replacement failed', [
                        'error' => $updated_content->get_error_message(),
                        'search_pattern_preview' => substr( $search_pattern, 0, 100 )
                    ] );
                    return $updated_content; // Return the error
                }

                $lines = explode( "\n", $updated_content );
                return true;
            }
        }

        // FUZZY MATCHING - DISABLED FOR SAFETY
        // Fuzzy matching was causing mass duplication bugs when patterns weren't unique.
        // Example: Trying to match "</div>" would match ALL 95+ occurrences and create duplicates.
        // Better to fail with clear error than silently corrupt files.
        // Re-enable only after adding uniqueness validation to prevent corruption.

        $this->logger->info( 'DIFF DEBUG: Fuzzy matching is disabled for safety' );

        $this->logger->error( 'DIFF DEBUG: All exact replacement methods FAILED', [
            'old_text_preview' => substr( $old_text, 0, 100 ),
            'had_target_line' => isset( $change['target_line'] ),
            'had_context_before' => !empty( $change['context_before'] ),
            'had_context_after' => !empty( $change['context_after'] ),
            'suggestion' => 'Pattern not found with exact matching. File may have changed or pattern may not be unique. Try providing more specific context or a longer unique string.'
        ] );

        return new WP_Error(
            'match_failed',
            'Could not find exact match for replacement. The pattern may appear multiple times or the file may have changed since analysis. Try being more specific about the location (e.g., "in the header section" or "after the hero image").'
        );
    }

    /**
     * Normalize whitespace for fuzzy matching
     */
    private function normalize_whitespace( $text ) {
        // Remove leading/trailing whitespace from each line
        $lines = explode( "\n", $text );
        $normalized_lines = array_map( 'trim', $lines );

        // Remove empty lines
        $normalized_lines = array_filter( $normalized_lines, function( $line ) {
            return !empty( $line );
        });

        // Join with single newlines
        return implode( "\n", $normalized_lines );
    }

    /**
     * Find fuzzy match for old text in content
     */
    private function find_fuzzy_match( $content, $old_text ) {
        $normalized_old = $this->normalize_whitespace( $old_text );
        $normalized_content = $this->normalize_whitespace( $content );

        // Check if normalized versions match
        if ( strpos( $normalized_content, $normalized_old ) === false ) {
            return false;
        }

        // Find the actual text with original whitespace
        $content_lines = explode( "\n", $content );
        $old_lines = explode( "\n", $old_text );
        $normalized_old_lines = array_filter( array_map( 'trim', $old_lines ), function( $line ) {
            return !empty( $line );
        });

        // Look for consecutive matching lines in content
        for ( $i = 0; $i <= count( $content_lines ) - count( $normalized_old_lines ); $i++ ) {
            $found_match = true;
            $match_start = $i;
            $match_end = $i;
            $old_line_index = 0;

            for ( $j = $i; $j < count( $content_lines ) && $old_line_index < count( $normalized_old_lines ); $j++ ) {
                $content_line = trim( $content_lines[$j] );

                // Skip empty lines in content
                if ( empty( $content_line ) ) {
                    continue;
                }

                if ( $content_line === $normalized_old_lines[$old_line_index] ) {
                    if ( $old_line_index === 0 ) {
                        $match_start = $j;
                    }
                    $match_end = $j;
                    $old_line_index++;
                } else {
                    $found_match = false;
                    break;
                }
            }

            if ( $found_match && $old_line_index === count( $normalized_old_lines ) ) {
                // Extract the original text from content with its whitespace
                $matched_lines = array_slice( $content_lines, $match_start, $match_end - $match_start + 1 );
                return implode( "\n", $matched_lines );
            }
        }

        return false;
    }

    /**
     * Find fuzzy match with context
     */
    private function find_fuzzy_match_with_context( $content, $old_text, $context, $position ) {
        $normalized_context = $this->normalize_whitespace( $context );
        $normalized_old = $this->normalize_whitespace( $old_text );
        $normalized_content = $this->normalize_whitespace( $content );

        if ( $position === 'before' ) {
            $search_pattern = $normalized_context . "\n" . $normalized_old;
        } else {
            $search_pattern = $normalized_old . "\n" . $normalized_context;
        }

        if ( strpos( $normalized_content, $search_pattern ) === false ) {
            return false;
        }

        // Find the actual text with original whitespace
        $content_lines = explode( "\n", $content );
        $context_lines = array_filter( array_map( 'trim', explode( "\n", $context ) ), function( $line ) {
            return !empty( $line );
        });
        $old_lines = array_filter( array_map( 'trim', explode( "\n", $old_text ) ), function( $line ) {
            return !empty( $line );
        });

        if ( $position === 'before' ) {
            $search_lines = array_merge( $context_lines, $old_lines );
        } else {
            $search_lines = array_merge( $old_lines, $context_lines );
        }

        // Look for consecutive matching lines
        for ( $i = 0; $i <= count( $content_lines ) - count( $search_lines ); $i++ ) {
            $found_match = true;
            $match_start = $i;
            $match_end = $i;
            $search_line_index = 0;

            for ( $j = $i; $j < count( $content_lines ) && $search_line_index < count( $search_lines ); $j++ ) {
                $content_line = trim( $content_lines[$j] );

                // Skip empty lines in content
                if ( empty( $content_line ) ) {
                    continue;
                }

                if ( $content_line === $search_lines[$search_line_index] ) {
                    if ( $search_line_index === 0 ) {
                        $match_start = $j;
                    }
                    $match_end = $j;
                    $search_line_index++;
                } else {
                    $found_match = false;
                    break;
                }
            }

            if ( $found_match && $search_line_index === count( $search_lines ) ) {
                // Extract the matched portion
                $matched_lines = array_slice( $content_lines, $match_start, $match_end - $match_start + 1 );
                return implode( "\n", $matched_lines );
            }
        }

        return false;
    }

    /**
     * Save edit history - REMOVED
     * Now using unified version control system instead of post meta
     */

    /**
     * Format changes for frontend display (match theme editor format)
     */
    private function format_changes_for_frontend( $result ) {
        return [
            [
                'type' => 'page_content',
                'description' => sprintf( 'Updated page content (%d changes)', $result['changes_applied'] ),
                'file' => 'page_content.html',
                'changes_count' => $result['changes_applied']
            ]
        ];
    }

    /**
     * Save page version for version control
     *
     * @param int $page_id Page post ID
     * @param string $content_before Page content before changes
     * @param string $content_after Page content after changes
     * @param array $diff_data Diff data from AI response
     * @param string $user_prompt Original user prompt
     * @return int|WP_Error Version ID or error
     */
    private function save_page_version( $page_id, $content_before, $content_after, $diff_data, $user_prompt ) {
        try {
            // Generate change description
            $change_description = $diff_data['user_message'] ?? 'Page content updated';

            // Prepare version control data
            $version_control = AI_Version_Control::get_instance();
            $version_id = $version_control->save_version( 'page', $page_id,
                $content_before,
                $content_after,
                array(
                    'change_description' => $change_description,
                    'user_prompt' => $user_prompt,
                    'ai_model' => AI_Config_Manager::AI_MODEL,
                    'change_metadata' => array(
                        'content_length_before' => strlen( $content_before ),
                        'content_length_after' => strlen( $content_after ),
                        'changes_applied' => count( $diff_data['files']['page_content.html']['changes'] ?? [] ),
                        'edit_type' => 'page_content',
                        'has_diff_data' => ! empty( $diff_data )
                    )
                )
            );

            if ( is_wp_error( $version_id ) ) {
                $this->logger->error( 'Failed to save page version', array(
                    'page_id' => $page_id,
                    'error' => $version_id->get_error_message()
                ) );
                return $version_id;
            }

            $this->logger->info( 'Page version saved successfully', array(
                'page_id' => $page_id,
                'version_id' => $version_id,
                'content_length_before' => strlen( $content_before ),
                'content_length_after' => strlen( $content_after ),
                'change_description' => $change_description
            ) );

            return $version_id;

        } catch ( Exception $e ) {
            $this->logger->error( 'Exception saving page version', array(
                'page_id' => $page_id,
                'error' => $e->getMessage()
            ) );
            return new WP_Error( 'version_save_exception', $e->getMessage() );
        }
    }

    /**
     * REST API: Undo page change
     */
    public function rest_undo_page_change( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        $this->logger->info( 'Processing page undo request', array(
            'page_id' => $page_id
        ) );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Page not found'
            ), 404 );
        }

        $version_control = AI_Version_Control::get_instance();
        $result = $version_control->undo_change( 'page', $page_id );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        // Get button states for frontend
        $button_states = $this->get_page_version_button_states( $page_id );

        // Trigger cache bust so browsers fetch fresh content after undo
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $page_id );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Undo successful',
            'can_undo' => $button_states['can_undo'],
            'can_redo' => $button_states['can_redo'],
            'version' => array(
                'id' => $result->id,
                'version_number' => $result->version_number,
                'change_description' => $result->change_description,
                'created_at' => $result->created_at
            )
        ) );
    }

    /**
     * REST API: Redo page change
     */
    public function rest_redo_page_change( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        $this->logger->info( 'Processing page redo request', array(
            'page_id' => $page_id
        ) );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Page not found'
            ), 404 );
        }

        $version_control = AI_Version_Control::get_instance();
        $result = $version_control->redo_change( 'page', $page_id );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        // Get button states for frontend
        $button_states = $this->get_page_version_button_states( $page_id );

        // Trigger cache bust so browsers fetch fresh content after redo
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $page_id );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Redo successful',
            'can_undo' => $button_states['can_undo'],
            'can_redo' => $button_states['can_redo'],
            'version' => array(
                'id' => $result->id,
                'version_number' => $result->version_number,
                'change_description' => $result->change_description,
                'created_at' => $result->created_at
            )
        ) );
    }

    /**
     * REST API: Get page version history
     */
    public function rest_get_page_versions( WP_REST_Request $request ) {
        $page_id = intval( $request->get_param( 'page_id' ) );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Page not found'
            ), 404 );
        }

        $version_control = AI_Version_Control::get_instance();

        // Create a proper REST request for the central API
        $history_request = new WP_REST_Request( 'GET', '/ai-builder/v1/versions/page/' . $page_id . '/history' );
        $history_request->set_param( 'entity_type', 'page' );
        $history_request->set_param( 'entity_id', $page_id );
        $response = $version_control->rest_get_version_history( $history_request );

        // Enhance with button states
        $button_states = $this->get_page_version_button_states( $page_id );

        if ( $response instanceof WP_REST_Response ) {
            $data = $response->get_data();
            $data['can_undo'] = $button_states['can_undo'];
            $data['can_redo'] = $button_states['can_redo'];
            return new WP_REST_Response( $data );
        }

        return $response;
    }

    /**
     * Get page version button states (can undo/redo)
     */
    private function get_page_version_button_states( $page_id ) {
        global $wpdb;

        $version_control = AI_Version_Control::get_instance();
        $table_name = $wpdb->prefix . 'ai_version_control';

        // Get current version
        $current_version = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE entity_type = 'page' AND entity_id = %d AND is_current = 1",
            $page_id
        ) );

        if ( ! $current_version ) {
            return array( 'can_undo' => false, 'can_redo' => false );
        }

        // Can undo if version number > 1
        $can_undo = $current_version->version_number > 1;

        // Can redo if there are versions with higher numbers
        $next_version_exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name}
             WHERE entity_type = 'page' AND entity_id = %d AND version_number > %d",
            $page_id, $current_version->version_number
        ) ) > 0;

        return array(
            'can_undo' => $can_undo,
            'can_redo' => $next_version_exists
        );
    }

    /**
     * Get AI-enhanced context with page content + detected theme files
     *
     * @param string $prompt User prompt
     * @param string $page_content Current page content
     * @param int $page_id Page ID for scanning page files
     * @return array Context files (page content + AI-detected theme files)
     */
    private function get_ai_enhanced_context( $prompt, $page_content, $page_id, $user_context = array() ) {
        // Let AI detection decide which files to include (including page_content)
        $context = [];

        // CRITICAL FIX: Enhance prompt with element context BEFORE file detection
        $enhanced_prompt = $prompt;
        if ( ! empty( $user_context['selected_element'] ) ) {
            $element_info = $user_context['selected_element'];
            if ( ! empty( $user_context['element_type'] ) ) {
                $element_info .= ' (' . $user_context['element_type'] . ')';
            }
            $enhanced_prompt = sprintf(
                "User selected element: %s. %s",
                $element_info,
                $prompt
            );


            // MVP SOLUTION: Tell AI EXACTLY which files contain the element
            $element_files = $this->find_element_files( $user_context['selected_element'], $page_id );

            if ( ! empty( $element_files ) ) {
                $files_list = implode( ', ', $element_files );
                $enhanced_prompt .= sprintf(
                    "\n\n🎯 ELEMENT LOCATION: The element '%s' exists in: %s\nYou MUST include these files in your selection.",
                    $user_context['selected_element'],
                    $files_list
                );
            } else {
            }
        }

        // Use AI to detect if theme files are needed (same logic as theme editor)
        $theme_path = get_template_directory();

        try {
            // Log theme path before scanning

            // Scan theme directory for available files
            $scanned_files = $this->scan_theme_files( $theme_path );

            // Log scan results
            if ( count( $scanned_files ) > 0 ) {
            }

            // Scan page directory for available files
            $page_files = $this->scan_page_files( $page_id );

            // Log page files
            if ( count( $page_files ) > 0 ) {
            }

            // Generate metadata WITHOUT loading full file contents (for intelligent selection)
            $file_metadata = $this->generate_file_metadata( $scanned_files, $page_files, $page_id );
            $available_files = array_column( $file_metadata, 'name' );

            // Log metadata breakdown
            $metadata_by_type = array_count_values( array_column( $file_metadata, 'type' ) );

            if ( empty( $available_files ) ) {
                $this->logger->warning( 'No files found to scan' );
                return $context;
            }

            $this->logger->info( 'Generated file metadata for detection', [
                'files_count' => count( $file_metadata ),
                'total_size' => array_sum( array_column( $file_metadata, 'size' ) )
            ] );

            // Use AI to detect required files (with metadata for smart selection)
            // CRITICAL: Use enhanced_prompt with element context, not raw prompt
            $file_detection = $this->get_vercel_client()->detect_files( $enhanced_prompt, $available_files, $theme_path, $file_metadata );

            if ( ! is_wp_error( $file_detection ) && ! empty( $file_detection['required_files'] ) ) {
                // Load detected files (both theme files, page files, and page content)
                foreach ( $file_detection['required_files'] as $filename ) {
                    if ( $filename === 'page_content.html' ) {
                        // Add page content only if Claude requested it
                        $context[$filename] = $page_content;
                    } elseif ( isset( $page_files[$filename] ) ) {
                        // Add page file content
                        $context[$filename] = $page_files[$filename];
                    } elseif ( strpos( $filename, 'theme:' ) === 0 ) {
                        // Handle theme files with theme: prefix
                        $actual_filename = substr( $filename, 6 ); // Strip "theme:" prefix
                        if ( isset( $scanned_files[$actual_filename] ) ) {
                            $file_path = $scanned_files[$actual_filename];
                            if ( file_exists( $file_path ) ) {
                                $context[$filename] = file_get_contents( $file_path );
                            }
                        }
                    } elseif ( isset( $scanned_files[$filename] ) ) {
                        // Fallback: theme file without prefix (add prefix to context key)
                        $file_path = $scanned_files[$filename];
                        if ( file_exists( $file_path ) ) {
                            $context['theme:' . $filename] = file_get_contents( $file_path );
                        }
                    }
                }

                $this->logger->info( 'Dreamformer editor: Added AI-detected files', [
                    'detected_files' => $file_detection['required_files'],
                    'total_context_files' => count( $context ),
                    'ai_reasoning' => $file_detection['reasoning'] ?? 'not_provided'
                ] );
            } else {
                $this->logger->info( 'Dreamformer editor: AI detection returned no files', [
                    'detection_result' => is_wp_error( $file_detection ) ? $file_detection->get_error_message() : 'no_files_detected'
                ] );
            }

        } catch ( Exception $e ) {
            $this->logger->error( 'Dreamformer editor: AI file detection failed', [
                'error' => $e->getMessage(),
                'fallback' => 'using_page_only_context'
            ] );
        }

        // SMART CSS EXTRACTION: If context is too large, extract relevant parts
        $context = $this->apply_smart_css_extraction( $context, $user_context );

        return $context;
    }

    /**
     * Apply smart CSS extraction to reduce token usage
     *
     * @param array $context Current context with file contents
     * @param array $user_context User context (selected_element, viewport, etc.)
     * @return array Optimized context
     */
    private function apply_smart_css_extraction( $context, $user_context ) {

        // Calculate total size
        $total_size = 0;
        foreach ( $context as $filename => $content ) {
            $total_size += strlen( $content );
        }

        $estimated_tokens = $total_size / 2;

        // If under 150k tokens, send as-is
        if ( $estimated_tokens < 150000 ) {
            $this->logger->debug( 'Context size acceptable, no extraction needed', [
                'total_size' => $this->format_bytes( $total_size ),
                'estimated_tokens' => number_format( $estimated_tokens )
            ] );
            return $context;
        }

        // Extract CSS files that are too large
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-css-extractor.php';
        $extractor = new AI_CSS_Extractor();

        foreach ( $context as $filename => $content ) {
            // Only extract CSS files
            if ( ! preg_match( '/\.css$/i', $filename ) ) {
                continue;
            }

            $file_size = strlen( $content );

            // Extract if file is >100KB
            if ( $file_size > 100000 ) {
                $this->logger->info( 'Applying CSS extraction to large file', [
                    'filename' => $filename,
                    'original_size' => $this->format_bytes( $file_size )
                ] );

                $extracted = $extractor->extract( $content, $user_context );
                $context[ $filename ] = $extracted;
            }
        }

        // Recalculate size after extraction
        $new_total_size = 0;
        foreach ( $context as $content ) {
            $new_total_size += strlen( $content );
        }

        $this->logger->info( 'Smart CSS extraction applied', [
            'original_size' => $this->format_bytes( $total_size ),
            'extracted_size' => $this->format_bytes( $new_total_size ),
            'reduction' => round( ( 1 - ( $new_total_size / $total_size ) ) * 100, 1 ) . '%'
        ] );

        return $context;
    }

    /**
     * Scan theme directory for available files (replicate AI_Simple_File_Editor logic)
     *
     * @param string $theme_path Theme directory path
     * @return array Array of filename => full_path
     */
    private function scan_theme_files( $theme_path ) {
        $files = [];
        $allowed_extensions = [ '.php', '.css', '.js', '.html', '.htm', '.json', '.xml', '.txt', '.md', '.scss', '.sass', '.less', '.twig', '.mustache', '.hbs', '.svg', '.webmanifest' ];

        // Check if directory exists
        if ( ! is_dir( $theme_path ) ) {
            return $files;
        }

        // PERFORMANCE: Cache scan results with auto-invalidation
        // Cache key includes directory mtime (catches file add/remove)
        $dir_mtime = filemtime( $theme_path );
        $cache_key = 'theme_scan_' . md5( $theme_path . '_' . $dir_mtime );
        $cache_time_key = $cache_key . '_time';

        $cached = get_transient( $cache_key );
        $cache_time = get_transient( $cache_time_key );

        // Use cache if exists and less than 5 minutes old
        if ( $cached !== false && $cache_time !== false ) {
            $cache_age = time() - $cache_time;
            if ( $cache_age < 300 ) {
                return $cached;
            }
        }

        try {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $theme_path, RecursiveDirectoryIterator::SKIP_DOTS ),
                RecursiveIteratorIterator::SELF_FIRST
            );

            $total_iterated = 0;
            $skipped_dirs = 0;
            $skipped_hidden = 0;
            $skipped_excluded = 0;
            $skipped_extension = 0;

            foreach ( $iterator as $file ) {
                $total_iterated++;

                if ( $file->isDir() ) {
                    $skipped_dirs++;
                    continue;
                }

                if ( strpos( $file->getFilename(), '.' ) === 0 ) {
                    $skipped_hidden++;
                    continue;
                }

                $path_string = $file->getPathname();
                if ( preg_match( '/\/(node_modules|vendor|\.git|\.svn|cache|tmp|temp)\//i', $path_string ) ) {
                    $skipped_excluded++;
                    continue;
                }

                $extension = '.' . strtolower( $file->getExtension() );
                if ( ! in_array( $extension, $allowed_extensions, true ) ) {
                    $skipped_extension++;
                    continue;
                }

                $relative_path = str_replace( $theme_path . '/', '', $file->getPathname() );
                $files[$relative_path] = $file->getPathname();
            }

            // Log iteration results

            // PERFORMANCE: Cache results for 1 hour
            set_transient( $cache_key, $files, HOUR_IN_SECONDS );
            set_transient( $cache_time_key, time(), HOUR_IN_SECONDS );

        } catch ( Exception $e ) {
            $this->logger->error( 'Theme file scanning failed', [ 'error' => $e->getMessage() ] );
        }

        return $files;
    }

    /**
     * Scan page directory for available files
     *
     * @param int $page_id Page ID
     * @return array Array of filename => content
     */
    private function scan_page_files( $page_id, $is_draft = null ) {
        // Auto-detect draft mode if not specified
        if ( $is_draft === null ) {
            $is_draft = $this->is_draft_mode( $page_id );
        }

        $page_dir = $this->get_page_directory( $page_id, $is_draft );


        if ( !is_dir( $page_dir ) ) {
            return [];
        }

        // NOTE: Page files are NOT cached because:
        // 1. They're small (few files per page)
        // 2. They change frequently during editing
        // 3. We need AI to see changes immediately
        // Only theme files are cached (they're large and rarely change)

        $files = [];
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $page_dir, RecursiveDirectoryIterator::SKIP_DOTS )
        );

        foreach ( $iterator as $file ) {
            if ( $file->isFile() && !$file->isDir() ) {
                $relative_path = str_replace( $page_dir . '/', '', $file->getPathname() );

                // Security: Prevent directory traversal
                if ( strpos( $relative_path, '..' ) !== false ) {
                    continue;
                }

                // Skip .htaccess security files
                if ( basename( $relative_path ) === '.htaccess' ) {
                    continue;
                }

                $files[$relative_path] = file_get_contents( $file->getPathname() );
            }
        }

        return $files;
    }

    /**
     * Extract CSS selectors using cached AST parser
     *
     * @param string $file_path Full path to CSS file
     * @return array Array of CSS selectors
     */
    private function extract_css_selectors_cached( $file_path ) {
        // Check cache first - cache key includes modification time for auto-invalidation
        $cache_key = 'css_meta_' . md5( $file_path . filemtime( $file_path ) );
        $cached = get_transient( $cache_key );

        if ( $cached !== false ) {
            return $cached;
        }

        // Parse with existing AST parser
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/css-parser/class-ai-css-ast-parser.php';
        $parser = new AI_CSS_AST_Parser();

        $content = file_get_contents( $file_path );
        if ( $content === false ) {
            return [];
        }

        $document = $parser->parse_css_content( $content, $file_path );

        if ( is_wp_error( $document ) ) {
            return [];
        }

        $selectors = [];
        foreach ( $document->getAllDeclarationBlocks() as $block ) {
            foreach ( $block->getSelectors() as $selector ) {
                $selectors[] = $selector->getSelector();
            }
        }

        // Cache for 1 hour
        set_transient( $cache_key, $selectors, HOUR_IN_SECONDS );

        return $selectors;
    }

    /**
     * Extract CSS selectors from content string (for page files already loaded)
     *
     * @param string $content CSS content
     * @return array Array of CSS selectors
     */
    private function extract_css_selectors_from_content( $content ) {
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/css-parser/class-ai-css-ast-parser.php';
        $parser = new AI_CSS_AST_Parser();

        $document = $parser->parse_css_content( $content );

        if ( is_wp_error( $document ) ) {
            return [];
        }

        $selectors = [];
        foreach ( $document->getAllDeclarationBlocks() as $block ) {
            foreach ( $block->getSelectors() as $selector ) {
                $selectors[] = $selector->getSelector();
            }
        }

        return $selectors;
    }

    /**
     * Add conflict markers to metadata array
     * Detects when same CSS selectors exist in multiple files
     *
     * @param array $metadata Metadata array (passed by reference)
     */
    private function add_conflict_markers( &$metadata ) {
        $selector_map = [];

        // Build map: selector => [file names]
        foreach ( $metadata as $file ) {
            if ( empty( $file['css_selectors'] ) ) {
                continue;
            }

            foreach ( $file['css_selectors'] as $selector ) {
                if ( ! isset( $selector_map[ $selector ] ) ) {
                    $selector_map[ $selector ] = [];
                }
                $selector_map[ $selector ][] = $file['name'];
            }
        }

        // Mark conflicts in metadata
        foreach ( $metadata as &$file ) {
            if ( empty( $file['css_selectors'] ) ) {
                continue;
            }

            $conflicts = [];
            $conflict_selectors = [];

            foreach ( $file['css_selectors'] as $selector ) {
                if ( count( $selector_map[ $selector ] ) > 1 ) {
                    // This selector exists in multiple files - it's a conflict
                    foreach ( $selector_map[ $selector ] as $conflict_file ) {
                        if ( $conflict_file !== $file['name'] ) {
                            $conflicts[] = $conflict_file;
                        }
                    }
                    $conflict_selectors[] = $selector;
                }
            }

            if ( ! empty( $conflicts ) ) {
                $file['conflicts'] = array_unique( $conflicts );
                $file['conflict_selectors'] = array_unique( $conflict_selectors );
            }
        }
    }

    /**
     * Generate file metadata WITHOUT loading full contents
     * This allows Claude to make intelligent file selection decisions
     *
     * @param array $theme_files Theme files from scan_theme_files
     * @param array $page_files Page files from scan_page_files
     * @param int $page_id Page ID
     * @return array File metadata
     */
    private function generate_file_metadata( $theme_files, $page_files, $page_id ) {
        $metadata = [];

        // IMPORTANT: page_content.html is EXCLUDED from file detection
        // Reason: It's a build artifact auto-generated by render_split_files()
        // Claude should ONLY edit split files (styles.css, app.js, index.html, etc.)
        // This prevents:
        // 1. Loading expensive 320KB blob when 20KB split file is enough (97% cost savings)
        // 2. Split files getting out of sync (no reverse sync exists)
        // 3. Duplicate content being sent to Claude (page_content contains all split files combined)

        // Add theme files (use filesize for speed - don't read contents)
        foreach ( $theme_files as $relative_path => $full_path ) {
            if ( ! file_exists( $full_path ) ) {
                continue;
            }

            $extension = pathinfo( $full_path, PATHINFO_EXTENSION );
            $size = filesize( $full_path );

            $file_meta = [
                'name' => 'theme:' . $relative_path,
                'type' => 'theme',
                'extension' => $extension,
                'size' => $size,
                'size_formatted' => $this->format_bytes( $size )
            ];

            // Add CSS metadata using AST parser (cached for performance)
            if ( $extension === 'css' ) {
                $selectors = $this->extract_css_selectors_cached( $full_path );
                if ( ! empty( $selectors ) ) {
                    $file_meta['css_selectors'] = $selectors;
                }
            }

            $metadata[] = $file_meta;
        }

        // Add page files (we already have content loaded from scan_page_files)
        foreach ( $page_files as $filename => $content ) {
            $extension = pathinfo( $filename, PATHINFO_EXTENSION );

            $file_meta = [
                'name' => $filename,
                'type' => 'page',
                'extension' => $extension,
                'size' => strlen( $content ),
                'size_formatted' => $this->format_bytes( strlen( $content ) )
            ];

            // Add CSS metadata from already-loaded content (no cache needed)
            if ( $extension === 'css' ) {
                $selectors = $this->extract_css_selectors_from_content( $content );
                if ( ! empty( $selectors ) ) {
                    $file_meta['css_selectors'] = $selectors;
                }
            }

            $metadata[] = $file_meta;
        }

        // Detect and mark CSS selector conflicts
        $this->add_conflict_markers( $metadata );

        return $metadata;
    }

    /**
     * Format bytes to human readable size
     *
     * @param int $bytes Number of bytes
     * @return string Formatted size (e.g., "51KB", "327KB")
     */
    private function format_bytes( $bytes ) {
        if ( $bytes < 1024 ) {
            return $bytes . 'B';
        }
        if ( $bytes < 1048576 ) {
            return round( $bytes / 1024, 1 ) . 'KB';
        }
        return round( $bytes / 1048576, 1 ) . 'MB';
    }

    /**
     * Create a new page file
     *
     * @param int $page_id Page ID
     * @param string $filename Filename to create
     * @param array $file_data File data with content
     * @return true|WP_Error True on success, WP_Error on failure
     */
    private function create_page_file( $page_id, $filename, $file_data ) {
        // Always write to draft during editing
        $page_dir = $this->get_page_directory( $page_id, true );

        // Create page directory if needed
        if ( !is_dir( $page_dir ) ) {
            if ( !wp_mkdir_p( $page_dir ) ) {
                $this->logger->error( 'Failed to create page directory', [
                    'page_id' => $page_id,
                    'page_dir' => $page_dir
                ] );
                return new WP_Error(
                    'dir_create_failed',
                    'Failed to create page directory'
                );
            }

            // Apply WordPress-defined permissions (respects FS_CHMOD_DIR constant)
            $dir_perms = defined( 'FS_CHMOD_DIR' ) ? FS_CHMOD_DIR : ( fileperms( WP_CONTENT_DIR ) & 0777 | 0755 );
            @chmod( $page_dir, $dir_perms );

            // Add security .htaccess
            $htaccess_content = "Options -Indexes\n<FilesMatch '\\.php$'>\n    Order Allow,Deny\n    Deny from all\n</FilesMatch>";
            file_put_contents( $page_dir . '/.htaccess', $htaccess_content );
        }

        // Security: Sanitize filename
        $filename = str_replace( ['../', '..\\', '..'], '', $filename );
        $file_path = $page_dir . '/' . $filename;

        // Create subdirectories if needed
        $dir = dirname( $file_path );
        if ( !is_dir( $dir ) ) {
            if ( !wp_mkdir_p( $dir ) ) {
                $this->logger->error( 'Failed to create subdirectory', [
                    'page_id' => $page_id,
                    'dir' => $dir
                ] );
                return new WP_Error(
                    'subdir_create_failed',
                    sprintf( 'Failed to create subdirectory for: %s', $filename )
                );
            }

            // Apply WordPress-defined permissions
            $dir_perms = defined( 'FS_CHMOD_DIR' ) ? FS_CHMOD_DIR : ( fileperms( WP_CONTENT_DIR ) & 0777 | 0755 );
            @chmod( $dir, $dir_perms );
        }

        // Extract content from NEW file (should be single change with full content)
        $changes = $file_data['changes'] ?? [];
        if ( empty( $changes ) ) {
            return new WP_Error( 'no_changes', 'No changes provided for new file' );
        }

        $content = $changes[0]['new'] ?? '';

        // Write file
        $result = file_put_contents( $file_path, $content );

        if ( $result === false ) {
            $this->logger->error( 'Failed to write new page file', [
                'page_id' => $page_id,
                'filename' => $filename,
                'file_path' => $file_path
            ] );
            return new WP_Error(
                'file_write_failed',
                sprintf( 'Failed to write file: %s', $filename )
            );
        }

        $this->logger->info( 'Created new page file', [
            'page_id' => $page_id,
            'filename' => $filename,
            'size' => strlen( $content )
        ] );

        return true;
    }

    /**
     * Update a page file with changes from diff
     *
     * @param int $page_id Page ID
     * @param string $filename Filename to update
     * @param array $file_data File data with changes
     * @return true|WP_Error True on success, WP_Error on failure
     */
    private function update_page_file( $page_id, $filename, $file_data ) {
        // Always write to draft during editing
        $page_dir = $this->get_page_directory( $page_id, true );
        $file_path = $page_dir . '/' . $filename;

        if ( !file_exists( $file_path ) ) {
            return $this->create_page_file( $page_id, $filename, $file_data );
        }

        // Read current content and apply changes (same logic as page_content.html)
        $current_content = file_get_contents( $file_path );
        if ( $current_content === false ) {
            $this->logger->error( 'Failed to read page file for update', [
                'page_id' => $page_id,
                'filename' => $filename,
                'file_path' => $file_path
            ] );
            return new WP_Error(
                'file_read_failed',
                sprintf( 'Failed to read file: %s', $filename )
            );
        }

        $lines = explode( "\n", $current_content );
        $changes_failed = 0;

        foreach ( $file_data['changes'] as $change ) {
            $change_result = $this->apply_single_change( $lines, $change );
            if ( is_wp_error( $change_result ) ) {
                $changes_failed++;
                $this->logger->error( 'Failed to apply change to page file', [
                    'page_id' => $page_id,
                    'filename' => $filename,
                    'error' => $change_result->get_error_message()
                ] );
            }
        }

        // Write updated content
        $updated_content = implode( "\n", $lines );
        $result = file_put_contents( $file_path, $updated_content );

        if ( $result === false ) {
            $this->logger->error( 'Failed to write updated page file', [
                'page_id' => $page_id,
                'filename' => $filename,
                'file_path' => $file_path
            ] );
            return new WP_Error(
                'file_write_failed',
                sprintf( 'Failed to write file: %s', $filename )
            );
        }

        $this->logger->info( 'Updated page file', [
            'page_id' => $page_id,
            'filename' => $filename,
            'size' => strlen( $updated_content ),
            'changes_failed' => $changes_failed
        ] );

        return true;
    }

    /**
     * Safely replace first occurrence of pattern with uniqueness validation
     * Prevents mass duplication bug by checking pattern appears exactly once
     *
     * @param string $content  Source content to search in
     * @param string $old_text Pattern to find
     * @param string $new_text Replacement text
     * @return string|WP_Error Modified content or error if pattern not unique
     */
    private function str_replace_once( $content, $old_text, $new_text ) {
        // Count occurrences of the pattern
        $count = substr_count( $content, $old_text );

        // Pattern not found
        if ( $count === 0 ) {
            $this->logger->warning( 'SAFETY: Pattern not found in content', [
                'pattern_preview' => substr( $old_text, 0, 100 )
            ] );

            return new WP_Error(
                'pattern_not_found',
                sprintf(
                    'Pattern not found in content. Pattern: "%s"',
                    substr( $old_text, 0, 100 ) . ( strlen( $old_text ) > 100 ? '...' : '' )
                )
            );
        }

        // Pattern appears multiple times - NOT SAFE TO REPLACE
        if ( $count > 1 ) {
            $this->logger->warning( 'SAFETY: Pattern appears multiple times, refusing to prevent corruption', [
                'pattern_preview' => substr( $old_text, 0, 100 ),
                'occurrences' => $count
            ] );

            return new WP_Error(
                'non_unique_pattern',
                sprintf(
                    'Pattern appears %d times in the content. ' .
                    'Please provide more context to make the pattern unique. ' .
                    'This safety check prevents accidental mass duplication. ' .
                    'Pattern: "%s"',
                    $count,
                    substr( $old_text, 0, 100 ) . ( strlen( $old_text ) > 100 ? '...' : '' )
                )
            );
        }

        // Pattern is unique (count === 1) - SAFE to replace
        $this->logger->info( 'SAFETY: Pattern is unique, safe to replace', [
            'pattern_preview' => substr( $old_text, 0, 100 )
        ] );

        $pos = strpos( $content, $old_text );
        return substr_replace(
            $content,
            $new_text,
            $pos,
            strlen( $old_text )
        );
    }

    /**
     * Check if file is a page file
     *
     * @param int $page_id Page ID
     * @param string $filename Filename to check
     * @return bool True if it's a page file
     */
    private function is_page_file( $page_id, $filename ) {
        $upload_dir = wp_upload_dir();
        $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id;
        $file_path = $page_dir . '/' . $filename;

        return file_exists( $file_path );
    }

    /**
     * Analyze diff to detect theme file changes BEFORE applying
     *
     * @param int $page_id Page ID
     * @param array $diff_data Parsed diff data
     * @return array Analysis results with theme_files, page_files, has_theme_changes
     */
    private function analyze_diff_for_theme_files( $page_id, $diff_data ) {
        $theme_files = [];
        $page_files = [];

        if ( empty( $diff_data['files'] ) ) {
            return [
                'theme_files' => [],
                'page_files' => [],
                'has_theme_changes' => false
            ];
        }

        foreach ( $diff_data['files'] as $filename => $file_data ) {
            // Clean filename (remove prefixes)
            $clean_name = $filename;

            // Strip "NEW:" prefix for new files
            $is_new_file = false;
            if ( strpos( $clean_name, 'NEW:' ) === 0 ) {
                $clean_name = substr( $clean_name, 4 );
                $is_new_file = true;
            }

            // Strip "theme:" prefix if present
            if ( strpos( $clean_name, 'theme:' ) === 0 ) {
                $clean_name = substr( $clean_name, 6 );
            }

            // SPECIAL CASE: New files with page-like naming patterns
            // If creating section-*, component-*, block-*, or common page file types, assume page file
            if ( $is_new_file ) {
                if ( preg_match( '/^(section|component|block|layout|partial)-.+\.(html|htm)$/i', $clean_name ) ) {
                    $page_files[] = $clean_name;
                    $this->logger->debug( 'Detected new page file by pattern', [ 'file' => $clean_name ] );
                    continue;
                }
                // Common page asset files
                if ( preg_match( '/^(styles?|app|script|main)\.(css|js)$/i', $clean_name ) ) {
                    $page_files[] = $clean_name;
                    $this->logger->debug( 'Detected new page asset by pattern', [ 'file' => $clean_name ] );
                    continue;
                }
            }

            // Use existing is_page_file() logic to check if file exists in page directory
            if ( $this->is_page_file( $page_id, $clean_name ) ) {
                $page_files[] = $clean_name;
                $this->logger->debug( 'Detected page file (exists in page dir)', [ 'file' => $clean_name ] );
            } else {
                // Not a page file → Must be theme file!
                $theme_files[] = $clean_name;
                $this->logger->debug( 'Detected theme file', [ 'file' => $clean_name ] );
            }
        }

        $has_theme_changes = ! empty( $theme_files );

        if ( $has_theme_changes ) {
            $this->logger->info( 'Theme file changes detected in diff', [
                'page_id' => $page_id,
                'theme_files' => $theme_files,
                'page_files' => $page_files
            ] );
        }

        return [
            'theme_files' => $theme_files,
            'page_files' => $page_files,
            'has_theme_changes' => $has_theme_changes
        ];
    }

    /**
     * Regenerate page content from split files
     *
     * @param int $page_id Page ID
     * @return true|WP_Error True on success, WP_Error on failure
     */
    private function regenerate_page_content( $page_id ) {
        $page_files = $this->scan_page_files( $page_id );

        if ( empty( $page_files ) ) {
            // Check if page directory exists but is empty vs doesn't exist
            $page_dir = $this->get_page_directory( $page_id, false );
            if ( is_dir( $page_dir ) ) {
                // Directory exists but no files - this is an error condition
                $this->logger->error( 'Page directory exists but contains no files', [
                    'page_id' => $page_id,
                    'page_dir' => $page_dir
                ] );
                return new WP_Error(
                    'empty_page_directory',
                    'Page directory exists but contains no files. Content may be corrupted.'
                );
            }
            // No split files and no directory - page uses _ai_protected_content directly
            // This is OK for legacy pages
            return true;
        }

        // Render split files into combined content
        $combined_content = $this->render_split_files( $page_files );

        if ( empty( $combined_content ) ) {
            $this->logger->error( 'Failed to render split files - empty content returned', [
                'page_id' => $page_id,
                'file_count' => count( $page_files )
            ] );
            return new WP_Error(
                'render_failed',
                'Failed to render page content from files.'
            );
        }

        // Update both _ai_protected_content AND post_content for display
        update_post_meta( $page_id, '_ai_protected_content', $combined_content );

        // CRITICAL: Also update post_content so the page actually displays the new content
        $update_result = wp_update_post( [
            'ID' => $page_id,
            'post_content' => $combined_content
        ], true );

        if ( is_wp_error( $update_result ) ) {
            $this->logger->error( 'Failed to update post content', [
                'page_id' => $page_id,
                'error' => $update_result->get_error_message()
            ] );
            return $update_result;
        }

        $this->logger->info( 'Regenerated page content from split files', [
            'page_id' => $page_id,
            'file_count' => count( $page_files ),
            'content_size' => strlen( $combined_content )
        ] );

        return true;
    }

    /**
     * Regenerate draft content for preview ONLY (doesn't update post_content)
     *
     * @param int $page_id Page ID
     * @return true|WP_Error True on success, WP_Error on failure
     */
    private function regenerate_draft_preview( $page_id ) {
        $draft_files = $this->scan_page_files( $page_id, true );  // Scan DRAFT directory

        if ( empty( $draft_files ) ) {
            // Check if draft directory exists but is empty
            $draft_dir = $this->get_page_directory( $page_id, true );
            if ( is_dir( $draft_dir ) ) {
                // Directory exists but no files - this is an error condition
                $this->logger->error( 'Draft directory exists but contains no files', [
                    'page_id' => $page_id,
                    'draft_dir' => $draft_dir
                ] );
                return new WP_Error(
                    'empty_draft_directory',
                    'Draft directory exists but contains no files. Draft may be corrupted.'
                );
            }
            // No draft directory - this shouldn't happen if we're in draft mode
            $this->logger->warning( 'regenerate_draft_preview called but no draft directory exists', [
                'page_id' => $page_id
            ] );
            return new WP_Error(
                'no_draft_directory',
                'Draft directory does not exist.'
            );
        }

        // Render draft files into combined content
        $draft_content = $this->render_split_files( $draft_files );

        if ( empty( $draft_content ) ) {
            $this->logger->error( 'Failed to render draft files - empty content returned', [
                'page_id' => $page_id,
                'file_count' => count( $draft_files )
            ] );
            return new WP_Error(
                'render_failed',
                'Failed to render draft content from files.'
            );
        }

        // Store in separate meta field - preview filter will check this first
        update_post_meta( $page_id, '_ai_draft_content', $draft_content );

        // CRITICAL: Clear WordPress object cache to ensure iframe sees fresh content
        // Without this, persistent caching (Redis/Memcached) might serve stale data
        wp_cache_delete( $page_id, 'post_meta' );

        $this->logger->info( 'Regenerated draft content for preview', [
            'page_id' => $page_id,
            'file_count' => count( $draft_files ),
            'content_size' => strlen( $draft_content )
        ] );

        return true;
    }

    /**
     * Update CSS property in draft styles.css file (for manual edits)
     *
     * @param int $page_id Page ID
     * @param string $selector CSS selector
     * @param string $property CSS property name
     * @param string $value CSS property value
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private function update_draft_css_property( $page_id, $selector, $property, $value ) {
        // Get draft directory
        $draft_dir = $this->get_page_directory( $page_id, true );
        $css_file = $draft_dir . '/styles.css';

        // Check if CSS file exists
        if ( ! file_exists( $css_file ) ) {
            return new WP_Error( 'css_file_not_found', 'Draft CSS file not found' );
        }

        // Read current CSS content
        $css_content = file_get_contents( $css_file );
        if ( $css_content === false ) {
            return new WP_Error( 'css_read_failed', 'Failed to read CSS file' );
        }

        // Update the CSS property
        $updated_css = $this->update_css_in_style_block( $css_content, $selector, $property, $value );

        // If no changes made (selector not found), add new rule
        if ( $updated_css === $css_content ) {
            // Selector doesn't exist, append new rule
            $updated_css .= "\n\n" . $selector . " {\n    " . $property . ": " . $value . ";\n}";
        }

        // Write updated CSS back to file
        $result = file_put_contents( $css_file, $updated_css );
        if ( $result === false ) {
            return new WP_Error( 'css_write_failed', 'Failed to write CSS file' );
        }

        $this->logger->info( 'Updated draft CSS file', [
            'page_id' => $page_id,
            'file' => 'styles.css',
            'selector' => $selector,
            'property' => $property
        ] );

        return true;
    }

    /**
     * Update text content in draft index.html file (for manual edits)
     *
     * @param int $page_id Page ID
     * @param string $selector CSS selector
     * @param string $new_text New text content
     * @param int $match_index Index of element to update (if multiple matches)
     * @param string $element_type Type of element being updated
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private function update_draft_html_content( $page_id, $selector, $new_text, $match_index = 0, $element_type = '' ) {
        // Get draft directory
        $draft_dir = $this->get_page_directory( $page_id, true );
        $html_file = $draft_dir . '/index.html';

        // Check if HTML file exists
        if ( ! file_exists( $html_file ) ) {
            return new WP_Error( 'html_file_not_found', 'Draft HTML file not found' );
        }

        // Read current HTML content
        $html_content = file_get_contents( $html_file );
        if ( $html_content === false ) {
            return new WP_Error( 'html_read_failed', 'Failed to read HTML file' );
        }

        // Update the text content using DOMDocument
        $updated_html = $this->update_text_in_html_file( $html_content, $selector, $new_text, $match_index );

        if ( is_wp_error( $updated_html ) ) {
            return $updated_html;
        }

        // Write updated HTML back to file
        $result = file_put_contents( $html_file, $updated_html );
        if ( $result === false ) {
            return new WP_Error( 'html_write_failed', 'Failed to write HTML file' );
        }

        $this->logger->info( 'Updated draft HTML file', [
            'page_id' => $page_id,
            'file' => 'index.html',
            'selector' => $selector,
            'match_index' => $match_index,
            'text_length' => strlen( $new_text )
        ] );

        return true;
    }

    /**
     * Update text content in HTML file content
     *
     * @param string $html_content HTML content
     * @param string $selector CSS selector
     * @param string $new_text New text content
     * @param int $match_index Index of element to update
     * @return string|WP_Error Updated HTML content or WP_Error on failure
     */
    private function update_text_in_html_file( $html_content, $selector, $new_text, $match_index = 0 ) {
        // Use DOMDocument for reliable HTML parsing
        $dom = $this->load_html_content( $html_content );
        if ( ! $dom ) {
            return new WP_Error( 'html_parse_failed', 'Failed to parse HTML content' );
        }

        $xpath = new DOMXPath( $dom );
        $nodes = $this->find_nodes_by_selector( $xpath, $selector );

        if ( count( $nodes ) === 0 ) {
            return new WP_Error( 'selector_not_found', 'Selector not found in HTML' );
        }

        // Update the specific match by index
        $current_index = 0;
        $updated = false;
        foreach ( $nodes as $node ) {
            if ( $current_index === $match_index ) {
                // Update the direct text content of this node
                foreach ( $node->childNodes as $child ) {
                    if ( $child->nodeType === XML_TEXT_NODE ) {
                        $child->nodeValue = $new_text;
                        $updated = true;
                        break; // Only update the first text node
                    }
                }
                break;
            }
            $current_index++;
        }

        if ( ! $updated ) {
            return new WP_Error( 'text_node_not_found', 'No text node found in element' );
        }

        // Return the full HTML document (not just body)
        return $dom->saveHTML();
    }

    /**
     * Render split files into combined content
     *
     * @param array $files Array of filename => content
     * @return string Combined content
     */
    private function render_split_files( $files ) {
        $html = '';

        // PRODUCTION FIX: Extract HTML ONLY, no scripts or styles
        // CSS and JS are loaded via enqueue_ai_page_assets() instead
        // This prevents WordPress from stripping <script>/<style> tags
        // and displaying code as visible text (Hostinger/Bluehost issue)
        foreach ( $files as $filename => $content ) {
            $extension = pathinfo( $filename, PATHINFO_EXTENSION );

            // Only extract HTML - skip CSS and JS files
            if ( $extension === 'html' || $extension === 'htm' ) {
                $html .= $content . "\n";
            }
        }

        // Return pure HTML only - all assets enqueued via wp_enqueue_scripts
        // Selection handler added via wp_add_inline_script() in enqueue_ai_page_assets()
        return $html;
    }

    /**
     * Get Dreamformer selection handler JavaScript
     * This code is injected into every page to enable element selection via postMessage
     * even when iframe is in isolated process (no allow-same-origin)
     *
     * @return string Selection handler JavaScript code
     */
    private function get_dreamformer_selection_handler() {
        return <<<'JAVASCRIPT'
// ========================================
// DREAMFORMER EDITOR INTEGRATION v2.0
// Selection handler for isolated iframe
// ========================================
(function() {
    'use strict';

    // Skip if not in iframe
    if (window.self === window.top) {
        return;
    }

    let selectedElement = null;

    /**
     * Generate CSS selector for element
     */
    function getElementSelector(element) {
        if (element.id) {
            return '#' + element.id;
        } else if (element.className && element.className.trim()) {
            // Filter out the editor's selection highlight class
            const classes = element.className.trim().split(/\s+/)
                .filter(c => c !== 'ai-editor-selected')
                .join('.');
            if (classes) {
                return '.' + classes;
            }
        }
        return element.tagName.toLowerCase();
    }

    /**
     * Get computed styles for element
     */
    function getComputedStyles(element) {
        const styles = window.getComputedStyle(element);
        return {
            // Visual styles
            background: styles.backgroundColor || 'transparent',
            color: styles.color || 'inherit',
            fontSize: styles.fontSize || '16px',

            // TIER 1: Critical for debugging positioning/layout issues
            position: styles.position,
            top: styles.top,
            left: styles.left,
            right: styles.right,
            bottom: styles.bottom,

            // Spacing (often the issue!)
            marginTop: styles.marginTop,
            marginBottom: styles.marginBottom,
            marginLeft: styles.marginLeft,
            marginRight: styles.marginRight,
            paddingTop: styles.paddingTop,
            paddingBottom: styles.paddingBottom,
            paddingLeft: styles.paddingLeft,
            paddingRight: styles.paddingRight,

            // Dimensions
            width: styles.width,
            maxWidth: styles.maxWidth,
            height: styles.height,

            // Display & layout
            display: styles.display,
            textAlign: styles.textAlign,
            overflow: styles.overflow,

            // Visibility & layering
            zIndex: styles.zIndex,
            opacity: styles.opacity,
            visibility: styles.visibility,

            // Transforms
            transform: styles.transform
        };
    }

    /**
     * Find parent section
     */
    function getParentSection(element) {
        const section = element.closest('section, .wp-block-group, .entry-content, header, footer, main, article');

        if (!section) {
            return {
                selector: 'body',
                index: 0
            };
        }

        // Calculate section index among similar sections
        const sectionTag = section.tagName.toLowerCase();
        const sectionClasses = section.className || '';
        const sectionId = section.id || '';

        // Find all similar sections
        let selectorPattern = sectionTag;
        if (sectionClasses) {
            // Filter out empty strings to avoid invalid selectors like '.class1..class2'
            selectorPattern += '.' + sectionClasses.trim().split(/\s+/).filter(c => c).join('.');
        }

        const allSimilar = document.querySelectorAll(selectorPattern);
        const sectionIndex = Array.from(allSimilar).indexOf(section) + 1;

        // Build selector with nth-of-type
        let sectionSelector = sectionTag;
        if (sectionId) {
            sectionSelector = '#' + sectionId;
        } else if (sectionClasses) {
            // Filter out empty strings to avoid invalid selectors
            sectionSelector += '.' + sectionClasses.trim().split(/\s+/).filter(c => c).join('.');
            if (sectionIndex > 1) {
                sectionSelector += ':nth-of-type(' + sectionIndex + ')';
            }
        }

        return {
            selector: sectionSelector,
            index: sectionIndex,
            tag: sectionTag,
            classes: sectionClasses,
            id: sectionId
        };
    }

    /**
     * Deselect current element
     */
    function deselectElement() {
        if (selectedElement) {
            selectedElement.classList.remove('ai-editor-selected');
            selectedElement = null;
            return true;
        }
        return false;
    }

    /**
     * Send message to parent editor
     */
    function notifyParent(data) {
        try {
            window.parent.postMessage(data, '*');
        } catch (error) {
            console.error('[Dreamformer] Failed to send message to parent:', error);
        }
    }

    /**
     * Element selection handler (Ctrl+Click)
     */
    document.addEventListener('click', function(e) {
        // Only intercept Ctrl+click (or Cmd+click on Mac)
        if (!e.ctrlKey && !e.metaKey) {
            return; // Let normal clicks through
        }

        e.preventDefault();
        e.stopPropagation();

        const element = e.target;

        // TOGGLE: If clicking already selected element, deselect
        if (element === selectedElement) {
            deselectElement();
            notifyParent({ type: 'element-deselected' });
            return;
        }

        // Deselect previous
        deselectElement();

        // Select new element
        selectedElement = element;
        element.classList.add('ai-editor-selected');

        // Get element info
        const selector = getElementSelector(element);
        const section = getParentSection(element);
        const computed = getComputedStyles(element);

        // Send to parent editor
        notifyParent({
            type: 'element-selected',
            selector: selector,
            element_type: element.tagName.toLowerCase(),
            element_classes: element.className || '',
            parent_section: section.selector,
            section_index: section.index,
            computed_styles: computed,
            html: element.outerHTML.substring(0, 200),
            text: (element.textContent || '').substring(0, 50).trim()
        });

    }, true); // Use capture phase

    /**
     * ESC key handler
     */
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' || e.keyCode === 27) {
            if (deselectElement()) {
                e.preventDefault();
                e.stopPropagation();
                notifyParent({ type: 'element-deselected' });
            }
        }
    });

    /**
     * Click empty space handler
     */
    document.addEventListener('click', function(e) {
        if (e.ctrlKey || e.metaKey) return;

        if (e.target === document.body || e.target === document.documentElement) {
            if (deselectElement()) {
                notifyParent({ type: 'element-deselected' });
            }
        }
    });

    /**
     * Listen for manual style updates from parent
     */
    window.addEventListener('message', function(e) {
        if (e.data.type === 'update-manual-styles') {
            let styleEl = document.getElementById('manual-editor-styles');
            if (!styleEl) {
                styleEl = document.createElement('style');
                styleEl.id = 'manual-editor-styles';
                document.head.appendChild(styleEl);
                // Initialize rule tracking object
                styleEl.__manualRules = {};
            }

            // Initialize tracking object if missing (for pages that already have the element)
            if (!styleEl.__manualRules) {
                styleEl.__manualRules = {};
            }

            // Store rule keyed by selector
            const selector = e.data.selector;
            if (!styleEl.__manualRules[selector]) {
                styleEl.__manualRules[selector] = {};
            }

            // Update specific property (replaces old value if exists)
            styleEl.__manualRules[selector][e.data.property] = e.data.value;

            // Rebuild all rules with !important for proper specificity
            let cssText = '';
            for (const [sel, properties] of Object.entries(styleEl.__manualRules)) {
                cssText += sel + ' {\n';
                for (const [prop, val] of Object.entries(properties)) {
                    cssText += '  ' + prop + ': ' + val + ' !important;\n';
                }
                cssText += '}\n\n';
            }

            // Replace entire stylesheet content (not append)
            styleEl.textContent = cssText;

        }

        // Handle scroll position save/restore requests
        if (e.data.type === 'get-scroll-position') {
            // Get current scroll position
            const position = {
                x: window.pageXOffset || document.documentElement.scrollLeft || 0,
                y: window.pageYOffset || document.documentElement.scrollTop || 0
            };

            // Send back to parent
            notifyParent({
                type: 'scroll-position-response',
                position: position
            });

        }

        if (e.data.type === 'restore-scroll-position') {
            // Restore scroll position
            const pos = e.data.position;
            if (pos && typeof pos.x === 'number' && typeof pos.y === 'number') {
                window.scrollTo(pos.x, pos.y);
            }
        }
    });

    /**
     * Inject selection styles
     */
    const style = document.createElement('style');
    style.id = 'dreamformer-selection-styles';
    style.textContent = `
        .ai-editor-selected {
            position: relative !important;
            outline: 3px solid #667eea !important;
            outline-offset: 4px !important;
            box-shadow: 0 0 0 1px rgba(102, 126, 234, 0.3),
                        0 0 20px rgba(102, 126, 234, 0.4),
                        inset 0 0 0 2px rgba(102, 126, 234, 0.1) !important;
            background: rgba(102, 126, 234, 0.05) !important;
            transition: all 0.3s ease !important;
            z-index: 9999 !important;
        }

        .ai-editor-selected::before {
            content: "★ SELECTED • ESC or Ctrl/⌘+click to deselect" !important;
            position: absolute !important;
            top: -32px !important;
            left: -3px !important;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
            color: white !important;
            padding: 4px 12px !important;
            font-size: 11px !important;
            font-weight: 600 !important;
            border-radius: 4px !important;
            letter-spacing: 0.5px !important;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.4) !important;
            z-index: 10000 !important;
            animation: ai-pulse 2s infinite !important;
            pointer-events: none !important;
            white-space: nowrap !important;
        }

        @keyframes ai-pulse {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.9; transform: scale(1.05); }
        }

        body.ai-selection-mode * {
            cursor: crosshair !important;
        }

        body.ai-selection-mode *:hover:not(.ai-editor-selected) {
            outline: 2px dashed rgba(102, 126, 234, 0.5) !important;
            outline-offset: 2px !important;
            transition: outline 0.2s ease !important;
        }
    `;
    document.head.appendChild(style);

    /**
     * Ctrl key cursor feedback
     */
    document.addEventListener('keydown', function(e) {
        if ((e.ctrlKey || e.metaKey) && !document.body.classList.contains('ai-selection-mode')) {
            document.body.classList.add('ai-selection-mode');
        }
    });

    document.addEventListener('keyup', function(e) {
        if (!e.ctrlKey && !e.metaKey && document.body.classList.contains('ai-selection-mode')) {
            document.body.classList.remove('ai-selection-mode');
        }
    });

    // Send heartbeat for freeze detection (extends existing heartbeat)
    setInterval(function() {
        notifyParent({
            type: 'dreamformer_heartbeat',
            timestamp: Date.now()
        });
    }, 1000);


    // Notify parent that handler is ready
    notifyParent({
        type: 'handler-loaded',
        version: '2.0'
    });
})();
JAVASCRIPT;
    }

    /**
     * Get page directory path (draft or live)
     *
     * @param int $page_id Page ID
     * @param bool $is_draft Whether to get draft directory
     * @return string Directory path
     */
    private function get_page_directory( $page_id, $is_draft = false ) {
        $upload_dir = wp_upload_dir();
        $suffix = $is_draft ? '-draft' : '';
        return $upload_dir['basedir'] . '/ai-pages/page-' . $page_id . $suffix;
    }

    /**
     * Check if page is in draft mode
     *
     * @param int $page_id Page ID
     * @return bool True if draft mode active
     */
    private function is_draft_mode( $page_id ) {
        return get_post_meta( $page_id, '_ai_draft_active', true ) === '1';
    }

    /**
     * Copy directory recursively with cleanup on failure
     *
     * @param string $source Source directory
     * @param string $dest Destination directory
     * @return bool Success
     */
    private function copy_directory( $source, $dest ) {
        if ( !is_dir( $source ) ) {
            $this->logger->error( 'Copy failed: Source directory does not exist', [
                'source' => $source
            ] );
            return false;
        }

        // Track if we created the destination directory (for cleanup on failure)
        $created_dest = false;

        if ( !is_dir( $dest ) ) {
            if ( !mkdir( $dest, 0755, true ) ) {
                $this->logger->error( 'Copy failed: Cannot create destination directory', [
                    'dest' => $dest
                ] );
                return false;
            }
            $created_dest = true;
        }

        if ( !is_writable( $dest ) ) {
            $this->logger->error( 'Copy failed: Destination not writable', [
                'dest' => $dest
            ] );
            // Cleanup if we created it
            if ( $created_dest ) {
                $this->delete_directory( $dest );
            }
            return false;
        }

        try {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $source, RecursiveDirectoryIterator::SKIP_DOTS ),
                RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ( $iterator as $item ) {
                $target = $dest . '/' . $iterator->getSubPathName();
                if ( $item->isDir() ) {
                    if ( !is_dir( $target ) && !mkdir( $target, 0755, true ) ) {
                        $this->logger->error( 'Copy failed: Cannot create subdirectory', [
                            'target' => $target
                        ] );
                        // CLEANUP: Remove partial directory on failure
                        if ( $created_dest ) {
                            $this->delete_directory( $dest );
                            $this->logger->info( 'Cleaned up partial directory after copy failure', [
                                'dest' => $dest
                            ] );
                        }
                        return false;
                    }
                } else {
                    if ( !copy( $item, $target ) ) {
                        $this->logger->error( 'Copy failed: Cannot copy file', [
                            'source_file' => $item->getPathname(),
                            'target_file' => $target
                        ] );
                        // CLEANUP: Remove partial directory on failure
                        if ( $created_dest ) {
                            $this->delete_directory( $dest );
                            $this->logger->info( 'Cleaned up partial directory after copy failure', [
                                'dest' => $dest
                            ] );
                        }
                        return false;
                    }
                }
            }

            return true;

        } catch ( Exception $e ) {
            $this->logger->error( 'Copy failed with exception', [
                'error' => $e->getMessage()
            ] );
            // CLEANUP: Remove partial directory on failure
            if ( $created_dest ) {
                $this->delete_directory( $dest );
                $this->logger->info( 'Cleaned up partial directory after copy exception', [
                    'dest' => $dest
                ] );
            }
            return false;
        }
    }

    /**
     * Delete directory recursively
     *
     * @param string $dir Directory to delete
     * @return bool Success
     */
    private function delete_directory( $dir ) {
        if ( !is_dir( $dir ) ) {
            return true;
        }

        try {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $dir, RecursiveDirectoryIterator::SKIP_DOTS ),
                RecursiveIteratorIterator::CHILD_FIRST
            );

            foreach ( $iterator as $item ) {
                if ( $item->isDir() ) {
                    rmdir( $item->getPathname() );
                } else {
                    unlink( $item->getPathname() );
                }
            }

            return rmdir( $dir );

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to delete directory', [
                'dir' => $dir,
                'error' => $e->getMessage()
            ] );
            return false;
        }
    }

    /**
     * Cleanup old backup directories
     *
     * @param int $page_id Page ID
     * @param int $keep_count Number of backups to keep
     */
    private function cleanup_old_backups( $page_id, $keep_count = 3 ) {
        $upload_dir = wp_upload_dir();
        $base_dir = $upload_dir['basedir'] . '/ai-pages';

        if ( !is_dir( $base_dir ) ) {
            return;
        }

        // Find all backup directories for this page
        $pattern = 'page-' . $page_id . '-backup-*';
        $backups = glob( $base_dir . '/' . $pattern );

        if ( count( $backups ) <= $keep_count ) {
            return; // Nothing to cleanup
        }

        // Sort by modification time (newest first)
        usort( $backups, function( $a, $b ) {
            return filemtime( $b ) - filemtime( $a );
        } );

        // Remove old backups (keep only $keep_count newest)
        $to_delete = array_slice( $backups, $keep_count );

        foreach ( $to_delete as $backup_dir ) {
            $this->delete_directory_recursive( $backup_dir );
            $this->logger->info( 'Cleaned up old backup', [
                'backup_dir' => $backup_dir
            ] );
        }
    }

    /**
     * Recursively delete a directory
     *
     * @param string $dir Directory path
     * @return bool Success
     */
    private function delete_directory_recursive( $dir ) {
        if ( !is_dir( $dir ) ) {
            return false;
        }

        $files = array_diff( scandir( $dir ), [ '.', '..' ] );

        foreach ( $files as $file ) {
            $path = $dir . '/' . $file;
            if ( is_dir( $path ) ) {
                $this->delete_directory_recursive( $path );
            } else {
                unlink( $path );
            }
        }

        return rmdir( $dir );
    }

    private function build_search_patterns( $selector ) {
        $patterns = array();

        // Original selector (CSS format - for stylesheets and JS)
        $patterns[] = $selector;

        // ID selector: #footer-newsletter → id="footer-newsletter"
        if ( strpos( $selector, '#' ) === 0 ) {
            $id = substr( $selector, 1 ); // Remove #
            $patterns[] = 'id="' . $id . '"';
            $patterns[] = "id='" . $id . "'";
            $patterns[] = 'id=' . $id; // Without quotes (rare but valid)
        }

        // Class selector: .newsletter-form → class="newsletter-form"
        if ( strpos( $selector, '.' ) === 0 ) {
            $class = substr( $selector, 1 ); // Remove .
            $patterns[] = 'class="' . $class . '"';
            $patterns[] = "class='" . $class . "'";
            // Also match within class lists (e.g., class="form newsletter-form")
            $patterns[] = 'class="' . $class . ' ';
            $patterns[] = ' ' . $class . '"';
            $patterns[] = ' ' . $class . ' ';
        }

        return array_unique( $patterns );
    }

    /**
     * Find which files contain a specific element/selector
     * Returns exact file names where the element exists
     *
     * @param string $selector CSS selector (e.g., '#footer-newsletter')
     * @param int $page_id Page ID
     * @return array Array of file names where element was found
     */
    private function find_element_files( $selector, $page_id ) {
        if ( empty( $selector ) ) {
            return array();
        }

        $found_in = array();

        // Get files (already loaded by scan functions, so this is essentially free)
        $theme_path = get_template_directory();
        $theme_files = $this->scan_theme_files( $theme_path );
        $page_files = $this->scan_page_files( $page_id );

        // Build search patterns (CSS selector → multiple formats)
        $search_patterns = $this->build_search_patterns( $selector );


        // Search theme files
        foreach ( $theme_files as $relative_path => $full_path ) {
            // Read file content (scan_theme_files returns paths, not content)
            if ( ! file_exists( $full_path ) || ! is_readable( $full_path ) ) {
                continue;
            }

            $content = file_get_contents( $full_path );
            if ( $content === false ) {
                continue;
            }

            // Check if any pattern matches
            foreach ( $search_patterns as $pattern ) {
                if ( stripos( $content, $pattern ) !== false ) {
                    if ( ! in_array( 'theme:' . $relative_path, $found_in ) ) {
                        $found_in[] = 'theme:' . $relative_path;
                    }
                    break; // Don't check other patterns for this file
                }
            }
        }

        // Search page files
        foreach ( $page_files as $file_name => $content ) {
            foreach ( $search_patterns as $pattern ) {
                if ( stripos( $content, $pattern ) !== false ) {
                    if ( ! in_array( $file_name, $found_in ) ) {
                        $found_in[] = $file_name;
                    }
                    break;
                }
            }
        }

        if ( empty( $found_in ) ) {
        } else {
        }


        return $found_in;
    }

    /**
     * REST API: Save section reference (Clone Section feature)
     *
     * Simplified: No AI extraction - just saves reference to source page + selector.
     * AI extraction happens at apply-time when we have full context.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function rest_save_snippet( WP_REST_Request $request ) {
        global $wpdb;

        $name = sanitize_text_field( $request->get_param( 'name' ) );
        $description = sanitize_textarea_field( $request->get_param( 'description' ) );
        $selector = sanitize_text_field( $request->get_param( 'selector' ) );
        $page_id = intval( $request->get_param( 'page_id' ) ?: 0 );

        // Validation
        if ( empty( $name ) || empty( $selector ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Name and selector are required'
            ], 400 );
        }

        if ( $page_id <= 0 ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Page ID is required'
            ], 400 );
        }

        // Verify source page exists
        $page = get_post( $page_id );
        if ( ! $page ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Source page not found'
            ], 404 );
        }

        try {
            // Get a simple HTML preview (for display in UI, not for applying)
            $html_preview = $this->get_section_html_preview( $page_id, $selector );

            // Save reference to database (NO AI extraction)
            $snippets_table = $wpdb->prefix . 'dreamformer_snippets';
            $result = $wpdb->insert(
                $snippets_table,
                [
                    'name' => $name,
                    'description' => $description,
                    'selector' => $selector,
                    'html_code' => $html_preview,  // Preview only, not used for applying
                    'css_code' => '',              // Not used in clone approach
                    'js_code' => '',               // Not used in clone approach
                    'saved_from_page_id' => $page_id,
                    'saved_from_theme_id' => null,
                    'user_id' => get_current_user_id(),
                ],
                [ '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%d' ]
            );

            if ( $result === false ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Database insert failed: ' . $wpdb->last_error
                ], 500 );
            }

            return new WP_REST_Response( [
                'success' => true,
                'message' => 'Section saved successfully',
                'snippet_id' => $wpdb->insert_id,
                'source_page' => $page->post_title
            ], 200 );

        } catch ( Exception $e ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $e->getMessage()
            ], 500 );
        }
    }

    /**
     * Get HTML preview of a section for display purposes
     *
     * @param int $page_id Page ID
     * @param string $selector CSS selector
     * @return string HTML preview (truncated if needed)
     */
    private function get_section_html_preview( $page_id, $selector ) {
        // Get page HTML content (check draft first, then live)
        $draft_dir = $this->get_page_directory( $page_id, true );
        $live_dir = $this->get_page_directory( $page_id, false );

        $html_file = $draft_dir . '/index.html';
        if ( ! file_exists( $html_file ) ) {
            $html_file = $live_dir . '/index.html';
        }
        if ( ! file_exists( $html_file ) ) {
            return '';
        }

        $html = file_get_contents( $html_file );
        if ( empty( $html ) ) {
            return '';
        }

        // Simple extraction using DOMDocument
        libxml_use_internal_errors( true );
        $doc = new DOMDocument();
        $doc->loadHTML( '<?xml encoding="UTF-8">' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
        $xpath = new DOMXPath( $doc );

        // Convert simple selector to XPath
        $xpath_query = $this->selector_to_xpath( $selector );
        if ( empty( $xpath_query ) ) {
            libxml_clear_errors();
            return '';
        }

        $nodes = $xpath->query( $xpath_query );
        if ( $nodes && $nodes->length > 0 ) {
            $preview = $doc->saveHTML( $nodes->item( 0 ) );
            libxml_clear_errors();
            // Truncate if too long (just for preview)
            return strlen( $preview ) > 5000 ? substr( $preview, 0, 5000 ) . '...' : $preview;
        }

        libxml_clear_errors();
        return '';
    }

    /**
     * Convert simple CSS selector to XPath
     * Supports: #id, .class, tag, tag.class, tag#id
     *
     * @param string $selector CSS selector
     * @return string XPath query
     */
    private function selector_to_xpath( $selector ) {
        $selector = trim( $selector );
        if ( empty( $selector ) ) {
            return '';
        }

        // #id
        if ( preg_match( '/^#([\w-]+)$/', $selector, $m ) ) {
            return "//*[@id='" . $m[1] . "']";
        }

        // .class (including .class1.class2)
        if ( preg_match( '/^\.([\w-]+(?:\.[\w-]+)*)$/', $selector, $m ) ) {
            $classes = explode( '.', $m[1] );
            $conditions = array_map( function( $c ) {
                return "contains(concat(' ', normalize-space(@class), ' '), ' " . $c . " ')";
            }, $classes );
            return "//*[" . implode( ' and ', $conditions ) . "]";
        }

        // tag#id
        if ( preg_match( '/^([\w-]+)#([\w-]+)$/', $selector, $m ) ) {
            return "//" . $m[1] . "[@id='" . $m[2] . "']";
        }

        // tag.class
        if ( preg_match( '/^([\w-]+)\.([\w-]+)$/', $selector, $m ) ) {
            return "//" . $m[1] . "[contains(concat(' ', normalize-space(@class), ' '), ' " . $m[2] . " ')]";
        }

        // Simple tag
        if ( preg_match( '/^[\w-]+$/', $selector ) ) {
            return "//" . $selector;
        }

        // [data-attribute="value"]
        if ( preg_match( '/^\[([\w-]+)=["\']?([\w-]+)["\']?\]$/', $selector, $m ) ) {
            return "//*[@" . $m[1] . "='" . $m[2] . "']";
        }

        return '';
    }

    /**
     * REST API: List all saved sections for current user
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function rest_list_snippets( WP_REST_Request $request ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $snippets_table = $wpdb->prefix . 'dreamformer_snippets';

        $snippets = $wpdb->get_results( $wpdb->prepare(
            "SELECT s.*, p.post_title as source_page_title
             FROM {$snippets_table} s
             LEFT JOIN {$wpdb->posts} p ON s.saved_from_page_id = p.ID
             WHERE s.user_id = %d
             ORDER BY s.created_at DESC",
            $user_id
        ), ARRAY_A );

        if ( $snippets === null ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Database query failed'
            ], 500 );
        }

        // Format response
        $formatted = array_map( function( $s ) {
            return [
                'id' => intval( $s['id'] ),
                'name' => $s['name'],
                'description' => $s['description'],
                'selector' => $s['selector'],
                'source_page_id' => intval( $s['saved_from_page_id'] ),
                'source_page_title' => $s['source_page_title'] ?: 'Deleted Page',
                'html_preview' => $s['html_code'] ? substr( $s['html_code'], 0, 200 ) . '...' : '',
                'created_at' => $s['created_at']
            ];
        }, $snippets );

        return new WP_REST_Response( [
            'success' => true,
            'sections' => $formatted,
            'count' => count( $formatted )
        ], 200 );
    }

    /**
     * REST API: Get single section details
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function rest_get_snippet( WP_REST_Request $request ) {
        global $wpdb;

        $snippet_id = intval( $request->get_param( 'id' ) );
        $user_id = get_current_user_id();
        $snippets_table = $wpdb->prefix . 'dreamformer_snippets';

        $snippet = $wpdb->get_row( $wpdb->prepare(
            "SELECT s.*, p.post_title as source_page_title
             FROM {$snippets_table} s
             LEFT JOIN {$wpdb->posts} p ON s.saved_from_page_id = p.ID
             WHERE s.id = %d AND s.user_id = %d",
            $snippet_id,
            $user_id
        ), ARRAY_A );

        if ( ! $snippet ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Section not found'
            ], 404 );
        }

        return new WP_REST_Response( [
            'success' => true,
            'section' => [
                'id' => intval( $snippet['id'] ),
                'name' => $snippet['name'],
                'description' => $snippet['description'],
                'selector' => $snippet['selector'],
                'source_page_id' => intval( $snippet['saved_from_page_id'] ),
                'source_page_title' => $snippet['source_page_title'] ?: 'Deleted Page',
                'html_preview' => $snippet['html_code'],
                'created_at' => $snippet['created_at']
            ]
        ], 200 );
    }

    /**
     * REST API: Delete a saved section
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function rest_delete_snippet( WP_REST_Request $request ) {
        global $wpdb;

        $snippet_id = intval( $request->get_param( 'id' ) );
        $user_id = get_current_user_id();
        $snippets_table = $wpdb->prefix . 'dreamformer_snippets';

        // Verify ownership
        $exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT id FROM {$snippets_table} WHERE id = %d AND user_id = %d",
            $snippet_id,
            $user_id
        ) );

        if ( ! $exists ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Section not found or access denied'
            ], 404 );
        }

        // Delete
        $result = $wpdb->delete(
            $snippets_table,
            [ 'id' => $snippet_id, 'user_id' => $user_id ],
            [ '%d', '%d' ]
        );

        if ( $result === false ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Failed to delete section'
            ], 500 );
        }

        return new WP_REST_Response( [
            'success' => true,
            'message' => 'Section deleted successfully'
        ], 200 );
    }

    /**
     * REST API: Apply a saved section to another page
     *
     * This is the main Clone Section feature - fetches source files,
     * sends to AI to integrate into target page, applies diffs.
     * Costs 6 credits.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function rest_apply_snippet( WP_REST_Request $request ) {
        global $wpdb;

        $snippet_id = intval( $request->get_param( 'id' ) );
        $target_page_id = intval( $request->get_param( 'target_page_id' ) );
        $user_id = get_current_user_id();
        $snippets_table = $wpdb->prefix . 'dreamformer_snippets';

        // Get the saved section
        $snippet = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$snippets_table} WHERE id = %d AND user_id = %d",
            $snippet_id,
            $user_id
        ), ARRAY_A );

        if ( ! $snippet ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Section not found'
            ], 404 );
        }

        $source_page_id = intval( $snippet['saved_from_page_id'] );
        $selector = $snippet['selector'];

        // Verify source page still exists
        if ( ! get_post( $source_page_id ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Source page no longer exists'
            ], 404 );
        }

        // Verify target page exists
        if ( ! get_post( $target_page_id ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Target page not found'
            ], 404 );
        }

        // Note: Credit validation happens on Vercel side (source of truth)
        // Vercel API will return 402 if insufficient credits

        try {
            // Get source page files
            $source_files = $this->get_page_files_for_apply( $source_page_id );
            if ( is_wp_error( $source_files ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to read source page: ' . $source_files->get_error_message()
                ], 500 );
            }

            // Get target page files
            $target_files = $this->get_page_files_for_apply( $target_page_id );
            if ( is_wp_error( $target_files ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to read target page: ' . $target_files->get_error_message()
                ], 500 );
            }

            // Start Vercel job (returns immediately with job_id)
            $result = $this->get_vercel_client()->apply_section( $selector, $source_files, $target_files );

            if ( is_wp_error( $result ) ) {
                return new WP_REST_Response( [
                    'success' => false,
                    'message' => 'Failed to start AI integration: ' . $result->get_error_message()
                ], 500 );
            }

            // Return job_id immediately - browser will poll for status
            return new WP_REST_Response( [
                'success' => true,
                'job_id' => $result['job_id'],
                'status' => 'processing',
                'poll_interval_ms' => 2000,
                // Metadata needed when applying completed result
                'snippet_id' => $snippet_id,
                'target_page_id' => $target_page_id
            ], 202 );

        } catch ( Exception $e ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $e->getMessage()
            ], 500 );
        }
    }

    /**
     * Poll apply section job status
     */
    public function rest_poll_apply_section_status( WP_REST_Request $request ) {
        // CRITICAL: Disable ALL caching plugins for this polling endpoint
        do_action( 'litespeed_control_set_nocache', 'Dreamformer polling endpoint' );
        if ( ! defined( 'DONOTCACHEPAGE' ) ) {
            define( 'DONOTCACHEPAGE', true );
        }

        $job_id = $request->get_param( 'job_id' );

        $status_data = $this->get_vercel_client()->poll_apply_section_status( $job_id );

        if ( is_wp_error( $status_data ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $status_data->get_error_message()
            ], 500 );
        }

        // No-cache headers
        $response = new WP_REST_Response( $status_data, 200 );
        $response->header( 'Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0, private' );
        $response->header( 'Pragma', 'no-cache' );
        $response->header( 'Expires', '0' );
        $response->header( 'X-LiteSpeed-Cache-Control', 'no-cache' );

        return $response;
    }

    /**
     * REST endpoint: Apply completed section result
     */
    public function rest_apply_section_result( WP_REST_Request $request ) {
        $snippet_id = intval( $request->get_param( 'snippet_id' ) );
        $target_page_id = intval( $request->get_param( 'target_page_id' ) );
        $files = $request->get_param( 'files' );

        if ( ! $snippet_id || ! $target_page_id || ! $files ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => 'Missing required parameters'
            ], 400 );
        }

        $result = $this->apply_completed_section( $snippet_id, $target_page_id, $files );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( [
                'success' => false,
                'message' => $result->get_error_message()
            ], 500 );
        }

        return new WP_REST_Response( [
            'success' => true,
            'message' => 'Section applied successfully',
            'credits_used' => 6
        ], 200 );
    }

    /**
     * Apply completed section result to target page
     */
    private function apply_completed_section( $snippet_id, $target_page_id, $files ) {
        global $wpdb;
        $snippets_table = $wpdb->prefix . 'dreamformer_snippets';
        $user_id = get_current_user_id();

        // Apply the returned diffs to target page
        $apply_result = $this->apply_section_diffs( $target_page_id, $files );
        if ( is_wp_error( $apply_result ) ) {
            return $apply_result;
        }

        // Sync credits locally
        $credit_manager = new Dreamformer_Credit_Manager();
        $credit_manager->deduct_credits( $user_id, 6, 'apply_section' );

        // Increment usage count
        $wpdb->query( $wpdb->prepare(
            "UPDATE {$snippets_table} SET usage_count = usage_count + 1 WHERE id = %d",
            $snippet_id
        ) );

        // Trigger cache bust
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $target_page_id );
        }

        return true;
    }

    /**
     * Get page files for apply section feature
     *
     * @param int $page_id Page ID
     * @return array|WP_Error Files array or error
     */
    private function get_page_files_for_apply( $page_id ) {
        $files = [];

        // Check draft first, then live
        $draft_dir = $this->get_page_directory( $page_id, true );
        $live_dir = $this->get_page_directory( $page_id, false );

        // Determine which directory to use
        $page_dir = is_dir( $draft_dir ) ? $draft_dir : $live_dir;

        // Get HTML
        $html_path = $page_dir . '/index.html';
        if ( file_exists( $html_path ) ) {
            $files['index.html'] = file_get_contents( $html_path );
        } else {
            return new WP_Error( 'file_not_found', 'Page HTML file not found' );
        }

        // Get CSS
        $css_path = $page_dir . '/styles.css';
        if ( file_exists( $css_path ) ) {
            $files['styles.css'] = file_get_contents( $css_path );
        } else {
            $files['styles.css'] = '';
        }

        // Get JS
        $js_path = $page_dir . '/app.js';
        if ( file_exists( $js_path ) ) {
            $files['app.js'] = file_get_contents( $js_path );
        } else {
            $files['app.js'] = '';
        }

        return $files;
    }

    /**
     * Apply section diffs to target page files
     *
     * @param int $page_id Target page ID
     * @param array $diffs Diffs from AI
     * @return true|WP_Error Success or error
     */
    private function apply_section_diffs( $page_id, $diffs ) {
        // Create version snapshot before applying
        if ( class_exists( 'AI_Version_Control' ) ) {
            $version_control = AI_Version_Control::get_instance();
            $version_control->create_version( 'page', $page_id, 'Before applying cloned section' );
        }

        // Write to draft directory (create if needed)
        $draft_dir = $this->get_page_directory( $page_id, true );
        if ( ! is_dir( $draft_dir ) ) {
            wp_mkdir_p( $draft_dir );
        }

        // Apply each file diff
        foreach ( $diffs as $filename => $content ) {
            if ( empty( $content ) ) {
                continue;
            }

            $file_path = $draft_dir . '/' . $filename;
            $result = file_put_contents( $file_path, $content );

            if ( $result === false ) {
                return new WP_Error( 'write_failed', "Failed to write {$filename}" );
            }
        }

        // Mark page as having draft
        update_post_meta( $page_id, '_ai_has_draft', '1' );
        update_post_meta( $page_id, '_ai_draft_active', '1' );

        return true;
    }

    /**
     * Get active AI theme ID
     *
     * Checks if the currently active WordPress theme is an AI-generated theme
     * and returns its post ID if so.
     *
     * @return int|null Theme post ID or null if not an AI theme
     */
    private function get_active_ai_theme_id() {
        $active_theme_slug = get_stylesheet();

        // Check if it's an AI theme (starts with 'ai-theme-')
        if ( strpos( $active_theme_slug, 'ai-theme-' ) === 0 ) {
            // Query for ai_theme post with matching slug
            $themes = get_posts( array(
                'post_type' => 'ai_theme',
                'meta_key' => 'theme_directory_slug',
                'meta_value' => $active_theme_slug,
                'posts_per_page' => 1,
                'fields' => 'ids'
            ) );

            return ! empty( $themes ) ? $themes[0] : null;
        }

        // For regular (non-AI) themes, use a hash of the slug as entity_id
        // This allows versioning to work for ALL themes, not just AI-generated ones
        // Use crc32 to get a consistent integer ID from the theme slug
        $theme_hash = crc32( $active_theme_slug );

        // Store slug mapping for reverse lookup during undo
        update_option( 'ai_theme_slug_' . $theme_hash, $active_theme_slug );

        return $theme_hash;
    }

    /**
     * Link two versions bidirectionally
     *
     * @param int $page_version_id Page version ID
     * @param int $theme_version_id Theme version ID
     */
    private function link_versions( $page_version_id, $theme_version_id ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_version_control';

        try {
            // Update page version with theme link
            $wpdb->update(
                $table_name,
                array( 'linked_version_id' => $theme_version_id ),
                array( 'id' => $page_version_id ),
                array( '%d' ),
                array( '%d' )
            );

            // Update theme version with page link
            $wpdb->update(
                $table_name,
                array( 'linked_version_id' => $page_version_id ),
                array( 'id' => $theme_version_id ),
                array( '%d' ),
                array( '%d' )
            );

            $this->logger->info( 'Versions linked bidirectionally', array(
                'page_version_id' => $page_version_id,
                'theme_version_id' => $theme_version_id
            ) );

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to link versions', array(
                'page_version_id' => $page_version_id,
                'theme_version_id' => $theme_version_id,
                'error' => $e->getMessage()
            ) );
        }
    }

    /**
     * Save linked theme version when theme files are modified during page editing
     *
     * @param int $theme_id Theme post ID
     * @param string $theme_files_before JSON encoded theme files before changes
     * @param int $page_id Page ID (for metadata)
     * @param string $user_prompt User's original prompt
     * @return int|null Theme version ID or null on failure
     */
    private function save_linked_theme_version( $theme_id, $theme_files_before, $page_id, $user_prompt ) {
        try {
            // Capture theme files after changes
            $theme_path = get_template_directory();
            $theme_files_snapshot = $this->scan_theme_files( $theme_path );
            $theme_files_after = array();
            foreach ( $theme_files_snapshot as $relative_path => $full_path ) {
                if ( file_exists( $full_path ) && is_readable( $full_path ) ) {
                    $theme_files_after[$relative_path] = file_get_contents( $full_path );
                }
            }

            // Save theme version
            $version_control = AI_Version_Control::get_instance();
            $theme_version_id = $version_control->save_version( 'theme', $theme_id,
                $theme_files_before,  // Already JSON encoded from frontend
                json_encode( $theme_files_after ),
                array(
                    'change_description' => 'Theme modified via page editor',
                    'user_prompt' => $user_prompt,
                    'ai_model' => AI_Config_Manager::AI_MODEL,
                    'change_metadata' => array(
                        'modified_via_page' => $page_id,
                        'edit_type' => 'linked_page_edit'
                    )
                )
            );

            if ( ! is_wp_error( $theme_version_id ) ) {
                $this->logger->info( 'Linked theme version saved', array(
                    'theme_id' => $theme_id,
                    'theme_version_id' => $theme_version_id,
                    'page_id' => $page_id
                ) );
                return $theme_version_id;
            }

            return null;

        } catch ( Exception $e ) {
            $this->logger->error( 'Failed to save linked theme version', array(
                'theme_id' => $theme_id,
                'page_id' => $page_id,
                'error' => $e->getMessage()
            ) );
            return null;
        }
    }

    /**
     * REST API: Get code files for manual editing
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function rest_get_code_files( $request ) {
        $page_id = (int) $request['page_id'];

        $this->logger->info( 'Code editor: Loading files', array( 'page_id' => $page_id ) );

        // Verify page exists and is AI-generated
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_Error( 'invalid_page', 'Page not found', array( 'status' => 404 ) );
        }

        if ( ! get_post_meta( $page_id, 'ai_generated_page', true ) ) {
            return new WP_Error( 'not_ai_page', 'This page is not AI-generated and cannot be edited with code editor', array( 'status' => 400 ) );
        }

        // Check if page uses split files format
        $has_split_files = get_post_meta( $page_id, '_ai_split_files', true );
        if ( ! $has_split_files ) {
            return new WP_Error( 'no_split_files', 'This page does not use split files format. Please regenerate the page.', array( 'status' => 400 ) );
        }

        // Load files from draft directory (auto-creates if needed)
        $files = $this->scan_page_files( $page_id, true ); // true = draft mode

        // If no draft exists, copy from published
        if ( empty( $files ) ) {
            $this->logger->info( 'No draft files found, copying from published', array( 'page_id' => $page_id ) );

            $published_files = $this->scan_page_files( $page_id, false ); // false = published

            if ( ! empty( $published_files ) ) {
                // Create draft directory
                $draft_dir = $this->get_page_directory( $page_id, true );
                if ( ! is_dir( $draft_dir ) ) {
                    wp_mkdir_p( $draft_dir );

                    // Apply WordPress-defined permissions (respects FS_CHMOD_DIR constant)
                    $dir_perms = defined( 'FS_CHMOD_DIR' ) ? FS_CHMOD_DIR : ( fileperms( WP_CONTENT_DIR ) & 0777 | 0755 );
                    @chmod( $draft_dir, $dir_perms );
                }

                // Copy files to draft
                foreach ( $published_files as $filename => $content ) {
                    file_put_contents( $draft_dir . '/' . $filename, $content );
                }

                $files = $published_files;

                // Enable draft mode
                update_post_meta( $page_id, '_ai_has_draft', true );
            }
        }

        if ( empty( $files ) ) {
            return new WP_Error( 'no_files', 'No files found for this page', array( 'status' => 404 ) );
        }

        $this->logger->info( 'Code editor: Files loaded', array(
            'page_id' => $page_id,
            'file_count' => count( $files ),
            'files' => array_keys( $files )
        ) );

        return new WP_REST_Response( array(
            'success' => true,
            'files' => $files,
            'page_dir' => $this->get_page_directory( $page_id, true ),
            'page_title' => $page->post_title,
            'has_draft' => get_post_meta( $page_id, '_ai_has_draft', true )
        ), 200 );
    }

    /**
     * REST API: Save code files from manual editing
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function rest_save_code_files( $request ) {
        $page_id = (int) $request['page_id'];
        $files = $request['files'];

        $this->logger->info( 'Code editor: Saving files', array(
            'page_id' => $page_id,
            'file_count' => count( $files )
        ) );

        // Verify page exists
        $page = get_post( $page_id );
        if ( ! $page || $page->post_type !== 'page' ) {
            return new WP_Error( 'invalid_page', 'Page not found', array( 'status' => 404 ) );
        }

        if ( ! get_post_meta( $page_id, 'ai_generated_page', true ) ) {
            return new WP_Error( 'not_ai_page', 'This page is not AI-generated', array( 'status' => 400 ) );
        }

        // Validate files
        if ( ! is_array( $files ) || empty( $files ) ) {
            return new WP_Error( 'invalid_files', 'Files must be a non-empty object', array( 'status' => 400 ) );
        }

        // Get draft directory
        $page_dir = $this->get_page_directory( $page_id, true ); // true = draft

        // Ensure directory exists
        if ( ! is_dir( $page_dir ) ) {
            wp_mkdir_p( $page_dir );

            // Apply WordPress-defined permissions (respects FS_CHMOD_DIR constant)
            $dir_perms = defined( 'FS_CHMOD_DIR' ) ? FS_CHMOD_DIR : ( fileperms( WP_CONTENT_DIR ) & 0777 | 0755 );
            @chmod( $page_dir, $dir_perms );
        }

        // Whitelist allowed files (security)
        $allowed_files = array( 'index.html', 'styles.css', 'app.js' );
        $saved_files = array();

        // Save individual files
        foreach ( $files as $filename => $content ) {
            // Sanitize filename (security - prevent path traversal)
            $filename = basename( $filename );
            $filename = str_replace( array( '..', '/', '\\' ), '', $filename );

            // Only allow whitelisted files
            if ( ! in_array( $filename, $allowed_files, true ) ) {
                $this->logger->warning( 'Code editor: Attempted to save unauthorized file', array(
                    'page_id' => $page_id,
                    'filename' => $filename
                ) );
                continue;
            }

            // Save file
            $file_path = $page_dir . '/' . $filename;
            $result = file_put_contents( $file_path, $content );

            if ( $result !== false ) {
                $saved_files[] = $filename;
                $this->logger->debug( 'Code editor: File saved', array(
                    'page_id' => $page_id,
                    'filename' => $filename,
                    'size' => strlen( $content )
                ) );
            } else {
                $this->logger->error( 'Code editor: Failed to save file', array(
                    'page_id' => $page_id,
                    'filename' => $filename
                ) );
            }
        }

        if ( empty( $saved_files ) ) {
            return new WP_Error( 'save_failed', 'No files were saved', array( 'status' => 500 ) );
        }

        // CRITICAL: Recombine files and update database
        // This is what makes the changes visible in preview
        $combined_content = $this->render_split_files( $files );

        // Enable draft mode
        update_post_meta( $page_id, '_ai_has_draft', true );
        update_post_meta( $page_id, '_ai_draft_content', $combined_content );

        // Create version snapshot for undo/redo
        // This integrates with existing version control system
        $version_control = AI_Version_Control::get_instance();
        if ( $version_control ) {
            $version_control->save_version(
                'page',
                $page_id,
                '', // content_before - version control will handle this
                json_encode( $files ),
                array(
                    'change_description' => 'Manual code edit',
                    'user_prompt' => 'Code editor changes',
                    'ai_model' => 'manual_edit',
                    'change_metadata' => array(
                        'edit_type' => 'code_editor',
                        'files_modified' => $saved_files
                    )
                )
            );
        }

        $this->logger->info( 'Code editor: Save complete', array(
            'page_id' => $page_id,
            'files_saved' => $saved_files,
            'combined_length' => strlen( $combined_content )
        ) );

        // Trigger cache bust so browsers fetch fresh content
        if ( function_exists( 'ai_site_builder_trigger_cache_bust' ) ) {
            ai_site_builder_trigger_cache_bust( 'page', $page_id );
        }

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Files saved successfully',
            'files_saved' => $saved_files,
            'combined_content_length' => strlen( $combined_content ),
            'preview_url' => get_permalink( $page_id ),
            'has_draft' => true
        ), 200 );
    }

}