<?php
/**
 * AI CSS AST Parser - WordPress-compliant AST-based CSS parser
 * 
 * Uses sabberworm/php-css-parser for AST parsing
 * Provides structured CSS analysis for manual editor functionality
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use Sabberworm\CSS\Parser;
use Sabberworm\CSS\CSSList\Document;
use Sabberworm\CSS\RuleSet\DeclarationBlock;
use Sabberworm\CSS\Property\Selector;
use Sabberworm\CSS\Rule\Rule;
use Sabberworm\CSS\Value\Size;
use Sabberworm\CSS\Value\Color;

class AI_CSS_AST_Parser {
    
    private $logger;
    private $parsed_documents = array();
    
    public function __construct() {
        $this->logger = AI_Logger::get_instance( 'AI_CSS_AST_Parser' );
    }
    
    /**
     * Parse CSS content into AST
     * 
     * @param string $css_content CSS content to parse
     * @param string $file_path Optional file path for caching
     * @return Document|WP_Error Parsed CSS document or error
     */
    public function parse_css_content( $css_content, $file_path = null ) {
        try {
            // Check cache if file path provided
            if ( $file_path && isset( $this->parsed_documents[ $file_path ] ) ) {
                return $this->parsed_documents[ $file_path ];
            }
            
            // Parse CSS into AST
            $parser = new Parser( $css_content );
            $document = $parser->parse();
            
            // Cache the parsed document
            if ( $file_path ) {
                $this->parsed_documents[ $file_path ] = $document;
            }
            
            return $document;
            
        } catch ( Exception $e ) {
            $this->logger->error( 'CSS parsing failed', array( 
                'error' => $e->getMessage(),
                'file' => $file_path 
            ) );
            return new WP_Error( 'css_parse_error', 'Failed to parse CSS: ' . $e->getMessage() );
        }
    }
    
    /**
     * Find CSS rules matching a selector
     * 
     * @param Document $document Parsed CSS document
     * @param string $target_selector Selector to match
     * @return array Array of matching rules with details
     */
    public function find_rules_for_selector( $document, $target_selector ) {
        $matching_rules = array();
        
        if ( is_wp_error( $document ) ) {
            return $matching_rules;
        }
        
        // Traverse all rule sets in the document
        foreach ( $document->getAllDeclarationBlocks() as $block ) {
            $selectors = $block->getSelectors();
            
            foreach ( $selectors as $selector ) {
                // Get selector string using proper method
                $selector_string = $selector->getSelector();
                
                if ( $this->selector_matches( $selector_string, $target_selector ) ) {
                    $matching_rules[] = $this->extract_rule_details( $block, $selector_string );
                }
            }
        }
        
        return $matching_rules;
    }
    
    /**
     * Extract editable properties from a rule
     * 
     * @param DeclarationBlock $block CSS declaration block
     * @param string $selector CSS selector
     * @return array Structured rule details
     */
    private function extract_rule_details( $block, $selector ) {
        $properties = array();
        
        foreach ( $block->getRules() as $rule ) {
            $property_name = $rule->getRule();
            $value = $rule->getValue();
            
            // Handle both object and string values
            $value_string = is_string( $value ) ? $value : $value->render( \Sabberworm\CSS\OutputFormat::createCompact() );
            
            $properties[ $property_name ] = array(
                'value' => $value_string,
                'type' => $this->detect_property_type( $property_name, $value ),
                'important' => $rule->getIsImportant(),
                'parsed_value' => $this->parse_property_value( $value )
            );
        }
        
        return array(
            'selector' => $selector,
            'properties' => $properties,
            'specificity' => $this->calculate_specificity( $selector ),
            'editable' => true
        );
    }
    
    /**
     * Detect property type for manual editor controls
     * 
     * @param string $property CSS property name
     * @param mixed $value CSS value object
     * @return string Property type (color, size, spacing, etc.)
     */
    private function detect_property_type( $property, $value ) {
        // Color properties
        $color_properties = array(
            'color', 'background-color', 'border-color', 'outline-color',
            'text-decoration-color', 'box-shadow', 'text-shadow'
        );
        
        if ( in_array( $property, $color_properties ) || strpos( $property, 'color' ) !== false ) {
            return 'color';
        }
        
        // Size properties
        $size_properties = array(
            'font-size', 'width', 'height', 'max-width', 'max-height',
            'min-width', 'min-height', 'line-height'
        );
        
        if ( in_array( $property, $size_properties ) ) {
            return 'size';
        }
        
        // Spacing properties
        if ( strpos( $property, 'margin' ) !== false || strpos( $property, 'padding' ) !== false ) {
            return 'spacing';
        }
        
        // Border properties
        if ( strpos( $property, 'border' ) !== false ) {
            return 'border';
        }
        
        // Font properties
        if ( strpos( $property, 'font' ) !== false ) {
            return 'typography';
        }
        
        return 'general';
    }
    
    /**
     * Parse property value into structured format
     * 
     * @param mixed $value CSS value object
     * @return array Parsed value details
     */
    private function parse_property_value( $value ) {
        $parsed = array(
            'raw' => is_string( $value ) ? $value : $value->render( \Sabberworm\CSS\OutputFormat::createCompact() ),
            'unit' => null,
            'numeric' => null
        );
        
        // Handle Size values (e.g., 16px, 2em, 100%)
        if ( $value instanceof Size ) {
            $parsed['numeric'] = $value->getSize();
            $parsed['unit'] = $value->getUnit();
        }
        
        // Handle Color values
        if ( $value instanceof Color ) {
            $parsed['color_info'] = 'color_value';
            // Color parsing can be enhanced later with proper method calls
        }
        
        return $parsed;
    }
    
    /**
     * Update a CSS property value
     * 
     * @param Document $document CSS document
     * @param string $selector Target selector
     * @param string $property Property to update
     * @param string $new_value New value
     * @return bool|WP_Error Success or error
     */
    public function update_property( $document, $selector, $property, $new_value ) {
        try {
            $updated = false;
            
            foreach ( $document->getAllDeclarationBlocks() as $block ) {
                $selectors = $block->getSelectors();
                
                foreach ( $selectors as $sel ) {
                    if ( $sel->getSelector() === $selector ) {
                        // Find and update the property
                        foreach ( $block->getRules() as $rule ) {
                            if ( $rule->getRule() === $property ) {
                                $rule->setValue( $new_value );
                                $updated = true;
                                break 2;
                            }
                        }
                    }
                }
            }
            
            if ( ! $updated ) {
                return new WP_Error( 'property_not_found', 'Property not found in selector' );
            }
            
            return true;
            
        } catch ( Exception $e ) {
            return new WP_Error( 'update_failed', $e->getMessage() );
        }
    }
    
    /**
     * Serialize AST back to CSS string
     * 
     * @param Document $document CSS document
     * @return string CSS string
     */
    public function serialize_to_css( $document ) {
        return $document->render();
    }
    
    /**
     * Check if CSS selector matches target
     * Enhanced version compatible with AST
     * 
     * @param string $css_selector CSS selector from AST
     * @param string $target_selector Target selector to match
     * @return bool True if matches
     */
    private function selector_matches( $css_selector, $target_selector ) {
        $css_selector = trim( $css_selector );
        $target_selector = trim( $target_selector );
        
        // Exact match
        if ( $css_selector === $target_selector ) {
            return true;
        }
        
        // Extract classes from both selectors
        preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $target_selector, $target_classes );
        preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $css_selector, $css_classes );
        
        // Check if any target class matches CSS classes
        if ( ! empty( $target_classes[1] ) && ! empty( $css_classes[1] ) ) {
            $common_classes = array_intersect( $target_classes[1], $css_classes[1] );
            if ( ! empty( $common_classes ) ) {
                return true;
            }
        }
        
        // Check element type match
        preg_match( '/^([a-z]+)/', $target_selector, $target_element );
        preg_match( '/^([a-z]+)/', $css_selector, $css_element );
        
        if ( ! empty( $target_element[1] ) && ! empty( $css_element[1] ) ) {
            if ( $target_element[1] === $css_element[1] ) {
                return true;
            }
        }
        
        // Check ID match
        preg_match( '/#([a-zA-Z0-9_-]+)/', $target_selector, $target_id );
        preg_match( '/#([a-zA-Z0-9_-]+)/', $css_selector, $css_id );
        
        if ( ! empty( $target_id[1] ) && ! empty( $css_id[1] ) ) {
            if ( $target_id[1] === $css_id[1] ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Calculate CSS specificity
     * 
     * @param string $selector CSS selector
     * @return int Specificity score
     */
    private function calculate_specificity( $selector ) {
        $specificity = 0;
        
        // Count IDs (weight: 100)
        $specificity += substr_count( $selector, '#' ) * 100;
        
        // Count classes and attributes (weight: 10)
        $specificity += substr_count( $selector, '.' ) * 10;
        $specificity += substr_count( $selector, '[' ) * 10;
        
        // Count elements (weight: 1)
        preg_match_all( '/\b[a-z]+\b/', $selector, $elements );
        $specificity += count( $elements[0] );
        
        return $specificity;
    }
}