<?php
/**
 * AI Site Builder - Theme Post Types
 * Theme-focused post types and meta fields for the new architecture
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Theme_Post_Types {
    
    public function __construct() {
        add_action( 'init', [ $this, 'register_post_types' ] );
        add_action( 'init', [ $this, 'register_meta_fields' ] );
        add_action( 'init', [ $this, 'add_theme_support' ] );
    }
    
    /**
     * Register AI theme post type
     */
    public function register_post_types() {
        // AI Generated Themes post type
        register_post_type( 'ai_theme', [
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'themes.php', // Integrate with WordPress themes menu
            'supports' => [ 'title', 'editor', 'revisions' ],
            'capabilities' => [
                'edit_post' => 'switch_themes',
                'delete_post' => 'switch_themes', 
                'read_post' => 'switch_themes',
                'edit_posts' => 'switch_themes',
                'edit_others_posts' => 'switch_themes',
                'publish_posts' => 'switch_themes',
                'read_private_posts' => 'switch_themes'
            ],
            'labels' => [
                'name' => __( 'AI Themes', 'ai-site-builder' ),
                'singular_name' => __( 'AI Theme', 'ai-site-builder' ),
                'add_new' => __( 'Add Theme', 'ai-site-builder' ),
                'add_new_item' => __( 'Add New Theme', 'ai-site-builder' ),
                'edit_item' => __( 'Edit Theme', 'ai-site-builder' ),
                'new_item' => __( 'New AI Theme', 'ai-site-builder' ),
                'view_item' => __( 'Preview Theme', 'ai-site-builder' ),
                'search_items' => __( 'Search Themes', 'ai-site-builder' ),
                'not_found' => __( 'No AI themes found.', 'ai-site-builder' ),
                'not_found_in_trash' => __( 'No AI themes found in trash', 'ai-site-builder' ),
                'menu_name' => __( 'AI Themes', 'ai-site-builder' )
            ]
        ] );
    }
    
    /**
     * Register theme-specific meta fields
     */
    public function register_meta_fields() {
        // Theme directory slug (ai-theme-{id})
        register_meta( 'post', 'theme_directory_slug', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'sanitize_callback' => 'sanitize_title',
            'show_in_rest' => false
        ] );
        
        // Business/site configuration
        register_meta( 'post', 'site_configuration', [
            'object_subtype' => 'ai_theme',
            'type' => 'array',
            'single' => true,
            'sanitize_callback' => [ $this, 'sanitize_site_config' ],
            'show_in_rest' => false
        ] );
        
        // Theme activation status
        register_meta( 'post', 'theme_status', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'default' => 'draft',
            'sanitize_callback' => [ $this, 'sanitize_theme_status' ],
            'show_in_rest' => false
        ] );
        
        // Theme version for updates
        register_meta( 'post', 'theme_version', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'default' => '1.0.0',
            'sanitize_callback' => 'sanitize_text_field',
            'show_in_rest' => false
        ] );
        
        // Original generation prompt
        register_meta( 'post', 'generation_prompt', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'sanitize_callback' => 'sanitize_textarea_field',
            'show_in_rest' => false
        ] );
        
        // AI model used for generation
        register_meta( 'post', 'ai_model_used', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'sanitize_callback' => 'sanitize_text_field',
            'show_in_rest' => false
        ] );
        
        // Theme generation timestamp
        register_meta( 'post', 'theme_generated_at', [
            'object_subtype' => 'ai_theme',
            'type' => 'string',
            'single' => true,
            'sanitize_callback' => 'sanitize_text_field',
            'show_in_rest' => false
        ] );
        
        // WordPress Customizer defaults
        register_meta( 'post', 'customizer_defaults', [
            'object_subtype' => 'ai_theme',
            'type' => 'array',
            'single' => true,
            'sanitize_callback' => [ $this, 'sanitize_customizer_defaults' ],
            'show_in_rest' => false
        ] );
    }
    
    /**
     * Sanitize site configuration
     */
    public function sanitize_site_config( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        
        $allowed_keys = [
            'business_name', 'business_type', 'target_audience',
            'business_goals', 'style_preference', 'color_preference',
            'industry', 'features_required'
        ];
        
        $sanitized = [];
        foreach ( $allowed_keys as $key ) {
            if ( isset( $value[$key] ) ) {
                $sanitized[$key] = sanitize_text_field( $value[$key] );
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitize theme status
     */
    public function sanitize_theme_status( $value ) {
        $allowed_statuses = [ 'draft', 'generating', 'ready', 'active', 'inactive', 'error' ];
        return in_array( $value, $allowed_statuses ) ? $value : 'draft';
    }
    
    /**
     * Sanitize customizer defaults
     */
    public function sanitize_customizer_defaults( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        
        // Sanitize customizer data recursively
        return $this->sanitize_array_recursive( $value );
    }
    
    /**
     * Recursively sanitize array data
     */
    private function sanitize_array_recursive( $array ) {
        $sanitized = [];
        foreach ( $array as $key => $value ) {
            $clean_key = sanitize_key( $key );
            if ( is_array( $value ) ) {
                $sanitized[$clean_key] = $this->sanitize_array_recursive( $value );
            } else {
                $sanitized[$clean_key] = sanitize_text_field( $value );
            }
        }
        return $sanitized;
    }
    
    /**
     * Add theme support for generated themes
     */
    public function add_theme_support() {
        // These will be available for generated themes
        add_theme_support( 'title-tag' );
        add_theme_support( 'post-thumbnails' );
        add_theme_support( 'automatic-feed-links' );
        add_theme_support( 'html5', [ 'search-form', 'comment-form', 'comment-list', 'gallery', 'caption' ] );
        add_theme_support( 'customize-selective-refresh-widgets' );
    }
}