<?php
/**
 * AI Site Builder - Theme Manager
 * Handles WordPress theme creation, installation, and management
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


class AI_Theme_Manager {
    
    private $themes_directory;
    private $logger;
    
    public function __construct() {
        $this->themes_directory = get_theme_root();
        $this->logger = AI_Logger::get_instance( 'AI_Theme_Manager' );
        add_action( 'init', [ $this, 'init_hooks' ] );
        $this->init_sync_hooks();
    }

    /**
     * Initialize theme synchronization hooks
     */
    private function init_sync_hooks() {
        // Sync when WordPress theme changes (via Appearance > Themes)
        add_action( 'switch_theme', array( $this, 'sync_wordpress_to_dreamformer' ), 10, 3 );

        // Sync when Dreamformer themes are loaded (check for mismatches)
        add_action( 'wp_loaded', array( $this, 'check_theme_sync' ) );
    }
    
    /**
     * Initialize WordPress hooks
     */
    public function init_hooks() {
        // Theme management hooks
        add_action( 'switch_theme', [ $this, 'on_theme_switch' ] );
        add_filter( 'theme_mod_custom_css', [ $this, 'inject_ai_theme_css' ] );
        
        // Admin hooks for theme management
        add_action( 'admin_init', [ $this, 'handle_theme_actions' ] );
    }
    
    /**
     * Create dynamic WordPress theme directory structure based on AI architecture
     */
    public function create_theme_structure( $theme_id, $theme_name, $ai_architecture = null ) {
        $theme_slug = 'ai-theme-' . $theme_id;
        $theme_path = $this->themes_directory . '/' . $theme_slug;
        
        // Security check - ensure we're creating in themes directory
        if ( strpos( realpath( dirname( $theme_path ) ), realpath( $this->themes_directory ) ) !== 0 ) {
            return new WP_Error( 'invalid_path', 'Invalid theme path' );
        }
        
        // Create theme directory with proper permissions
        $dir_created = AI_File_Manager::ensure_directory( $theme_path );
        if ( is_wp_error( $dir_created ) ) {
            return new WP_Error( 'theme_creation_failed', 'Cannot create theme directory' );
        }
        
        // Create directories based on AI architecture or use default structure
        if ( $ai_architecture && isset( $ai_architecture['directory_structure'] ) ) {
            $this->create_ai_directory_structure( $theme_path, $ai_architecture['directory_structure'] );
        } else {
            // Fallback to minimal WordPress structure
            $subdirectories = [
                'assets',
                'assets/css',
                'assets/js',
                'assets/images',
                'template-parts',
                'inc',
                'languages'
            ];
            
            foreach ( $subdirectories as $dir ) {
                AI_File_Manager::ensure_directory( $theme_path . '/' . $dir );
            }
        }
        
        // Create security files for all directories
        $this->create_security_files_recursive( $theme_path );
        
        // Store theme slug for reference
        update_post_meta( $theme_id, 'theme_directory_slug', $theme_slug );
        
        return $theme_path;
    }
    
    /**
     * Create directory structure based on AI architecture design
     * 
     * @param string $theme_path Base theme directory
     * @param array $structure AI-designed directory structure
     */
    private function create_ai_directory_structure( $theme_path, $structure ) {
        foreach ( $structure as $dir_path => $dir_config ) {
            $full_path = $theme_path . '/' . ltrim( $dir_path, '/' );
            
            // Create directory
            AI_File_Manager::ensure_directory( $full_path );
            
            // If directory has subdirectories, create them recursively
            if ( is_array( $dir_config ) && isset( $dir_config['subdirectories'] ) ) {
                foreach ( $dir_config['subdirectories'] as $subdir ) {
                    AI_File_Manager::ensure_directory( $full_path . '/' . $subdir );
                }
            }
        }
    }
    
    /**
     * Install WordPress theme files from AI generation - Creative Freedom Mode Only
     */
    public function install_theme_files( $theme_id, $generated_files ) {
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );
        $theme_path = $this->themes_directory . '/' . $theme_slug;
        
        if ( ! file_exists( $theme_path ) ) {
            return new WP_Error( 'theme_not_found', 'Theme directory not found' );
        }
        
        // Debug: Log what we received
        if ( is_array( $generated_files ) ) {
            $this->logger->debug( 'Received generated files', array(
                'keys' => array_keys( $generated_files )
            ) );
        } else {
            $this->logger->error( 'Invalid generated files type', array(
                'type' => gettype( $generated_files )
            ) );
        }
        
        // REQUIRE AI file manifest - no backward compatibility
        if ( ! isset( $generated_files['file_manifest'] ) ) {
            return new WP_Error( 'no_file_manifest', 'AI must provide file_manifest with creative architecture. Legacy format not supported.' );
        }
        
        $this->logger->info( 'Processing file manifest', array(
            'file_count' => count( $generated_files['file_manifest'] )
        ) );
        
        // Install all files as specified by AI architecture
        return $this->install_ai_file_manifest( $theme_path, $generated_files['file_manifest'] );
    }
    
    /**
     * Validate WordPress theme structure and compliance
     */
    public function validate_theme_structure( $theme_slug ) {
        $this->logger->info( 'Validating theme structure', [
            'theme_slug' => $theme_slug
        ] );

        // Force WordPress to refresh theme cache
        wp_clean_themes_cache();

        // Check if theme exists in filesystem first
        $theme_root = get_theme_root();
        $theme_path = $theme_root . '/' . $theme_slug;

        if ( ! is_dir( $theme_path ) ) {
            $this->logger->error( 'Theme directory not found', [
                'theme_slug' => $theme_slug,
                'theme_path' => $theme_path,
                'theme_root' => $theme_root
            ] );
            return new WP_Error( 'theme_directory_missing', "Theme directory '{$theme_path}' not found" );
        }

        // Check if theme exists in WordPress registry
        $wp_theme = wp_get_theme( $theme_slug );
        if ( ! $wp_theme->exists() ) {
            $this->logger->error( 'Theme not found in WordPress registry', [
                'theme_slug' => $theme_slug,
                'wp_theme_exists' => $wp_theme->exists(),
                'theme_path' => $theme_path,
                'style_css_exists' => file_exists( $theme_path . '/style.css' ),
                'index_php_exists' => file_exists( $theme_path . '/index.php' )
            ] );

            // Try to get error details from the theme object
            $error_data = array(
                'theme_errors' => $wp_theme->errors(),
                'theme_data' => array(
                    'name' => $wp_theme->get( 'Name' ),
                    'version' => $wp_theme->get( 'Version' ),
                    'description' => $wp_theme->get( 'Description' )
                )
            );

            $this->logger->error( 'Theme validation details', $error_data );

            return new WP_Error( 'theme_not_found', "Theme '{$theme_slug}' not found in WordPress registry" );
        }

        $theme_path = $wp_theme->get_stylesheet_directory();
        $this->logger->info( 'Theme found in WordPress registry', [
            'theme_slug' => $theme_slug,
            'theme_name' => $wp_theme->get( 'Name' ),
            'theme_path' => $theme_path,
            'theme_exists' => $wp_theme->exists()
        ] );

        // Check required files exist using WordPress theme path
        $required_files = [ 'style.css', 'index.php' ];
        foreach ( $required_files as $file ) {
            $file_path = $theme_path . '/' . $file;
            $file_exists = file_exists( $file_path );

            $this->logger->info( 'Checking required file', [
                'file' => $file,
                'full_path' => $file_path,
                'exists' => $file_exists,
                'is_readable' => is_readable( $file_path )
            ] );

            if ( ! $file_exists ) {
                $this->logger->error( 'Required file missing', [
                    'file' => $file,
                    'full_path' => $file_path,
                    'theme_path' => $theme_path,
                    'theme_slug' => $theme_slug
                ] );
                return new WP_Error( 'missing_required_file', "Missing required file: {$file} in {$theme_path}" );
            }
        }
        
        // Validate style.css header
        $style_content = AI_File_Manager::read( $theme_path . '/style.css' );
        if ( is_wp_error( $style_content ) ) {
            return $style_content;
        }
        if ( ! $this->validate_style_css_header( $style_content ) ) {
            return new WP_Error( 'invalid_style_header', 'Invalid style.css header' );
        }
        
        // PHP syntax validation - TEMPORARILY DISABLED
        // $php_files = glob( $theme_path . '/*.php' );
        // if ( $php_files ) {
        //     foreach ( $php_files as $php_file ) {
        //         if ( ! $this->validate_php_syntax( $php_file ) ) {
        //             return new WP_Error( 'php_syntax_error', "PHP syntax error in: " . basename( $php_file ) );
        //         }
        //     }
        // }
        
        return true;
    }
    
    /**
     * Activate AI-generated theme in WordPress
     */
    public function activate_theme( $theme_id ) {
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );

        if ( empty( $theme_slug ) ) {
            return new WP_Error( 'theme_slug_missing', 'Theme slug not found in database' );
        }

        $this->logger->info( 'Attempting to activate theme', [
            'theme_id' => $theme_id,
            'theme_slug' => $theme_slug
        ] );

        // Validate theme before activation
        $validation = $this->validate_theme_structure( $theme_slug );
        if ( is_wp_error( $validation ) ) {
            $this->logger->error( 'Theme validation failed', [
                'theme_slug' => $theme_slug,
                'error' => $validation->get_error_message()
            ] );
            return $validation;
        }

        // Store current theme for rollback
        $previous_theme = get_option( 'stylesheet' );

        // Switch to the generated theme
        try {
            switch_theme( $theme_slug );

            // Verify the switch was successful
            if ( get_option( 'stylesheet' ) !== $theme_slug ) {
                throw new Exception( 'Theme switch failed - WordPress did not activate the theme' );
            }

        } catch ( Exception $e ) {
            $this->logger->error( 'Theme activation failed', [
                'theme_slug' => $theme_slug,
                'error' => $e->getMessage()
            ] );

            // Rollback to previous theme
            if ( $previous_theme && $previous_theme !== $theme_slug ) {
                switch_theme( $previous_theme );
            }

            return new WP_Error( 'activation_failed', 'Theme activation failed: ' . $e->getMessage() );
        }
        
        // Update theme status
        update_post_meta( $theme_id, 'theme_status', 'active' );
        
        // Apply customizer defaults if they exist
        $customizer_defaults = get_post_meta( $theme_id, 'customizer_defaults', true );
        if ( ! empty( $customizer_defaults ) ) {
            $this->apply_customizer_defaults( $theme_slug, $customizer_defaults );
        }
        
        // Trigger theme activation hooks
        do_action( 'ai_theme_activated', $theme_id, $theme_slug );
        
        return [
            'success' => true,
            'theme_slug' => $theme_slug,
            'customizer_url' => admin_url( 'customize.php?theme=' . $theme_slug ),
            'theme_name' => get_post_field( 'post_title', $theme_id )
        ];
    }
    
    /**
     * Write theme file with security checks
     */
    private function write_theme_file( $file_path, $content ) {
        // Security: Ensure file is within theme directory
        $theme_dir = realpath( $this->themes_directory );
        $target_dir = realpath( dirname( $file_path ) );
        
        if ( $target_dir === false || strpos( $target_dir, $theme_dir ) !== 0 ) {
            return new WP_Error( 'invalid_file_path', 'File path outside theme directory' );
        }
        
        // Security: Validate file extension
        $allowed_extensions = [ 
            'php', 'css', 'js', 'json', 'txt', 'md',
            'svg', 'png', 'jpg', 'jpeg', 'gif', 'webp',  // Image files
            'woff', 'woff2', 'ttf', 'eot',  // Font files
            'xml', 'yml', 'yaml'  // Config files
        ];
        $extension = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
        
        if ( ! in_array( $extension, $allowed_extensions ) ) {
            $this->logger->warning( 'File extension not allowed', array(
                'extension' => $extension,
                'file_path' => $file_path
            ) );
            return new WP_Error( 'invalid_file_extension', 'File extension not allowed: ' . $extension );
        }
        
        // Write file with error handling
        $write_result = AI_File_Manager::write( $file_path, $content );
        if ( is_wp_error( $write_result ) ) {
            return new WP_Error( 'file_write_failed', 'Could not write theme file' );
        }
        
        return true;
    }
    
    /**
     * Create security files for theme
     */
    private function create_security_files( $theme_path ) {
        // Add index.php to all directories for security
        $directories = [ '', '/assets', '/template-parts', '/inc' ];
        foreach ( $directories as $dir ) {
            $index_file = $theme_path . $dir . '/index.php';
            if ( ! file_exists( $index_file ) ) {
                AI_File_Manager::write( $index_file, "<?php\n// Silence is golden.\n" );
            }
        }
    }
    
    /**
     * Create security files recursively for AI-generated directory structure
     * 
     * @param string $theme_path Base theme directory
     */
    private function create_security_files_recursive( $theme_path ) {
        // Only create security files in asset subdirectories, NOT in root or template directories
        $asset_directories = array(
            'assets',
            'assets/css', 
            'assets/js',
            'assets/images',
            'languages',
            'inc'
        );
        
        foreach ( $asset_directories as $asset_dir ) {
            $full_asset_path = $theme_path . '/' . $asset_dir;
            $security_file = $full_asset_path . '/index.php';
            
            // Only create security file if directory exists and doesn't have an index.php
            if ( is_dir( $full_asset_path ) && ! file_exists( $security_file ) ) {
                AI_File_Manager::write( $security_file, "<?php\n// Silence is golden.\n" );
            }
        }
        
        // CRITICAL: Do NOT create security files in root directory or template-parts
        // These directories need real theme files, not placeholders!
    }
    
    /**
     * Install files based on AI-generated file manifest
     * 
     * @param string $theme_path Base theme directory
     * @param array $file_manifest AI-generated file manifest with paths and content
     * @return bool|WP_Error Success or error
     */
    private function install_ai_file_manifest( $theme_path, $file_manifest ) {
        $this->logger->info( 'Installing theme files', array(
            'file_count' => count( $file_manifest ),
            'theme_path' => $theme_path
        ) );
        
        $files_written = 0;
        $files_skipped = 0;
        $files_failed = 0;
        
        foreach ( $file_manifest as $relative_path => $file_content ) {
            $this->logger->debug( 'Processing file', array(
                'path' => $relative_path,
                'content_length' => strlen( $file_content )
            ) );
            
            // Skip empty content
            if ( empty( $file_content ) ) {
                $this->logger->warning( 'Skipping empty file', array( 'path' => $relative_path ) );
                $files_skipped++;
                continue;
            }
            
            // Build full file path
            $full_path = $theme_path . '/' . ltrim( $relative_path, '/' );
            
            // Ensure directory exists
            $directory = dirname( $full_path );
            if ( ! file_exists( $directory ) ) {
                AI_File_Manager::ensure_directory( $directory );
                $this->logger->debug( 'Created directory', array( 'directory' => $directory ) );
            }
            
            // Write file content
            $this->logger->debug( 'Writing file', array(
                'path' => $full_path,
                'size' => strlen( $file_content )
            ) );
            $result = $this->write_theme_file( $full_path, $file_content );
            if ( is_wp_error( $result ) ) {
                $this->logger->wp_error( $result, 'Failed to write file: ' . $relative_path );
                $files_failed++;
                return $result;
            }
            
            $files_written++;
            $this->logger->debug( 'Successfully wrote file', array( 'path' => $relative_path ) );
        }
        
        $this->logger->info( 'File installation complete', array(
            'written' => $files_written,
            'skipped' => $files_skipped,
            'failed' => $files_failed
        ) );
        
        return true;
    }
    
    /**
     * Validate style.css header format
     */
    private function validate_style_css_header( $style_content ) {
        return ( strpos( $style_content, 'Theme Name:' ) !== false );
    }
    
    /**
     * Validate PHP file syntax
     */
    private function validate_php_syntax( $file_path ) {
        $output = [];
        $return_var = 0;
        exec( 'php -l ' . escapeshellarg( $file_path ), $output, $return_var );
        return $return_var === 0;
    }
    
    /**
     * Install theme assets (CSS, JS, images)
     */
    // REMOVED: install_theme_assets() - No longer needed
    // All assets are now handled via AI file_manifest in install_ai_file_manifest()
    
    /**
     * Install theme images
     */
    private function install_theme_images( $theme_path, $images ) {
        foreach ( $images as $filename => $image_data ) {
            if ( is_string( $image_data ) && preg_match( '/^data:image\/(\w+);base64,/', $image_data ) ) {
                // Handle base64 encoded images
                $image_path = $theme_path . '/assets/images/' . sanitize_file_name( $filename );
                $decoded_image = base64_decode( preg_replace( '/^data:image\/\w+;base64,/', '', $image_data ) );
                $write_result = AI_File_Manager::write( $image_path, $decoded_image );
                if ( is_wp_error( $write_result ) ) {
                    $this->logger->wp_error( $write_result, 'Failed to save image' );
                    continue;
                }
            }
        }
    }
    
    /**
     * Generate theme screenshot for WordPress themes page
     */
    private function generate_theme_screenshot( $theme_id, $theme_path ) {
        // For now, create a placeholder screenshot
        // In Phase 2, this will be enhanced to capture actual theme preview
        $screenshot_path = $theme_path . '/screenshot.png';
        
        if ( ! file_exists( $screenshot_path ) ) {
            // Create a simple placeholder image
            $this->create_placeholder_screenshot( $screenshot_path, $theme_id );
        }
    }
    
    /**
     * Create placeholder screenshot
     */
    private function create_placeholder_screenshot( $screenshot_path, $theme_id ) {
        // Create a simple 1200x900 placeholder image
        if ( function_exists( 'imagecreate' ) ) {
            $img = imagecreate( 1200, 900 );
            $background = imagecolorallocate( $img, 240, 240, 240 );
            $text_color = imagecolorallocate( $img, 100, 100, 100 );
            
            $theme_name = get_post_field( 'post_title', $theme_id );
            imagestring( $img, 5, 500, 440, $theme_name ?: 'AI Generated Theme', $text_color );
            
            imagepng( $img, $screenshot_path );
            imagedestroy( $img );
        }
    }
    
    /**
     * Apply customizer defaults
     */
    private function apply_customizer_defaults( $theme_slug, $defaults ) {
        if ( ! is_array( $defaults ) ) {
            return;
        }
        
        foreach ( $defaults as $setting => $value ) {
            set_theme_mod( $setting, $value );
        }
    }
    
    /**
     * Handle theme switch event
     */
    public function on_theme_switch( $new_name ) {
        // Update status of previously active AI theme
        $this->update_previous_theme_status( $new_name );
    }
    
    /**
     * Update status of previous AI theme when switching
     */
    private function update_previous_theme_status( $new_theme ) {
        global $wpdb;
        
        // Find any AI themes marked as active
        $active_themes = $wpdb->get_results( $wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
             WHERE meta_key = 'theme_status' 
             AND meta_value = 'active'"
        ) );
        
        foreach ( $active_themes as $theme ) {
            $theme_slug = get_post_meta( $theme->post_id, 'theme_directory_slug', true );
            
            // If this AI theme is no longer active, update its status
            if ( get_option( 'stylesheet' ) !== $theme_slug ) {
                update_post_meta( $theme->post_id, 'theme_status', 'inactive' );
            }
        }
    }
    
    /**
     * Inject AI theme CSS modifications
     */
    public function inject_ai_theme_css( $css ) {
        // Hook for future AI theme CSS injection
        return $css;
    }
    
    /**
     * Handle theme actions from admin
     */
    public function handle_theme_actions() {
        // Handle preview, activation, deletion actions
        // This will be expanded in future phases
    }

    // ========== THEME SYNCHRONIZATION METHODS ==========

    /**
     * Sync WordPress theme change to Dreamformer database
     * Triggered when theme is changed via Appearance > Themes
     */
    public function sync_wordpress_to_dreamformer( $new_theme_name, $new_theme, $old_theme ) {
        $this->logger->info( 'WordPress theme switched', [
            'new_theme' => $new_theme_name,
            'old_theme' => $old_theme ? $old_theme->get_stylesheet() : 'none'
        ] );

        // Find the corresponding Dreamformer theme
        $dreamformer_theme = $this->find_dreamformer_theme_by_slug( $new_theme_name );

        if ( $dreamformer_theme ) {
            // Update Dreamformer database to mark this theme as active
            $this->set_dreamformer_theme_active( $dreamformer_theme->ID );

            $this->logger->info( 'Synced WordPress theme to Dreamformer', [
                'theme_slug' => $new_theme_name,
                'dreamformer_theme_id' => $dreamformer_theme->ID
            ] );
        } else {
            $this->logger->warning( 'No matching Dreamformer theme found for WordPress theme', [
                'wordpress_theme' => $new_theme_name
            ] );
        }
    }

    /**
     * Check for theme sync mismatches on page load
     */
    public function check_theme_sync() {
        // Only check in admin or when explicitly requested
        if ( ! is_admin() && ! isset( $_GET['check_theme_sync'] ) ) {
            return;
        }

        $wp_active_theme = get_option( 'stylesheet' );
        $dreamformer_active_theme = $this->get_active_dreamformer_theme();

        // If there's a mismatch, log it and optionally auto-fix
        if ( $dreamformer_active_theme && $dreamformer_active_theme->theme_slug !== $wp_active_theme ) {
            $this->logger->warning( 'Theme sync mismatch detected', [
                'wordpress_active' => $wp_active_theme,
                'dreamformer_active' => $dreamformer_active_theme->theme_slug,
                'dreamformer_theme_id' => $dreamformer_active_theme->ID
            ] );

            // Auto-fix: Update Dreamformer to match WordPress
            $this->auto_fix_theme_sync( $wp_active_theme );
        }
    }

    /**
     * Auto-fix theme synchronization issues
     */
    private function auto_fix_theme_sync( $wp_active_theme ) {
        // Find matching Dreamformer theme
        $matching_theme = $this->find_dreamformer_theme_by_slug( $wp_active_theme );

        if ( $matching_theme ) {
            $this->set_dreamformer_theme_active( $matching_theme->ID );

            $this->logger->info( 'Auto-fixed theme sync', [
                'wordpress_theme' => $wp_active_theme,
                'dreamformer_theme_id' => $matching_theme->ID
            ] );
        }
    }

    /**
     * Find Dreamformer theme by WordPress theme slug
     */
    private function find_dreamformer_theme_by_slug( $theme_slug ) {
        $themes = get_posts( [
            'post_type' => 'ai_theme',
            'post_status' => [ 'draft', 'publish' ],
            'posts_per_page' => -1,
            'meta_query' => [
                [
                    'key' => 'theme_directory_slug',
                    'value' => $theme_slug,
                    'compare' => '='
                ]
            ]
        ] );

        return ! empty( $themes ) ? $themes[0] : null;
    }

    /**
     * Get currently active Dreamformer theme
     */
    private function get_active_dreamformer_theme() {
        $themes = get_posts( [
            'post_type' => 'ai_theme',
            'post_status' => [ 'draft', 'publish' ],
            'posts_per_page' => 1,
            'meta_query' => [
                [
                    'key' => 'theme_status',
                    'value' => 'active',
                    'compare' => '='
                ]
            ]
        ] );

        if ( ! empty( $themes ) ) {
            $theme = $themes[0];
            $theme->theme_slug = get_post_meta( $theme->ID, 'theme_directory_slug', true );
            return $theme;
        }

        return null;
    }

    /**
     * Set a Dreamformer theme as active and deactivate others
     */
    private function set_dreamformer_theme_active( $theme_id ) {
        // Deactivate all other themes
        $all_themes = get_posts( [
            'post_type' => 'ai_theme',
            'post_status' => [ 'draft', 'publish' ],
            'posts_per_page' => -1
        ] );

        foreach ( $all_themes as $theme ) {
            update_post_meta( $theme->ID, 'theme_status', $theme->ID == $theme_id ? 'active' : 'ready' );
        }

        $this->logger->info( 'Updated Dreamformer theme active status', [
            'active_theme_id' => $theme_id
        ] );
    }

    /**
     * Clean up orphaned theme database entries
     * Remove themes from database that no longer exist in filesystem
     */
    public function cleanup_orphaned_themes() {
        $themes = get_posts( [
            'post_type' => 'ai_theme',
            'post_status' => [ 'draft', 'publish' ],
            'posts_per_page' => -1
        ] );

        $cleaned_count = 0;

        foreach ( $themes as $theme ) {
            $theme_slug = get_post_meta( $theme->ID, 'theme_directory_slug', true );

            if ( $theme_slug ) {
                $theme_path = get_theme_root() . '/' . $theme_slug;

                // Check if theme directory still exists
                if ( ! file_exists( $theme_path ) || ! is_dir( $theme_path ) ) {
                    // Theme directory doesn't exist - remove from database
                    wp_delete_post( $theme->ID, true );
                    $cleaned_count++;

                    $this->logger->info( 'Removed orphaned theme from database', [
                        'theme_id' => $theme->ID,
                        'theme_slug' => $theme_slug,
                        'missing_path' => $theme_path
                    ] );
                }
            } else {
                // Theme has no slug - remove it
                wp_delete_post( $theme->ID, true );
                $cleaned_count++;

                $this->logger->info( 'Removed theme with missing slug from database', [
                    'theme_id' => $theme->ID
                ] );
            }
        }

        $this->logger->info( 'Theme cleanup completed', [
            'themes_removed' => $cleaned_count
        ] );

        return $cleaned_count;
    }

    /**
     * Update theme database entries with current template info
     * Sync theme names and descriptions from style.css headers
     */
    public function sync_theme_metadata() {
        $themes = get_posts( [
            'post_type' => 'ai_theme',
            'post_status' => [ 'draft', 'publish' ],
            'posts_per_page' => -1
        ] );

        $updated_count = 0;

        foreach ( $themes as $theme ) {
            $theme_slug = get_post_meta( $theme->ID, 'theme_directory_slug', true );

            if ( $theme_slug ) {
                $theme_path = get_theme_root() . '/' . $theme_slug;
                $style_css_path = $theme_path . '/style.css';

                if ( file_exists( $style_css_path ) ) {
                    $theme_data = wp_get_theme( $theme_slug );

                    // Update post title with theme name from style.css
                    if ( $theme_data->get( 'Name' ) && $theme_data->get( 'Name' ) !== $theme->post_title ) {
                        wp_update_post( [
                            'ID' => $theme->ID,
                            'post_title' => $theme_data->get( 'Name' )
                        ] );
                        $updated_count++;

                        $this->logger->info( 'Updated theme metadata', [
                            'theme_id' => $theme->ID,
                            'old_name' => $theme->post_title,
                            'new_name' => $theme_data->get( 'Name' )
                        ] );
                    }
                }
            }
        }

        return $updated_count;
    }

    /**
     * Detect and import WordPress themes that don't exist in Dreamformer Studio
     */
    public function import_wordpress_themes() {
        $imported_count = 0;
        $wp_themes = wp_get_themes();

        foreach ( $wp_themes as $theme_slug => $theme_data ) {
            // Check if this theme already exists in Dreamformer
            $existing_theme = $this->find_dreamformer_theme_by_slug( $theme_slug );

            if ( ! $existing_theme ) {
                // Check if this looks like a Dreamformer-compatible theme
                $theme_path = get_theme_root() . '/' . $theme_slug;
                $style_css_path = $theme_path . '/style.css';

                if ( file_exists( $style_css_path ) ) {
                    // Create new theme entry in Dreamformer Studio
                    $theme_post_id = wp_insert_post( [
                        'post_type' => 'ai_theme',
                        'post_status' => 'publish',
                        'post_title' => $theme_data->get( 'Name' ),
                        'meta_input' => [
                            'theme_directory_slug' => $theme_slug,
                            'business_type' => 'imported',
                            'style_preference' => 'existing',
                            'theme_status' => 'ready'
                        ]
                    ] );

                    if ( $theme_post_id && ! is_wp_error( $theme_post_id ) ) {
                        $imported_count++;

                        $this->logger->info( 'Imported WordPress theme to Dreamformer Studio', [
                            'theme_slug' => $theme_slug,
                            'theme_name' => $theme_data->get( 'Name' ),
                            'post_id' => $theme_post_id
                        ] );
                    }
                }
            }
        }

        return $imported_count;
    }
}