<?php
/**
 * AI Theme Editor - Clean Ultra-Fast Version
 * Handles AI-powered theme modifications through simple file replacement
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Theme_Editor {

    private $theme_manager;
    private $logger;
    
    // Ultra-fast simple file editor
    private $file_editor;
    
    // AST parser for manual editing
    private $ast_parser;
    private $use_ast_parser = true; // Feature flag: true for AST, false for regex
    
    public function __construct() {
        $this->logger = AI_Logger::get_instance( 'AI_Theme_Editor' );
        
        // AI service removed - now using Vercel APIs only via file_editor
        $this->theme_manager = new AI_Theme_Manager();
        
        // Initialize ultra-fast file editor
        $this->file_editor = new AI_Simple_File_Editor();
        
        // Initialize AST parser if enabled
        if ( $this->use_ast_parser && class_exists( 'AI_CSS_AST_Parser' ) ) {
            $this->ast_parser = new AI_CSS_AST_Parser();
        }
        
        $this->init_hooks();
    }
    
    private function init_hooks() {
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
    }
    
    /**
     * Register REST API routes for AI editing
     */
    public function register_rest_routes() {
        // Preview session endpoint (required by frontend)
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/preview-session', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_create_preview_session' ),
            'permission_callback' => function() { 
                return current_user_can( 'edit_theme_options' ); 
            }
        ) );
        
        // Edit history endpoint (required by frontend) 
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/history', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_get_edit_history' ),
            'permission_callback' => function() { 
                return current_user_can( 'edit_theme_options' ); 
            }
        ) );
        
        // Theme list endpoint (required by theme library)
        register_rest_route( 'ai-builder/v1', '/themes', array(
            'methods' => 'GET',
            'callback' => array( $this, 'rest_list_themes' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_theme_options' );
            }
        ) );

        // Theme cleanup endpoint
        register_rest_route( 'ai-builder/v1', '/themes/cleanup', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_cleanup_themes' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_theme_options' );
            }
        ) );

        
        // Screenshot upload endpoint
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/screenshot', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_upload_screenshot' ),
            'permission_callback' => function() { 
                return current_user_can( 'edit_theme_options' ); 
            }
        ) );
        
        // CSS rules analysis endpoint - Enhanced context system
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/css-rules', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_analyze_css_rules' ),
            'permission_callback' => function() { 
                return current_user_can( 'edit_theme_options' ); 
            },
            'args' => array(
                'selector' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS selector to analyze'
                )
            )
        ) );
        
        // Manual CSS property update endpoint
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/update-property', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_update_css_property' ),
            'permission_callback' => function() { 
                return current_user_can( 'edit_theme_options' ); 
            },
            'args' => array(
                'selector' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS selector to update'
                ),
                'property' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS property name'
                ),
                'value' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'New CSS value'
                ),
                'file' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => 'CSS file to modify'
                )
            )
        ) );

        // VERSION CONTROL ENDPOINTS

        // Note: Undo/Redo/Versions now handled by unified version control system

        // Theme activation endpoint
        register_rest_route( 'ai-builder/v1', '/themes/(?P<theme_id>\d+)/activate', array(
            'methods' => 'POST',
            'callback' => array( $this, 'rest_activate_theme' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_theme_options' );
            }
        ) );
    }

    /**
     * Build technical changes summary from edit results
     * 
     * @param array $edit_result Results from the file editing operation
     * @return array Array of technical change descriptions
     */
    private function build_technical_changes_summary( $edit_result ) {
        $changes = array();
        
        // Handle multiple files if available
        if ( isset( $edit_result['files_modified'] ) && is_array( $edit_result['files_modified'] ) ) {
            foreach ( $edit_result['files_modified'] as $file_path ) {
                $changes[] = array(
                    'file' => basename( $file_path ),
                    'description' => 'Modified ' . basename( $file_path )
                );
            }
        } else if ( isset( $edit_result['file_updated'] ) ) {
            // Single file modification
            $file_name = basename( $edit_result['file_updated'] );
            $bytes = $edit_result['bytes_written'] ?? null;
            $changes_count = $edit_result['changes_applied'] ?? null;
            
            $description = 'Modified ' . $file_name;
            if ( $bytes ) {
                $description .= ' (' . number_format( $bytes ) . ' bytes)';
            }
            if ( $changes_count ) {
                $description .= ' • ' . $changes_count . ' changes applied';
            }
            
            $changes[] = array(
                'file' => $file_name,
                'description' => $description
            );
        }
        
        // Fallback if no specific files identified
        if ( empty( $changes ) ) {
            $changes[] = array(
                'file' => 'theme files',
                'description' => 'Theme files updated'
            );
        }
        
        return $changes;
    }

    /**
     * Get preview URL for theme
     */
    private function get_preview_url( $theme_slug ) {
        return add_query_arg( array(
            'theme' => $theme_slug,
            'preview' => 1,
            'preview_nonce' => wp_create_nonce( 'theme_preview_' . $theme_slug )
        ), home_url() );
    }
    
    /**
     * REST: Create preview session (required by frontend)
     */
    public function rest_create_preview_session( $request ) {
        $theme_id = $request->get_param( 'theme_id' );
        
        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_REST_Response( array( 
                'success' => false,
                'message' => 'Theme not found' 
            ), 404 );
        }
        
        // Get theme directory
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );
        if ( ! $theme_slug ) {
            return new WP_REST_Response( array( 
                'success' => false,
                'message' => 'Theme directory not found' 
            ), 400 );
        }
        
        // Generate simple preview session
        $session_id = 'preview_' . $theme_id . '_' . time();
        $preview_url = $this->get_preview_url( $theme_slug );
        
        return rest_ensure_response( array(
            'success' => true,
            'session_id' => $session_id,
            'preview_url' => $preview_url
        ) );
    }
    
    /**
     * REST: Get edit history (required by frontend)
     */
    public function rest_get_edit_history( $request ) {
        $theme_id = $request->get_param( 'theme_id' );
        
        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_REST_Response( array( 
                'success' => false,
                'message' => 'Theme not found' 
            ), 404 );
        }
        
        // Return empty history - clean start for users
        $history = array();
        
        return rest_ensure_response( array(
            'success' => true,
            'history' => $history
        ) );
    }
    
    /**
     * Save edit history (simplified)
     */
    private function save_edit_history( $theme_id, $edit_data ) {
        $this->logger->info( 'Edit completed', array(
            'theme_id' => $theme_id,
            'file_updated' => $edit_data['file_updated'] ?? 'unknown',
            'bytes_written' => $edit_data['bytes_written'] ?? 0
        ) );
    }
    
    /**
     * REST endpoint: Analyze CSS rules for a selector - Enhanced context system
     */
    public function rest_analyze_css_rules( $request ) {
        $theme_id = (int) $request['theme_id'];
        $selector = sanitize_text_field( $request['selector'] );
        
        // Get theme post and validate
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_Error( 'theme_not_found', 'Theme not found', array( 'status' => 404 ) );
        }
        
        // Get theme path using consistent meta key
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );
        if ( ! $theme_slug ) {
            return new WP_Error( 'theme_slug_missing', 'Theme directory slug not found', array( 'status' => 404 ) );
        }
        
        $theme_path = get_theme_root() . '/' . $theme_slug;
        if ( ! is_dir( $theme_path ) ) {
            return new WP_Error( 'theme_path_invalid', 'Theme directory not found', array( 'status' => 404 ) );
        }
        
        // Analyze CSS rules for the selector
        $css_analysis = $this->analyze_css_rules_for_selector( $theme_path, $selector );
        if ( is_wp_error( $css_analysis ) ) {
            return $css_analysis;
        }
        
        return rest_ensure_response( array(
            'success' => true,
            'selector' => $selector,
            'css_rules' => $css_analysis['rules'] ?? array(),
            'theme_files' => $this->get_theme_file_list( $theme_path ),
            'debug_info' => array(
                'files_found' => $css_analysis['files_found'] ?? 0,
                'theme_path' => $theme_path,
                'total_rules_checked' => $css_analysis['total_rules'] ?? 0,
                'files_processed' => $css_analysis['files_processed'] ?? array(),
                'css_file_paths' => $css_analysis['css_file_paths'] ?? array()
            )
        ) );
    }
    
    /**
     * Analyze CSS rules for a specific selector across theme files
     * Uses AST parser if enabled, falls back to regex parser
     */
    private function analyze_css_rules_for_selector( $theme_path, $selector ) {
        // Use AST parser if available
        if ( $this->use_ast_parser && $this->ast_parser ) {
            return $this->analyze_css_rules_with_ast( $theme_path, $selector );
        }
        
        // Fallback to original regex-based parser
        return $this->analyze_css_rules_with_regex( $theme_path, $selector );
    }
    
    /**
     * AST-based CSS rules analysis - Enhanced version
     */
    private function analyze_css_rules_with_ast( $theme_path, $selector ) {
        $css_rules = array();
        $total_rules_checked = 0;
        $files_processed = array();
        $css_files = $this->discover_css_files( $theme_path );
        
        if ( empty( $css_files ) ) {
            return array(
                'rules' => array(),
                'files_found' => 0,
                'total_rules' => 0,
                'files_processed' => array(),
                'css_file_paths' => array()
            );
        }
        
        foreach ( $css_files as $css_file ) {
            $filename = basename( $css_file );
            $css_content = file_get_contents( $css_file );
            
            if ( $css_content === false ) {
                $files_processed[] = $filename . ' (failed to read)';
                continue;
            }
            
            // Parse CSS with AST
            $document = $this->ast_parser->parse_css_content( $css_content, $css_file );
            if ( is_wp_error( $document ) ) {
                $files_processed[] = $filename . ' (parse failed)';
                continue;
            }
            
            // Find matching rules using AST
            $matching_rules = $this->ast_parser->find_rules_for_selector( $document, $selector );
            $total_rules_checked += count( $matching_rules );
            $files_processed[] = $filename . ' (' . count( $matching_rules ) . ' AST rules)';
            
            // Format rules for response
            foreach ( $matching_rules as $rule ) {
                $css_rules[] = array(
                    'file' => $filename,
                    'line' => 1, // AST doesn't provide line numbers easily
                    'selector' => $rule['selector'],
                    'css' => $this->format_ast_rule_as_css( $rule ),
                    'properties' => $rule['properties'],
                    'specificity' => $rule['specificity'],
                    'editable' => true,
                    'ast_source' => true // Flag to indicate this came from AST
                );
            }
        }
        
        return array(
            'rules' => $css_rules,
            'files_found' => count( $css_files ),
            'total_rules' => $total_rules_checked,
            'files_processed' => $files_processed,
            'css_file_paths' => $css_files,
            'parser_type' => 'AST'
        );
    }
    
    /**
     * Original regex-based CSS rules analysis - Fallback
     */
    private function analyze_css_rules_with_regex( $theme_path, $selector ) {
        $css_rules = array();
        $total_rules_checked = 0;
        
        // Find all CSS files in theme - check common locations
        $css_files = array();
        
        // Check root directory
        $root_files = glob( $theme_path . '/*.css' );
        if ( $root_files ) {
            $css_files = array_merge( $css_files, $root_files );
        }
        
        // Check common CSS subdirectories
        $css_directories = array(
            'assets/css',
            'css', 
            'styles',
            'assets/styles'
        );
        
        foreach ( $css_directories as $css_dir ) {
            $dir_path = $theme_path . '/' . $css_dir;
            if ( is_dir( $dir_path ) ) {
                $dir_files = glob( $dir_path . '/*.css' );
                if ( $dir_files ) {
                    $css_files = array_merge( $css_files, $dir_files );
                }
            }
        }
        
        if ( empty( $css_files ) ) {
            return new WP_Error( 'no_css_files', 'No CSS files found in theme' );
        }
        
        // Debug: log all CSS files found
        
        $files_processed = array();
        
        foreach ( $css_files as $css_file ) {
            $filename = basename( $css_file );
            $css_content = file_get_contents( $css_file );
            
            if ( $css_content === false ) {
                $files_processed[] = $filename . ' (failed to read)';
                continue;
            }
            
            // Parse CSS and find matching rules
            $matching_rules = $this->parse_css_for_selector( $css_content, $selector, $filename );
            $total_rules_checked += count( $matching_rules );
            $files_processed[] = $filename . ' (' . count( $matching_rules ) . ' potential rules)';
            
            if ( ! empty( $matching_rules ) ) {
                $css_rules = array_merge( $css_rules, $matching_rules );
            }
        }
        
        return array(
            'rules' => $css_rules,
            'files_found' => count( $css_files ),
            'total_rules' => $total_rules_checked,
            'files_processed' => $files_processed,
            'css_file_paths' => $css_files
        );
    }
    
    /**
     * Parse CSS content to find rules matching a selector
     */
    private function parse_css_for_selector( $css_content, $target_selector, $filename ) {
        $rules = array();
        $lines = explode( "\n", $css_content );
        
        // Simple CSS parser - matches selectors and extracts properties
        $current_selector = '';
        $in_rule = false;
        $rule_start_line = 0;
        $properties = array();
        $brace_count = 0;
        $in_media_query = false;
        $media_query_text = '';
        
        foreach ( $lines as $line_num => $line ) {
            $line = trim( $line );
            
            // Skip comments and empty lines
            if ( empty( $line ) || strpos( $line, '/*' ) === 0 ) {
                continue;
            }
            
            // Handle media queries
            if ( strpos( $line, '@media' ) !== false ) {
                $in_media_query = true;
                $media_query_text = trim( str_replace( '{', '', $line ) );
                // Don't continue here - let it fall through to brace handling
            }
            
            // Check for opening brace
            if ( strpos( $line, '{' ) !== false ) {
                $brace_count++;
                
                // If this is a media query opening, just track it
                if ( $in_media_query && $brace_count == 1 ) {
                    continue;
                }
                
                // If this is a selector inside media query or regular selector
                if ( ! $in_rule && ( ( $in_media_query && $brace_count == 2 ) || ( ! $in_media_query && $brace_count == 1 ) ) ) {
                    // This is a selector line
                    $current_selector = trim( str_replace( '{', '', $line ) );
                    $in_rule = true;
                    $rule_start_line = $line_num + 1;
                    $properties = array();
                }
                continue;
            }
            
            // Check for closing brace
            if ( strpos( $line, '}' ) !== false ) {
                $brace_count--;
                
                // Check if we're ending a rule (inside or outside media query)
                $rule_ending = ( $in_media_query && $brace_count == 1 ) || ( ! $in_media_query && $brace_count == 0 );
                
                if ( $in_rule && $rule_ending ) {
                    // End of current rule
                    $matches = $this->selector_matches( $current_selector, $target_selector );
                    // Debug: log what we're comparing
                    if ( $matches ) {
                        // Format properties as CSS text
                        if ( $in_media_query && ! empty( $media_query_text ) ) {
                            $css_text = $media_query_text . " {\n    " . $current_selector . " {\n";
                            foreach ( $properties as $property => $value ) {
                                $css_text .= "        " . $property . ": " . $value . ";\n";
                            }
                            $css_text .= "    }\n}";
                            $display_selector = $media_query_text . ' ' . $current_selector;
                        } else {
                            $css_text = $current_selector . " {\n";
                            foreach ( $properties as $property => $value ) {
                                $css_text .= "    " . $property . ": " . $value . ";\n";
                            }
                            $css_text .= "}";
                            $display_selector = $current_selector;
                        }
                        
                        $rules[] = array(
                            'file' => $filename,
                            'line' => $rule_start_line,
                            'selector' => $display_selector,
                            'css' => $css_text,
                            'properties' => $properties,
                            'specificity' => $this->calculate_css_specificity( $current_selector )
                        );
                    }
                    
                    $in_rule = false;
                    $current_selector = '';
                    $properties = array();
                }
                
                // If we're closing media query completely
                if ( $brace_count == 0 && $in_media_query ) {
                    $in_media_query = false;
                    $media_query_text = '';
                }
                continue;
            }
            
            // Collect properties within rule (adjust brace count for media queries)
            $property_brace_level = $in_media_query ? 2 : 1;
            if ( $in_rule && strpos( $line, ':' ) !== false && $brace_count == $property_brace_level ) {
                $parts = explode( ':', $line, 2 );
                if ( count( $parts ) === 2 ) {
                    $property = trim( $parts[0] );
                    $value = trim( rtrim( $parts[1], ';' ) );
                    if ( ! empty( $property ) && ! empty( $value ) ) {
                        $properties[ $property ] = $value;
                    }
                }
            }
        }
        
        return $rules;
    }
    
    /**
     * Check if a CSS selector could match the target element - MVP generous matching
     */
    private function selector_matches( $css_selector, $target_selector ) {
        $css_selector = trim( $css_selector );
        $target_selector = trim( $target_selector );
        
        // Exact match
        if ( $css_selector === $target_selector ) {
            return true;
        }
        
        // For MVP: be very generous with matching to show more CSS rules
        
        // If target is an element (like 'div', 'h1'), match any CSS rule containing that element
        if ( ! strpos( $target_selector, '.' ) && ! strpos( $target_selector, '#' ) ) {
            // Match element names in CSS selector
            if ( preg_match( '/\b' . preg_quote( $target_selector, '/' ) . '\b/', $css_selector ) ) {
                return true;
            }
        }
        
        // If target has classes, check if any class matches
        if ( strpos( $target_selector, '.' ) !== false ) {
            // Extract classes from target (e.g., ".hero.section" -> ["hero", "section"])
            preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $target_selector, $target_matches );
            $target_classes = $target_matches[1] ?? array();
            
            // Extract classes from CSS selector
            preg_match_all( '/\.([a-zA-Z0-9_-]+)/', $css_selector, $css_matches );
            $css_classes = $css_matches[1] ?? array();
            
            // Check if any target class matches any CSS class
            foreach ( $target_classes as $target_class ) {
                foreach ( $css_classes as $css_class ) {
                    if ( $target_class === $css_class ) {
                        return true;
                    }
                }
            }
        }
        
        // If target has ID, check if ID matches
        if ( strpos( $target_selector, '#' ) !== false ) {
            $id = str_replace( '#', '', $target_selector );
            if ( strpos( $css_selector, $id ) !== false ) {
                return true;
            }
        }
        
        // For MVP: Add more generous matching patterns
        
        // Handle button classes specifically (common case)
        if ( strpos( $target_selector, 'button' ) !== false && strpos( $css_selector, 'button' ) !== false ) {
            return true;
        }
        
        // Handle wrap/container classes specifically  
        if ( strpos( $target_selector, 'wrap' ) !== false && strpos( $css_selector, 'wrap' ) !== false ) {
            return true;
        }
        
        // Handle section classes
        if ( strpos( $target_selector, 'section' ) !== false && strpos( $css_selector, 'section' ) !== false ) {
            return true;
        }
        
        // Generic partial matching for common element types
        $common_elements = array( 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'p', 'a', 'img', 'nav', 'header', 'footer', 'main' );
        foreach ( $common_elements as $element ) {
            if ( strpos( $target_selector, $element ) !== false && strpos( $css_selector, $element ) !== false ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Discover all CSS files in theme directory
     */
    private function discover_css_files( $theme_path ) {
        $css_files = array();
        
        // Check root directory
        $root_files = glob( $theme_path . '/*.css' );
        if ( $root_files ) {
            $css_files = array_merge( $css_files, $root_files );
        }
        
        // Check common CSS subdirectories
        $css_directories = array(
            'assets/css',
            'css', 
            'styles',
            'assets/styles'
        );
        
        foreach ( $css_directories as $css_dir ) {
            $dir_path = $theme_path . '/' . $css_dir;
            if ( is_dir( $dir_path ) ) {
                $dir_files = glob( $dir_path . '/*.css' );
                if ( $dir_files ) {
                    $css_files = array_merge( $css_files, $dir_files );
                }
            }
        }
        
        return $css_files;
    }
    
    /**
     * Format AST rule as CSS text for display
     */
    private function format_ast_rule_as_css( $rule ) {
        $css_text = $rule['selector'] . " {\n";
        
        foreach ( $rule['properties'] as $property => $details ) {
            $important = $details['important'] ? ' !important' : '';
            $css_text .= "    " . $property . ": " . $details['value'] . $important . ";\n";
        }
        
        $css_text .= "}";
        return $css_text;
    }
    
    /**
     * Calculate CSS specificity (simplified)
     */
    private function calculate_css_specificity( $selector ) {
        $specificity = 0;
        
        // Count IDs (most specific)
        $specificity += substr_count( $selector, '#' ) * 100;
        
        // Count classes and attributes
        $specificity += ( substr_count( $selector, '.' ) + substr_count( $selector, '[' ) ) * 10;
        
        // Count elements
        $specificity += preg_match_all( '/\b[a-zA-Z][\w-]*\b/', $selector );
        
        return $specificity;
    }
    
    /**
     * Get list of theme files for reference
     */
    private function get_theme_file_list( $theme_path ) {
        $files = array();
        $all_files = array_merge(
            glob( $theme_path . '/*.css' ) ?: array(),
            glob( $theme_path . '/*.php' ) ?: array()
        );
        
        foreach ( $all_files as $file ) {
            $files[] = array(
                'name' => basename( $file ),
                'path' => $file,
                'size' => filesize( $file ),
                'modified' => filemtime( $file )
            );
        }
        
        return $files;
    }
    
    /**
     * REST endpoint: List all themes for theme library
     */
    public function rest_list_themes( $request ) {
        $themes = get_posts( array(
            'post_type' => 'ai_theme',
            'post_status' => array( 'draft', 'publish' ),
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => 'theme_directory_slug',
                    'compare' => 'EXISTS'
                )
            )
        ) );
        
        $theme_data = array();
        foreach ( $themes as $theme ) {
            $theme_slug = get_post_meta( $theme->ID, 'theme_directory_slug', true );
            $is_active = ( get_option( 'stylesheet' ) === $theme_slug );
            
            $theme_data[] = array(
                'id' => $theme->ID,
                'business_name' => $theme->post_title,
                'business_type' => get_post_meta( $theme->ID, 'business_type', true ) ?: 'general',
                'style_preference' => get_post_meta( $theme->ID, 'style_preference', true ) ?: 'modern',
                'status' => get_post_meta( $theme->ID, 'theme_status', true ) ?: 'ready',
                'is_active' => $is_active,
                'created' => $theme->post_date,
                'preview_available' => file_exists( get_theme_root() . '/' . $theme_slug ),
                'theme_slug' => $theme_slug,
                'screenshot' => get_post_meta( $theme->ID, 'screenshot', true )
            );
        }
        
        return rest_ensure_response( array(
            'success' => true,
            'themes' => $theme_data
        ) );
    }

    /**
     * Clean up orphaned themes and sync metadata
     */
    public function rest_cleanup_themes( $request ) {
        $theme_manager = new AI_Theme_Manager();

        // Clean up orphaned themes
        $orphaned_count = $theme_manager->cleanup_orphaned_themes();

        // Import WordPress themes that don't exist in Dreamformer Studio
        $imported_count = $theme_manager->import_wordpress_themes();

        // Sync metadata for remaining themes
        $updated_count = $theme_manager->sync_theme_metadata();

        return rest_ensure_response( array(
            'success' => true,
            'message' => "Cleanup completed: {$orphaned_count} orphaned themes removed, {$imported_count} themes imported, {$updated_count} themes updated",
            'orphaned_removed' => $orphaned_count,
            'themes_imported' => $imported_count,
            'metadata_updated' => $updated_count
        ) );
    }


    /**
     * Handle screenshot upload for a theme
     */
    public function rest_upload_screenshot( $request ) {
        // Suppress any output that might corrupt JSON
        ob_start();
        
        $theme_id = $request->get_param( 'theme_id' );
        
        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'Theme not found'
            ) );
        }
        
        // Check if file was uploaded
        if ( empty( $_FILES['screenshot'] ) ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'No file uploaded'
            ) );
        }
        
        $file = $_FILES['screenshot'];
        
        // Check for upload errors
        if ( $file['error'] !== UPLOAD_ERR_OK ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'Upload error: ' . $file['error']
            ) );
        }
        
        // SECURITY: Multi-layer file validation

        // Layer 1: Validate file extension (prevents .php.jpg attacks)
        $allowed_extensions = array( 'jpg', 'jpeg', 'png', 'gif' );
        $extension = strtolower( pathinfo( $file['name'], PATHINFO_EXTENSION ) );

        if ( ! in_array( $extension, $allowed_extensions, true ) ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'Invalid file extension. Allowed: JPG, JPEG, PNG, GIF'
            ) );
        }

        // Layer 2: Verify actual file content (prevents MIME spoofing)
        $image_info = @getimagesize( $file['tmp_name'] );
        if ( $image_info === false ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'File is not a valid image. Upload failed security check.'
            ) );
        }

        // Layer 3: Validate MIME type matches extension
        $allowed_mime_types = array(
            'jpg'  => array( 'image/jpeg', 'image/jpg' ),
            'jpeg' => array( 'image/jpeg', 'image/jpg' ),
            'png'  => array( 'image/png' ),
            'gif'  => array( 'image/gif' )
        );

        if ( ! isset( $allowed_mime_types[ $extension ] ) ||
             ! in_array( $image_info['mime'], $allowed_mime_types[ $extension ], true ) ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'File extension does not match image type. Possible file manipulation detected.'
            ) );
        }

        // Layer 4: Check file size (5MB max)
        if ( $file['size'] > 5 * 1024 * 1024 ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'File too large. Maximum size is 5MB.'
            ) );
        }
        
        // Create screenshots directory if it doesn't exist
        $screenshots_dir = AI_SITE_BUILDER_PLUGIN_DIR . 'admin/images/screenshots/';
        if ( ! file_exists( $screenshots_dir ) ) {
            if ( ! wp_mkdir_p( $screenshots_dir ) ) {
                ob_end_clean();
                return rest_ensure_response( array(
                    'success' => false,
                    'message' => 'Failed to create screenshots directory'
                ) );
            }
        }
        
        // Check if directory is writable
        if ( ! is_writable( $screenshots_dir ) ) {
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'Screenshots directory is not writable'
            ) );
        }
        
        // Generate filename using validated extension
        // $extension already validated above, reuse it
        $filename = 'theme-' . $theme_id . '.' . $extension;
        $filepath = $screenshots_dir . $filename;
        
        // Remove existing screenshot if it exists
        $existing_files = glob( $screenshots_dir . 'theme-' . $theme_id . '.*' );
        foreach ( $existing_files as $existing_file ) {
            if ( file_exists( $existing_file ) ) {
                unlink( $existing_file );
            }
        }
        
        // Move uploaded file
        if ( move_uploaded_file( $file['tmp_name'], $filepath ) ) {
            // Verify file was actually created
            if ( ! file_exists( $filepath ) ) {
                ob_end_clean();
                return rest_ensure_response( array(
                    'success' => false,
                    'message' => 'File upload succeeded but file not found after upload'
                ) );
            }
            
            // Update theme meta
            update_post_meta( $theme_id, 'screenshot', $filename );
            
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => true,
                'message' => 'Screenshot uploaded successfully',
                'filename' => $filename
            ) );
        } else {
            $error_msg = 'Failed to save file';
            if ( ! is_uploaded_file( $file['tmp_name'] ) ) {
                $error_msg .= ' - Invalid upload file';
            } elseif ( ! file_exists( $file['tmp_name'] ) ) {
                $error_msg .= ' - Temporary file missing';
            } elseif ( ! is_readable( $file['tmp_name'] ) ) {
                $error_msg .= ' - Temporary file not readable';
            }
            
            ob_end_clean();
            return rest_ensure_response( array(
                'success' => false,
                'message' => $error_msg
            ) );
        }
    }
    
    /**
     * REST endpoint: Update CSS property manually
     */
    public function rest_update_css_property( $request ) {
        $theme_id = (int) $request['theme_id'];
        $selector = sanitize_text_field( $request['selector'] );
        $property = sanitize_text_field( $request['property'] );
        $value = sanitize_textarea_field( $request['value'] );
        $file = sanitize_text_field( $request['file'] );
        
        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_Error( 'theme_not_found', 'Theme not found', array( 'status' => 404 ) );
        }
        
        // Get theme path
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );
        if ( ! $theme_slug ) {
            return new WP_Error( 'theme_slug_missing', 'Theme directory slug not found', array( 'status' => 404 ) );
        }
        
        $theme_path = get_theme_root() . '/' . $theme_slug;
        if ( ! is_dir( $theme_path ) ) {
            return new WP_Error( 'theme_path_invalid', 'Theme directory not found', array( 'status' => 404 ) );
        }
        
        // Validate file path (security)
        $file_path = $theme_path . '/' . $file;
        if ( ! file_exists( $file_path ) || strpos( realpath( $file_path ), realpath( $theme_path ) ) !== 0 ) {
            return new WP_Error( 'invalid_file', 'Invalid file path', array( 'status' => 400 ) );
        }
        
        // Update property using AST if available
        if ( $this->use_ast_parser && $this->ast_parser ) {
            $result = $this->update_css_property_with_ast( $file_path, $selector, $property, $value );
        } else {
            return new WP_Error( 'ast_required', 'AST parser required for manual editing', array( 'status' => 500 ) );
        }
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }
        
        return rest_ensure_response( array(
            'success' => true,
            'message' => 'Property updated successfully',
            'selector' => $selector,
            'property' => $property,
            'value' => $value,
            'file' => $file
        ) );
    }
    
    /**
     * Update CSS property using AST parser
     */
    private function update_css_property_with_ast( $file_path, $selector, $property, $value ) {
        try {
            // Read current CSS content
            $css_content = file_get_contents( $file_path );
            if ( $css_content === false ) {
                return new WP_Error( 'file_read_error', 'Could not read CSS file' );
            }
            
            // Old backup system removed - using unified version control instead
            
            // Parse CSS with AST
            $document = $this->ast_parser->parse_css_content( $css_content );
            if ( is_wp_error( $document ) ) {
                return $document;
            }
            
            // Update property
            $update_result = $this->ast_parser->update_property( $document, $selector, $property, $value );
            if ( is_wp_error( $update_result ) ) {
                return $update_result;
            }
            
            // Serialize back to CSS
            $updated_css = $this->ast_parser->serialize_to_css( $document );
            
            // Write updated CSS using native PHP functions for reliability
            // Check if file is writable
            if ( ! is_writable( dirname( $file_path ) ) ) {
                return new WP_Error( 'file_permissions', 'CSS directory is not writable' );
            }
            
            // Write the file
            $bytes_written = file_put_contents( $file_path, $updated_css, LOCK_EX );
            
            if ( $bytes_written === false ) {
                return new WP_Error( 'file_write_error', 'Could not write updated CSS file' );
            }
            
            // Set proper file permissions
            $file_permissions = defined( 'FS_CHMOD_FILE' ) ? FS_CHMOD_FILE : 0644;
            @chmod( $file_path, $file_permissions );
            
            // Log the change
            $this->logger->info( 'CSS property updated manually', array(
                'file' => basename( $file_path ),
                'selector' => $selector,
                'property' => $property,
                'old_value' => 'unknown', // AST doesn't easily provide old value
                'new_value' => $value
            ) );
            
            return true;
            
        } catch ( Exception $e ) {
            return new WP_Error( 'update_exception', $e->getMessage() );
        }
    }

    /**
     * Save theme version for version control
     *
     * @param int $theme_id Theme post ID
     * @param string $theme_path Theme directory path
     * @param array $content_before File contents before changes
     * @param array $edit_result Results from file editor
     * @param string $user_prompt Original user prompt
     * @return int|WP_Error Version ID or error
     */
    private function save_theme_version( $theme_id, $theme_path, $content_before, $edit_result, $user_prompt ) {
        try {
            // Log input data to understand the problem
            $this->logger->debug( 'Version save input data', array(
                'content_before_keys' => is_array( $content_before ) ? array_keys( $content_before ) : 'not_array',
                'content_before_count' => is_array( $content_before ) ? count( $content_before ) : 0,
                'files_modified' => $edit_result['files_modified'] ?? 'none',
                'files_modified_count' => isset( $edit_result['files_modified'] ) ? count( $edit_result['files_modified'] ) : 0
            ) );

            // Get content of only the files that were actually modified
            $content_before_modified = array();
            $content_after_modified = array();

            if ( isset( $edit_result['files_modified'] ) && is_array( $edit_result['files_modified'] ) ) {
                foreach ( $edit_result['files_modified'] as $modified_file_path ) {
                    // Handle both relative and absolute paths from file editor
                    if ( strpos( $modified_file_path, $theme_path ) === 0 ) {
                        // Already a full path
                        $full_path = $modified_file_path;
                        $relative_path = str_replace( $theme_path . '/', '', $modified_file_path );
                    } else {
                        // Just a filename, construct full path
                        $relative_path = $modified_file_path;
                        $full_path = $theme_path . '/' . $modified_file_path;
                    }

                    // Log path resolution
                    $this->logger->debug( 'Processing modified file', array(
                        'modified_file_path' => $modified_file_path,
                        'relative_path' => $relative_path,
                        'full_path' => $full_path,
                        'content_before_has_key' => isset( $content_before[ $relative_path ] ),
                        'file_exists' => file_exists( $full_path )
                    ) );

                    // Get before content from our captured snapshot
                    if ( isset( $content_before[ $relative_path ] ) ) {
                        $content_before_modified[ $relative_path ] = $content_before[ $relative_path ];
                    }

                    // Get after content from the modified file
                    if ( file_exists( $full_path ) && is_readable( $full_path ) ) {
                        $content_after_modified[ $relative_path ] = file_get_contents( $full_path );
                    }
                }
            }

            // Log results before validation
            $this->logger->debug( 'Version save content collected', array(
                'content_before_modified_count' => count( $content_before_modified ),
                'content_after_modified_count' => count( $content_after_modified ),
                'content_before_modified_keys' => array_keys( $content_before_modified ),
                'content_after_modified_keys' => array_keys( $content_after_modified )
            ) );

            // Only save version if we have both before and after content
            if ( empty( $content_before_modified ) || empty( $content_after_modified ) ) {
                $this->logger->warning( 'Skipping version save - no valid content to version', array(
                    'theme_id' => $theme_id,
                    'files_modified' => $edit_result['files_modified'] ?? 'none',
                    'content_before_empty' => empty( $content_before_modified ),
                    'content_after_empty' => empty( $content_after_modified )
                ) );
                return null;
            }

            // Generate change description
            $change_description = $this->generate_theme_change_description( $edit_result );

            // Prepare version control data
            $version_control = AI_Version_Control::get_instance();
            $version_id = $version_control->save_version( 'theme', $theme_id,
                json_encode( $content_before_modified ),
                json_encode( $content_after_modified ),
                array(
                    'change_description' => $change_description,
                    'user_prompt' => $user_prompt,
                    'ai_model' => AI_Config_Manager::AI_MODEL,
                    'change_metadata' => array(
                        'files_modified' => array_keys( $content_after_modified ),
                        'files_targeted' => array_keys( $content_before ),
                        'total_targets' => count( $content_before ),
                        'actual_changes' => count( $content_after_modified ),
                        'edit_method' => $edit_result['method'] ?? 'unknown',
                        'changes_applied' => $edit_result['changes_applied'] ?? 0,
                        'execution_time' => $edit_result['execution_time'] ?? null
                    )
                )
            );

            if ( is_wp_error( $version_id ) ) {
                $this->logger->error( 'Failed to save theme version', array(
                    'theme_id' => $theme_id,
                    'error' => $version_id->get_error_message()
                ) );
                return $version_id;
            }

            $this->logger->info( 'Theme version saved successfully', array(
                'theme_id' => $theme_id,
                'version_id' => $version_id,
                'files_modified' => array_keys( $content_after_modified ),
                'change_description' => $change_description
            ) );

            return $version_id;

        } catch ( Exception $e ) {
            $this->logger->error( 'Exception saving theme version', array(
                'theme_id' => $theme_id,
                'error' => $e->getMessage()
            ) );
            return new WP_Error( 'version_save_exception', $e->getMessage() );
        }
    }

    /**
     * Generate theme change description from edit results
     *
     * @param array $edit_result Results from file editor
     * @return string Human-readable change description
     */
    private function generate_theme_change_description( $edit_result ) {
        $files_count = 0;
        $primary_file = 'theme files';

        if ( isset( $edit_result['files_modified'] ) && is_array( $edit_result['files_modified'] ) ) {
            $files_count = count( $edit_result['files_modified'] );
            if ( $files_count > 0 ) {
                $primary_file = basename( $edit_result['files_modified'][0] );
            }
        }

        if ( $files_count === 0 ) {
            return 'AI edit (no files modified)';
        } elseif ( $files_count === 1 ) {
            return 'AI edit: ' . $primary_file;
        } else {
            return sprintf( 'AI edit: %s and %d other file%s',
                $primary_file,
                $files_count - 1,
                ( $files_count - 1 ) === 1 ? '' : 's'
            );
        }
    }

    // NOTE: Theme undo functionality moved to unified version control system

    /**
     * REST API: Redo theme change
     */
    public function rest_redo_theme_change( WP_REST_Request $request ) {
        $theme_id = intval( $request->get_param( 'theme_id' ) );

        $this->logger->info( 'Processing theme redo request', array(
            'theme_id' => $theme_id
        ) );

        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Theme not found'
            ), 404 );
        }

        $version_control = AI_Version_Control::get_instance();
        $result = $version_control->redo_change( 'theme', $theme_id );

        if ( is_wp_error( $result ) ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ), 400 );
        }

        // Get button states for frontend
        $button_states = $this->get_theme_version_button_states( $theme_id );

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Redo successful',
            'can_undo' => $button_states['can_undo'],
            'can_redo' => $button_states['can_redo'],
            'version' => array(
                'id' => $result->id,
                'version_number' => $result->version_number,
                'change_description' => $result->change_description,
                'created_at' => $result->created_at
            )
        ) );
    }

    /**
     * REST API: Get theme version history
     */
    public function rest_get_theme_versions( WP_REST_Request $request ) {
        $theme_id = intval( $request->get_param( 'theme_id' ) );

        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return new WP_REST_Response( array(
                'success' => false,
                'message' => 'Theme not found'
            ), 404 );
        }

        $version_control = AI_Version_Control::get_instance();

        // Create a proper REST request for the central API
        $history_request = new WP_REST_Request( 'GET', '/ai-builder/v1/versions/theme/' . $theme_id . '/history' );
        $history_request->set_param( 'entity_type', 'theme' );
        $history_request->set_param( 'entity_id', $theme_id );
        $response = $version_control->rest_get_version_history( $history_request );

        // Enhance with button states
        $button_states = $this->get_theme_version_button_states( $theme_id );

        if ( $response instanceof WP_REST_Response ) {
            $data = $response->get_data();
            $data['can_undo'] = $button_states['can_undo'];
            $data['can_redo'] = $button_states['can_redo'];
            return new WP_REST_Response( $data );
        }

        return $response;
    }

    /**
     * Get theme version button states (can undo/redo)
     */
    private function get_theme_version_button_states( $theme_id ) {
        global $wpdb;

        $version_control = AI_Version_Control::get_instance();
        $table_name = $wpdb->prefix . 'ai_version_control';

        // Get current version
        $current_version = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$table_name}
             WHERE entity_type = 'theme' AND entity_id = %d AND is_current = 1",
            $theme_id
        ) );

        if ( ! $current_version ) {
            return array( 'can_undo' => false, 'can_redo' => false );
        }

        // Can undo if version number > 1
        $can_undo = $current_version->version_number > 1;

        // Can redo if there are versions with higher numbers
        $next_version_exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name}
             WHERE entity_type = 'theme' AND entity_id = %d AND version_number > %d",
            $theme_id, $current_version->version_number
        ) ) > 0;

        return array(
            'can_undo' => $can_undo,
            'can_redo' => $next_version_exists
        );
    }

    /**
     * REST endpoint: Activate theme
     */
    public function rest_activate_theme( $request ) {
        $theme_id = $request->get_param( 'theme_id' );

        // Verify theme exists
        $theme_post = get_post( $theme_id );
        if ( ! $theme_post || $theme_post->post_type !== 'ai_theme' ) {
            return rest_ensure_response( array(
                'success' => false,
                'message' => 'Theme not found'
            ) );
        }

        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );
        $this->logger->info( 'REST activation request started', [
            'theme_id' => $theme_id,
            'theme_slug' => $theme_slug,
            'theme_title' => $theme_post->post_title
        ] );

        // Get theme manager instance
        $theme_manager = new AI_Theme_Manager();

        // Attempt to activate the theme
        $result = $theme_manager->activate_theme( $theme_id );

        if ( is_wp_error( $result ) ) {
            return rest_ensure_response( array(
                'success' => false,
                'message' => $result->get_error_message()
            ) );
        }

        // Update theme status to active using theme manager sync
        $theme_manager = new AI_Theme_Manager();
        $theme_slug = get_post_meta( $theme_id, 'theme_directory_slug', true );

        // Use the theme manager's sync method to ensure consistency
        $theme_manager->sync_wordpress_to_dreamformer( $theme_slug, null, null );

        // Set all other themes to inactive
        $other_themes = get_posts( array(
            'post_type' => 'ai_theme',
            'posts_per_page' => -1,
            'exclude' => array( $theme_id ),
            'meta_query' => array(
                array(
                    'key' => 'theme_status',
                    'value' => 'active'
                )
            )
        ) );

        foreach ( $other_themes as $other_theme ) {
            update_post_meta( $other_theme->ID, 'theme_status', 'ready' );
        }

        return rest_ensure_response( array(
            'success' => true,
            'message' => 'Theme activated successfully'
        ) );
    }

}
