<?php
/**
 * AI Site Builder - Bundled Theme Installer
 *
 * Handles installation of bundled themes on plugin activation
 * Ensures clients have immediate functionality without requiring AI generation
 *
 * @package AI_Site_Builder
 * @since 1.0.1
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Bundled_Theme_Installer {

    private $logger;

    public function __construct() {
        $this->logger = AI_Logger::get_instance( 'BundledThemeInstaller' );

        // Hook into plugin activation
        add_action( 'ai_site_builder_plugin_activated', [ $this, 'install_bundled_themes' ] );
    }

    /**
     * Install all bundled themes on plugin activation
     */
    public function install_bundled_themes() {
        $this->logger->info( 'Starting bundled theme installation' );

        $bundled_themes_dir = AI_SITE_BUILDER_PLUGIN_DIR . 'templates/themes/';

        if ( ! is_dir( $bundled_themes_dir ) ) {
            $this->logger->warning( 'No bundled themes directory found at: ' . $bundled_themes_dir );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'Dreamformer: No bundled themes directory found at: ' . $bundled_themes_dir );
        }
            return;
        }

        $theme_directories = glob( $bundled_themes_dir . '*', GLOB_ONLYDIR );

        foreach ( $theme_directories as $theme_dir ) {
            $theme_name = basename( $theme_dir );
            $this->install_single_theme( $theme_name );
        }

        $this->logger->info( 'Bundled theme installation completed' );
    }

    /**
     * Install a single bundled theme
     *
     * @param string $theme_name Theme directory name
     * @return bool Success status
     */
    private function install_single_theme( $theme_name ) {
        $source_dir = AI_SITE_BUILDER_PLUGIN_DIR . 'templates/themes/' . $theme_name;
        $dest_dir = WP_CONTENT_DIR . '/themes/' . $theme_name;

        // Check if theme already exists
        if ( file_exists( $dest_dir ) ) {
            $this->logger->info( "Theme {$theme_name} already exists, skipping installation" );
            return true;
        }

        // Copy theme files
        $copy_result = $this->recursive_copy( $source_dir, $dest_dir );

        if ( $copy_result ) {
            $this->logger->info( "Successfully installed bundled theme: {$theme_name}" );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "Dreamformer: Successfully installed bundled theme: {$theme_name} to {$dest_dir}" );
        }

            // Create WordPress post entry for the theme (optional - for plugin management)
            $this->create_theme_post_entry( $theme_name );

            return true;
        } else {
            $this->logger->error( "Failed to install bundled theme: {$theme_name}" );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "Dreamformer: Failed to install bundled theme: {$theme_name} from {$source_dir} to {$dest_dir}" );
        }
            return false;
        }
    }

    /**
     * Create a WordPress post entry for the bundled theme
     * This allows the theme to be managed through the plugin interface
     *
     * @param string $theme_name Theme name
     */
    private function create_theme_post_entry( $theme_name ) {
        // Check if post already exists
        $existing_post = get_posts( [
            'post_type' => 'ai_theme',
            'meta_key' => 'theme_directory_slug',
            'meta_value' => $theme_name,
            'posts_per_page' => 1
        ] );

        if ( ! empty( $existing_post ) ) {
            return; // Post already exists
        }

        // Create new theme post
        $post_data = [
            'post_title' => $this->get_theme_display_name( $theme_name ),
            'post_content' => 'Bundled base theme included with Dreamformer plugin.',
            'post_status' => 'publish',
            'post_type' => 'ai_theme',
            'meta_input' => [
                'theme_directory_slug' => $theme_name,
                'theme_status' => 'installed',
                'theme_version' => '1.0.0',
                'generation_prompt' => 'Bundled theme - no AI generation required',
                'is_bundled_theme' => true,
                'bundled_theme_source' => 'plugin_templates'
            ]
        ];

        $post_id = wp_insert_post( $post_data );

        if ( $post_id ) {
            $this->logger->info( "Created WordPress post entry for bundled theme: {$theme_name} (ID: {$post_id})" );
        }
    }

    /**
     * Get display name for theme from style.css
     *
     * @param string $theme_name Theme directory name
     * @return string Display name
     */
    private function get_theme_display_name( $theme_name ) {
        $style_file = WP_CONTENT_DIR . '/themes/' . $theme_name . '/style.css';

        if ( file_exists( $style_file ) ) {
            $style_content = file_get_contents( $style_file );

            // Extract theme name from style.css header
            if ( preg_match( '/Theme Name:\s*(.+)/i', $style_content, $matches ) ) {
                return trim( $matches[1] );
            }
        }

        // Fallback to directory name
        return ucwords( str_replace( [ '-', '_' ], ' ', $theme_name ) );
    }

    /**
     * Recursively copy directory
     *
     * @param string $source Source directory
     * @param string $dest Destination directory
     * @return bool Success status
     */
    private function recursive_copy( $source, $dest ) {
        // Create destination directory
        if ( ! wp_mkdir_p( $dest ) ) {
            return false;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $source, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ( $iterator as $item ) {
            $dest_path = $dest . DIRECTORY_SEPARATOR . $iterator->getSubPathName();

            if ( $item->isDir() ) {
                wp_mkdir_p( $dest_path );
            } else {
                copy( $item, $dest_path );
            }
        }

        return true;
    }

    /**
     * Get list of available bundled themes
     *
     * @return array List of bundled theme names
     */
    public function get_bundled_themes() {
        $bundled_themes_dir = AI_SITE_BUILDER_PLUGIN_DIR . 'templates/themes/';

        if ( ! is_dir( $bundled_themes_dir ) ) {
            return [];
        }

        $theme_directories = glob( $bundled_themes_dir . '*', GLOB_ONLYDIR );
        $themes = [];

        foreach ( $theme_directories as $theme_dir ) {
            $theme_name = basename( $theme_dir );
            $themes[] = [
                'name' => $theme_name,
                'display_name' => $this->get_theme_display_name( $theme_name ),
                'is_installed' => file_exists( WP_CONTENT_DIR . '/themes/' . $theme_name )
            ];
        }

        return $themes;
    }

    /**
     * Manually install bundled themes (for use in admin interface)
     */
    public function manual_install() {
        if ( ! current_user_can( 'install_themes' ) ) {
            return new WP_Error( 'insufficient_permissions', 'You do not have permission to install themes.' );
        }

        $this->install_bundled_themes();
        return true;
    }
}