<?php

/**
 * Plugin Name: Dreamformer (Premium)
 * Plugin URI: https://dreamformer.ai
 * Description: Complete AI Website Builder - Build themes, pages, and plugins with AI-powered tools for WordPress
 * Version: 1.13.7
 * Update URI: https://api.freemius.com
 * Author: Dreamformer
 * Author URI: https://dreamformer.ai
 * License: Proprietary
 * Text Domain: dreamformer
 */
// Prevent direct access
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Define plugin constants
define( 'AI_SITE_BUILDER_VERSION', '1.13.7' );
define( 'AI_SITE_BUILDER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AI_SITE_BUILDER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'AI_SITE_BUILDER_PLUGIN_FILE', __FILE__ );
// Bundled theme slugs
define( 'AI_SITE_BUILDER_BASE_THEME', 'ai-base-theme' );
define( 'AI_SITE_BUILDER_SHOP_THEME', 'dreamformer-woocommerce' );
// Load Composer autoloader for bundled dependencies
if ( file_exists( AI_SITE_BUILDER_PLUGIN_DIR . 'vendor/autoload.php' ) ) {
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'vendor/autoload.php';
}
// Initialize bundled Action Scheduler (background task processing)
if ( file_exists( AI_SITE_BUILDER_PLUGIN_DIR . 'vendor/woocommerce/action-scheduler/action-scheduler.php' ) ) {
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'vendor/woocommerce/action-scheduler/action-scheduler.php';
}
/**
 * Enqueue custom CSS and JS for Freemius pricing page
 * Since Freemius ignores the custom CSS setting, we inject it directly
 */
function dreamformer_inject_pricing_css() {
    // Only on Freemius pricing page
    if ( !isset( $_GET['page'] ) || $_GET['page'] !== 'dreamformer-pricing' ) {
        return;
    }
    // Enqueue our custom pricing CSS
    wp_enqueue_style(
        'dreamformer-pricing-override',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/freemius-pricing-custom.css',
        array(),
        AI_SITE_BUILDER_VERSION,
        'all'
    );
    // Enqueue particle animation JS
    wp_enqueue_script(
        'dreamformer-pricing-particles',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/freemius-pricing-particles.js',
        array(),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // DISABLED: CSS-only solution is cleaner
    // wp_enqueue_script(
    // 	'dreamformer-hide-pricing-cards',
    // 	AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/hide-pricing-cards.js',
    // 	array(),
    // 	AI_SITE_BUILDER_VERSION . '-' . time(),
    // 	false
    // );
}

add_action( 'admin_enqueue_scripts', 'dreamformer_inject_pricing_css', 999999 );
// DISABLED: CSS-only solution works better
// function dreamformer_inline_nuke_pricing_cards() {
// 	// Disabled
// }
// add_action( 'admin_footer', 'dreamformer_inline_nuke_pricing_cards', 999999 );
/**
 * Freemius Integration for Payment Processing
 *
 * SDK Location: includes/freemius/
 * Download from: https://github.com/Freemius/wordpress-sdk
 */
if ( !function_exists( 'dreamformer_fs' ) ) {
    /**
     * Create a helper function for easy SDK access
     */
    function dreamformer_fs() {
        global $dreamformer_fs;
        if ( !isset( $dreamformer_fs ) ) {
            // Check if Freemius SDK is installed
            $freemius_path = AI_SITE_BUILDER_PLUGIN_DIR . 'includes/freemius/start.php';
            if ( !file_exists( $freemius_path ) ) {
                // Freemius SDK not installed yet - skip initialization
                // Download from: https://github.com/Freemius/wordpress-sdk
                // Extract to: includes/freemius/
                return null;
            }
            // Include Freemius SDK
            require_once $freemius_path;
            $dreamformer_fs = fs_dynamic_init( array(
                'id'             => '21657',
                'slug'           => 'dreamformer',
                'type'           => 'plugin',
                'public_key'     => 'pk_b8243f8f85f8d0147e71e3d7a916d',
                'is_premium'     => true,
                'has_addons'     => false,
                'has_paid_plans' => true,
                'menu'           => array(
                    'slug'    => 'dreamformer',
                    'account' => false,
                    'pricing' => true,
                    'contact' => false,
                    'support' => false,
                ),
                'is_live'        => true,
            ) );
        }
        return $dreamformer_fs;
    }

    // Init Freemius
    dreamformer_fs();
    // Hide Freemius Account submenu - we use our own custom Account page
    // Official Freemius pattern: https://freemius.com/help/documentation/wordpress-sdk/filters-actions-hooks/
    // Filter runs on every request (more reliable than cached config setting)
    if ( dreamformer_fs() ) {
        dreamformer_fs()->add_filter(
            'is_submenu_visible',
            function ( $is_visible, $menu_id ) {
                if ( 'account' === $menu_id ) {
                    return false;
                }
                return $is_visible;
            },
            10,
            2
        );
    }
    // Signal that SDK was initiated
    do_action( 'dreamformer_fs_loaded' );
}
// Load utility classes first (required by all other classes)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-logger.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-config-manager.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-file-manager.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/utils/class-ai-json-handler.php';
// Load theme system classes
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/class-ai-theme-post-types.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/class-ai-theme-manager.php';
// Load services (Phase 2)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-ai-vercel-client.php';
// AI service removed - now using Vercel APIs only
// Load Freemius webhook handler (handles subscription events)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-freemius-handler.php';
// Load Credit Manager (handles all credit operations)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
// Load ultra-fast simple file editor (replaces complex tool system)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-ai-simple-file-editor.php';
// Load CSS AST parser for manual editor functionality
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/css-parser/class-ai-css-ast-parser.php';
// Load bundled theme installer
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/class-ai-bundled-theme-installer.php';
// Load Phase 2: AI Theme Editor (depends on version control and tools)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/class-ai-theme-editor.php';
// Load Page Generation System (separate from theme system)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/pages/class-ai-page-post-types.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/pages/class-ai-page-generator.php';
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/pages/class-ai-dreamformer-page-editor.php';
// Load WooCommerce Integration System
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/woocommerce/class-ai-woocommerce-styler.php';
// Load Version Control System (unified for themes and pages)
require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/version-control/class-ai-version-control.php';
/**
 * Initialize the plugin - Theme-based Architecture
 */
function ai_site_builder_init() {
    // SECURITY: Proper error handling in REST API requests
    // Log errors but don't display them (prevents JSON corruption while maintaining debuggability)
    if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
        // Disable display, enable logging
        @ini_set( 'display_errors', 0 );
        @ini_set( 'log_errors', 1 );
        // Set custom error handler that logs to WordPress debug log
        set_error_handler( function (
            $errno,
            $errstr,
            $errfile,
            $errline
        ) {
            // Only log if WP_DEBUG is enabled
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( sprintf(
                    '[Dreamformer REST API] %s in %s on line %d',
                    $errstr,
                    $errfile,
                    $errline
                ) );
            }
            // Prevent default PHP error handler (which would output to page)
            return true;
        }, E_ALL );
    }
    // Initialize theme system
    new AI_Theme_Post_Types();
    new AI_Theme_Manager();
    // Initialize AI Theme Editor (REST API)
    new AI_Theme_Editor();
    // Initialize page generation system (separate from theme system)
    new AI_Page_Post_Types();
    new AI_Page_Generator();
    // Initialize Dreamformer page editor integration
    new AI_Dreamformer_Page_Editor();
    // Initialize WooCommerce integration (product styling)
    new AI_WooCommerce_Styler();
    // Initialize unified version control system
    AI_Version_Control::get_instance();
    // Initialize bundled theme installer
    new AI_Bundled_Theme_Installer();
    // Add CSS/JS cache busting filters for theme edits
    add_filter(
        'style_loader_src',
        'ai_site_builder_bust_css_cache',
        10,
        2
    );
    add_filter(
        'script_loader_src',
        'ai_site_builder_bust_js_cache',
        10,
        2
    );
    // Add ETag/Last-Modified headers for AI pages (smart caching)
    add_action( 'template_redirect', 'ai_site_builder_smart_cache_headers' );
}

add_action( 'plugins_loaded', 'ai_site_builder_init' );
/**
 * Universal theme compatibility filter
 *
 * Ensures AI-generated pages display correctly with ANY WordPress theme,
 * not just Dreamformer themes. This filter intercepts the_content() and
 * returns the protected AI content for AI-generated pages.
 */
function ai_site_builder_universal_content_filter(  $content  ) {
    // Only apply on singular pages on the frontend
    if ( !is_singular( 'page' ) || is_admin() ) {
        return $content;
    }
    // Check if this is an AI-generated page
    $is_ai_page = get_post_meta( get_the_ID(), 'ai_generated_page', true );
    if ( !$is_ai_page ) {
        return $content;
    }
    // Get the protected AI content
    $protected_content = get_post_meta( get_the_ID(), '_ai_protected_content', true );
    if ( $protected_content ) {
        // Remove wpautop for AI content (it breaks the HTML structure)
        remove_filter( 'the_content', 'wpautop' );
        return do_shortcode( $protected_content );
    }
    return $content;
}

add_filter( 'the_content', 'ai_site_builder_universal_content_filter', 5 );
/**
 * Trigger cache bust for theme or page assets
 *
 * Call this after AI edits files to ensure browsers fetch fresh CSS/JS.
 * Sets a transient that cache bust filters will pick up.
 *
 * @param string $type 'theme' or 'page'
 * @param int|string $identifier Theme slug or page ID
 */
function ai_site_builder_trigger_cache_bust(  $type = 'theme', $identifier = null  ) {
    $timestamp = time();
    if ( $type === 'theme' ) {
        $theme_slug = ( $identifier ?: get_stylesheet() );
        set_transient( 'ai_theme_cache_bust_' . $theme_slug, $timestamp, HOUR_IN_SECONDS );
    } elseif ( $type === 'page' && $identifier ) {
        set_transient( 'ai_page_cache_bust_' . intval( $identifier ), $timestamp, HOUR_IN_SECONDS );
    }
    // Clear opcache if available (some hosts cache filemtime results)
    if ( function_exists( 'opcache_invalidate' ) ) {
        // Invalidate specific files if we know them
        $upload_dir = wp_upload_dir();
        if ( $type === 'page' && $identifier ) {
            $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . intval( $identifier );
            @opcache_invalidate( $page_dir . '/styles.css', true );
            @opcache_invalidate( $page_dir . '/app.js', true );
        }
    }
}

/**
 * Smart cache headers for AI pages using ETag/Last-Modified
 *
 * Allows browsers to cache pages but validates before serving.
 * Returns 304 Not Modified if content unchanged (fast).
 * Returns fresh content if files have been modified.
 */
function ai_site_builder_smart_cache_headers() {
    // Only for singular pages
    if ( !is_singular( 'page' ) ) {
        return;
    }
    $page_id = get_the_ID();
    if ( !$page_id ) {
        return;
    }
    // Only for AI-generated pages
    if ( !get_post_meta( $page_id, 'ai_generated_page', true ) ) {
        return;
    }
    // Calculate last modified time from page files
    $upload_dir = wp_upload_dir();
    $page_dir = $upload_dir['basedir'] . '/ai-pages/page-' . $page_id;
    $last_modified = 0;
    // Check page files for most recent modification
    $files_to_check = array($page_dir . '/index.html', $page_dir . '/styles.css', $page_dir . '/app.js');
    foreach ( $files_to_check as $file ) {
        if ( file_exists( $file ) ) {
            $mtime = filemtime( $file );
            if ( $mtime > $last_modified ) {
                $last_modified = $mtime;
            }
        }
    }
    // Also check cache bust transient (in case filemtime is cached)
    $cache_bust = get_transient( 'ai_page_cache_bust_' . $page_id );
    if ( $cache_bust && $cache_bust > $last_modified ) {
        $last_modified = $cache_bust;
    }
    // Fallback to post modified date if no files found
    if ( $last_modified === 0 ) {
        $last_modified = strtotime( get_post_modified_time( 'c', true, $page_id ) );
    }
    // Generate ETag from last modified time + page ID
    $etag = '"' . md5( $page_id . '-' . $last_modified ) . '"';
    // Check if browser sent If-Modified-Since or If-None-Match
    $if_modified_since = ( isset( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) ? strtotime( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) : false );
    $if_none_match = ( isset( $_SERVER['HTTP_IF_NONE_MATCH'] ) ? trim( $_SERVER['HTTP_IF_NONE_MATCH'] ) : false );
    // Determine if we can send 304 Not Modified
    $not_modified = false;
    if ( $if_none_match && $if_none_match === $etag ) {
        $not_modified = true;
    } elseif ( $if_modified_since && $if_modified_since >= $last_modified ) {
        $not_modified = true;
    }
    // Send cache headers
    header( 'Last-Modified: ' . gmdate( 'D, d M Y H:i:s', $last_modified ) . ' GMT' );
    header( 'ETag: ' . $etag );
    header( 'Cache-Control: no-cache, must-revalidate' );
    // Always validate, but can use 304
    // If content hasn't changed, send 304 and exit
    if ( $not_modified ) {
        status_header( 304 );
        exit;
    }
}

/**
 * Bust CSS cache by replacing version parameter with timestamp
 *
 * After AI edits theme files, we store a cache-bust timestamp in a transient.
 * This filter intercepts WordPress CSS enqueue and replaces the version parameter
 * with the timestamp, forcing browsers to fetch fresh CSS without hard refresh.
 *
 * @param string $src    Stylesheet URL
 * @param string $handle Stylesheet handle
 * @return string Modified URL with cache-bust version
 */
function ai_site_builder_bust_css_cache(  $src, $handle  ) {
    // Get current theme slug
    $theme_slug = get_stylesheet();
    // Check if we have a cache-bust timestamp for this theme
    $cache_bust = get_transient( 'ai_theme_cache_bust_' . $theme_slug );
    if ( $cache_bust ) {
        // Remove existing version parameter
        $src = remove_query_arg( 'ver', $src );
        // Add cache-bust timestamp as version
        $src = add_query_arg( 'ver', $cache_bust, $src );
    }
    return $src;
}

/**
 * Bust JS cache by replacing version parameter with timestamp
 *
 * Same as CSS cache bust but for JavaScript files.
 *
 * @param string $src    Script URL
 * @param string $handle Script handle
 * @return string Modified URL with cache-bust version
 */
function ai_site_builder_bust_js_cache(  $src, $handle  ) {
    // Get current theme slug
    $theme_slug = get_stylesheet();
    // Check if we have a cache-bust timestamp for this theme
    $cache_bust = get_transient( 'ai_theme_cache_bust_' . $theme_slug );
    if ( $cache_bust ) {
        // Only bust theme-related scripts
        $theme_url = get_template_directory_uri();
        if ( strpos( $src, $theme_url ) !== false ) {
            $src = remove_query_arg( 'ver', $src );
            $src = add_query_arg( 'ver', $cache_bust, $src );
        }
    }
    // Also check for page-specific cache bust
    if ( is_singular( 'page' ) && strpos( $handle, 'ai-page-' ) === 0 ) {
        $page_id = get_the_ID();
        $page_cache_bust = get_transient( 'ai_page_cache_bust_' . $page_id );
        if ( $page_cache_bust ) {
            $src = remove_query_arg( 'ver', $src );
            $src = add_query_arg( 'ver', $page_cache_bust, $src );
        }
    }
    return $src;
}

/**
 * Plugin activation hook
 */
function ai_site_builder_activate() {
    // Install bundled themes directly on activation
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/class-ai-bundled-theme-installer.php';
    $installer = new AI_Bundled_Theme_Installer();
    $installer->install_bundled_themes();
    // Create secure uploads directories for page drafts/backups
    $upload_dir = wp_upload_dir();
    $ai_pages_dir = $upload_dir['basedir'] . '/ai-pages';
    $ai_theme_versions_dir = $upload_dir['basedir'] . '/ai-theme-versions';
    // Create directories if they don't exist
    $secure_dirs = array($ai_pages_dir, $ai_theme_versions_dir);
    foreach ( $secure_dirs as $dir ) {
        if ( !is_dir( $dir ) ) {
            wp_mkdir_p( $dir );
        }
        // Add .htaccess to prevent direct access
        $htaccess_path = $dir . '/.htaccess';
        if ( !file_exists( $htaccess_path ) ) {
            $htaccess_content = "# Dreamformer - Deny Direct Access\n";
            $htaccess_content .= "# Prevents public access to draft pages, backups, and version snapshots\n";
            $htaccess_content .= "<Files *>\n";
            $htaccess_content .= "  Order Allow,Deny\n";
            $htaccess_content .= "  Deny from all\n";
            $htaccess_content .= "</Files>\n";
            file_put_contents( $htaccess_path, $htaccess_content );
        }
    }
    // Create database tables
    ai_site_builder_create_tables();
    // Set activation flag and default options
    add_option( 'ai_site_builder_activated', true );
    add_option( 'ai_site_builder_image_model', 'dall-e-3' );
    update_option( 'ai_site_builder_db_version', AI_SITE_BUILDER_VERSION );
    // Schedule cleanup for transients using Action Scheduler
    as_schedule_recurring_action(
        time(),
        DAY_IN_SECONDS,
        'ai_site_builder_cleanup_transients',
        [],
        'ai-site-builder'
    );
    // Schedule daily cleanup for expired rollover credits
    as_schedule_recurring_action(
        time(),
        DAY_IN_SECONDS,
        'dreamformer_cleanup_expired_rollover',
        [],
        'ai-site-builder'
    );
}

register_activation_hook( __FILE__, 'ai_site_builder_activate' );
/**
 * Check for plugin updates and run migrations
 * This ensures database tables are created/updated on plugin updates
 */
function ai_site_builder_check_upgrade() {
    $stored_version = get_option( 'ai_site_builder_db_version', '0' );
    if ( version_compare( $stored_version, AI_SITE_BUILDER_VERSION, '<' ) ) {
        // Run table creation (dbDelta handles updates safely)
        ai_site_builder_create_tables();
        // Regenerate .htaccess for ai-pages directory (fixes LiteSpeed caching issues)
        ai_site_builder_regenerate_htaccess();
        // v1.13.5: Fix available_credits computed column formula
        if ( version_compare( $stored_version, '1.13.5', '<' ) ) {
            ai_site_builder_migrate_1_13_5();
        }
        // Update stored version
        update_option( 'ai_site_builder_db_version', AI_SITE_BUILDER_VERSION );
        // Log upgrade for debugging
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( sprintf( 'Dreamformer: Upgraded from %s to %s - database tables and .htaccess updated', $stored_version, AI_SITE_BUILDER_VERSION ) );
        }
    }
}

/**
 * Migration for v1.13.5
 * Fixes available_credits computed column to use correct formula
 */
function ai_site_builder_migrate_1_13_5() {
    global $wpdb;
    $table = $wpdb->prefix . 'dreamformer_user_credits';
    // Check if table exists
    if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
        return;
    }
    // Drop and recreate the computed column with correct formula
    // Old formula: (total_credits - used_credits)
    // New formula: (monthly_credits - used_credits) + rollover_credits
    $wpdb->query( "ALTER TABLE {$table} DROP COLUMN IF EXISTS available_credits" );
    $wpdb->query( "ALTER TABLE {$table} ADD COLUMN available_credits INT GENERATED ALWAYS AS ((monthly_credits - used_credits) + rollover_credits) STORED" );
    // Also clean up duplicate license_key indexes if they exist
    $indexes = $wpdb->get_results( "SHOW INDEX FROM {$table} WHERE Key_name LIKE 'license_key_%'" );
    foreach ( $indexes as $index ) {
        if ( $index->Key_name !== 'license_key' ) {
            $wpdb->query( "ALTER TABLE {$table} DROP INDEX `{$index->Key_name}`" );
        }
    }
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'Dreamformer: v1.13.5 migration - Fixed available_credits computed column' );
    }
}

/**
 * Regenerate .htaccess for ai-pages directory
 * Called on plugin upgrade to apply new cache control headers
 */
function ai_site_builder_regenerate_htaccess() {
    $upload_dir = wp_upload_dir();
    $ai_pages_dir = $upload_dir['basedir'] . '/ai-pages';
    $htaccess_path = $ai_pages_dir . '/.htaccess';
    // Only regenerate if directory exists
    if ( !is_dir( $ai_pages_dir ) ) {
        return;
    }
    $htaccess_content = <<<'HTACCESS'
# Dreamformer AI Pages - Security Rules
# =======================================================
# CSS/JS are now injected inline into HTML (CDN-proof)
# These files are only accessed by PHP, not browsers
# =======================================================

# BLOCK: All source files (CSS, JS, HTML, JSON)
# These contain source code and should not be directly accessible
<FilesMatch "\.(css|js|html?|json)$">
  Order Allow,Deny
  Deny from all
</FilesMatch>

# ALLOW: Images (may be referenced in pages)
<FilesMatch "\.(jpg|jpeg|png|gif|svg|webp|ico)$">
  Order Allow,Deny
  Allow from all
</FilesMatch>

# Prevent directory listing
Options -Indexes

# Block PHP execution (security)
<FilesMatch "\.php$">
  Order Allow,Deny
  Deny from all
</FilesMatch>
HTACCESS;
    file_put_contents( $htaccess_path, $htaccess_content );
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'Dreamformer: Regenerated ai-pages .htaccess with LiteSpeed cache disable' );
    }
}

add_action( 'plugins_loaded', 'ai_site_builder_check_upgrade' );
/**
 * Create database tables for the plugin
 */
function ai_site_builder_create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    // Templates table - Theme-focused design
    $templates_table = $wpdb->prefix . 'ai_templates';
    $templates_sql = "CREATE TABLE {$templates_table} (\n\t\tid int(11) NOT NULL AUTO_INCREMENT,\n\t\tname varchar(255) NOT NULL,\n\t\ttype enum('theme') DEFAULT 'theme',\n\t\tcategory varchar(100) DEFAULT 'general',\n\t\tdescription text,\n\t\tcontent longtext NOT NULL,\n\t\tpreview_image varchar(255),\n\t\tuser_id int(11),\n\t\tis_public tinyint(1) DEFAULT 0,\n\t\tusage_count int(11) DEFAULT 0,\n\t\ttheme_version varchar(20) DEFAULT '1.0.0',\n\t\twordpress_version varchar(20) DEFAULT NULL,\n\t\tphp_version varchar(20) DEFAULT NULL,\n\t\tcreated_at timestamp DEFAULT CURRENT_TIMESTAMP,\n\t\tupdated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\n\t\tPRIMARY KEY (id),\n\t\tKEY idx_type (type),\n\t\tKEY idx_category (category),\n\t\tKEY idx_user_id (user_id),\n\t\tKEY idx_public (is_public),\n\t\tKEY idx_user_type (user_id, type)\n\t) {$charset_collate};";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $templates_sql );
    // Version control table - Unified system for themes and pages
    $version_control_table = $wpdb->prefix . 'ai_version_control';
    $version_control_sql = "CREATE TABLE {$version_control_table} (\n\t\tid bigint(20) unsigned NOT NULL AUTO_INCREMENT,\n\t\tentity_type varchar(20) NOT NULL,\n\t\tentity_id bigint(20) unsigned NOT NULL,\n\t\tversion_number int(11) NOT NULL DEFAULT 1,\n\t\tcontent_before longtext,\n\t\tcontent_after longtext NOT NULL,\n\t\tchange_description text,\n\t\tuser_prompt text,\n\t\tai_model varchar(50) DEFAULT '" . AI_Config_Manager::AI_MODEL . "',\n\t\tuser_id bigint(20) unsigned NOT NULL,\n\t\tcreated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,\n\t\tis_current tinyint(1) NOT NULL DEFAULT 1,\n\t\tchange_metadata json,\n\t\tparent_version_id bigint(20) unsigned NULL,\n\t\tlinked_version_id bigint(20) unsigned NULL,\n\t\tPRIMARY KEY (id),\n\t\tUNIQUE KEY unique_version (entity_type, entity_id, version_number),\n\t\tKEY entity_lookup (entity_type, entity_id),\n\t\tKEY version_lookup (entity_type, entity_id, version_number),\n\t\tKEY current_version (entity_type, entity_id, is_current),\n\t\tKEY user_versions (user_id, created_at),\n\t\tKEY parent_lookup (parent_version_id),\n\t\tKEY linked_lookup (linked_version_id)\n\t) {$charset_collate};";
    dbDelta( $version_control_sql );
    // Edit attempts table - Memory system for AI learning
    $edit_attempts_table = $wpdb->prefix . 'ai_edit_attempts';
    $edit_attempts_sql = "CREATE TABLE {$edit_attempts_table} (\n\t\tid bigint(20) unsigned NOT NULL AUTO_INCREMENT,\n\t\tpage_id bigint(20) unsigned NOT NULL,\n\t\tsession_id varchar(64) NOT NULL,\n\t\tuser_prompt text,\n\t\tapplied_diff longtext,\n\t\tuser_feedback text,\n\t\ttimestamp datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,\n\t\tPRIMARY KEY (id),\n\t\tKEY page_session_lookup (page_id, session_id, timestamp)\n\t) {$charset_collate};";
    dbDelta( $edit_attempts_sql );
    // Snippets table - Reusable component library
    $snippets_table = $wpdb->prefix . 'dreamformer_snippets';
    $snippets_sql = "CREATE TABLE {$snippets_table} (\n\t\tid bigint(20) unsigned NOT NULL AUTO_INCREMENT,\n\t\tname varchar(255) NOT NULL,\n\t\tdescription text,\n\t\tselector varchar(255),\n\t\thtml_code longtext NOT NULL,\n\t\tcss_code longtext,\n\t\tjs_code longtext,\n\t\tpreview_image varchar(255),\n\t\tcategory varchar(100) DEFAULT 'general',\n\t\tsaved_from_page_id bigint(20) unsigned,\n\t\tsaved_from_theme_id bigint(20) unsigned,\n\t\tusage_count int(11) DEFAULT 0,\n\t\tuser_id bigint(20) unsigned NOT NULL,\n\t\tcreated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,\n\t\tupdated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\n\t\tPRIMARY KEY (id),\n\t\tKEY user_snippets (user_id, created_at),\n\t\tKEY category_lookup (category),\n\t\tKEY usage_lookup (usage_count)\n\t) {$charset_collate};";
    dbDelta( $snippets_sql );
    // Freemius payment system tables
    // NOTE: GENERATED columns require MySQL 5.7.6+ or MariaDB 10.2.1+
    // Table 1: User credits - Track subscription and credit balance
    // Simple monthly credit allocation (Final Pricing Model)
    $user_credits_table = $wpdb->prefix . 'dreamformer_user_credits';
    $user_credits_sql = "CREATE TABLE {$user_credits_table} (\n\t\tid BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,\n\t\tuser_id BIGINT UNSIGNED NOT NULL UNIQUE,\n\t\tlicense_key VARCHAR(255) NOT NULL UNIQUE,\n\t\tplan_id VARCHAR(50) NOT NULL,\n\t\tmonthly_credits INT NOT NULL DEFAULT 0,\n\t\tused_credits INT NOT NULL DEFAULT 0,\n\t\trollover_credits INT NOT NULL DEFAULT 0,\n\t\trollover_expiry DATE NULL,\n\t\tavailable_credits INT GENERATED ALWAYS AS (\n\t\t\t(monthly_credits - used_credits) + rollover_credits\n\t\t) STORED,\n\t\tsubscription_status ENUM('active', 'cancelled', 'expired', 'trial', 'free') DEFAULT 'active',\n\t\tbilling_cycle_start DATE NOT NULL,\n\t\tbilling_cycle_end DATE NOT NULL,\n\t\tcreated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n\t\tupdated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\n\t\tINDEX idx_user_id (user_id),\n\t\tINDEX idx_license_key (license_key),\n\t\tINDEX idx_subscription_status (subscription_status),\n\t\tINDEX idx_rollover_expiry (rollover_expiry)\n\t) ENGINE=InnoDB {$charset_collate};";
    dbDelta( $user_credits_sql );
    // Table 2: Usage log - Track every API call with token counts
    $usage_log_table = $wpdb->prefix . 'dreamformer_usage_log';
    $usage_log_sql = "CREATE TABLE {$usage_log_table} (\n\t\tid BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,\n\t\tuser_id BIGINT UNSIGNED NOT NULL,\n\t\tendpoint VARCHAR(100) NOT NULL,\n\t\trequest_type VARCHAR(50) NOT NULL,\n\t\tinput_tokens INT DEFAULT 0,\n\t\toutput_tokens INT DEFAULT 0,\n\t\tthinking_tokens INT DEFAULT 0,\n\t\ttotal_tokens INT GENERATED ALWAYS AS (input_tokens + output_tokens + thinking_tokens) STORED,\n\t\tcredits_deducted INT NOT NULL,\n\t\testimated_cost DECIMAL(10,4),\n\t\tsuccess BOOLEAN DEFAULT TRUE,\n\t\terror_message TEXT,\n\t\tcreated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n\t\tINDEX idx_user_date (user_id, created_at),\n\t\tINDEX idx_endpoint (endpoint),\n\t\tINDEX idx_user_credits (user_id, credits_deducted)\n\t) ENGINE=InnoDB {$charset_collate};";
    dbDelta( $usage_log_sql );
    // Table 3: Rate limits - Prevent abuse/infinite loops
    $rate_limits_table = $wpdb->prefix . 'dreamformer_rate_limits';
    $rate_limits_sql = "CREATE TABLE {$rate_limits_table} (\n\t\tid BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,\n\t\tuser_id BIGINT UNSIGNED NOT NULL,\n\t\tlimit_type ENUM('daily', 'monthly') NOT NULL,\n\t\tlimit_threshold INT NOT NULL,\n\t\trequests_made INT DEFAULT 0,\n\t\treset_at TIMESTAMP NOT NULL,\n\t\tcreated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n\t\tupdated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\n\t\tUNIQUE KEY unique_user_limit (user_id, limit_type),\n\t\tINDEX idx_reset (reset_at)\n\t) ENGINE=InnoDB {$charset_collate};";
    dbDelta( $rate_limits_sql );
    // Set database version for future migrations
    update_option( 'ai_version_control_db_version', '1.1' );
    update_option( 'dreamformer_credits_db_version', '1.0' );
}

/**
 * Cleanup progress transients (scheduled daily)
 */
function ai_site_builder_cleanup_progress_transients() {
    global $wpdb;
    // Delete expired AI progress transients
    $wpdb->query( "DELETE FROM {$wpdb->options} \n\t\t WHERE option_name LIKE '_transient_ai_progress_%'\n\t\t OR option_name LIKE '_transient_timeout_ai_progress_%'" );
}

add_action( 'ai_site_builder_cleanup_transients', 'ai_site_builder_cleanup_progress_transients' );
/**
 * Cleanup expired rollover credits (scheduled daily)
 *
 * Sets rollover_credits to 0 for any user whose rollover has expired.
 * This ensures "Credits roll over for 1 month" is enforced properly.
 */
function dreamformer_cleanup_expired_rollover() {
    global $wpdb;
    $table = $wpdb->prefix . 'dreamformer_user_credits';
    $result = $wpdb->query( "UPDATE {$table}\n\t\t SET rollover_credits = 0, rollover_expiry = NULL\n\t\t WHERE rollover_expiry IS NOT NULL\n\t\t AND rollover_expiry < CURDATE()" );
    if ( $result > 0 ) {
        error_log( sprintf( 'Dreamformer: Cleared expired rollover credits for %d users', $result ) );
    }
}

add_action( 'dreamformer_cleanup_expired_rollover', 'dreamformer_cleanup_expired_rollover' );
// Ensure cron is scheduled for existing installations
add_action( 'init', function () {
    if ( function_exists( 'as_has_scheduled_action' ) && !as_has_scheduled_action( 'dreamformer_cleanup_expired_rollover' ) ) {
        as_schedule_recurring_action(
            time(),
            DAY_IN_SECONDS,
            'dreamformer_cleanup_expired_rollover',
            [],
            'ai-site-builder'
        );
    }
} );
/**
 * SECURITY: Rate-limited permission check for public REST API endpoints
 *
 * Implements IP-based rate limiting to prevent abuse of license validation
 * and credit deduction endpoints.
 *
 * @param WP_REST_Request $request The REST request object
 * @return bool|WP_Error True if allowed, WP_Error if rate limit exceeded
 */
function dreamformer_rest_permission_check(  $request  ) {
    // Get client IP address
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    // Sanitize IP for transient key
    $ip_hash = md5( $ip );
    $rate_limit_key = 'dreamformer_rate_limit_' . $ip_hash;
    // Get current attempt count
    $attempts = get_transient( $rate_limit_key );
    // Rate limit: 60 requests per minute per IP
    $max_attempts = 60;
    if ( $attempts && $attempts >= $max_attempts ) {
        return new WP_Error('rate_limit_exceeded', 'Too many requests. Please try again later.', array(
            'status' => 429,
        ));
    }
    // Increment attempt count (expires after 1 minute)
    set_transient( $rate_limit_key, ( $attempts ? $attempts + 1 : 1 ), MINUTE_IN_SECONDS );
    // Allow the request
    return true;
}

/**
 * Register REST API routes for license validation and credit operations
 * Called by Vercel API for per-user authentication and credit tracking
 *
 * @since 4.0.0
 */
add_action( 'rest_api_init', function () {
    // License validation endpoint
    register_rest_route( 'dreamformer/v1', '/license/validate', array(
        'methods'             => 'POST',
        'callback'            => 'dreamformer_validate_license_rest',
        'permission_callback' => 'dreamformer_rest_permission_check',
    ) );
    // Credit deduction endpoint
    register_rest_route( 'dreamformer/v1', '/credits/deduct', array(
        'methods'             => 'POST',
        'callback'            => 'dreamformer_deduct_credits_rest',
        'permission_callback' => 'dreamformer_rest_permission_check',
    ) );
    // Credit sync endpoint (called by Vercel after KV deduction)
    register_rest_route( 'dreamformer/v1', '/credits/sync', array(
        'methods'             => 'POST',
        'callback'            => 'dreamformer_sync_credits_rest',
        'permission_callback' => 'dreamformer_rest_permission_check',
    ) );
} );
/**
 * REST API: Validate license key
 *
 * Called by Vercel API to verify user license and get credit balance
 *
 * @param WP_REST_Request $request Request object
 * @return WP_REST_Response|WP_Error
 */
function dreamformer_validate_license_rest(  WP_REST_Request $request  ) {
    global $wpdb;
    $table = $wpdb->prefix . 'dreamformer_user_credits';
    // Get license key from request
    $license_key = $request->get_param( 'license_key' );
    if ( empty( $license_key ) ) {
        return new WP_Error('missing_license_key', 'License key is required', array(
            'status' => 400,
        ));
    }
    // Query user credits by license key
    $credits = $wpdb->get_row( $wpdb->prepare( "SELECT user_id, plan_id, available_credits, subscription_status, billing_cycle_end\n\t\t\t FROM {$table}\n\t\t\t WHERE license_key = %s", $license_key ), ARRAY_A );
    if ( !$credits ) {
        return new WP_Error('invalid_license', 'License key not found', array(
            'status' => 401,
        ));
    }
    // Check if subscription is active, free, or trial
    $valid_statuses = array('active', 'free', 'trial');
    if ( !in_array( $credits['subscription_status'], $valid_statuses, true ) ) {
        return new WP_Error('inactive_subscription', sprintf( 'Subscription status "%s" is not active', $credits['subscription_status'] ), array(
            'status' => 401,
        ));
    }
    // Check if billing cycle hasn't expired
    if ( strtotime( $credits['billing_cycle_end'] ) < time() ) {
        return new WP_Error('expired_subscription', 'Billing cycle has expired', array(
            'status' => 401,
        ));
    }
    // Return validation success with user data
    return new WP_REST_Response(array(
        'valid'             => true,
        'user_id'           => (int) $credits['user_id'],
        'available_credits' => (int) $credits['available_credits'],
        'plan_id'           => $credits['plan_id'],
    ), 200);
}

/**
 * REST API: Deduct credits after API call
 *
 * Called by Vercel API after successful generation to deduct credits
 *
 * @param WP_REST_Request $request Request object
 * @return WP_REST_Response|WP_Error
 */
function dreamformer_deduct_credits_rest(  WP_REST_Request $request  ) {
    // Get parameters
    $license_key = $request->get_param( 'license_key' );
    $credits_to_deduct = $request->get_param( 'credits' );
    $endpoint = $request->get_param( 'endpoint' );
    $usage_data = $request->get_param( 'usage_data' );
    if ( empty( $license_key ) || empty( $credits_to_deduct ) ) {
        return new WP_Error('missing_parameters', 'License key and credits are required', array(
            'status' => 400,
        ));
    }
    // Get user_id from license key
    global $wpdb;
    $credits_table = $wpdb->prefix . 'dreamformer_user_credits';
    $user_row = $wpdb->get_row( $wpdb->prepare( "SELECT user_id FROM {$credits_table} WHERE license_key = %s", $license_key ), ARRAY_A );
    if ( !$user_row ) {
        return new WP_Error('invalid_license', 'License key not found', array(
            'status' => 401,
        ));
    }
    $user_id = (int) $user_row['user_id'];
    // Use Credit Manager to deduct credits
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
    $credit_manager = new Dreamformer_Credit_Manager();
    $result = $credit_manager->deduct_credits(
        $user_id,
        (int) $credits_to_deduct,
        $endpoint,
        ( is_array( $usage_data ) ? $usage_data : array() )
    );
    if ( is_wp_error( $result ) ) {
        return new WP_Error($result->get_error_code(), $result->get_error_message(), array(
            'status' => 402,
        ));
    }
    // Get updated credits
    $updated_credits = $credit_manager->get_user_credits( $user_id );
    return new WP_REST_Response(array(
        'success'           => true,
        'credits_deducted'  => (int) $credits_to_deduct,
        'available_credits' => (int) $updated_credits['available_credits'],
    ), 200);
}

/**
 * REST API: Sync credits from Vercel KV to WordPress DB
 *
 * Called by Vercel API after KV credit deduction for real-time display sync
 * This ensures WordPress admin shows accurate credit usage immediately
 *
 * @param WP_REST_Request $request Request object
 * @return WP_REST_Response|WP_Error
 */
function dreamformer_sync_credits_rest(  WP_REST_Request $request  ) {
    global $wpdb;
    $table = $wpdb->prefix . 'dreamformer_user_credits';
    // Get parameters from Vercel
    $license_key = $request->get_param( 'license_key' );
    $used_credits = $request->get_param( 'used_credits' );
    $remaining_credits = $request->get_param( 'remaining_credits' );
    $monthly_limit = $request->get_param( 'monthly_limit' );
    // Validate required parameters
    if ( empty( $license_key ) || !is_numeric( $used_credits ) ) {
        return new WP_Error('missing_parameters', 'license_key and used_credits are required', array(
            'status' => 400,
        ));
    }
    // Find user record by license key
    $current = $wpdb->get_row( $wpdb->prepare( "SELECT user_id, monthly_credits, rollover_credits FROM {$table} WHERE license_key = %s", $license_key ), ARRAY_A );
    if ( !$current ) {
        return new WP_Error('license_not_found', 'No user found with this license key', array(
            'status' => 404,
        ));
    }
    // Calculate rollover credits
    // KV returns: remaining = (monthly_limit - used_credits) + rollover
    // So: rollover = remaining - (monthly_limit - used_credits)
    $monthly_available = $monthly_limit - $used_credits;
    $rollover_credits = max( 0, $remaining_credits - $monthly_available );
    // Update WordPress DB to match KV state
    $update_result = $wpdb->update(
        $table,
        array(
            'used_credits'     => (int) $used_credits,
            'rollover_credits' => (int) $rollover_credits,
            'monthly_credits'  => (int) $monthly_limit,
        ),
        array(
            'license_key' => $license_key,
        ),
        array('%d', '%d', '%d'),
        array('%s')
    );
    if ( false === $update_result ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'Dreamformer: Credit sync failed - ' . $wpdb->last_error );
        }
        return new WP_Error('update_failed', 'Failed to update credits: ' . $wpdb->last_error, array(
            'status' => 500,
        ));
    }
    // Clear transient cache to force refresh
    delete_transient( 'dreamformer_credits_checked_' . $current['user_id'] );
    // Log success
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( sprintf(
            'Dreamformer: Credits synced from KV - License: %s, Used: %d, Rollover: %d',
            substr( $license_key, 0, 8 ) . '...',
            $used_credits,
            $rollover_credits
        ) );
    }
    return new WP_REST_Response(array(
        'success'          => true,
        'used_credits'     => (int) $used_credits,
        'rollover_credits' => (int) $rollover_credits,
        'monthly_limit'    => (int) $monthly_limit,
    ), 200);
}

/**
 * Plugin deactivation hook
 */
function ai_site_builder_deactivate() {
    // Clean up Action Scheduler jobs
    if ( function_exists( 'as_unschedule_all_actions' ) ) {
        as_unschedule_all_actions( 'ai_site_builder_cleanup_transients', [], 'ai-site-builder' );
        as_unschedule_all_actions( 'ai_process_page_generation', [], 'ai-site-builder' );
    }
    // Clean up temporary transients
    ai_site_builder_cleanup_progress_transients();
}

register_deactivation_hook( __FILE__, 'ai_site_builder_deactivate' );
/**
 * Add admin menu
 */
function ai_site_builder_admin_menu() {
    add_menu_page(
        __( 'Dreamformer', 'dreamformer' ),
        __( 'Dreamformer', 'dreamformer' ),
        'manage_options',
        'dreamformer',
        'ai_site_builder_admin_page',
        'dashicons-admin-site-alt3',
        30
    );
    // Account submenu - shows credits and plan info
    add_submenu_page(
        'dreamformer',
        __( 'Account', 'dreamformer' ),
        __( 'Account', 'dreamformer' ),
        'manage_options',
        'dreamformer-account',
        'dreamformer_account_page_render'
    );
}

add_action( 'admin_menu', 'ai_site_builder_admin_menu' );
/**
 * Render minimal Account page
 */
function dreamformer_account_page_render() {
    // Get credit info from local database
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
    $credit_manager = new Dreamformer_Credit_Manager();
    $credits_info = $credit_manager->get_user_credits();
    // Defaults
    $available = 0;
    $monthly = 0;
    $used = 0;
    $rollover = 0;
    $plan_id = 'free';
    $status = 'free';
    $billing_end = null;
    if ( $credits_info ) {
        $available = (int) $credits_info['available_credits'];
        $monthly = (int) $credits_info['monthly_credits'];
        $used = (int) $credits_info['used_credits'];
        $rollover = (int) ($credits_info['rollover_credits'] ?? 0);
        $plan_id = $credits_info['plan_id'];
        $status = $credits_info['subscription_status'];
        $billing_end = $credits_info['billing_cycle_end'] ?? null;
    }
    $remaining = max( 0, $monthly - $used );
    $percentage = ( $monthly > 0 ? max( 0, min( 100, round( $remaining / $monthly * 100 ) ) ) : 0 );
    $plan_label = ucfirst( str_replace( array('-', '_'), ' ', $plan_id ) );
    $pricing_url = admin_url( 'admin.php?page=dreamformer-pricing' );
    // Bar color based on percentage
    $bar_color = ( $percentage < 20 ? '#dc2626' : (( $percentage < 50 ? '#f59e0b' : '#10b981' )) );
    ?>
	<div class="wrap dreamformer-account-wrap">
		<h1><?php 
    esc_html_e( 'Account', 'dreamformer' );
    ?></h1>

		<div class="dreamformer-account-card">
			<!-- Credits Section -->
			<div class="account-section">
				<h2><?php 
    esc_html_e( 'Credits', 'dreamformer' );
    ?></h2>
				<div class="credits-big"><?php 
    echo esc_html( number_format( $available ) );
    ?></div>
				<div class="credits-label"><?php 
    esc_html_e( 'available', 'dreamformer' );
    ?></div>

				<div class="credits-bar">
					<div class="credits-bar-fill" style="width: <?php 
    echo esc_attr( max( 2, $percentage ) );
    ?>%; background: <?php 
    echo esc_attr( $bar_color );
    ?>;"></div>
				</div>

				<div class="credits-details">
					<span><?php 
    echo esc_html( number_format( $remaining ) );
    ?> / <?php 
    echo esc_html( number_format( $monthly ) );
    ?> <?php 
    esc_html_e( 'monthly', 'dreamformer' );
    ?></span>
					<?php 
    if ( $rollover > 0 ) {
        ?>
						<span class="rollover">+<?php 
        echo esc_html( number_format( $rollover ) );
        ?> <?php 
        esc_html_e( 'rollover', 'dreamformer' );
        ?></span>
					<?php 
    }
    ?>
				</div>

				<?php 
    if ( $billing_end ) {
        ?>
					<div class="billing-reset">
						<?php 
        printf( esc_html__( 'Resets %s', 'dreamformer' ), esc_html( date_i18n( 'M j', strtotime( $billing_end ) ) ) );
        ?>
					</div>
				<?php 
    }
    ?>
			</div>

			<!-- Plan Section -->
			<div class="account-section">
				<h2><?php 
    esc_html_e( 'Plan', 'dreamformer' );
    ?></h2>
				<div class="plan-name"><?php 
    echo esc_html( $plan_label );
    ?></div>
				<div class="plan-status status-<?php 
    echo esc_attr( $status );
    ?>"><?php 
    echo esc_html( ucfirst( $status ) );
    ?></div>

				<div class="plan-actions">
					<?php 
    $needs_upgrade = $plan_id === 'free' || $status === 'cancelled' || $status === 'expired' || $percentage < 30;
    if ( $needs_upgrade ) {
        ?>
						<a href="<?php 
        echo esc_url( $pricing_url );
        ?>" class="button button-primary"><?php 
        esc_html_e( 'Upgrade', 'dreamformer' );
        ?></a>
					<?php 
    } else {
        ?>
						<a href="<?php 
        echo esc_url( $pricing_url );
        ?>" class="button"><?php 
        esc_html_e( 'Change Plan', 'dreamformer' );
        ?></a>
					<?php 
    }
    ?>
				</div>
			</div>
		</div>

		<p class="account-help">
			<?php 
    esc_html_e( 'Need help?', 'dreamformer' );
    ?>
			<a href="https://dreamformer.ai/support" target="_blank"><?php 
    esc_html_e( 'Contact Support', 'dreamformer' );
    ?></a>
		</p>
	</div>
	<?php 
}

/**
 * Enqueue admin styles for Dreamformer
 */
function dreamformer_admin_styles(  $hook  ) {
    // Only load on Dreamformer pages
    if ( strpos( $hook, 'dreamformer' ) !== false ) {
        // Always enqueue main admin styles
        wp_enqueue_style(
            'dreamformer-admin',
            AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/dreamformer-admin.css',
            array(),
            AI_SITE_BUILDER_VERSION
        );
        // Enqueue Lucide Icons (required for all Dreamformer pages)
        wp_enqueue_script(
            'lucide-icons',
            AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/vendor/lucide.min.js',
            array(),
            AI_SITE_BUILDER_VERSION,
            true
        );
        // Initialize Lucide icons after page load
        wp_add_inline_script( 'lucide-icons', 'document.addEventListener("DOMContentLoaded", function() { if (typeof lucide !== "undefined") { lucide.createIcons(); } });' );
        // Enqueue Code Editor (for page editing only)
        wp_enqueue_style(
            'dreamformer-code-editor',
            AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/dreamformer-code-editor.css',
            array(),
            AI_SITE_BUILDER_VERSION
        );
        wp_enqueue_script(
            'dreamformer-code-editor',
            AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/dreamformer-code-editor.js',
            array('jquery'),
            AI_SITE_BUILDER_VERSION,
            true
        );
        // Check if setup is complete
        $setup_complete = get_option( 'dreamformer_setup_complete', false );
        if ( !$setup_complete ) {
            // Setup wizard assets
            wp_enqueue_style(
                'dreamformer-setup-wizard',
                AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/setup-wizard.css',
                array('dreamformer-admin'),
                AI_SITE_BUILDER_VERSION
            );
            wp_enqueue_script(
                'dreamformer-setup-wizard',
                AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/setup-wizard.js',
                array('jquery', 'lucide-icons'),
                AI_SITE_BUILDER_VERSION,
                true
            );
        }
    }
}

add_action( 'admin_enqueue_scripts', 'dreamformer_admin_styles' );
/**
 * Display credit balance admin notice on Dreamformer pages
 */
function dreamformer_credit_balance_notice() {
    $screen = get_current_screen();
    // Only show on Dreamformer admin pages
    if ( !$screen || strpos( $screen->id, 'dreamformer' ) === false ) {
        return;
    }
    $user_id = get_current_user_id();
    if ( !$user_id ) {
        return;
    }
    // Get user credit info
    require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
    $credit_manager = new Dreamformer_Credit_Manager();
    $credits_info = $credit_manager->get_user_credits( $user_id );
    if ( !$credits_info ) {
        // User has no subscription - show upgrade notice
        ?>
		<div class="notice notice-info" style="border-left-color: #2563eb; padding: 12px 15px;">
			<p style="margin: 0;">
				<strong>💳 No active subscription</strong> -
				<a href="<?php 
        echo esc_url( admin_url( 'admin.php?page=dreamformer-pricing' ) );
        ?>">Upgrade to a plan</a>
				to start using AI page generation.
			</p>
		</div>
		<?php 
        return;
    }
    $available_credits = (int) $credits_info['available_credits'];
    $monthly_credits = (int) $credits_info['monthly_credits'];
    $used_credits = (int) $credits_info['used_credits'];
    $rollover_credits = (int) ($credits_info['rollover_credits'] ?? 0);
    $rollover_expiry = $credits_info['rollover_expiry'] ?? null;
    $plan_id = $credits_info['plan_id'];
    // Format plan name
    $plan_display = ucfirst( str_replace( '-', ' ', $plan_id ) );
    // "pro-200" → "Pro 200"
    $monthly_available = $monthly_credits - $used_credits;
    $percentage = ( $monthly_credits > 0 ? $monthly_available / $monthly_credits * 100 : 0 );
    // Determine notice type based on credit level
    $notice_type = 'notice-success';
    $color = '#16a34a';
    // Green
    if ( $percentage < 20 ) {
        $notice_type = 'notice-error';
        $color = '#dc2626';
        // Red
    } elseif ( $percentage < 50 ) {
        $notice_type = 'notice-warning';
        $color = '#ea580c';
        // Orange
    }
    ?>
	<div class="notice <?php 
    echo esc_attr( $notice_type );
    ?>" style="border-left-color: <?php 
    echo esc_attr( $color );
    ?>; padding: 12px 15px;">
		<p style="margin: 0;">
			<strong>💎 Credits:</strong> <?php 
    echo esc_html( number_format( $available_credits ) );
    ?> total
			<?php 
    if ( $rollover_credits > 0 ) {
        ?>
				<span style="font-size: 11px; color: #666;">(<?php 
        echo esc_html( number_format( $monthly_available ) );
        ?> monthly + <span style="color: #16a34a; font-weight: 600;"><?php 
        echo esc_html( number_format( $rollover_credits ) );
        ?> rollover</span><?php 
        if ( $rollover_expiry ) {
            ?> · expires <?php 
            echo esc_html( date( 'M j', strtotime( $rollover_expiry ) ) );
        }
        ?>)</span>
			<?php 
    } else {
        ?>
				<span style="font-size: 11px; color: #666;">(<?php 
        echo esc_html( number_format( $monthly_available ) );
        ?> / <?php 
        echo esc_html( number_format( $monthly_credits ) );
        ?>)</span>
			<?php 
    }
    ?>
			&nbsp;|&nbsp;
			<strong>Plan:</strong> <?php 
    echo esc_html( $plan_display );
    ?>
			<?php 
    if ( $percentage < 50 ) {
        ?>
				&nbsp;|&nbsp;
				<a href="<?php 
        echo esc_url( admin_url( 'admin.php?page=dreamformer-pricing' ) );
        ?>">Upgrade Plan</a>
			<?php 
    }
    ?>
		</p>
	</div>
	<?php 
}

add_action( 'admin_notices', 'dreamformer_credit_balance_notice' );
/**
 * Redirect "Add New AI Theme" to main AI Site Builder page
 */
function ai_site_builder_redirect_new_post() {
    global $pagenow;
    // Redirect "Add New" to Dreamformer
    if ( $pagenow === 'post-new.php' && isset( $_GET['post_type'] ) && $_GET['post_type'] === 'ai_theme' ) {
        wp_redirect( admin_url( 'admin.php?page=dreamformer' ) );
        exit;
    }
    // Redirect "Edit" to AI Site Builder with theme ID
    if ( $pagenow === 'post.php' && isset( $_GET['post'] ) && isset( $_GET['action'] ) && $_GET['action'] === 'edit' ) {
        $post_id = intval( $_GET['post'] );
        $post = get_post( $post_id );
        if ( $post && $post->post_type === 'ai_theme' ) {
            wp_redirect( admin_url( 'admin.php?page=dreamformer&theme_id=' . $post_id ) );
            exit;
        }
    }
}

add_action( 'admin_init', 'ai_site_builder_redirect_new_post' );
/**
 * Customize admin bar new content menu
 */
function ai_site_builder_admin_bar_menu(  $wp_admin_bar  ) {
    // Remove the default "AI Theme" item and replace with our custom one
    $wp_admin_bar->remove_node( 'new-ai_theme' );
    // Add our custom "Create with AI" item
    $wp_admin_bar->add_node( array(
        'parent' => 'new-content',
        'id'     => 'new-ai-theme',
        'title'  => __( 'AI Generated Theme', 'ai-site-builder' ),
        'href'   => admin_url( 'admin.php?page=dreamformer' ),
    ) );
}

add_action( 'admin_bar_menu', 'ai_site_builder_admin_bar_menu', 81 );
/**
 * Handle setup wizard reset
 */
function ai_site_builder_handle_setup_reset() {
    if ( isset( $_GET['page'] ) && $_GET['page'] === 'dreamformer' && isset( $_GET['reset_setup'] ) && $_GET['reset_setup'] === '1' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'reset_dreamformer_setup' ) ) {
        delete_option( 'dreamformer_setup_complete' );
        delete_option( 'dreamformer_quick_start_dismissed' );
        wp_redirect( admin_url( 'admin.php?page=dreamformer' ) );
        exit;
    }
}

add_action( 'admin_init', 'ai_site_builder_handle_setup_reset' );
/**
 * Render admin page - Phase 4: Clean Theme-Focused Interface
 */
function ai_site_builder_admin_page() {
    // Check if setup is complete
    $setup_complete = get_option( 'dreamformer_setup_complete', false );
    if ( !$setup_complete ) {
        // Show setup wizard
        include AI_SITE_BUILDER_PLUGIN_DIR . 'admin/partials/setup-wizard.php';
        return;
    }
    // Main dashboard
    ?>
	<div class="wrap">
		<h1><?php 
    _e( 'Dreamformer Studio', 'ai-site-builder' );
    ?></h1>

		<?php 
    // Show Quick Start box on first dashboard visit
    $quick_start_dismissed = get_option( 'dreamformer_quick_start_dismissed', false );
    if ( !$quick_start_dismissed ) {
        include AI_SITE_BUILDER_PLUGIN_DIR . 'admin/partials/quick-start-box.php';
    }
    ?>

		<!-- Page Library Tab -->
		<div id="pages-tab" class="tab-content active">
			<!-- Page Library Explanation -->
			<div class="pages-explanation">
				<h3>Build your dream</h3>
				<p>Create beautiful pages and websites by chatting with AI</p>
			</div>

			<!-- Page Generation Section - Hybrid Chat + Chips UI -->
			<div class="page-generation-section">
				<div class="page-prompt-container">
					<!-- Textarea wrapper with attach button inside -->
					<div class="textarea-wrapper">
						<textarea
							id="page-prompt-input"
							class="page-prompt-input"
							placeholder="What kind of page do you want to create? Describe the content, style, purpose, and any specific features you need..."
							rows="4"></textarea>

						<!-- Attach button inside textarea (bottom-left corner) -->
						<button type="button" class="btn-attach-file" data-action="attach-file">
							<i data-lucide="paperclip" class="icon-inline icon-sm"></i>
							<span class="attach-label">Attach</span>
						</button>
					</div>

					<!-- Hidden file input -->
					<input type="file"
						   id="page-attachment-input"
						   accept=".pdf,.png,.jpg,.jpeg,.md,.txt,.doc,.docx,.csv,.json,.xml,.html"
						   style="display: none;">

					<!-- Attachment preview (shows after file selected, below textarea) -->
					<div id="attachment-preview" style="display: none;">
						<div class="attachment-preview-item">
							<img class="attachment-thumbnail" style="display: none; max-width: 60px; max-height: 60px; border-radius: 6px;">
							<i data-lucide="file-text" class="attachment-icon icon-inline" style="display: none; font-size: 24px;"></i>
							<div class="attachment-info">
								<span class="attachment-filename"></span>
								<span class="attachment-size"></span>
							</div>
							<button type="button" class="btn-remove-attachment" data-action="remove-attachment" title="Remove attachment">
								<i data-lucide="x" class="icon-inline icon-sm"></i>
							</button>
						</div>
					</div>

					<!-- Actions row: Style + Generate (below textarea) -->
					<div class="prompt-actions">
						<div class="style-reference-field">
							<label for="reference-page-select">
								<i data-lucide="palette" class="icon-inline icon-sm"></i>
								Style:
							</label>
							<select id="reference-page-select" name="reference_page_id">
								<option value="">No reference - fresh style</option>
							</select>
						</div>
						<button class="btn-primary btn-generate-page" data-action="generate-from-prompt">
							<i data-lucide="sparkles" class="icon-inline"></i>
							Generate Page
						</button>
					</div>
				</div>

				<div class="template-suggestions">
					<label class="suggestions-label">Quick start:</label>
					<div class="template-chips">
						<button class="template-chip" data-type="landing" data-action="select-template">
							<i data-lucide="rocket" class="chip-icon"></i>
							Landing
						</button>
						<button class="template-chip" data-type="about" data-action="select-template">
							<i data-lucide="users" class="chip-icon"></i>
							About
						</button>
						<button class="template-chip" data-type="contact" data-action="select-template">
							<i data-lucide="mail" class="chip-icon"></i>
							Contact
						</button>
						<button class="template-chip" data-type="blog" data-action="select-template">
							<i data-lucide="pen-line" class="chip-icon"></i>
							Blog
						</button>
						<button class="template-chip" data-type="services" data-action="select-template">
							<i data-lucide="briefcase" class="chip-icon"></i>
							Services
						</button>
						<button class="template-chip" data-type="app" data-action="select-template">
							<i data-lucide="zap" class="chip-icon"></i>
							App
						</button>
					</div>
				</div>
			</div>

			<!-- Pages Section Header with Actions -->
			<div class="pages-tab-header">
				<h3 class="pages-section-title">Your Pages</h3>
				<div style="display: flex; gap: 8px;">
					<button class="btn-secondary" data-action="import-page" style="white-space: nowrap;">
						<span class="dashicons dashicons-upload"></span>
						Import
					</button>
					<button class="btn-secondary refresh-pages-btn" data-action="refresh-pages" style="white-space: nowrap;">
						<span class="dashicons dashicons-update"></span>
						Refresh
					</button>
				</div>
			</div>

			<!-- Pages Grid Container -->
			<div class="pages-section">
				<div class="pages-grid" id="ai-pages-grid">
					<p class="loading-text">Loading your pages...</p>
				</div>
			</div>

			<!-- Generation Messages -->
			<div class="generation-message" id="pages-message"></div>
		</div>

	<!-- Theme Library Interface (rendered by theme-library.js) -->
	<div id="theme-library-interface"></div>

	</div>

	<style>
	.dreamformer-tabs {
		margin: 20px 0;
		border-bottom: 1px solid #ccd0d4;
	}
	.dreamformer-tabs .tab-link {
		display: inline-block;
		padding: 10px 20px;
		text-decoration: none;
		color: #50575e;
		border-bottom: 2px solid transparent;
		margin-right: 10px;
	}
	.dreamformer-tabs .tab-link.active {
		color: #0073aa;
		border-bottom-color: #0073aa;
	}
	.tab-content {
		margin-top: 20px;
	}
	.page-library-header {
		margin-bottom: 30px;
	}
	.page-grid {
		display: grid;
		grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
		gap: 20px;
	}
	.page-card {
		border: 1px solid #ddd;
		border-radius: 8px;
		overflow: hidden;
		background: white;
		box-shadow: 0 2px 4px rgba(0,0,0,0.1);
		transition: transform 0.2s;
	}
	.page-card:hover {
		transform: translateY(-2px);
		box-shadow: 0 4px 8px rgba(0,0,0,0.15);
	}
	.page-preview {
		height: 200px;
		overflow: hidden;
		background: #f9f9f9;
	}
	.page-preview iframe {
		width: 100%;
		height: 100%;
		border: none;
		transform: scale(0.5);
		transform-origin: 0 0;
		width: 200%;
		height: 200%;
	}
	.page-info {
		padding: 15px;
	}
	.page-info h3 {
		margin: 0 0 10px 0;
		font-size: 16px;
	}
	.page-meta {
		margin-bottom: 15px;
		font-size: 12px;
		color: #666;
	}
	.page-meta .page-status {
		padding: 2px 6px;
		border-radius: 3px;
		background: #eee;
		margin-right: 10px;
	}
	.page-meta .status-publish {
		background: #d4edda;
		color: #155724;
	}
	.page-meta .status-draft {
		background: #fff3cd;
		color: #856404;
	}
	.page-actions {
		display: flex;
		gap: 10px;
	}
	.page-actions .button {
		flex: 1;
		text-align: center;
	}
	.no-pages {
		text-align: center;
		color: #666;
		font-style: italic;
		margin: 50px 0;
	}
	</style>

	<script>
		// Pass WordPress data to JavaScript
		window.aiSiteBuilder = {
			restUrl: '<?php 
    echo esc_js( rest_url() );
    ?>',
			nonce: '<?php 
    echo esc_js( wp_create_nonce( 'wp_rest' ) );
    ?>',
			restNonce: '<?php 
    echo esc_js( wp_create_nonce( 'wp_rest' ) );
    ?>',
			adminUrl: '<?php 
    echo esc_js( admin_url() );
    ?>',
			homeUrl: '<?php 
    echo esc_js( home_url() );
    ?>',
			pluginUrl: '<?php 
    echo esc_js( AI_SITE_BUILDER_PLUGIN_URL );
    ?>'
		};

		// Page edit button functionality
		jQuery(document).ready(function($) {
			$('.edit-page-btn').on('click', function() {
				const pageId = $(this).data('page-id');
				const pageTitle = $(this).data('page-title');

				// Redirect to unified editor with page mode
				window.location.href = window.aiSiteBuilder.adminUrl + 'admin.php?page=dreamformer&edit=page&id=' + pageId;
			});
		});

	</script>
	<?php 
}

/**
 * Enqueue Phase 4 admin assets on AI Site Builder pages only
 */
function ai_site_builder_admin_enqueue_scripts(  $hook_suffix  ) {
    // Only enqueue on Dreamformer admin pages
    $ai_pages = array('toplevel_page_dreamformer', 'dreamformer_page_dreamformer-settings');
    // More flexible check - also check if we're on the Dreamformer pages
    $is_ai_page = in_array( $hook_suffix, $ai_pages ) || isset( $_GET['page'] ) && ($_GET['page'] === 'dreamformer' || $_GET['page'] === 'dreamformer-settings');
    if ( !$is_ai_page ) {
        return;
    }
    // Enqueue Theme Library CSS
    wp_enqueue_style(
        'ai-site-builder-theme-library',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/theme-library.css',
        array(),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue Theme Library JavaScript
    wp_enqueue_script(
        'ai-site-builder-theme-library',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/theme-library.js',
        array('jquery'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Get user credits for editor display
    $available_credits = 0;
    $user_id = get_current_user_id();
    if ( $user_id ) {
        require_once AI_SITE_BUILDER_PLUGIN_DIR . 'includes/services/class-dreamformer-credit-manager.php';
        $credit_manager = new Dreamformer_Credit_Manager();
        $credits_info = $credit_manager->get_user_credits( $user_id );
        if ( $credits_info ) {
            $available_credits = (int) $credits_info['available_credits'];
        }
    }
    // Localize script data for JavaScript files
    wp_localize_script( 'ai-site-builder-theme-library', 'aiSiteBuilder', array(
        'restUrl'          => rest_url(),
        'nonce'            => wp_create_nonce( 'wp_rest' ),
        'restNonce'        => wp_create_nonce( 'wp_rest' ),
        'adminUrl'         => admin_url(),
        'homeUrl'          => home_url(),
        'pluginUrl'        => AI_SITE_BUILDER_PLUGIN_URL,
        'userHasPremium'   => function_exists( 'dreamformer_fs' ) && dreamformer_fs() && dreamformer_fs()->is_premium(),
        'pricingUrl'       => admin_url( 'admin.php?page=dreamformer-pricing' ),
        'availableCredits' => $available_credits,
    ) );
    // Enqueue Pages Tab JavaScript
    wp_enqueue_script(
        'dreamformer-pages',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/dreamformer-pages.js',
        array('jquery'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Enqueue Page Generation Streaming JavaScript
    wp_enqueue_script(
        'page-generation-streaming',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/page-generation-streaming.js',
        array('jquery'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Enqueue Page Generation Streaming CSS
    wp_enqueue_style(
        'page-generation-streaming-css',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/page-generation-streaming.css',
        array(),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue Theme Editor CSS
    wp_enqueue_style(
        'ai-site-builder-theme-editor',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/theme-editor.css',
        array('ai-site-builder-theme-library'),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue Lucide Icons CSS
    wp_enqueue_style(
        'ai-site-builder-lucide-icons',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/lucide-icons.css',
        array('ai-site-builder-theme-editor'),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue Emotional Interactions CSS
    wp_enqueue_style(
        'ai-site-builder-emotional-interactions',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/emotional-interactions.css',
        array('ai-site-builder-lucide-icons'),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue Celebration Animations CSS
    wp_enqueue_style(
        'ai-site-builder-celebration-animations',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/css/celebration-animations.css',
        array('ai-site-builder-emotional-interactions'),
        AI_SITE_BUILDER_VERSION
    );
    // Enqueue WordPress Media scripts for image upload
    wp_enqueue_media();
    // Enqueue Manual Editor Property Controls (must load before theme editor)
    wp_enqueue_script(
        'ai-site-builder-property-controls',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/manual-editor/property-controls.js',
        array('jquery'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Enqueue Theme Editor JavaScript
    wp_enqueue_script(
        'ai-site-builder-theme-editor',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/theme-editor.js',
        array('jquery', 'ai-site-builder-theme-library', 'ai-site-builder-property-controls'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Enqueue Celebration System JavaScript
    wp_enqueue_script(
        'ai-site-builder-celebrations',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/celebrations.js',
        array('jquery', 'ai-site-builder-theme-editor'),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Enqueue Lucide Icons (hosted locally without source maps for clean console)
    wp_enqueue_script(
        'lucide-icons',
        AI_SITE_BUILDER_PLUGIN_URL . 'admin/js/vendor/lucide.min.js',
        array(),
        AI_SITE_BUILDER_VERSION,
        true
    );
    // Initialize Lucide icons after page load
    wp_add_inline_script( 'lucide-icons', 'document.addEventListener("DOMContentLoaded", function() { if (typeof lucide !== "undefined") { lucide.createIcons(); } });' );
}

add_action( 'admin_enqueue_scripts', 'ai_site_builder_admin_enqueue_scripts' );