/**
 * AI Site Builder - Theme Library
 * Phase 1: Theme Management Dashboard
 *
 * NOTE: This file is NOT CURRENTLY IN USE (as of Dec 2025)
 * - Loaded as dependency but #themes-tab element doesn't exist in HTML
 * - Only pages tab is active, theme editing is not exposed to users
 * - Safe to remove during future cleanup if theme editing is not planned
 * - NOTE: admin/css/theme-library.css IS used (contains shared page styles)
 */

(function($) {
    'use strict';

class AIThemeLibrary {
    constructor() {
        this.apiBase = aiSiteBuilder.restUrl + 'ai-builder/v1/themes/';
        this.themes = [];
        this.currentView = 'library'; // 'library' or 'generator'
        this.abortController = null; // Track fetch requests
        this.isInitialized = false; // Track initialization state
        this.initializationInProgress = false; // Prevent race conditions
        this.init();
    }

    init() {
        this.bindEvents();
        this.checkInitialState(); // Check if should initialize now
    }

    // Check if themes tab is visible or should be shown
    checkInitialState() {
        const urlParams = new URLSearchParams(window.location.search);
        const isPageEditMode = urlParams.get('edit') === 'page' && urlParams.get('id');
        const isThemesTabVisible = this.isThemesTabActive();

        if (!isPageEditMode && isThemesTabVisible) {
            this.initializeThemeLibrary();
        }
    }


    // Check if themes tab is currently active
    isThemesTabActive() {
        const themesTab = $('#themes-tab');
        const result = themesTab.length > 0 &&
               themesTab.hasClass('active') &&
               themesTab.is(':visible');

        return result;
    }
    
    bindEvents() {
        // Theme library actions
        $(document).on('click', '#back-to-library-btn', () => this.showThemeLibrary());
        $(document).on('click', '.theme-preview-btn', (e) => this.previewTheme($(e.target).data('theme-id')));
        $(document).on('click', '.theme-edit-btn', (e) => this.editTheme($(e.target).data('theme-id')));
        $(document).on('click', '.theme-activate-btn', (e) => this.activateTheme($(e.target).data('theme-id')));
        $(document).on('click', '.theme-delete-btn', (e) => this.deleteTheme($(e.target).data('theme-id')));
        $(document).on('click', '.theme-duplicate-btn', (e) => this.duplicateTheme($(e.target).data('theme-id')));
        $(document).on('click', '.theme-screenshot-btn', (e) => this.uploadScreenshot($(e.target).data('theme-id')));
        $(document).on('change', '.screenshot-upload-input', (e) => this.handleScreenshotUpload(e));
        
        // Search and filter
        $(document).on('input', '#theme-search', () => this.filterThemes());
        $(document).on('change', '#theme-filter-type', () => this.filterThemes());
    }
    
    async initializeThemeLibrary() {
        // Prevent multiple simultaneous initializations
        if (this.isInitialized || this.initializationInProgress) {
            return;
        }

        this.initializationInProgress = true;

        try {
            // Validate required dependencies
            if (typeof aiSiteBuilder === 'undefined' || !aiSiteBuilder.restUrl || !aiSiteBuilder.nonce) {
                throw new Error('Required WordPress REST API configuration missing');
            }

            // Ensure container exists
            const container = $('#theme-library-interface');
            if (container.length === 0) {
                throw new Error('Theme library container not found in DOM');
            }


            await this.loadThemes();

            this.renderLibraryInterface();

            // Mark as successfully initialized
            this.isInitialized = true;
            this.currentView = 'library';

        } catch (error) {
            console.error('🎨 Theme library initialization failed:', error);

            // Still render interface even if themes fail to load
            this.renderLibraryInterface();
            this.showError('Failed to initialize theme library: ' + error.message);

            // Reset initialization state so it can be retried
            this.isInitialized = false;
        } finally {
            this.initializationInProgress = false;
        }
    }

    // Legacy method for backward compatibility
    async showThemeLibrary() {
        return this.initializeThemeLibrary();
    }
    
    
    async loadThemes() {
        try {
            // Cancel any existing request
            if (this.abortController) {
                this.abortController.abort();
            }

            // Create new abort controller
            this.abortController = new AbortController();
            
            const response = await fetch(this.apiBase, {
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin',
                signal: this.abortController.signal
            });


            const result = await response.json();
            
            if (response.ok && result.success) {
                this.themes = result.themes || [];
            } else {
                console.error('🎨 API error:', result);
                this.showError('Failed to load themes: ' + (result.message || 'Unknown error'));
            }

        } catch (error) {
            // Ignore abort errors
            if (error.name === 'AbortError') {
                return;
            }
            console.error('🎨 Theme loading error:', error);
            this.showError('Failed to load themes: ' + error.message);
            throw error; // Re-throw to trigger catch in initializeThemeLibrary
        }
    }
    
    renderLibraryInterface() {
        const libraryHtml = `
            <div class="ai-theme-library">
                <div class="library-header">
                    <div class="library-title-section">
                        <h2>Dreamformer</h2>
                        <p class="description">Complete websites built with AI - themes, nalysis of Can , and functionality combined.</p>
                    </div>
                </div>

                <div class="themes-explanation">
                    <h3>Site Framework & Infrastructure</h3>
                    <p>Themes control your website's structural foundation - navigation, header, footer, plugin integrations, and technical hooks that enable site functionality. Edit themes to modify the underlying infrastructure and site-wide elements that support all pages.</p>
                </div>

                ${this.renderThemeFilters()}
                <div id="themes-container">
                    ${this.renderThemeGrid()}
                </div>
            </div>
        `;
        
        $('#theme-library-interface').html(libraryHtml);
    }
    
    renderThemeFilters() {
        const businessTypes = [...new Set(this.themes.map(t => t.business_type).filter(Boolean))];
        
        return `
            <div class="library-filters">
                <div class="filter-group">
                    <input type="text" id="theme-search" placeholder="Search themes..." class="regular-text">
                </div>
                <div class="filter-group">
                    <select id="theme-filter-type">
                        <option value="">All Types</option>
                        ${businessTypes.map(type => 
                            `<option value="${type}">${this.formatBusinessType(type)}</option>`
                        ).join('')}
                    </select>
                </div>
                <div class="filter-stats">
                    ${this.themes.length} theme${this.themes.length !== 1 ? 's' : ''}
                </div>
            </div>
        `;
    }
    
    renderThemeGrid() {
        if (this.themes.length === 0) {
            return this.renderEmptyState();
        }
        
        // For initial render, show all themes (filters will be applied after DOM is ready)
        const themesToShow = this.themes;
        
        return `
            <div class="themes-grid">
                ${themesToShow.map(theme => this.renderThemeCard(theme)).join('')}
            </div>
        `;
    }
    
    renderEmptyState() {
        return `
            <div class="themes-empty-state">
                <div class="empty-icon">
                    <span class="dashicons dashicons-admin-appearance"></span>
                </div>
                <h3>No themes found</h3>
                <p>No themes are currently available in your library.</p>
            </div>
        `;
    }
    
    renderThemeCard(theme) {
        const statusClass = this.getStatusClass(theme.status);
        const statusText = this.getStatusText(theme.status);
        const isActive = theme.is_active;
        
        return `
            <div class="theme-card ${theme.status !== 'ready' ? statusClass : ''}">
                <div class="theme-preview-wrapper">
                    ${this.renderThemePreview(theme)}
                    <div class="theme-overlay">
                        ${theme.preview_available ? 
                            `<button class="button theme-preview-btn" data-theme-id="${theme.id}">
                                <span class="dashicons dashicons-visibility"></span> Preview
                            </button>` : 
                            '<span class="preview-unavailable">Preview not ready</span>'
                        }
                    </div>
                </div>
                
                <div class="theme-info">
                    <div class="theme-header">
                        <h3 class="theme-title">${theme.business_name || `Theme ${theme.id}`}</h3>
                        ${theme.status !== 'ready' ? `<span class="theme-status ${statusClass}">${statusText}</span>` : ''}
                        ${isActive ? '<span class="theme-active-badge">Active</span>' : ''}
                    </div>
                    
                    <div class="theme-meta">
                        ${theme.business_type ? `<span class="business-type">${this.formatBusinessType(theme.business_type)}</span>` : ''}
                        ${theme.style_preference ? `<span class="style-pref">${theme.style_preference}</span>` : ''}
                        <span class="theme-date">Created ${this.formatDate(theme.created)}</span>
                    </div>
                    
                    <div class="theme-actions">
                        ${!isActive ?
                            `<button class="button button-primary theme-activate-btn" data-theme-id="${theme.id}">Activate</button>` :
                            ''}
                        ${(theme.status === 'ready' || theme.status === 'unknown') || isActive ?
                            `<button class="button theme-edit-btn" data-theme-id="${theme.id}">
                                <span class="dashicons dashicons-edit"></span> Edit
                            </button>` :
                            ''}
                        <button class="button theme-screenshot-btn" data-theme-id="${theme.id}">
                            <span class="dashicons dashicons-camera"></span> Screenshot
                        </button>
                        <button class="button theme-duplicate-btn" data-theme-id="${theme.id}">
                            <span class="dashicons dashicons-admin-page"></span> Clone
                        </button>
                        ${!isActive ? 
                            `<button class="button theme-delete-btn" data-theme-id="${theme.id}">
                                <span class="dashicons dashicons-trash"></span> Delete
                            </button>` : 
                            ''}
                    </div>
                </div>
            </div>
        `;
    }
    
    renderThemePreview(theme) {
        // Check if theme has a screenshot
        const screenshotUrl = this.getThemeScreenshot(theme);
        
        if (screenshotUrl) {
            return `
                <div class="theme-screenshot">
                    <img src="${screenshotUrl}" alt="${theme.business_name || 'Theme'} Screenshot" />
                </div>
            `;
        }
        
        // Fallback to beautiful mock website preview
        return `
            <div class="theme-preview-placeholder">
                <div class="preview-content">
                    <div class="preview-header">
                        <div class="preview-logo"></div>
                        <div class="preview-nav">
                            <div class="preview-nav-item"></div>
                            <div class="preview-nav-item"></div>
                            <div class="preview-nav-item"></div>
                        </div>
                    </div>
                    <div class="preview-body">
                        <div class="preview-hero">
                            <div class="preview-hero-text">
                                <div class="preview-title"></div>
                                <div class="preview-subtitle"></div>
                            </div>
                        </div>
                        <div class="preview-sections">
                            <div class="preview-section">
                                <div class="preview-section-header"></div>
                                <div class="preview-cards">
                                    <div class="preview-card"></div>
                                    <div class="preview-card"></div>
                                    <div class="preview-card"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
    
    getThemeScreenshot(theme) {
        // Check for screenshot in multiple formats
        const baseUrl = aiSiteBuilder.pluginUrl + 'admin/images/screenshots/';
        const themeSlug = theme.theme_slug || `theme-${theme.id}`;
        
        // Check if screenshot exists (we'll add logic to verify file exists)
        if (theme.screenshot) {
            return baseUrl + theme.screenshot;
        }
        
        // Check for common screenshot names
        const possibleScreenshots = [
            `${themeSlug}.png`,
            `${themeSlug}.jpg`,
            `theme-${theme.id}.png`,
            `theme-${theme.id}.jpg`
        ];
        
        // For now, return the first possibility - in production you'd check if file exists
        return null; // Will show fallback preview
    }
    
    
    getFilteredThemes() {
        const searchInput = $('#theme-search');
        const typeInput = $('#theme-filter-type');
        
        const search = searchInput.length ? (searchInput.val() || '').toLowerCase() : '';
        const typeFilter = typeInput.length ? (typeInput.val() || '') : '';
        
        return this.themes.filter(theme => {
            const businessName = theme.business_name || '';
            const businessType = theme.business_type || '';
            
            const matchesSearch = !search || 
                businessName.toLowerCase().includes(search) ||
                businessType.toLowerCase().includes(search);
                
            const matchesType = !typeFilter || businessType === typeFilter;
            
            return matchesSearch && matchesType;
        });
    }
    
    filterThemes() {
        const filteredThemes = this.getFilteredThemes();
        const gridHtml = filteredThemes.length > 0 ? 
            `<div class="themes-grid">${filteredThemes.map(theme => this.renderThemeCard(theme)).join('')}</div>` :
            '<div class="themes-no-results"><p>No themes match your search criteria.</p></div>';
            
        const container = $('#themes-container');
        if (container.length) {
            container.html(gridHtml);
        }
        
        // Update stats
        const statsElement = $('.filter-stats');
        if (statsElement.length) {
            statsElement.text(`${filteredThemes.length} theme${filteredThemes.length !== 1 ? 's' : ''}`);
        }
    }
    
    async previewTheme(themeId) {
        const theme = this.themes.find(t => t.id === themeId);
        if (!theme || !theme.preview_available) {
            this.showError('Theme preview is not available');
            return;
        }
        
        try {
            const response = await fetch(`${this.apiBase}${themeId}/preview`, {
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                window.open(result.preview_url, '_blank');
            } else {
                this.showError('Preview failed: ' + (result.message || 'Unknown error'));
            }
            
        } catch (error) {
            console.error('Preview error:', error);
            this.showError('Preview failed: ' + error.message);
        }
    }
    
    editTheme(themeId) {
        // Open the AI Theme Editor
        if (window.aiThemeEditor) {
            window.aiThemeEditor.openEditor(themeId);
        } else {
            // Fallback to WordPress customizer
            this.showSuccess('Opening WordPress Customizer as fallback.');
            const theme = this.themes.find(t => t.id === themeId);
            if (theme && theme.theme_slug) {
                const customizerUrl = `${window.location.origin}/wp-admin/customize.php?theme=${theme.theme_slug}`;
                window.open(customizerUrl, '_blank');
            }
        }
    }
    
    async activateTheme(themeId) {
        const theme = this.themes.find(t => t.id === themeId);
        if (!theme) {
            this.showError('Theme not found');
            return;
        }

        if (!confirm(`Are you sure you want to activate "${theme.business_name || 'this theme'}"? It will become your active WordPress theme.`)) {
            return;
        }

        try {
            const activateUrl = `${this.apiBase}${themeId}/activate`;

            const response = await fetch(activateUrl, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();

            if (response.ok && result.success) {
                this.showSuccess(`Theme "${theme.business_name || 'theme'}" activated successfully!`);
                // Reload themes to update statuses
                await this.loadThemes();
                $('#themes-container').html(this.renderThemeGrid());
            } else {
                console.error('🎨 Activation failed:', result);
                this.showError('Activation failed: ' + (result.message || 'Unknown error'));
            }
            
        } catch (error) {
            console.error('Activation error:', error);
            this.showError('Activation failed: ' + error.message);
        }
    }
    
    async deleteTheme(themeId) {
        const theme = this.themes.find(t => t.id === themeId);
        if (!theme) return;
        
        if (!confirm(`Are you sure you want to delete "${theme.business_name || 'this theme'}"? This action cannot be undone.`)) {
            return;
        }
        
        try {
            const response = await fetch(`${this.apiBase}${themeId}`, {
                method: 'DELETE',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                this.showSuccess('Theme deleted successfully');
                await this.loadThemes();
                $('#themes-container').html(this.renderThemeGrid());
            } else {
                this.showError('Delete failed: ' + (result.message || 'Unknown error'));
            }
            
        } catch (error) {
            console.error('Delete error:', error);
            this.showError('Delete failed: ' + error.message);
        }
    }
    
    duplicateTheme(themeId) {
        // Feature planned for future release
        this.showSuccess('Theme duplication will be implemented in a future update.');
    }
    
    uploadScreenshot(themeId) {
        // Create a hidden file input
        const input = $('<input type="file" class="screenshot-upload-input" accept="image/*" style="display: none;">');
        input.attr('data-theme-id', themeId);
        $('body').append(input);
        input.click();
    }
    
    async handleScreenshotUpload(event) {
        const file = event.target.files[0];
        const themeId = $(event.target).data('theme-id');
        
        if (!file) return;
        
        // Validate file type
        if (!file.type.startsWith('image/')) {
            this.showError('Please select an image file.');
            return;
        }
        
        // Validate file size (max 5MB)
        if (file.size > 5 * 1024 * 1024) {
            this.showError('File size must be less than 5MB.');
            return;
        }
        
        try {
            const formData = new FormData();
            formData.append('screenshot', file);
            formData.append('theme_id', themeId);
            
            const response = await fetch(`${this.apiBase}${themeId}/screenshot`, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin',
                body: formData
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                this.showSuccess('Screenshot uploaded successfully!');
                // Reload themes to show the new screenshot
                await this.loadThemes();
                $('#themes-container').html(this.renderThemeGrid());
            } else {
                this.showError('Upload failed: ' + (result.message || 'Unknown error'));
            }
            
        } catch (error) {
            console.error('Screenshot upload error:', error);
            this.showError('Upload failed: ' + error.message);
        } finally {
            // Clean up the file input
            $(event.target).remove();
        }
    }
    
    // Utility methods
    getStatusClass(status) {
        const statusClasses = {
            'draft': 'status-draft',
            'generating': 'status-generating',
            'active': 'status-active',
            'error': 'status-error'
        };
        return statusClasses[status] || 'status-unknown';
    }
    
    getStatusText(status) {
        const statusTexts = {
            'draft': 'Draft',
            'generating': 'Generating...',
            'active': 'Active',
            'error': 'Error'
        };
        return statusTexts[status] || 'Unknown';
    }
    
    formatBusinessType(type) {
        return type.charAt(0).toUpperCase() + type.slice(1).replace(/[-_]/g, ' ');
    }
    
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString();
    }
    
    showError(message, details = null) {
        console.error('🎨 Theme Library Error:', message, details);

        // Create more detailed error message
        const detailsHtml = details ? `<div class="error-details" style="margin-top: 8px; font-size: 12px; opacity: 0.8;">${details}</div>` : '';

        let errorDiv = $('#ai-error-notification');
        if (errorDiv.length === 0) {
            errorDiv = $(`<div id="ai-error-notification" class="notice notice-error is-dismissible"></div>`);
            $('#theme-library-interface').prepend(errorDiv);
        }

        errorDiv.html(`
            <p>
                <span class="dashicons dashicons-warning" style="color: #d63638; margin-right: 8px;"></span>
                <strong>Theme Library Error:</strong> ${message}
                ${detailsHtml}
            </p>
        `).show();

        // Auto-remove after 10 seconds
        setTimeout(() => {
            errorDiv.fadeOut();
        }, 10000);
    }
    
    showSuccess(message) {
        let successDiv = $('#ai-success-notification');
        if (successDiv.length === 0) {
            successDiv = $('<div id="ai-success-notification" class="notice notice-success is-dismissible"></div>');
            $('#theme-library-interface').prepend(successDiv);
        }
        
        successDiv.html(`<p>${message}</p>`).show();
        
        setTimeout(() => {
            successDiv.fadeOut();
        }, 7000);
        
    }
}

// Initialize when document is ready
$(document).ready(function() {
    // Only initialize once
    if (!window.aiThemeLibrary) {
        window.aiThemeLibrary = new AIThemeLibrary();
    }
});

})(jQuery);