/**
 * =====================================================
 * UNIFIED EDITOR - Handles BOTH Theme & Page Editing
 * =====================================================
 *
 * This is the MAIN EDITOR interface used for:
 * - Theme editing: admin.php?page=dreamformer (default)
 * - Page editing: admin.php?page=dreamformer&edit=page&id=334
 *
 * Features:
 * - Split-screen interface with live preview iframe
 * - AI chat for conversational editing
 * - Manual property controls
 * - CSS rules inspection
 * - Element selection (Ctrl/⌘+click)
 * - Version control (undo/redo)
 * - Draft/Live separation for pages
 *
 * Mode Detection:
 * - Checks URL params for edit=page to enable page mode
 * - this.isPageMode flag switches between theme/page logic
 *
 * NOTE: dreamformer-pages.js is a DIFFERENT file - it only
 * handles the page list/grid view, NOT editing.
 * =====================================================
 */

(function($) {
    'use strict';

// Feature flag: Use polling instead of streaming for page edits
// Set to true for Hostinger and other hosts with HTTP/2 timeout issues
const USE_EDIT_POLLING = true;

class AIThemeEditor {
    constructor() {
        this.themeId = null;
        this.pageId = null;
        this.sessionId = null;
        this.previewUrl = null;
        this.editHistory = [];
        this.isPageMode = false;
        this.apiBase = aiSiteBuilder.restUrl + 'ai-builder/v1/themes/';
        this.isProcessing = false;
        this.currentContext = {};
        this.contextHideTimer = null;
        // Conversation history for context continuity
        this.conversationHistory = [];
        this.maxHistoryMessages = 10; // Keep last 10 messages (5 exchanges)
        // Abort controller for cancelling requests
        this.currentAbortController = null;
        this.currentReader = null;
        this.currentProgressMsg = null;
        this.currentLivePreviewMsg = null;

        // Initialize Lucide icons helper
        this.initializeLucideIcons();

        this.init();
    }
    
    init() {
        this.bindEvents();
    }


    
    async openEditor(id, mode = 'theme') {
        // Set mode and ID based on what we're editing
        this.isPageMode = (mode === 'page');
        if (this.isPageMode) {
            this.pageId = id;
            this.apiBase = aiSiteBuilder.restUrl + 'ai-builder/v1/pages/';
        } else {
            this.themeId = id;
            this.apiBase = aiSiteBuilder.restUrl + 'ai-builder/v1/themes/';
        }

        // Initialize viewport properties
        this.currentViewport = 'desktop';
        this.viewportWidths = {
            desktop: '100%',
            tablet: '768px',
            mobile: '375px'
        };

        // Initialize manual property controls (both theme and page mode)
        // Always reinitialize to ensure correct mode and ID
        if (typeof ManualPropertyControls !== 'undefined') {
            window.manualPropertyControls = new ManualPropertyControls(id, this.isPageMode);
        }

        // Initialize image upload functionality
        this.setupImageUpload();

        // Show loading state
        this.showLoadingState();

        // Create preview session
        await this.createPreviewSession();

        // Load edit history
        await this.loadEditHistory();

        // Render editor interface
        this.renderEditorInterface();

        // Initialize sidebar resize functionality
        this.initSidebarResize();

        // Restore sidebar visibility state
        this.restoreSidebarState();

        // Initialize version control button states
        this.updateVersionControlButtons();

        // Initialize postMessage listener for iframe communication
        this.initPostMessageListener();

        // Check draft status for pages
        if (this.isPageMode) {
            await this.checkDraftStatus();
        }
    }
    
    bindEvents() {
        // Chat input events
        $(document).on('submit', '#ai-chat-form', (e) => {
            e.preventDefault();
            this.handleChatSubmit();
        });

        $(document).on('keydown', '#ai-chat-input', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.handleChatSubmit();
            }
        });

        // Auto-resize textarea as user types
        $(document).on('input', '#ai-chat-input', (e) => {
            const textarea = e.target;
            textarea.style.height = 'auto';
            textarea.style.height = Math.min(textarea.scrollHeight, 200) + 'px';
        });

        // Copy message button
        $(document).on('click', '.message-copy-btn', (e) => {
            e.preventDefault();
            const $btn = $(e.currentTarget);
            const message = $btn.data('message');

            // Copy to clipboard
            navigator.clipboard.writeText(message).then(() => {
                // Visual feedback
                const $icon = $btn.find('[data-lucide]');
                if ($icon.length && $icon[0].parentElement) {
                    $icon.attr('data-lucide', 'check');
                    if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                    $btn.addClass('copied');

                    setTimeout(() => {
                        if ($icon.length && $icon[0].parentElement) {
                            $icon.attr('data-lucide', 'clipboard');
                            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                        }
                        $btn.removeClass('copied');
                    }, 2000);
                }
            }).catch(err => {
                console.error('Failed to copy:', err);
            });
        });

        // Clear chat button
        $(document).on('click', '.chat-clear-btn', () => {
            this.clearChat();
        });

        // Cancel/Stop button (when it's in stop mode)
        $(document).on('click', '.chat-send-btn.stop-mode', (e) => {
            e.preventDefault();
            this.cancelCurrentRequest();
        });

        // Toggle sidebar
        $(document).on('click', '#toggle-sidebar-btn', () => {
            this.toggleSidebar();
        });

        // Editor controls
        $(document).on('click', '#close-editor-btn', () => this.closeEditor());
        $(document).on('click', '#refresh-preview-btn', () => this.refreshPreview());
        // Rollback removed - using unified version control undo/redo instead
        $(document).on('click', '#save-theme-btn', () => this.saveTheme());

        // Viewport switching
        $(document).on('click', '.viewport-btn', (e) => {
            const $btn = $(e.currentTarget);
            const viewport = $btn.data('viewport');
            const width = $btn.data('width');

            // Update UI
            $('.viewport-btn').removeClass('active');
            $btn.addClass('active');

            // Store viewport
            this.currentViewport = viewport;

            // Resize iframe
            const $iframe = $('#theme-preview-iframe');

            if (width === '100%') {
                $iframe.css({
                    width: '100%',
                    maxWidth: '100%',
                    margin: '0'
                });
            } else {
                $iframe.css({
                    width: width + 'px',
                    maxWidth: '100%',
                    margin: '0 auto'
                });
            }
        });
        
        // Version control
        $(document).on('click', '#undo-btn', () => this.undo());
        $(document).on('click', '#redo-btn', () => this.redo());
        $(document).on('click', '#versions-btn', () => this.showVersionHistory());

        // Save Snippet handlers
        $(document).on('click', '#save-snippet-btn', () => this.openSaveSnippetModal());
        $(document).on('click', '.cancel-snippet-btn', () => this.closeSaveSnippetModal());
        $(document).on('click', '.confirm-snippet-btn', () => this.saveSnippet());

        // Handle Enter key in snippet name input
        $(document).on('keypress', '#inline-snippet-name', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                this.saveSnippet();
            }
        });

        // Load Section handlers
        $(document).on('click', '#load-section-btn', () => this.openLoadSectionModal());
        $(document).on('click', '.close-section-browser', () => this.closeLoadSectionModal());
        $(document).on('click', '.apply-section-btn', (e) => {
            const sectionId = $(e.currentTarget).data('section-id');
            this.applySection(sectionId);
        });
        $(document).on('click', '.delete-section-btn', (e) => {
            const sectionId = $(e.currentTarget).data('section-id');
            this.deleteSection(sectionId);
        });

        // Retry last prompt after timeout/error
        $(document).on('click', '.retry-last-prompt-btn', () => {
            if (this.lastPrompt && !this.isProcessing) {
                $('#ai-chat-input').val(this.lastPrompt);
                this.handleChatSubmit();
            }
        });

        // Keyboard shortcuts
        $(document).on('keydown', (e) => {
            // Only handle shortcuts when editor is open and not typing in input
            if (!this.themeId || $(e.target).is('input, textarea')) {
                return;
            }
            
            if (e.ctrlKey || e.metaKey) {
                switch(e.key.toLowerCase()) {
                    case 'z':
                        e.preventDefault();
                        if (e.shiftKey) {
                            this.redo();
                        } else {
                            this.undo();
                        }
                        break;
                    case 'y':
                        e.preventDefault();
                        this.redo();
                        break;
                }
            }
        });
        
        // Preview iframe interactions - fixed event binding
    }
    
    showLoadingState() {
        const loadingHtml = `
            <div class="editor-loading">
                <div class="spinner"></div>
                <p><i data-lucide="sparkles" class="icon-inline icon-ai animate-pulse"></i> Preparing your creative workspace...</p>
            </div>
        `;
        $('#theme-library-interface').html(loadingHtml);
    }
    
    async createPreviewSession() {
        try {
            const id = this.isPageMode ? this.pageId : this.themeId;
            const response = await fetch(`${this.apiBase}${id}/preview-session`, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                this.sessionId = result.session_id;
                this.previewUrl = result.preview_url;
            } else {
                throw new Error(result.message || 'Failed to create preview session');
            }
            
        } catch (error) {
            this.showError('Failed to initialize preview: ' + error.message);
        }
    }
    
    async loadEditHistory() {
        try {
            const id = this.isPageMode ? this.pageId : this.themeId;
            const response = await fetch(`${this.apiBase}${id}/history`, {
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (response.ok && result.success) {
                this.editHistory = result.history || [];
            }
            
        } catch (error) {
            // Non-critical error, continue without history
        }
    }
    
    renderEditorInterface() {
        const editorHtml = `
            <div class="ai-theme-editor">
                <div class="editor-header">
                    <div class="editor-title">
                        <button id="close-editor-btn" class="button">
                            <i data-lucide="arrow-left" class="icon-inline icon-secondary"></i>
                            Back
                        </button>
                        <h2>Dreamformer</h2>
                        <span class="editor-status">✨ Ready for magic</span>
                        ${aiSiteBuilder.availableCredits < 5 ? `
                        <a href="${aiSiteBuilder.pricingUrl}" class="low-credits-warning" title="Click to upgrade">
                            <i data-lucide="alert-triangle" class="icon-inline icon-sm"></i>
                            ${aiSiteBuilder.availableCredits} credit${aiSiteBuilder.availableCredits !== 1 ? 's' : ''} left
                        </a>
                        ` : ''}
                    </div>

                    <div class="viewport-controls">
                        <button class="viewport-btn active" data-viewport="desktop" data-width="100%" title="Desktop View">
                            <i data-lucide="monitor" class="icon-lg"></i>
                        </button>
                        <button class="viewport-btn" data-viewport="tablet" data-width="768" title="Tablet View (768px)">
                            <i data-lucide="tablet" class="icon-lg"></i>
                        </button>
                        <button class="viewport-btn" data-viewport="mobile" data-width="375" title="Mobile View (375px)">
                            <i data-lucide="smartphone" class="icon-lg"></i>
                        </button>
                        <button id="toggle-sidebar-btn" class="viewport-btn" title="Toggle sidebar">
                            <i data-lucide="eye" class="icon-lg icon-secondary"></i>
                        </button>
                    </div>

                    <div class="editor-actions">
                        <div class="version-controls">
                            <button id="undo-btn" class="button" title="Undo (Ctrl+Z)" disabled>
                                <i data-lucide="undo-2" class="icon-lg icon-secondary"></i>
                            </button>
                            <button id="redo-btn" class="button" title="Redo (Ctrl+Y)" disabled>
                                <i data-lucide="redo-2" class="icon-lg icon-secondary"></i>
                            </button>
                            <button id="versions-btn" class="button" title="Version History">
                                <i data-lucide="history" class="icon-lg icon-info"></i>
                            </button>
                        </div>
                        <!-- Rollback removed - unified version control provides proper undo/redo -->
                        ${this.isPageMode && aiSiteBuilder.userHasPremium ? `
                        <button id="toggle-code-editor-btn" class="button" title="Toggle Code Editor (Ctrl+E)">
                            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="icon-lg icon-secondary">
                                <polyline points="16 18 22 12 16 6"></polyline>
                                <polyline points="8 6 2 12 8 18"></polyline>
                            </svg>
                            <span>Code</span>
                        </button>
                        ` : ''}
                        <button id="refresh-preview-btn" class="button" title="Refresh preview">
                            <i data-lucide="refresh-cw" class="icon-lg icon-secondary"></i>
                        </button>
                        <button id="save-theme-btn" class="button button-primary">
                            Save ${this.isPageMode ? 'Page' : 'Theme'}
                        </button>
                    </div>
                </div>

                <div class="editor-main">
                    <div class="editor-preview">
                        <div class="preview-container">
                            <!--
                                Sandbox attributes intentionally include both allow-scripts and allow-same-origin.
                                This is required for:
                                - allow-scripts: Page functionality and selection handler
                                - allow-same-origin: Access to contentDocument.readyState for load detection
                                - allow-forms: Form interactions within preview
                                Browser security warning is expected and acceptable for editor functionality.
                            -->
                            <iframe id="theme-preview-iframe" src="${this.previewUrl}" frameborder="0" sandbox="allow-scripts allow-forms allow-same-origin"></iframe>
                            <div class="preview-overlay" style="display: none;">
                                <div class="spinner"></div>
                                <p><i data-lucide="sparkles" class="icon-inline icon-ai"></i> Working my magic...</p>
                            </div>
                        </div>

                        ${this.isPageMode ? `
                        <!-- Code Editor Container (hidden by default) -->
                        <div class="code-editor-container" id="code-editor-container" style="display: none;">
                            <div class="code-editor-header">
                                <div class="code-tabs" id="code-tabs">
                                    <button class="code-tab active" data-file="index.html">
                                        📄 index.html
                                    </button>
                                    <button class="code-tab" data-file="styles.css">
                                        🎨 styles.css
                                    </button>
                                    <button class="code-tab" data-file="app.js">
                                        ⚡ app.js
                                    </button>
                                </div>
                                <div class="code-status">
                                    <span id="editor-status">Ready</span>
                                    <span id="editor-position"></span>
                                    <div class="code-editor-help">
                                        <button class="code-editor-help__button" id="code-editor-help-btn" title="Keyboard Shortcuts">?</button>
                                        <div class="code-editor-help__tooltip" id="code-editor-help-tooltip">
                                            <div class="code-editor-help__title">
                                                ⌨️ Keyboard Shortcuts
                                            </div>
                                            <div class="code-editor-help__section">
                                                <div class="code-editor-help__label">Editing</div>
                                                <div class="code-editor-help__shortcuts">
                                                    <div class="code-editor-help__shortcut">
                                                        <span class="code-editor-help__shortcut-label">Undo</span>
                                                        <div class="code-editor-help__shortcut-keys">
                                                            <kbd class="code-editor-help__key">Ctrl</kbd>
                                                            <kbd class="code-editor-help__key">Z</kbd>
                                                        </div>
                                                    </div>
                                                    <div class="code-editor-help__shortcut">
                                                        <span class="code-editor-help__shortcut-label">Redo</span>
                                                        <div class="code-editor-help__shortcut-keys">
                                                            <kbd class="code-editor-help__key">Ctrl</kbd>
                                                            <kbd class="code-editor-help__key">Y</kbd>
                                                        </div>
                                                    </div>
                                                    <div class="code-editor-help__shortcut">
                                                        <span class="code-editor-help__shortcut-label">Save</span>
                                                        <div class="code-editor-help__shortcut-keys">
                                                            <kbd class="code-editor-help__key">Ctrl</kbd>
                                                            <kbd class="code-editor-help__key">S</kbd>
                                                        </div>
                                                    </div>
                                                    <div class="code-editor-help__shortcut">
                                                        <span class="code-editor-help__shortcut-label">Toggle Editor</span>
                                                        <div class="code-editor-help__shortcut-keys">
                                                            <kbd class="code-editor-help__key">Ctrl</kbd>
                                                            <kbd class="code-editor-help__key">E</kbd>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="code-editor-help__section">
                                                <div class="code-editor-help__label">Tips</div>
                                                <div class="code-editor-help__tips">
                                                    <div class="code-editor-help__tip">
                                                        <span class="code-editor-help__tip-icon">💾</span>
                                                        <span class="code-editor-help__tip-text">Auto-saves every 2 minutes</span>
                                                    </div>
                                                    <div class="code-editor-help__tip">
                                                        <span class="code-editor-help__tip-icon">📑</span>
                                                        <span class="code-editor-help__tip-text">Click tabs to switch files</span>
                                                    </div>
                                                    <div class="code-editor-help__tip">
                                                        <span class="code-editor-help__tip-icon">🔄</span>
                                                        <span class="code-editor-help__tip-text">Changes saved to draft only</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="code-editor-body">
                                <div id="monaco-editor-container"></div>
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    
                    <div class="editor-sidebar">
                        <!-- Resize Handle -->
                        <div class="sidebar-resize-handle" title="Drag to resize sidebar">
                            <div class="resize-handle-line"></div>
                        </div>

                        <!-- Tabbed Interface -->
                        <div class="sidebar-tabs">
                            <button type="button" class="tab-button active" data-tab="chat" title="AI Chat">
                                <span class="tab-icon"><i data-lucide="message-square" class="icon-ai"></i></span>
                            </button>
                            <button type="button" class="tab-button" data-tab="manual" title="Manual Editor">
                                <span class="tab-icon"><i data-lucide="palette" class="icon-primary"></i></span>
                            </button>
                        </div>

                        <!-- Chat Tab Content -->
                        <div class="editor-tab-content" id="chat-tab">
                            <div class="chat-container">
                                <div class="chat-messages" id="chat-messages">
                                    ${this.renderChatHistory()}
                                </div>
                                
                                <form id="ai-chat-form" class="chat-input-form">
                                    <div class="chat-input-wrapper">
                                        <input type="file" id="chat-image-upload" accept="image/*,video/*" style="display: none;">

                                        <!-- Floating action buttons above input -->
                                        <div class="chat-floating-actions">
                                            <button type="button" class="chat-upload-btn" title="Upload Image or Video">
                                                <i data-lucide="paperclip" class="icon-secondary"></i>
                                            </button>
                                            <button type="button" class="chat-media-btn" title="Choose from Media Library">
                                                <i data-lucide="image" class="icon-primary"></i>
                                            </button>
                                            <button type="button" id="save-snippet-btn" class="chat-save-snippet-btn" title="Save selected element as snippet" style="display: none;">
                                                <i data-lucide="save" class="icon-success"></i>
                                            </button>
                                            <button type="button" id="load-section-btn" class="chat-load-section-btn" title="Load section from library">
                                                <i data-lucide="folder-open" class="icon-primary"></i>
                                            </button>
                                            <button type="button" class="chat-clear-btn" title="Clear conversation">
                                                <i data-lucide="trash-2" class="icon-inline icon-danger"></i>
                                            </button>
                                        </div>

                                        <!-- Chat input container with full width -->
                                        <div class="chat-input-container">
                                            <textarea
                                                id="ai-chat-input"
                                                placeholder="E.g., 'Make the header background dark blue' or 'Add a testimonials section'"
                                                rows="3"
                                                ${this.isProcessing ? 'disabled' : ''}
                                            ></textarea>
                                            <button type="submit" class="chat-send-btn" ${this.isProcessing ? 'disabled' : ''}>
                                                ${this.isProcessing ?
                                                    '<i data-lucide="loader-2" class="icon-info animate-spin"></i>' :
                                                    '<i data-lucide="send" class="icon-primary"></i>'
                                                }
                                            </button>
                                        </div>
                                    </div>
                                    <div id="chat-upload-status" class="upload-status" style="display: none;"></div>
                                </form>
                            </div>
                        </div>

                        <!-- Manual Editor Tab Content -->
                        <div class="editor-tab-content" id="manual-tab">
                            <div class="manual-editor-content">
                                <div class="manual-editor-header">
                                    <h3><i data-lucide="palette" class="icon-inline icon-ai"></i> Manual Editor</h3>
                                    <p class="manual-description">Click an element in the preview to edit its properties</p>
                                </div>
                                <div id="manual-controls-content" class="manual-controls-wrapper">
                                    <p class="no-selection">Select an element to see editing options</p>
                                </div>
                            </div>
                        </div>

                        <div class="history-panel" id="history-panel" style="display: none;">
                            <div class="history-header">
                                <h3>Edit History</h3>
                                <button class="close-history-btn" onclick="document.getElementById('history-panel').style.display='none'">
                                    <i data-lucide="x" class="icon-inline icon-danger"></i>
                                </button>
                            </div>
                            <div class="history-list">
                                ${this.renderHistoryList()}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        $('#theme-library-interface').html(editorHtml);

        // Initialize icons in the editor interface
        if (typeof lucide !== 'undefined') {
            const editorRoot = document.getElementById('theme-library-interface');
            if (editorRoot) lucide.createIcons({ root: editorRoot });
        }

        // Initialize tab switching after HTML is inserted
        this.initializeTabs();

        // Initialize Code Editor (for pages only, premium users only)
        if (this.isPageMode && aiSiteBuilder.userHasPremium && typeof DreamformerCodeEditor !== 'undefined') {
            window.dreamformerCodeEditor = new DreamformerCodeEditor(this.pageId);
        }

        // Show chat tab by default
        $('.editor-tab-content').hide();
        $('#chat-tab').show();

        // Properly bind iframe load event after DOM is ready
        setTimeout(() => {
            const iframe = document.getElementById('theme-preview-iframe');
            if (iframe) {
                // Use native addEventListener for reliability
                iframe.addEventListener('load', () => {
                    this.onPreviewLoad();
                });

                // Check if already loaded
                try {
                    if (iframe.contentDocument && iframe.contentDocument.readyState === 'complete') {
                        this.onPreviewLoad();
                    }
                } catch (e) {
                    // Silently handle cross-origin case
                }
            }
        }, 100);
    }
    
    renderChatHistory() {
        if (this.editHistory.length === 0) {
            return `
                <div class="chat-welcome">
                    <p><i data-lucide="sparkles" class="icon-inline icon-ai animate-bounce-in"></i> Welcome to Dreamformer AI!</p>
                    <p>Tell me what you'd like to change about your ${this.isPageMode ? 'page' : 'theme'}. I can help with:</p>
                    <ul>
                        <li>• Colors and styling</li>
                        <li>• Layout and spacing</li>
                        <li>• Adding new sections</li>
                        <li>• Typography and fonts</li>
                        <li>• Mobile responsiveness</li>
                    </ul>
                    <div class="selection-hint-inline">
                        <i data-lucide="lightbulb" class="icon-inline icon-warning"></i> <strong>Tip:</strong> Ctrl/⌘+click to select/deselect • Press ESC • Click empty space
                    </div>
                </div>
            `;
        }
        
        let html = '';
        this.editHistory.slice(0, 10).forEach(entry => {
            html += `
                <div class="chat-message user-message">
                    <strong>You:</strong> ${this.escapeHtml(entry.prompt)}
                </div>
                <div class="chat-message ai-message">
                    <strong>AI:</strong>
                    <div class="change-summary">
                        ${this.renderChangeSummary(entry.modifications)}
                    </div>
                </div>
            `;
        });
        
        return html;
    }
    
    renderChangeSummary(modifications) {
        if (!modifications || modifications.length === 0) {
            return '<em>No changes applied</em>';
        }
        
        return modifications.map(mod => 
            `<div class="change-item">
                <i data-lucide="check" class="icon-inline icon-success"></i>
                ${mod.description || `Modified ${mod.file}`}
            </div>`
        ).join('');
    }
    
    renderHistoryList() {
        if (this.editHistory.length === 0) {
            return '<p class="no-history">No edit history yet</p>';
        }
        
        return this.editHistory.map((entry) => `
            <div class="history-item">
                <div class="history-time">${this.formatTime(entry.timestamp)}</div>
                <div class="history-prompt">${this.escapeHtml(entry.prompt)}</div>
                <div class="history-changes">${entry.modifications ? entry.modifications.length : 0} changes</div>
            </div>
        `).join('');
    }
    
    async handleChatSubmit() {
        // Feature flag: Use polling for page edits (fixes Hostinger timeout issues)
        if (USE_EDIT_POLLING && this.isPageMode) {
            return this.handleChatSubmitPolling();
        }

        let input = $('#ai-chat-input').val().trim();

        if (!input || this.isProcessing) {
            return;
        }

        // Store the pending image for display
        const imageToSend = this.pendingImage;
        
        // If there's a pending media (image/video), append its URL for AI processing
        if (this.pendingImage) {
            // Add the media URL to the input for AI processing
            const separator = input ? '\n\n' : '';
            const mediaType = this.pendingImage.type === 'video' ? 'video' : 'image';
            input = input + separator + `Use this ${mediaType}: ${this.pendingImage.url}`;

            // Clear pending media after use
            this.removeImagePreview();
        }
        
        this.isProcessing = true;
        this.updateStatus('processing');
        
        // Add user message to chat (show the clean version without URL)
        const displayMessage = $('#ai-chat-input').val().trim();
        this.addChatMessage('user', displayMessage, imageToSend);
        
        // Clear input
        $('#ai-chat-input').val('').prop('disabled', true);

        // Transform send button to stop button
        const $sendBtn = $('.chat-send-btn');
        $sendBtn
            .addClass('stop-mode')
            .prop('disabled', false)
            .attr('title', 'Stop generation')
            .html('<i data-lucide="x" class="icon-danger"></i>');

        // Re-initialize Lucide for the new icon
        if (typeof lucide !== 'undefined' && $sendBtn.length) {
            lucide.createIcons({ root: $sendBtn[0] });
        }

        // Show preview overlay
        $('.preview-overlay').show();

        // Declare timeoutId and livePreviewMsg outside try block so finally can access them
        let timeoutId;
        let livePreviewMsg = null;

        try {
            // Add viewport context to prompt
            let enhancedPrompt = input;
            if (this.currentViewport === 'desktop') {
                enhancedPrompt = `[VIEWPORT: Desktop (1920px)] User is viewing in desktop mode. ${input}`;
            } else if (this.currentViewport === 'tablet') {
                enhancedPrompt = `[VIEWPORT: Tablet (768px)] User is viewing in tablet mode. ${input}`;
            } else if (this.currentViewport === 'mobile') {
                enhancedPrompt = `[VIEWPORT: Mobile (375px)] User is viewing in mobile mode. ${input}`;
            }

            // Set up timeout for long AI operations (15 minutes)
            const controller = new AbortController();
            this.currentAbortController = controller; // Store for cancellation

            // Store prompt for retry functionality
            this.lastPrompt = input;
            this.lastEnhancedPrompt = enhancedPrompt;

            // Track start time for elapsed counter
            const startTime = Date.now();
            let warningShown = { min2: false, min5: false, min10: false };

            // Update elapsed time every second in the progress message
            const elapsedInterval = setInterval(() => {
                const elapsed = Math.floor((Date.now() - startTime) / 1000);
                const mins = Math.floor(elapsed / 60);
                const secs = elapsed % 60;
                const timeStr = mins > 0 ? `${mins}m ${secs}s` : `${secs}s`;

                // Update time in progress message if it exists
                const $timeDisplay = this.currentProgressMsg?.find('.elapsed-time');
                if ($timeDisplay?.length) {
                    $timeDisplay.text(timeStr);
                }

                // Show warnings at key intervals
                if (mins >= 2 && !warningShown.min2) {
                    warningShown.min2 = true;
                    this.showWarning('Taking a bit longer than usual... Complex changes take more time.');
                }
                if (mins >= 5 && !warningShown.min5) {
                    warningShown.min5 = true;
                    this.showWarning('Still working after 5 minutes. You can cancel and try a simpler request.');
                }
                if (mins >= 10 && !warningShown.min10) {
                    warningShown.min10 = true;
                    this.showWarning('10 minutes elapsed. Consider cancelling - the request may be too complex.');
                }
            }, 1000);

            // Store interval for cleanup
            this.elapsedInterval = elapsedInterval;

            timeoutId = setTimeout(() => {
                controller.abort();
            }, 900000); // 15 minutes

            const id = this.isPageMode ? this.pageId : this.themeId;

            let response, result;

            // STREAMING MODE: Browser streams from WordPress (which proxies to Vercel with API key)
            // Create single progress message that will be updated (with elapsed time)
            const progressMsg = this.addChatMessage('ai', '<span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> Preparing... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">0s</span></span>');
            this.currentProgressMsg = progressMsg; // Store for cancellation

            progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="cloud" class="icon-inline icon-info animate-pulse"></i> Connecting to AI... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">0s</span></span>');

            // Stream directly from WordPress (which proxies to Vercel with API key)
            response = await fetch(`${this.apiBase}${id}/prepare-stream`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'text/event-stream',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin',
                signal: controller.signal,
                body: JSON.stringify({
                    prompt: enhancedPrompt,
                    context: {
                        ...this.currentContext,
                        conversation_history: this.conversationHistory,
                        viewport: this.currentViewport,
                        viewport_width: this.viewportWidths[this.currentViewport],
                        responsive_editing: this.currentViewport !== 'desktop'
                    }
                })
            });

            if (!response.ok) {
                throw new Error(`Streaming error: HTTP ${response.status}`);
            }

            // Step 3: Process SSE stream in real-time (no PHP buffering!)
            const reader = response.body.getReader();
            this.currentReader = reader; // Store for cancellation
            const decoder = new TextDecoder();
            let buffer = '';
            let diffResult = null;
            let liveContent = '';

            while (true) {
                const {done, value} = await reader.read();
                if (done) break;

                buffer += decoder.decode(value, {stream: true});
                const events = buffer.split('\n\n');
                buffer = events.pop();

                for (const eventData of events) {
                    if (!eventData.trim()) continue;

                    const lines = eventData.split('\n');
                    let eventType = 'message';
                    let data = '';

                    for (const line of lines) {
                        if (line.startsWith('event:')) {
                            eventType = line.substring(6).trim();
                        } else if (line.startsWith('data:')) {
                            data = line.substring(5).trim();
                        }
                    }

                    if (eventType === 'stage') {
                        // Update progress message in real-time with visual feedback (preserve elapsed time)
                        const currentTime = this.currentProgressMsg?.find('.elapsed-time').text() || '0s';
                        progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> ' + data + ' <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">' + currentTime + '</span></span>');

                        // Add pulse animation to show it's updating
                        progressMsg.addClass('progress-pulse');
                        setTimeout(() => progressMsg.removeClass('progress-pulse'), 300);

                        // Scroll to bottom to ensure visibility
                        $('#chat-messages').scrollTop($('#chat-messages')[0].scrollHeight);

                    } else if (eventType === 'content_chunk') {
                        // CHARACTER BY CHARACTER STREAMING! 🎉
                        const chunkData = JSON.parse(data);
                        liveContent += chunkData.chunk;

                        // Early detection: check if "conversational_response" appears in stream
                        const isConversationalDetected = liveContent.includes('conversational_response');

                        // Try to extract message from partial/complete JSON
                        let extractedMessage = '';
                        if (isConversationalDetected) {
                            // Try to parse complete JSON
                            try {
                                const parsed = JSON.parse(liveContent);
                                if (parsed.message) {
                                    extractedMessage = parsed.message;
                                }
                            } catch (e) {
                                // JSON not complete yet, try regex to extract partial message
                                const match = liveContent.match(/"message"\s*:\s*"((?:[^"\\]|\\.)*)/);
                                if (match) {
                                    extractedMessage = match[1]
                                        .replace(/\\n/g, '\n')
                                        .replace(/\\"/g, '"')
                                        .replace(/\\\\/g, '\\');
                                }
                            }
                        }

                        // Create and update message based on type
                        if (!livePreviewMsg) {
                            if (isConversationalDetected) {
                                // We have conversational content! Create clean chat message
                                const initialMessage = extractedMessage || 'Thinking...';
                                livePreviewMsg = this.addChatMessage('ai', initialMessage, null, null, true);
                                livePreviewMsg.data('is-conversational', true);
                                this.currentLivePreviewMsg = livePreviewMsg;
                            } else if (liveContent.includes('"type"') && liveContent.includes('"schema"')) {
                                // It's a code diff (has schema field)
                                livePreviewMsg = this.addChatMessage('ai',
                                    '<div class="live-ai-response">' +
                                    '<strong class="live-header">' +
                                    '<i data-lucide="wand-2" class="icon-inline icon-ai animate-pulse"></i> ' +
                                    'Crafting your vision into reality' +
                                    '</strong>' +
                                    '<pre class="live-content"></pre>' +
                                    '</div>'
                                );
                                this.currentLivePreviewMsg = livePreviewMsg;
                            }
                            // Else: wait for more chunks to determine type
                        } else if (livePreviewMsg.data('is-conversational')) {
                            // Update conversational message as more text streams in
                            if (extractedMessage) {
                                const $messageContent = livePreviewMsg.find('.message-content');
                                $messageContent.html(this.parseMarkdown(this.escapeHtml(extractedMessage)));

                                // Re-initialize icons
                                if (typeof lucide !== 'undefined') {
                                    lucide.createIcons({ root: livePreviewMsg[0] });
                                }
                            }
                        } else {
                            // Update technical details with raw JSON
                            const $liveContent = livePreviewMsg.find('.live-content');
                            $liveContent.text(liveContent);
                            $liveContent.scrollTop($liveContent[0].scrollHeight);
                        }

                        // Auto-scroll chat messages container to bottom
                        $('#chat-messages').scrollTop($('#chat-messages')[0].scrollHeight);

                    } else if (eventType === 'complete') {
                        diffResult = JSON.parse(data);

                        // Check if it's conversational
                        const isConversationalComplete = diffResult.type === 'conversational_response';

                        // Handle live preview message completion
                        if (livePreviewMsg) {
                            // Check if this message was marked as conversational during streaming
                            const isStreamConversational = livePreviewMsg.data('is-conversational');

                            if (isConversationalComplete || isStreamConversational) {
                                // For conversational: keep the message as-is, it's already in final format
                                // Just clear the reference so we don't try to transform it later
                                livePreviewMsg = null;
                            } else {
                                // For code changes: store the technical details
                                // The final message with user explanation + technical details will be created later
                                const technicalDetails = liveContent;
                                livePreviewMsg.data('technical-details', technicalDetails);
                                // Don't remove yet - we'll transform it later
                            }
                        }

                    } else if (eventType === 'metadata') {
                        // Store metadata from WordPress for apply step
                        const metadataFromWordPress = JSON.parse(data);
                        if (!diffResult) diffResult = {};
                        // Merge metadata into result for apply step
                        Object.assign(diffResult, metadataFromWordPress);

                    } else if (eventType === 'error') {
                        if (livePreviewMsg) {
                            livePreviewMsg.remove();
                        }
                        progressMsg.remove();
                        throw new Error(data);
                    }
                }
            }

            reader.releaseLock();

            if (!diffResult) {
                throw new Error('No result received from Vercel streaming');
            }

            // Check if this is a conversational response (no changes to apply)
            if (diffResult.type === 'conversational_response') {
                // Remove progress message only
                progressMsg.remove();

                // Don't remove or recreate livePreviewMsg - it's already the final message
                // (it was created during streaming at line 780 and kept at line 833)

                result = {
                    success: true,
                    is_conversational: true,
                    message: diffResult.message,
                    changes: [],
                    tool_usage: diffResult.tool_usage || {
                        tools_called: 0,
                        iterations_used: 0
                    }
                };
            } else {
                // Step 4: Send diff back to WordPress to apply changes (preserve elapsed time)
                const currentTime = this.currentProgressMsg?.find('.elapsed-time').text() || '0s';
                progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="sparkles" class="icon-inline icon-ai animate-pulse"></i> Weaving magic into your site... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">' + currentTime + '</span></span>');

                // Prepare the apply request payload
                const applyPayload = {
                    prompt: enhancedPrompt,
                    diff_result: diffResult,
                    content_before: diffResult.content_before,
                    theme_files_before: diffResult.theme_files_before,
                    active_theme_id: diffResult.active_theme_id
                };

                // First attempt - check if theme confirmation needed
                let applyResponse = await fetch(`${this.apiBase}${id}/apply-stream-diff`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify(applyPayload)
                });

                if (!applyResponse.ok) {
                    throw new Error(`Failed to apply: HTTP ${applyResponse.status}`);
                }

                result = await applyResponse.json();

                // THEME WARNING SYSTEM: Check if confirmation needed
                if (result.needs_confirmation) {

                    // Remove progress message before showing modal
                    progressMsg.remove();

                    // Show warning modal and wait for user decision
                    const confirmed = await this.showThemeWarningModal(
                        result.theme_files || [],
                        result.page_files || []
                    );

                    if (confirmed) {
                        // User confirmed - re-send with confirmation flag

                        // Show new progress message
                        const confirmProgressMsg = this.addChatMessage('ai', '<span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> Applying changes...</span>');

                        applyResponse = await fetch(`${this.apiBase}${id}/apply-stream-diff`, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-WP-Nonce': aiSiteBuilder.nonce
                            },
                            credentials: 'same-origin',
                            body: JSON.stringify({
                                ...applyPayload,
                                theme_changes_confirmed: true  // User said OK to theme changes
                            })
                        });

                        if (!applyResponse.ok) {
                            throw new Error(`Failed to apply: HTTP ${applyResponse.status}`);
                        }

                        result = await applyResponse.json();
                        confirmProgressMsg.remove();
                    } else {
                        // User cancelled - don't apply anything
                        this.addChatMessage('ai', '↶ Changes cancelled - no files were modified.');
                        this.isProcessing = false;
                        return;
                    }
                } else {
                    // No confirmation needed - remove progress message
                    progressMsg.remove();
                }

                if (!result || !result.success) {
                    throw new Error(result?.message || 'Failed to apply changes');
                }
            }

            clearTimeout(timeoutId);

            if (result.success) {
                // Check if this is a conversation response (no actual changes)
                if (result.is_conversational || result.is_conversation || (result.changes && result.changes.length === 0)) {
                    // Conversational message already displayed, nothing more to do
                } else {
                    // Real changes were applied - use user_message if available, otherwise fallback to message
                    let aiMessage = result.user_message || result.message || '✨ Perfect! Your changes are live!';
                    if (result.tool_usage && result.tool_usage.iterations_used > 0) {
                        const summary = result.tool_usage.exploration_summary;
                        aiMessage += `\n\n<small><i data-lucide="search" class="icon-inline icon-sm icon-info"></i> Explored ${summary?.files_examined?.length || 0} files with ${result.tool_usage.tools_called} tool calls in ${result.tool_usage.iterations_used} iterations</small>`;
                    }
                    // Enable markdown for AI-generated explanatory messages (user_message from GPT-5)
                    const isExplanatoryMessage = result.user_message || result.message !== 'Changes applied successfully!';

                    // Check if we have a livePreviewMsg with technical details to transform
                    if (livePreviewMsg && livePreviewMsg.data('technical-details')) {
                        // Transform the existing streaming message into the final message
                        const technicalDetails = livePreviewMsg.data('technical-details');

                        // Replace the entire message content
                        const $messageContent = livePreviewMsg.find('.message-content');

                        // Build the final message with user explanation + collapsible technical details
                        let finalHTML = '';

                        // Add the user message (parsed as markdown if explanatory)
                        if (isExplanatoryMessage) {
                            finalHTML += this.parseMarkdown(this.escapeHtml(aiMessage));
                        } else {
                            finalHTML += aiMessage.replace(/\n/g, '<br>');
                        }

                        // Add the change summary
                        if (result.changes && result.changes.length > 0) {
                            finalHTML += '<div class="change-summary">' + this.renderChangeSummary(result.changes) + '</div>';
                        }

                        // Add collapsible technical details
                        finalHTML += `
                            <div class="technical-details" style="margin-top: 12px;">
                                <div class="tech-details-header" style="cursor: pointer; user-select: none; padding: 8px; background: #f5f5f5; border-radius: 4px;">
                                    <i data-lucide="wrench" class="icon-inline icon-secondary"></i>
                                    <strong>Technical Details</strong>
                                    <i data-lucide="chevron-down" class="icon-inline toggle-icon" style="float: right;"></i>
                                </div>
                                <pre class="tech-details-content" style="display: none; margin-top: 8px; padding: 12px; background: #f9f9f9; border-radius: 4px; max-height: 400px; overflow: auto;">${this.escapeHtml(technicalDetails)}</pre>
                            </div>
                        `;

                        $messageContent.html(finalHTML);

                        // Add toggle functionality
                        livePreviewMsg.find('.tech-details-header').on('click', function() {
                            const $content = $(this).next('.tech-details-content');
                            const $icon = $(this).find('.toggle-icon');

                            if ($content.is(':visible')) {
                                $content.slideUp(200);
                                $icon.attr('data-lucide', 'chevron-down');
                                if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                            } else {
                                $content.slideDown(200);
                                $icon.attr('data-lucide', 'chevron-up');
                                if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                            }
                        });

                        // Initialize icons in the transformed message
                        if (typeof lucide !== 'undefined') {
                            lucide.createIcons({ root: livePreviewMsg[0] });
                        }

                        // Clear the reference
                        livePreviewMsg = null;
                    } else {
                        // No livePreviewMsg, create a new message as usual
                        this.addChatMessage('ai', aiMessage, null, result.changes, isExplanatoryMessage);
                    }
                    
                    // Update history with tool usage info
                    this.editHistory.unshift({
                        prompt: input,
                        modifications: result.changes,
                        tool_usage: result.tool_usage,
                        method: 'tool_based',
                        timestamp: new Date().toISOString()
                    });
                    
                    // Old rollback system removed - version control handles this now

                    // Update version control buttons after successful changes
                    this.updateVersionControlButtons();

                    // Refresh preview only if changes were made
                    // Force aggressive refresh to show changes immediately
                    this.forcePreviewRefresh();
                    
                    let successMessage = result.message || '🎉 Amazing! Your theme is looking fantastic!';
                    if (result.validation && result.validation.auto_fixes && result.validation.auto_fixes.length > 0) {
                        successMessage += ` (${result.validation.auto_fixes.length} auto-fixes applied)`;
                    }
                    this.showSuccess(successMessage);

                    // Celebrate AI completion
                    if (typeof DreamformerCelebrations !== 'undefined') {
                        DreamformerCelebrations.celebrateAICompletion();
                    }

                    // Update status to draft for pages (changes not yet published)
                    if (this.isPageMode) {
                        this.updateStatus('draft');
                    }
                }
            } else {
                throw new Error(result.message || 'Failed to apply changes');
            }
            
        } catch (error) {
            // Don't show error if user manually cancelled (they already see "Request cancelled" message)
            if (this.userCancelled) {
            } else {
                // Categorize errors with user-friendly messages and actionable advice
                let errorMessage = '';
                let advice = '';
                let showRetryButton = false;
                const msg = (error.message || '').toLowerCase();

                // 1. Timeout
                if (error.name === 'AbortError') {
                    errorMessage = 'The request timed out after 15 minutes.';
                    advice = 'Try breaking your request into smaller, simpler changes.';
                    showRetryButton = true;
                }
                // 2. Network/Connection errors
                else if (msg.includes('fetch') || msg.includes('network') || msg.includes('connection') || msg.includes('offline') || msg.includes('err_')) {
                    errorMessage = 'Could not connect to the server.';
                    advice = 'Check your internet connection and try again.';
                    showRetryButton = true;
                }
                // 3. Server errors (500, 502, 503, 504)
                else if (msg.includes('500') || msg.includes('502') || msg.includes('503') || msg.includes('504') || msg.includes('server error') || msg.includes('internal error')) {
                    errorMessage = 'The server encountered a temporary problem.';
                    advice = 'Wait a moment and try again. If it persists, the AI service may be busy.';
                    showRetryButton = true;
                }
                // 4. Rate limiting (429)
                else if (msg.includes('429') || msg.includes('rate limit') || msg.includes('too many')) {
                    errorMessage = 'Too many requests in a short time.';
                    advice = 'Wait 30 seconds before trying again.';
                    showRetryButton = true;
                }
                // 5. AI content/safety errors
                else if (msg.includes('content') || msg.includes('safety') || msg.includes('filter') || msg.includes('blocked') || msg.includes('policy')) {
                    errorMessage = 'The AI could not process this request.';
                    advice = 'Try rephrasing your request differently.';
                    showRetryButton = true;
                }
                // 6. Tool errors
                else if (msg.includes('tool')) {
                    errorMessage = 'The AI had trouble analyzing your page.';
                    advice = 'Try a simpler edit or refresh the page.';
                    showRetryButton = true;
                }
                // 7. JSON/Parse errors (usually server issues)
                else if (msg.includes('json') || msg.includes('parse') || msg.includes('unexpected token') || msg.includes('syntax')) {
                    errorMessage = 'Received an unexpected response from the server.';
                    advice = 'This is usually temporary. Try again in a moment.';
                    showRetryButton = true;
                }
                // 8. Authentication errors
                else if (msg.includes('401') || msg.includes('403') || msg.includes('unauthorized') || msg.includes('forbidden')) {
                    errorMessage = 'Your session may have expired.';
                    advice = 'Try refreshing the page and logging in again.';
                    showRetryButton = false;
                }
                // 9. Fallback - unknown error
                else {
                    errorMessage = 'Something went wrong.';
                    advice = 'Try again, or simplify your request.';
                    showRetryButton = true;
                }

                // Build user-friendly message HTML
                let messageHtml = `
                    <div class="friendly-error">
                        <div style="display: flex; align-items: flex-start; gap: 8px;">
                            <i data-lucide="alert-circle" class="icon-inline icon-danger" style="margin-top: 2px;"></i>
                            <div>
                                <strong>${errorMessage}</strong>
                                <p style="margin: 6px 0 0; opacity: 0.8; font-size: 0.9em;">${advice}</p>
                            </div>
                        </div>
                    </div>
                `;

                // Add retry button if applicable
                if (showRetryButton && this.lastPrompt) {
                    messageHtml += `
                        <div style="margin-top: 12px;">
                            <button class="retry-last-prompt-btn" style="padding: 8px 16px; background: #4f46e5; color: white; border: none; border-radius: 6px; cursor: pointer; display: inline-flex; align-items: center; gap: 6px;">
                                <i data-lucide="refresh-cw" class="icon-inline icon-sm"></i> Retry
                            </button>
                        </div>
                    `;
                }

                this.addChatMessage('ai', messageHtml);
                this.showError(errorMessage);
            }
        } finally {
            // Clear timeout to prevent it firing after request completes/fails
            if (timeoutId) {
                clearTimeout(timeoutId);
            }

            // Clear elapsed time interval
            if (this.elapsedInterval) {
                clearInterval(this.elapsedInterval);
                this.elapsedInterval = null;
            }

            // Clean up abort controller and reader references
            this.currentAbortController = null;
            this.currentReader = null;
            this.currentProgressMsg = null;
            this.currentLivePreviewMsg = null;

            // Reset cancellation flag
            this.userCancelled = false;

            // Reset UI
            this.isProcessing = false;
            this.updateStatus('ready');
            $('#ai-chat-input').prop('disabled', false);

            // Transform stop button back to send button
            const $sendBtn = $('.chat-send-btn');
            $sendBtn
                .removeClass('stop-mode')
                .prop('disabled', false)
                .attr('title', 'Send message')
                .html('<i data-lucide="arrow-right" class="icon-primary"></i>');

            // Re-initialize Lucide for the new icon
            if (typeof lucide !== 'undefined') {
                lucide.createIcons({ root: $sendBtn[0] });
            }

            $('.preview-overlay').hide();
        }
    }

    /**
     * Handle chat submit using polling (alternative to streaming)
     * Used for hosts with HTTP/2 timeout issues (Hostinger, etc.)
     */
    async handleChatSubmitPolling() {
        let input = $('#ai-chat-input').val().trim();

        if (!input || this.isProcessing) {
            return;
        }

        // Store the pending image for display
        const imageToSend = this.pendingImage;

        // If there's a pending media (image/video), append its URL for AI processing
        if (this.pendingImage) {
            const separator = input ? '\n\n' : '';
            const mediaType = this.pendingImage.type === 'video' ? 'video' : 'image';
            input = input + separator + `Use this ${mediaType}: ${this.pendingImage.url}`;
            this.removeImagePreview();
        }

        this.isProcessing = true;
        this.updateStatus('processing');

        // Add user message to chat
        const displayMessage = $('#ai-chat-input').val().trim();
        this.addChatMessage('user', displayMessage, imageToSend);

        // Clear input
        $('#ai-chat-input').val('').prop('disabled', true);

        // Transform send button to stop button
        const $sendBtn = $('.chat-send-btn');
        $sendBtn
            .addClass('stop-mode')
            .prop('disabled', false)
            .attr('title', 'Stop generation')
            .html('<i data-lucide="x" class="icon-danger"></i>');

        if (typeof lucide !== 'undefined') {
            lucide.createIcons({ root: $sendBtn[0] });
        }

        // Show preview overlay
        $('.preview-overlay').show();

        let pollInterval = null;
        let livePreviewMsg = null;

        try {
            // Add viewport context to prompt
            let enhancedPrompt = input;
            if (this.currentViewport === 'desktop') {
                enhancedPrompt = `[VIEWPORT: Desktop (1920px)] User is viewing in desktop mode. ${input}`;
            } else if (this.currentViewport === 'tablet') {
                enhancedPrompt = `[VIEWPORT: Tablet (768px)] User is viewing in tablet mode. ${input}`;
            } else if (this.currentViewport === 'mobile') {
                enhancedPrompt = `[VIEWPORT: Mobile (375px)] User is viewing in mobile mode. ${input}`;
            }

            // Store prompt for retry functionality
            this.lastPrompt = input;
            this.lastEnhancedPrompt = enhancedPrompt;

            // Track start time for elapsed counter
            const startTime = Date.now();
            let warningShown = { min2: false, min5: false, min10: false };

            // Update elapsed time every second in the progress message
            const elapsedInterval = setInterval(() => {
                const elapsed = Math.floor((Date.now() - startTime) / 1000);
                const mins = Math.floor(elapsed / 60);
                const secs = elapsed % 60;
                const timeStr = mins > 0 ? `${mins}m ${secs}s` : `${secs}s`;

                // Update time in progress message if it exists
                const $timeDisplay = this.currentProgressMsg?.find('.elapsed-time');
                if ($timeDisplay?.length) {
                    $timeDisplay.text(timeStr);
                }

                // Show warnings at key intervals
                if (mins >= 2 && !warningShown.min2) {
                    warningShown.min2 = true;
                    this.showWarning('Taking a bit longer than usual... Complex changes take more time.');
                }
                if (mins >= 5 && !warningShown.min5) {
                    warningShown.min5 = true;
                    this.showWarning('Still working after 5 minutes. You can cancel and try a simpler request.');
                }
                if (mins >= 10 && !warningShown.min10) {
                    warningShown.min10 = true;
                    this.showWarning('10 minutes elapsed. Consider cancelling - the request may be too complex.');
                }
            }, 1000);

            // Store interval for cleanup
            this.elapsedInterval = elapsedInterval;

            // Create progress message (with elapsed time)
            const progressMsg = this.addChatMessage('ai', '<span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> Preparing... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">0s</span></span>');
            this.currentProgressMsg = progressMsg;

            // Step 1: Create polling job
            progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="cloud" class="icon-inline icon-info animate-pulse"></i> Connecting to AI... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">0s</span></span>');

            const createResponse = await fetch(`${this.apiBase}${this.pageId}/edit-polling`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin',
                body: JSON.stringify({
                    prompt: enhancedPrompt,
                    context: {
                        ...this.currentContext,
                        conversation_history: this.conversationHistory,
                        viewport: this.currentViewport,
                        viewport_width: this.viewportWidths[this.currentViewport],
                        responsive_editing: this.currentViewport !== 'desktop'
                    }
                })
            });

            const createData = await createResponse.json();

            if (!createData.success) {
                throw new Error(createData.message || 'Failed to start generation');
            }

            const jobId = createData.job_id;
            // Save metadata from create response for apply step
            const createMetadata = {
                content_before: createData.content_before,
                theme_files_before: createData.theme_files_before,
                active_theme_id: createData.active_theme_id
            };

            // Step 2: Poll for status (preserve elapsed time)
            const currentTime1 = this.currentProgressMsg?.find('.elapsed-time').text() || '0s';
            progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> 🧠 Analyzing your request... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">' + currentTime1 + '</span></span>');

            let lastContent = '';
            let pollCount = 0;
            const maxPolls = 300; // 10 minutes max (300 * 2 seconds)

            // Return a promise that resolves when polling completes
            const pollResult = await new Promise((resolve, reject) => {
                pollInterval = setInterval(async () => {
                    try {
                        pollCount++;

                        // Safety: Stop polling after max attempts
                        if (pollCount > maxPolls) {
                            clearInterval(pollInterval);
                            reject(new Error('Generation timeout after 10 minutes - please try again'));
                            return;
                        }

                        // Poll WordPress proxy endpoint
                        const pollUrl = aiSiteBuilder.restUrl + 'ai-builder/v1/pages/poll-edit-status/' + jobId;
                        const pollResponse = await fetch(pollUrl, {
                            method: 'GET',
                            headers: {
                                'Accept': 'application/json',
                                'X-WP-Nonce': aiSiteBuilder.nonce
                            },
                            credentials: 'same-origin'
                        });

                        if (!pollResponse.ok) {
                            console.warn(`Poll attempt ${pollCount} failed: HTTP ${pollResponse.status}`);
                            return; // Continue polling on transient errors
                        }

                        const statusData = await pollResponse.json();

                        // Update progress message with stage (preserve elapsed time)
                        if (statusData.stage) {
                            const currentTimeStage = this.currentProgressMsg?.find('.elapsed-time').text() || '0s';
                            progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> ' + statusData.stage + ' <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">' + currentTimeStage + '</span></span>');
                        }

                        // Handle different statuses
                        if (statusData.status === 'processing') {
                            // Show live content if available (fake streaming effect)
                            if (statusData.content_so_far && statusData.content_so_far !== lastContent) {
                                const newContent = statusData.content_so_far.substring(lastContent.length);

                                // Create live preview message if first content
                                if (!livePreviewMsg && newContent.length > 0) {
                                    livePreviewMsg = this.addChatMessage('ai', '');
                                    livePreviewMsg.find('.message-content').html('<pre class="live-diff-preview" style="white-space: pre-wrap; max-height: 300px; overflow: auto; background: #f5f5f5; padding: 12px; border-radius: 4px;"></pre>');
                                    this.currentLivePreviewMsg = livePreviewMsg;
                                }

                                if (livePreviewMsg) {
                                    const $preview = livePreviewMsg.find('.live-diff-preview');
                                    $preview.append(this.escapeHtml(newContent));
                                    $preview.scrollTop($preview[0].scrollHeight);
                                }

                                lastContent = statusData.content_so_far;
                            }

                        } else if (statusData.status === 'completed') {
                            clearInterval(pollInterval);
                            resolve(statusData);

                        } else if (statusData.status === 'failed') {
                            clearInterval(pollInterval);
                            const errorMessage = statusData.error?.message || statusData.error || 'Generation failed';
                            reject(new Error(errorMessage));
                        }

                    } catch (error) {
                        console.error('Poll error:', error);
                        // Continue polling on transient errors
                    }
                }, 2000); // Poll every 2 seconds
            });

            // Step 3: Apply the diff result
            const rawResult = pollResult.result;

            if (!rawResult) {
                throw new Error('No result received from polling');
            }

            // Check if this is a conversational response
            // API format: {conversational_response: {type: "...", message: "..."}, model_used: "..."}
            if (rawResult.conversational_response) {
                progressMsg.remove();

                const convMessage = rawResult.conversational_response.message || rawResult.conversational_response;

                // Show the conversational message
                if (livePreviewMsg) {
                    livePreviewMsg.find('.message-content').html(this.parseMarkdown(this.escapeHtml(convMessage)));
                } else {
                    this.addChatMessage('ai', convMessage, null, [], true);
                }

                // Update conversation history
                this.conversationHistory.push({ user: displayMessage, ai: convMessage });
                if (this.conversationHistory.length > this.maxHistoryMessages) {
                    this.conversationHistory.shift();
                }

            } else {
                // API format for diffs: {diff_json: "{...}", user_message: "...", model_used: "..."}
                // DON'T parse diff_json - send wrapper to PHP like streaming does
                // PHP expects diff_result['diff_json'] to be a string
                const diffResult = rawResult;
                // Real changes - apply the diff (preserve elapsed time)
                const currentTime2 = this.currentProgressMsg?.find('.elapsed-time').text() || '0s';
                progressMsg.html('<strong>AI:</strong> <span class="streaming-progress"><i data-lucide="sparkles" class="icon-inline icon-ai animate-pulse"></i> Weaving magic into your site... <span class="elapsed-time" style="opacity: 0.6; font-size: 0.85em;">' + currentTime2 + '</span></span>');

                const applyPayload = {
                    prompt: enhancedPrompt,
                    diff_result: diffResult,
                    content_before: createMetadata.content_before,
                    theme_files_before: createMetadata.theme_files_before,
                    active_theme_id: createMetadata.active_theme_id
                };

                // Apply the changes
                let applyResponse = await fetch(`${this.apiBase}${this.pageId}/apply-stream-diff`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify(applyPayload)
                });

                if (!applyResponse.ok) {
                    throw new Error(`Failed to apply: HTTP ${applyResponse.status}`);
                }

                let result = await applyResponse.json();

                // Handle theme confirmation if needed
                if (result.needs_confirmation) {
                    progressMsg.remove();

                    const confirmed = await this.showThemeWarningModal(
                        result.theme_files || [],
                        result.page_files || []
                    );

                    if (confirmed) {
                        const confirmProgressMsg = this.addChatMessage('ai', '<span class="streaming-progress"><i data-lucide="loader-2" class="icon-inline icon-secondary animate-spin"></i> Applying changes...</span>');

                        applyResponse = await fetch(`${this.apiBase}${this.pageId}/apply-stream-diff`, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-WP-Nonce': aiSiteBuilder.nonce
                            },
                            credentials: 'same-origin',
                            body: JSON.stringify({
                                ...applyPayload,
                                theme_changes_confirmed: true
                            })
                        });

                        if (!applyResponse.ok) {
                            throw new Error(`Failed to apply: HTTP ${applyResponse.status}`);
                        }

                        result = await applyResponse.json();
                        confirmProgressMsg.remove();
                    } else {
                        this.addChatMessage('ai', '↶ Changes cancelled - no files were modified.');
                        return;
                    }
                } else {
                    progressMsg.remove();
                }

                if (!result || !result.success) {
                    throw new Error(result?.message || 'Failed to apply changes');
                }

                // Show success message
                let aiMessage = result.user_message || result.message || '✨ Perfect! Your changes are live!';
                const isExplanatoryMessage = result.user_message || result.message !== 'Changes applied successfully!';

                // Transform live preview into final message with technical details
                if (livePreviewMsg && lastContent) {
                    const $messageContent = livePreviewMsg.find('.message-content');

                    let finalHTML = '';
                    if (isExplanatoryMessage) {
                        finalHTML += this.parseMarkdown(this.escapeHtml(aiMessage));
                    } else {
                        finalHTML += aiMessage.replace(/\n/g, '<br>');
                    }

                    if (result.changes && result.changes.length > 0) {
                        finalHTML += '<div class="change-summary">' + this.renderChangeSummary(result.changes) + '</div>';
                    }

                    finalHTML += `
                        <div class="technical-details" style="margin-top: 12px;">
                            <div class="tech-details-header" style="cursor: pointer; user-select: none; padding: 8px; background: #f5f5f5; border-radius: 4px;">
                                <i data-lucide="wrench" class="icon-inline icon-secondary"></i>
                                <strong>Technical Details</strong>
                                <i data-lucide="chevron-down" class="icon-inline toggle-icon" style="float: right;"></i>
                            </div>
                            <pre class="tech-details-content" style="display: none; margin-top: 8px; padding: 12px; background: #f9f9f9; border-radius: 4px; max-height: 400px; overflow: auto;">${this.escapeHtml(lastContent)}</pre>
                        </div>
                    `;

                    $messageContent.html(finalHTML);

                    livePreviewMsg.find('.tech-details-header').on('click', function() {
                        const $content = $(this).next('.tech-details-content');
                        const $icon = $(this).find('.toggle-icon');

                        if ($content.is(':visible')) {
                            $content.slideUp(200);
                            $icon.attr('data-lucide', 'chevron-down');
                            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                        } else {
                            $content.slideDown(200);
                            $icon.attr('data-lucide', 'chevron-up');
                            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
                        }
                    });

                    if (typeof lucide !== 'undefined') {
                        lucide.createIcons({ root: livePreviewMsg[0] });
                    }
                } else {
                    this.addChatMessage('ai', aiMessage, null, result.changes, isExplanatoryMessage);
                }

                // Update history
                this.editHistory.unshift({
                    prompt: input,
                    modifications: result.changes,
                    method: 'polling',
                    timestamp: new Date().toISOString()
                });

                // Update version control and refresh preview
                this.updateVersionControlButtons();
                this.forcePreviewRefresh();

                let successMessage = result.message || '🎉 Amazing! Your page is looking fantastic!';
                this.showSuccess(successMessage);

                if (typeof DreamformerCelebrations !== 'undefined') {
                    DreamformerCelebrations.celebrateAICompletion();
                }

                this.updateStatus('draft');

                // Update conversation history
                this.conversationHistory.push({ user: displayMessage, ai: aiMessage });
                if (this.conversationHistory.length > this.maxHistoryMessages) {
                    this.conversationHistory.shift();
                }
            }

        } catch (error) {
            if (pollInterval) {
                clearInterval(pollInterval);
            }

            // Check if this is an insufficient credits error
            if (error.message && error.message.toLowerCase().includes('insufficient credits')) {
                this.addChatMessage('ai', '<i data-lucide="alert-circle" class="icon-inline icon-sm icon-warning"></i> ' + error.message);
                // Note: lucide icons already initialized by addChatMessage()

                // Show upgrade modal
                if (typeof showDreamformerUpgradeModal === 'function') {
                    showDreamformerUpgradeModal({
                        title: 'Out of Credits',
                        message: error.message + ' Upgrade your plan to get more credits for AI editing.',
                        features: [
                            {
                                icon: 'zap',
                                label: 'More Monthly Credits',
                                description: 'Make more AI edits each month'
                            },
                            {
                                icon: 'sparkles',
                                label: 'Page Generation',
                                description: 'Create complete AI-powered pages'
                            },
                            {
                                icon: 'palette',
                                label: 'Theme Editing',
                                description: 'Customize your theme with AI assistance'
                            }
                        ],
                        upgradeUrl: aiSiteBuilder.pricingUrl
                    });
                }
            } else {
                // Categorize errors with user-friendly messages and actionable advice
                let errorMessage = '';
                let advice = '';
                let showRetryButton = false;
                const msg = (error.message || '').toLowerCase();

                // 1. Timeout
                if (msg.includes('timeout')) {
                    errorMessage = 'The request timed out.';
                    advice = 'Try breaking your request into smaller, simpler changes.';
                    showRetryButton = true;
                }
                // 2. Network/Connection errors
                else if (msg.includes('fetch') || msg.includes('network') || msg.includes('connection') || msg.includes('offline') || msg.includes('err_')) {
                    errorMessage = 'Could not connect to the server.';
                    advice = 'Check your internet connection and try again.';
                    showRetryButton = true;
                }
                // 3. Server errors (500, 502, 503, 504)
                else if (msg.includes('500') || msg.includes('502') || msg.includes('503') || msg.includes('504') || msg.includes('server error') || msg.includes('internal error')) {
                    errorMessage = 'The server encountered a temporary problem.';
                    advice = 'Wait a moment and try again. If it persists, the AI service may be busy.';
                    showRetryButton = true;
                }
                // 4. Rate limiting (429)
                else if (msg.includes('429') || msg.includes('rate limit') || msg.includes('too many')) {
                    errorMessage = 'Too many requests in a short time.';
                    advice = 'Wait 30 seconds before trying again.';
                    showRetryButton = true;
                }
                // 5. AI content/safety errors
                else if (msg.includes('content') || msg.includes('safety') || msg.includes('filter') || msg.includes('blocked') || msg.includes('policy')) {
                    errorMessage = 'The AI could not process this request.';
                    advice = 'Try rephrasing your request differently.';
                    showRetryButton = true;
                }
                // 6. Tool errors
                else if (msg.includes('tool')) {
                    errorMessage = 'The AI had trouble analyzing your page.';
                    advice = 'Try a simpler edit or refresh the page.';
                    showRetryButton = true;
                }
                // 7. JSON/Parse errors
                else if (msg.includes('json') || msg.includes('parse') || msg.includes('unexpected token') || msg.includes('syntax')) {
                    errorMessage = 'Received an unexpected response from the server.';
                    advice = 'This is usually temporary. Try again in a moment.';
                    showRetryButton = true;
                }
                // 8. Authentication errors
                else if (msg.includes('401') || msg.includes('403') || msg.includes('unauthorized') || msg.includes('forbidden')) {
                    errorMessage = 'Your session may have expired.';
                    advice = 'Try refreshing the page and logging in again.';
                    showRetryButton = false;
                }
                // 9. Fallback
                else {
                    errorMessage = 'Something went wrong.';
                    advice = 'Try again, or simplify your request.';
                    showRetryButton = true;
                }

                // Build user-friendly message HTML
                let messageHtml = `
                    <div class="friendly-error">
                        <div style="display: flex; align-items: flex-start; gap: 8px;">
                            <i data-lucide="alert-circle" class="icon-inline icon-danger" style="margin-top: 2px;"></i>
                            <div>
                                <strong>${errorMessage}</strong>
                                <p style="margin: 6px 0 0; opacity: 0.8; font-size: 0.9em;">${advice}</p>
                            </div>
                        </div>
                    </div>
                `;

                // Add retry button if applicable
                if (showRetryButton && this.lastPrompt) {
                    messageHtml += `
                        <div style="margin-top: 12px;">
                            <button class="retry-last-prompt-btn" style="padding: 8px 16px; background: #4f46e5; color: white; border: none; border-radius: 6px; cursor: pointer; display: inline-flex; align-items: center; gap: 6px;">
                                <i data-lucide="refresh-cw" class="icon-inline icon-sm"></i> Retry
                            </button>
                        </div>
                    `;
                }

                this.addChatMessage('ai', messageHtml);
                this.showError(errorMessage);
            }

        } finally {
            // Clean up
            if (pollInterval) {
                clearInterval(pollInterval);
            }

            // Clear elapsed time interval
            if (this.elapsedInterval) {
                clearInterval(this.elapsedInterval);
                this.elapsedInterval = null;
            }

            this.currentProgressMsg = null;
            this.currentLivePreviewMsg = null;

            // Reset UI
            this.isProcessing = false;
            this.updateStatus('ready');
            $('#ai-chat-input').prop('disabled', false);

            const $sendBtn = $('.chat-send-btn');
            $sendBtn
                .removeClass('stop-mode')
                .prop('disabled', false)
                .attr('title', 'Send message')
                .html('<i data-lucide="arrow-right" class="icon-primary"></i>');

            if (typeof lucide !== 'undefined') {
                lucide.createIcons({ root: $sendBtn[0] });
            }

            $('.preview-overlay').hide();
        }
    }

    addChatMessage(type, message, imageData = null, changes = null, isConversational = false) {
        // Check if this is a special message that needs HTML rendering
        const isProgressMessage = message.includes('live-ai-response') ||
                                 message.includes('streaming-progress') ||
                                 message.includes('snippet-save-form');

        // Check if message contains Lucide icons (should not be escaped)
        const hasLucideIcon = message.includes('data-lucide=');

        // For AI messages, check if it's conversational and parse markdown
        let processedMessage;
        if (isProgressMessage || hasLucideIcon) {
            // Don't escape HTML for progress/live messages or messages with icons - they need to render
            processedMessage = message;
        } else {
            processedMessage = this.escapeHtml(message);
            if (type === 'ai' && isConversational) {
                processedMessage = this.parseMarkdown(processedMessage);
            }
        }

        // Create media preview if image/video data provided
        let imagePreview = '';
        if (imageData && type === 'user') {
            const isVideo = imageData.type === 'video';
            const mediaElement = isVideo
                ? `<video src="${imageData.url}" controls style="max-width: 200px; max-height: 150px; border-radius: 6px;"></video>`
                : `<img src="${imageData.url}" alt="${imageData.alt || imageData.filename}">`;
            const icon = isVideo ? '<i data-lucide="video" class="icon-inline icon-sm icon-primary"></i>' : '<i data-lucide="image" class="icon-inline icon-sm icon-primary"></i>';

            imagePreview = `
                <div class="message-image-preview">
                    ${mediaElement}
                    <span class="image-label">${icon} ${imageData.filename}</span>
                </div>
            `;
        }

        // Generate timestamp
        const now = new Date();
        const timeString = now.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' });
        const timestamp = now.getTime();

        // Add copy button for non-progress messages
        const copyButton = !isProgressMessage ? `
            <button class="message-copy-btn" data-message="${this.escapeHtml(message).replace(/"/g, '&quot;')}" title="Copy message">
                <i data-lucide="clipboard" class="icon-inline icon-secondary"></i>
            </button>
        ` : '';

        const messageHtml = `
            <div class="chat-message ${type}-message" data-timestamp="${timestamp}">
                <div class="message-header">
                    <strong>${type === 'user' ? 'You' : 'AI'}:</strong>
                    <span class="message-timestamp">${timeString}</span>
                    ${copyButton}
                </div>
                <div class="message-content">${processedMessage}</div>
                ${imagePreview}
                ${changes ? `<div class="change-summary">${this.renderChangeSummary(changes)}</div>` : ''}
            </div>
        `;

        const $messageElement = $(messageHtml);
        $('#chat-messages').append($messageElement);

        // Initialize icons only in this new message (scoped, not entire page)
        if (typeof lucide !== 'undefined') {
            lucide.createIcons({ root: $messageElement[0] });
        }

        // Store in conversation history (skip progress and live streaming messages)
        if (!isProgressMessage) {
            const historyEntry = {};
            if (type === 'user') {
                historyEntry.user = message;
            } else {
                historyEntry.ai = message;
            }

            this.conversationHistory.push(historyEntry);

            // Keep only last N messages to avoid token bloat
            if (this.conversationHistory.length > this.maxHistoryMessages) {
                this.conversationHistory.shift(); // Remove oldest
            }
        }

        // Scroll to bottom
        $('#chat-messages').scrollTop($('#chat-messages')[0].scrollHeight);

        // Return element for progress updates
        return $messageElement;
    }

    /**
     * Refresh preview iframe with cache busting
     */
    refreshPreview() {
        const iframe = document.getElementById('theme-preview-iframe');
        if (!iframe) return;

        // Show loading overlay
        $('.preview-overlay').show();

        // Get base URL and add cache busting parameters
        const baseUrl = this.previewUrl ? this.previewUrl.split('?')[0].split('#')[0] : iframe.src.split('?')[0].split('#')[0];
        const cacheBuster = Date.now();

        let newUrl;
        if (this.isPageMode && this.previewUrl) {
            // For pages, preserve preview parameters
            const separator = this.previewUrl.includes('?') ? '&' : '?';
            newUrl = `${this.previewUrl}${separator}v=${cacheBuster}&refresh=${cacheBuster}&nocache=${Math.random()}`;
        } else {
            // For themes, use the old logic
            newUrl = `${baseUrl}?v=${cacheBuster}&refresh=${cacheBuster}&nocache=${Math.random()}`;
        }

        // Force iframe reload
        iframe.src = 'about:blank';

        // Small delay to ensure blank page loads, then load new URL
        setTimeout(() => {
            iframe.src = newUrl;

            // Set a timeout to hide loading if iframe takes too long
            setTimeout(() => {
                $('.preview-overlay').fadeOut();
            }, 5000);
        }, 100);

        // Hide loading overlay when iframe loads
        iframe.onload = () => {
            $('.preview-overlay').fadeOut();
        };
    }
    
    /**
     * Force aggressive preview refresh after AI changes
     */
    forcePreviewRefresh() {
        const iframe = document.getElementById('theme-preview-iframe');
        if (!iframe) return;

        // Show loading overlay with specific message
        $('.preview-overlay').show().html('<i data-lucide="sparkles" class="icon-inline icon-ai animate-pulse"></i> Bringing your changes to life...');

        // Use the stored preview URL for current editing context (page or theme)
        const baseUrl = this.previewUrl ? this.previewUrl.split('?')[0].split('#')[0] : iframe.src.split('?')[0].split('#')[0];
        const timestamp = Date.now();
        const random = Math.random().toString(36).substring(2, 11);

        // Debug logging

        // For pages, preserve the original preview parameters and add cache busting
        let newUrl;
        if (this.isPageMode && this.previewUrl) {
            // Keep the page preview URL structure intact, just add cache busting
            const separator = this.previewUrl.includes('?') ? '&' : '?';
            newUrl = `${this.previewUrl}${separator}ai_refresh=${timestamp}&force=${random}&nocache=${Math.random()}&_=${timestamp}`;
        } else {
            // For themes, use the old logic
            newUrl = `${baseUrl}?ai_refresh=${timestamp}&force=${random}&nocache=${Math.random()}&_=${timestamp}`;
        }

        
        // Clear iframe completely first
        iframe.src = 'about:blank';
        
        // Wait longer for complete clear, then reload
        setTimeout(() => {
            iframe.src = newUrl;
            
            // Fallback timeout
            setTimeout(() => {
                $('.preview-overlay').fadeOut();
            }, 8000);
        }, 300);
        
        // Enhanced onload handler
        iframe.onload = () => {
            // Wait a moment for page to settle
            setTimeout(() => {
                $('.preview-overlay').fadeOut();
            }, 500);
        };
    }

    /**
     * Show theme warning modal and wait for user confirmation
     *
     * @param {Array} themeFiles List of theme files to be modified
     * @param {Array} pageFiles List of page files to be modified (optional)
     * @returns {Promise<boolean>} True if confirmed, false if cancelled
     */
    showThemeWarningModal(themeFiles, pageFiles = []) {
        return new Promise((resolve) => {
            // Create modal HTML
            const modalHTML = `
                <div id="theme-warning-modal" class="modal-overlay">
                    <div class="modal-dialog">
                        <div class="modal-header">
                            <i data-lucide="sparkles" class="theme-icon"></i>
                            <h3>Theme Changes Ahead</h3>
                        </div>
                        <div class="modal-body">
                            <p class="info-text">
                                <strong>Just a heads up!</strong> You're editing your site's <span class="highlight-theme">theme</span> (not just this page). These changes will go <span class="highlight-live">live immediately</span> and be visible across all pages:
                            </p>
                            <ul class="theme-files-list">
                                ${themeFiles.map(file => `
                                    <li>
                                        <i data-lucide="file-code"></i>
                                        <strong>${this.escapeHtml(file)}</strong>
                                    </li>
                                `).join('')}
                            </ul>
                            ${pageFiles.length > 0 ? `
                                <div class="info-box">
                                    <i data-lucide="info"></i>
                                    <p>Page-specific changes (${pageFiles.map(f => this.escapeHtml(f)).join(', ')}) will go to draft as usual.</p>
                                </div>
                            ` : ''}
                            <div class="undo-reminder">
                                <i data-lucide="rotate-ccw"></i>
                                <p>Don't worry, you can undo if needed.</p>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button class="btn-cancel" id="cancel-theme-changes">
                                <i data-lucide="x"></i> Cancel
                            </button>
                            <button class="btn-apply" id="confirm-theme-changes">
                                <i data-lucide="check"></i> Apply Theme Changes
                            </button>
                        </div>
                    </div>
                </div>
            `;

            // Append to body
            $('body').append(modalHTML);
            const $modal = $('#theme-warning-modal');

            // Initialize Lucide icons
            if (typeof lucide !== 'undefined') {
                lucide.createIcons({ root: $modal[0] });
            }

            // Show modal with fade in
            $modal.fadeIn(200);

            // Handle confirm
            $('#confirm-theme-changes').off('click').on('click', () => {
                $modal.fadeOut(200, () => $modal.remove());
                resolve(true);
            });

            // Handle cancel
            $('#cancel-theme-changes').off('click').on('click', () => {
                $modal.fadeOut(200, () => $modal.remove());
                resolve(false);
            });

            // Handle ESC key
            $(document).off('keydown.theme-warning').on('keydown.theme-warning', (e) => {
                if (e.key === 'Escape') {
                    $modal.fadeOut(200, () => $modal.remove());
                    $(document).off('keydown.theme-warning');
                    resolve(false);
                }
            });
        });
    }

    /**
     * Helper function to deselect currently selected element
     * @param {Document} iframeDoc - The iframe document
     * @returns {boolean} True if element was deselected, false if nothing was selected
     */
    deselectElement(iframeDoc) {
        const selectedElement = $(iframeDoc).find('.ai-editor-selected');
        if (selectedElement.length > 0) {
            selectedElement.removeClass('ai-editor-selected');
            this.currentContext = null;
            $('#ai-chat-input').attr('placeholder', 'Ask me to edit your page...');

            // Hide save snippet button
            $('#save-snippet-btn').fadeOut(200);
            return true;
        }
        return false;
    }

    onPreviewLoad() {

        // Hide loading overlay
        $('.preview-overlay').hide();

        // Selection handler is already injected during page generation
        // It will send 'handler-loaded' message when ready
        // Parent listens for messages in initPostMessageListener()

        // Reset handler loaded flag
        this.handlerLoaded = false;

        // Set timeout to detect if handler doesn't load
        setTimeout(() => {
            if (!this.handlerLoaded) {
                // Handler might still be loading - only show UI prompt, no console noise
                // Show helpful message
                this.showWarning(
                    'This page was created before editor v2.0. ' +
                    '<button class="regenerate-page-btn">Click here to update</button>'
                );

                // Bind regenerate button
                $(document).on('click', '.regenerate-page-btn', () => {
                    this.regeneratePageForCompatibility();
                });
            }
        }, 3000);
    }

    /**
     * Regenerate page to add selection handler
     */
    async regeneratePageForCompatibility() {
        if (!confirm('This will regenerate the page content to enable editor features. Continue?')) {
            return;
        }

        try {
            this.showSuccess('Updating page for editor compatibility...');

            const response = await fetch(
                `${aiSiteBuilder.restUrl}ai-builder/v1/pages/${this.pageId}/regenerate`,
                {
                    method: 'POST',
                    headers: { 'X-WP-Nonce': aiSiteBuilder.nonce }
                }
            );

            const result = await response.json();

            if (result.success) {
                this.showSuccess('Page updated! Refreshing editor...');
                this.forcePreviewRefresh();
            } else {
                this.showError('Failed to update page: ' + result.message);
            }
        } catch (error) {
            this.showError('Error updating page: ' + error.message);
        }
    }
    
    
    /**
     * Fetch real CSS rules for selected element from server
     */
    async fetchCSSRulesForElement(selector) {
        try {
            const id = this.isPageMode ? this.pageId : this.themeId;
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${id}/css-rules`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    selector: selector
                })
            });
            
            const result = await response.json();
            
            if (result.success && this.currentContext) {
                // Update context with real CSS rules
                this.currentContext.css_rules = result.css_rules;
                this.currentContext.css_loading = false;
                this.currentContext.theme_files = result.theme_files;

                // Trigger manual controls display if available
                if (window.manualPropertyControls && (result.css_rules?.length > 0 || result.text_content?.length > 0 || result.is_image)) {
                    window.manualPropertyControls.displayControlsForRules(result.css_rules || [], result.text_content || [], result);
                }
            } else {
                this.currentContext.css_loading = false;
            }
        } catch (error) {
            if (this.currentContext) {
                this.currentContext.css_loading = false;
            }
        }
    }

    
    getElementSelector(element) {
        let selector;

        if (element.id) {
            selector = '#' + element.id;
        } else if (element.className) {
            selector = '.' + element.className.split(' ').join('.');
        } else {
            selector = element.tagName.toLowerCase();
        }

        return selector;
    }
    
    toggleHistory() {
        $('#history-panel').toggle();
    }
    
    // Old rollback method removed - unified version control provides proper undo/redo
    
    async undo() {
        if (!this.getEntityId()) return;

        try {
            this.isProcessing = true;
            this.updateStatus('processing');

            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${this.getEntityType()}s/${this.getEntityId()}/undo`, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.addChatMessage('ai', '↶ Undo successful');
                this.forcePreviewRefresh();
                this.updateUndoRedoButtons(result.can_undo, result.can_redo);
            } else {
                this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Undo failed: ' + result.message);
            }
        } catch (error) {
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> An error occurred during undo');
        } finally {
            this.isProcessing = false;
            this.updateStatus('ready');
        }
    }
    
    async redo() {
        if (!this.getEntityId()) return;

        try {
            this.isProcessing = true;
            this.updateStatus('processing');

            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${this.getEntityType()}s/${this.getEntityId()}/redo`, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.addChatMessage('ai', '↷ Redo successful');
                this.forcePreviewRefresh();
                this.updateUndoRedoButtons(result.can_undo, result.can_redo);
            } else {
                this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Redo failed: ' + result.message);
            }
        } catch (error) {
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> An error occurred during redo');
        } finally {
            this.isProcessing = false;
            this.updateStatus('ready');
        }
    }
    
    updateUndoRedoButtons(canUndo, canRedo) {
        $('#undo-btn').prop('disabled', !canUndo);
        $('#redo-btn').prop('disabled', !canRedo);
    }

    // Helper methods for unified version control
    getEntityType() {
        return this.isPageMode ? 'page' : 'theme';
    }

    getEntityId() {
        return this.isPageMode ? this.pageId : this.themeId;
    }

    // Update version control buttons based on current state
    async updateVersionControlButtons() {
        if (!this.getEntityId()) return;

        try {
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${this.getEntityType()}s/${this.getEntityId()}/versions`, {
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });

            const result = await response.json();

            if (result.success) {
                this.updateUndoRedoButtons(result.can_undo || false, result.can_redo || false);
            } else {
                // Fallback - disable both buttons on error
                this.updateUndoRedoButtons(false, false);
            }
        } catch (error) {
            // Fallback - disable both buttons on error
            this.updateUndoRedoButtons(false, false);
        }
    }

    async showVersionHistory() {
        try {
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${this.getEntityType()}s/${this.getEntityId()}/versions`, {
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                credentials: 'same-origin'
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.renderVersionHistoryModal(result.versions);
            } else {
                this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Failed to load versions');
            }
        } catch (error) {
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Error loading version history');
        }
    }
    
    renderVersionHistoryModal(versions) {
        const entityTypeName = this.getEntityType().charAt(0).toUpperCase() + this.getEntityType().slice(1);

        const modalHtml = `
            <div id="version-history-modal">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>${entityTypeName} Version History</h3>
                        <button class="modal-close">&times;</button>
                    </div>
                    <div class="modal-body">
                        ${versions.length === 0 ? `
                            <div class="empty-state">
                                <p>No versions saved yet. Make some AI edits to see version history.</p>
                            </div>
                        ` : `
                            <div class="versions-list">
                                ${versions.map((version, index) => {
                                    const isCurrent = version.is_current == 1 || version.is_current === true;
                                    return `
                                    <div class="version-card ${isCurrent ? 'current-version' : ''}" data-version-id="${version.id}">
                                        <div class="version-top">
                                            <span class="version-number">v${version.version_number}</span>
                                            ${isCurrent ? '<span class="current-badge">Current</span>' : ''}
                                            <span class="version-time">${this.formatVersionTime(version.created_at)}</span>
                                        </div>
                                        <div class="version-description">${version.change_description || 'No description'}</div>
                                        <div class="version-bottom">
                                            <span class="version-user">by ${version.user_name}</span>
                                            ${!isCurrent ? `
                                                <button class="restore-btn restore-version-btn" data-version-id="${version.id}">
                                                    Restore
                                                </button>
                                            ` : ''}
                                        </div>
                                    </div>
                                `;}).join('')}
                            </div>
                        `}
                    </div>
                </div>
            </div>
        `;
        
        $('body').append(modalHtml);
        
        // Handle restore version
        $(document).on('click', '.restore-version-btn', async (e) => {
            const versionId = $(e.target).data('version-id');
            await this.restoreVersion(versionId);
            $('#version-history-modal').remove();
        });
        
        // Handle modal close
        $(document).on('click', '.modal-close, #version-history-modal', (e) => {
            if (e.target === e.currentTarget) {
                $('#version-history-modal').remove();
            }
        });
    }

    // Helper method to format version timestamps
    formatVersionTime(dateString) {
        const date = new Date(dateString);
        const now = new Date();
        const diffMs = now - date;
        const diffMins = Math.floor(diffMs / 60000);
        const diffHours = Math.floor(diffMs / 3600000);
        const diffDays = Math.floor(diffMs / 86400000);

        if (diffMins < 1) return 'Just now';
        if (diffMins < 60) return `${diffMins} minute${diffMins === 1 ? '' : 's'} ago`;
        if (diffHours < 24) return `${diffHours} hour${diffHours === 1 ? '' : 's'} ago`;
        if (diffDays < 7) return `${diffDays} day${diffDays === 1 ? '' : 's'} ago`;

        return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
    }

    async restoreVersion(versionId) {
        try {
            this.isProcessing = true;
            this.updateStatus('processing');

            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/versions/${versionId}/restore`, {
                method: 'POST',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce,
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin'
            });

            const result = await response.json();

            if (result.success) {
                this.addChatMessage('ai', '<i data-lucide="check-circle-2" class="icon-inline icon-sm icon-success animate-bounce-in"></i> Version restored successfully');
                this.refreshPreview();
                this.updateVersionControlButtons();
            } else {
                this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Version restore failed: ' + result.message);
            }
        } catch (error) {
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Error restoring version');
        } finally {
            this.isProcessing = false;
            this.updateStatus('ready');
        }
    }

    /**
     * Open Save Snippet Inline Form (in chat)
     */
    openSaveSnippetModal() {
        if (!this.currentContext || !this.currentContext.selected_element) {
            this.addChatMessage('ai', '<i data-lucide="alert-triangle" class="icon-inline icon-sm icon-warning"></i> Please select an element first (Ctrl/⌘+click)');
            return;
        }

        // Check if a form is already open
        if ($('.snippet-save-form').length > 0) {
            this.addChatMessage('ai', '<i data-lucide="alert-triangle" class="icon-inline icon-sm icon-warning"></i> A snippet save form is already open. Please complete or cancel it first.');
            return;
        }

        // Store selector for later use
        const selector = this.currentContext.selected_element;
        const elementInfo = `${this.currentContext.element_type}${selector}`;

        const formHtml = `
            <div class="snippet-save-form" data-selector="${selector}">
                <div class="form-header">
                    <span class="form-icon"><i data-lucide="save" class="icon-inline icon-sm icon-success"></i></span>
                    <strong>Save Snippet</strong>
                </div>

                <div class="form-group">
                    <label for="inline-snippet-name">Name *</label>
                    <input type="text"
                           id="inline-snippet-name"
                           class="snippet-name-input"
                           placeholder="e.g., Hero Section"
                           autocomplete="off">
                </div>

                <div class="form-group">
                    <label for="inline-snippet-desc">Description (optional)</label>
                    <textarea id="inline-snippet-desc"
                              class="snippet-desc-input"
                              rows="2"
                              placeholder="What does this component do?"></textarea>
                </div>

                <div class="form-info">
                    <small>Selected: <code>${elementInfo}</code></small>
                </div>

                <div class="form-actions">
                    <button class="cancel-snippet-btn">Cancel</button>
                    <button class="confirm-snippet-btn"><i data-lucide="save" class="icon-inline icon-sm icon-success"></i> Save Snippet</button>
                </div>
            </div>
        `;

        // Add as AI message
        this.addChatMessage('ai', formHtml);

        // Scroll to bottom
        const $messages = $('#chat-messages');
        $messages.scrollTop($messages[0].scrollHeight);

        // Focus name input after a short delay
        setTimeout(() => {
            $('#inline-snippet-name').focus();
        }, 100);
    }

    /**
     * Close Save Snippet Inline Form
     */
    closeSaveSnippetModal() {
        // Remove the form message
        $('.snippet-save-form').closest('.chat-message').remove();
    }

    /**
     * Save Snippet from Inline Form
     */
    async saveSnippet() {
        const name = $('#inline-snippet-name').val().trim();
        const description = $('#inline-snippet-desc').val().trim();

        // Validation
        if (!name) {
            $('#inline-snippet-name').focus();
            return;
        }

        // Get selector from form data attribute (stored when form was opened)
        const $form = $('.snippet-save-form');
        const selector = $form.attr('data-selector');

        if (!selector) {
            this.closeSaveSnippetModal();
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> No element selector found');
            return;
        }

        try {
            // Disable form while saving
            $('.snippet-save-form input, .snippet-save-form textarea, .snippet-save-form button')
                .prop('disabled', true);
            $('.confirm-snippet-btn').text('Saving...');

            // Lock chat during save
            $('#ai-chat-input').prop('disabled', true);
            $('.chat-send-btn').prop('disabled', true);
            $('#save-snippet-btn').prop('disabled', true);

            // Call REST API to save snippet
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets/save`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    name: name,
                    description: description,
                    selector: selector,
                    page_id: this.isPageMode ? this.pageId : null,
                    theme_id: !this.isPageMode ? this.themeId : null
                })
            });

            const result = await response.json();

            // Remove form and unlock chat
            this.closeSaveSnippetModal();
            $('#ai-chat-input').prop('disabled', false);
            $('.chat-send-btn').prop('disabled', false);
            $('#save-snippet-btn').prop('disabled', false);

            if (result.success) {
                // Add success message
                this.addChatMessage('ai', `<i data-lucide="check-circle-2" class="icon-inline icon-sm icon-success animate-bounce-in"></i> "${name}" saved to snippets library!`);

                // Celebrate snippet save with sparkles
                if (typeof DreamformerCelebrations !== 'undefined') {
                    DreamformerCelebrations.sparkle($('#save-snippet-btn'), { count: 8 });
                }
            } else {
                this.addChatMessage('ai', `<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Failed to save: ${result.message || 'Unknown error'}`);
            }

        } catch (error) {
            console.error('Save snippet error:', error);
            this.closeSaveSnippetModal();

            // Unlock chat on error
            $('#ai-chat-input').prop('disabled', false);
            $('.chat-send-btn').prop('disabled', false);
            $('#save-snippet-btn').prop('disabled', false);

            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Error saving snippet. Check console for details.');
        }
    }

    /**
     * Open Load Section Browser (in chat)
     * Shows list of saved sections that can be applied to current page
     */
    async openLoadSectionModal() {
        // Only works in page mode
        if (!this.isPageMode) {
            this.addChatMessage('ai', '<i data-lucide="alert-triangle" class="icon-inline icon-sm icon-warning"></i> Section library is only available when editing pages.');
            return;
        }

        // Check if browser is already open
        if ($('.section-browser').length > 0) {
            this.addChatMessage('ai', '<i data-lucide="alert-triangle" class="icon-inline icon-sm icon-warning"></i> Section browser is already open.');
            return;
        }

        // Show loading message
        this.addChatMessage('ai', '<i data-lucide="loader-2" class="icon-inline icon-sm icon-info animate-spin"></i> Loading your saved sections...');

        try {
            // Fetch sections from API
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets`, {
                method: 'GET',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                }
            });

            const result = await response.json();

            // Remove loading message
            $('#chat-messages .chat-message:last-child').remove();

            if (!result.success) {
                this.addChatMessage('ai', `<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger"></i> Failed to load sections: ${result.message || 'Unknown error'}`);
                return;
            }

            // Build the browser UI
            let browserHtml = `
                <div class="section-browser">
                    <div class="browser-header">
                        <span class="browser-icon"><i data-lucide="folder-open" class="icon-inline icon-sm icon-primary"></i></span>
                        <strong>Section Library</strong>
                        <button class="close-section-browser" title="Close">
                            <i data-lucide="x" class="icon-inline icon-sm"></i>
                        </button>
                    </div>
            `;

            if (result.sections && result.sections.length > 0) {
                browserHtml += '<div class="section-list">';

                for (const section of result.sections) {
                    // Skip sections from the same page (can't apply to self)
                    const isSamePage = section.source_page_id === this.pageId;

                    browserHtml += `
                        <div class="section-item ${isSamePage ? 'same-page' : ''}" data-section-id="${section.id}">
                            <div class="section-info">
                                <div class="section-name">${this.escapeHtml(section.name)}</div>
                                <div class="section-meta">
                                    <span class="source-page"><i data-lucide="file" class="icon-inline icon-xs"></i> ${this.escapeHtml(section.source_page_title)}</span>
                                    <span class="selector-tag">${this.escapeHtml(section.selector)}</span>
                                </div>
                                ${section.description ? `<div class="section-desc">${this.escapeHtml(section.description)}</div>` : ''}
                            </div>
                            <div class="section-actions">
                                ${isSamePage ?
                                    '<span class="same-page-label">Source page</span>' :
                                    `<button class="apply-section-btn" data-section-id="${section.id}" title="Apply to this page (6 credits)">
                                        <i data-lucide="plus-circle" class="icon-inline icon-sm icon-success"></i> Apply
                                    </button>`
                                }
                                <button class="delete-section-btn" data-section-id="${section.id}" title="Delete section">
                                    <i data-lucide="trash-2" class="icon-inline icon-sm icon-danger"></i>
                                </button>
                            </div>
                        </div>
                    `;
                }

                browserHtml += '</div>';
            } else {
                browserHtml += `
                    <div class="no-sections">
                        <i data-lucide="inbox" class="icon-lg icon-muted"></i>
                        <p>No saved sections yet</p>
                        <small>Select an element (Ctrl/⌘+click) and save it to start building your library.</small>
                    </div>
                `;
            }

            browserHtml += `
                    <div class="browser-footer">
                        <small><i data-lucide="info" class="icon-inline icon-xs"></i> Applying a section costs 6 credits</small>
                    </div>
                </div>
            `;

            // Add as AI message
            this.addChatMessage('ai', browserHtml);

            // Scroll to bottom
            const $messages = $('#chat-messages');
            $messages.scrollTop($messages[0].scrollHeight);

        } catch (error) {
            console.error('Load sections error:', error);
            // Remove loading message if it exists
            $('#chat-messages .chat-message:last-child').remove();
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Error loading sections. Check console for details.');
        }
    }

    /**
     * Close Load Section Browser
     */
    closeLoadSectionModal() {
        $('.section-browser').closest('.chat-message').remove();
    }

    /**
     * Apply a saved section to current page (with polling)
     */
    async applySection(sectionId) {
        if (!this.isPageMode || !this.pageId) {
            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger"></i> Cannot apply section - no page selected.');
            return;
        }

        // Disable all apply buttons while processing
        $('.apply-section-btn').prop('disabled', true);
        const $clickedBtn = $(`.apply-section-btn[data-section-id="${sectionId}"]`);
        $clickedBtn.html('<i data-lucide="loader-2" class="icon-inline icon-sm animate-spin"></i> Applying...');

        // Show progress message
        this.addChatMessage('ai', '<i data-lucide="sparkles" class="icon-inline icon-sm icon-ai animate-pulse"></i> Starting AI integration...');

        try {
            // Step 1: Start the job
            const startResponse = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets/${sectionId}/apply`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    target_page_id: this.pageId
                })
            });

            const startResult = await startResponse.json();

            if (!startResult.success || !startResult.job_id) {
                throw new Error(startResult.message || 'Failed to start job');
            }

            const jobId = startResult.job_id;
            const snippetId = startResult.snippet_id;
            const targetPageId = startResult.target_page_id;

            // Step 2: Poll for completion
            let pollCount = 0;
            const maxPolls = 300; // 10 minutes max

            while (pollCount < maxPolls) {
                pollCount++;
                await new Promise(resolve => setTimeout(resolve, 2000)); // Wait 2 seconds

                const pollResponse = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets/apply-status/${jobId}`, {
                    headers: { 'X-WP-Nonce': aiSiteBuilder.nonce }
                });

                // Handle HTTP errors (e.g., 404 job not found, 500 server error)
                if (!pollResponse.ok) {
                    const errorData = await pollResponse.json().catch(() => ({}));
                    throw new Error(errorData.message || `Poll failed (HTTP ${pollResponse.status})`);
                }

                const pollResult = await pollResponse.json();

                // Handle error responses without status field
                if (pollResult.error) {
                    throw new Error(pollResult.message || pollResult.error);
                }

                // Update progress message
                if (pollResult.stage) {
                    const $lastMessage = $('#chat-messages .chat-message:last-child');
                    $lastMessage.find('.message-content').html(
                        `<i data-lucide="sparkles" class="icon-inline icon-sm icon-ai animate-pulse"></i> ${pollResult.stage}`
                    );
                    if (typeof lucide !== 'undefined') lucide.createIcons({ root: $lastMessage[0] });
                }

                if (pollResult.status === 'completed') {
                    // Step 3: Apply the result
                    const applyResponse = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets/apply-result`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-WP-Nonce': aiSiteBuilder.nonce
                        },
                        body: JSON.stringify({
                            snippet_id: snippetId,
                            target_page_id: targetPageId,
                            files: pollResult.files
                        })
                    });

                    const applyResult = await applyResponse.json();

                    // Remove progress message
                    $('#chat-messages .chat-message:last-child').remove();

                    if (applyResult.success) {
                        // Close the browser
                        this.closeLoadSectionModal();

                        // Success message
                        this.addChatMessage('ai', `<i data-lucide="check-circle-2" class="icon-inline icon-sm icon-success animate-bounce-in"></i> Section applied successfully! (${applyResult.credits_used} credits used)`);

                        // Refresh the preview
                        this.refreshPreview();

                        // Celebrate
                        if (typeof DreamformerCelebrations !== 'undefined') {
                            DreamformerCelebrations.sparkle($('#load-section-btn'), { count: 12 });
                        }
                    } else {
                        throw new Error(applyResult.message || 'Failed to apply result');
                    }

                    // Re-enable buttons
                    $('.apply-section-btn').prop('disabled', false);
                    $clickedBtn.html('<i data-lucide="plus-circle" class="icon-inline icon-sm icon-success"></i> Apply');
                    return;
                }

                if (pollResult.status === 'failed') {
                    throw new Error(pollResult.error || 'AI integration failed');
                }
            }

            // Timeout
            throw new Error('Operation timed out after 10 minutes');

        } catch (error) {
            console.error('Apply section error:', error);

            // Remove progress message
            $('#chat-messages .chat-message:last-child').remove();

            // Re-enable buttons
            $('.apply-section-btn').prop('disabled', false);
            $clickedBtn.html('<i data-lucide="plus-circle" class="icon-inline icon-sm icon-success"></i> Apply');

            this.addChatMessage('ai', `<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> ${error.message || 'Error applying section'}`);
        }
    }

    /**
     * Delete a saved section
     */
    async deleteSection(sectionId) {
        // Confirm deletion
        if (!confirm('Delete this section from your library?')) {
            return;
        }

        const $item = $(`.section-item[data-section-id="${sectionId}"]`);
        const $deleteBtn = $item.find('.delete-section-btn');

        // Disable button while processing
        $deleteBtn.prop('disabled', true);
        $deleteBtn.html('<i data-lucide="loader-2" class="icon-inline icon-sm animate-spin"></i>');

        try {
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/snippets/${sectionId}`, {
                method: 'DELETE',
                headers: {
                    'X-WP-Nonce': aiSiteBuilder.nonce
                }
            });

            const result = await response.json();

            if (result.success) {
                // Animate removal
                $item.fadeOut(300, function() {
                    $(this).remove();

                    // Check if list is now empty
                    if ($('.section-item').length === 0) {
                        const $sectionList = $('.section-list');
                        $sectionList.html(`
                            <div class="no-sections">
                                <i data-lucide="inbox" class="icon-lg icon-muted"></i>
                                <p>No saved sections yet</p>
                                <small>Select an element (Ctrl/⌘+click) and save it to start building your library.</small>
                            </div>
                        `);
                        // Reinitialize Lucide icons only in the section list
                        if (typeof lucide !== 'undefined') lucide.createIcons({ root: $sectionList[0] });
                    }
                });
            } else {
                // Re-enable button
                $deleteBtn.prop('disabled', false);
                $deleteBtn.html('<i data-lucide="trash-2" class="icon-inline icon-sm icon-danger"></i>');

                this.addChatMessage('ai', `<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger"></i> Failed to delete: ${result.message || 'Unknown error'}`);
            }

        } catch (error) {
            console.error('Delete section error:', error);

            // Re-enable button
            $deleteBtn.prop('disabled', false);
            $deleteBtn.html('<i data-lucide="trash-2" class="icon-inline icon-sm icon-danger"></i>');

            this.addChatMessage('ai', '<i data-lucide="x-circle" class="icon-inline icon-sm icon-danger animate-shake"></i> Error deleting section. Check console for details.');
        }
    }

    async saveTheme() {
        // For themes, no action needed (auto-saved)
        if (!this.isPageMode) {
            this.showSuccess('Theme saved successfully!');
            return;
        }

        // For pages: Publish draft
        const $saveBtn = $('#save-theme-btn');

        // Prevent double-clicks
        if ($saveBtn.prop('disabled')) {
            return;
        }

        try {
            // Disable button and show loading state
            $saveBtn.prop('disabled', true);
            this.updateStatus('publishing');

            const response = await fetch(`${this.apiBase}${this.pageId}/publish-draft`, {
                method: 'POST',
                headers: { 'X-WP-Nonce': aiSiteBuilder.nonce },
                credentials: 'same-origin'
            });

            const result = await response.json();

            if (result.success) {
                this.updateStatus('published');
                this.showSuccess('Page published successfully! 🎉');

                // Celebrate the publish!
                if (typeof DreamformerCelebrations !== 'undefined') {
                    DreamformerCelebrations.celebrateSave($saveBtn);

                    // Check for milestone celebrations
                    this.checkMilestoneCelebration();
                }

                this.refreshPreview();
            } else {
                this.updateStatus('draft');
                this.showError(result.message || 'Failed to publish page');
            }
        } catch (error) {
            console.error('Publish error:', error);
            this.updateStatus('draft');
            this.showError('Failed to publish: ' + error.message);
        } finally {
            // Re-enable button
            $saveBtn.prop('disabled', false);
        }
    }

    async checkDraftStatus() {
        try {
            const response = await fetch(`${this.apiBase}${this.pageId}/draft-status`, {
                headers: { 'X-WP-Nonce': aiSiteBuilder.nonce },
                credentials: 'same-origin'
            });

            const result = await response.json();

            // Check if page has active draft
            if (result.success && result.is_draft) {
                this.updateStatus('draft');
            }
        } catch (error) {
            console.error('Failed to check draft status:', error);
        }
    }

    /**
     * Check and trigger milestone celebrations
     * Tracks published page count and celebrates achievements
     */
    checkMilestoneCelebration() {
        // Get current published count from localStorage
        let publishedCount = parseInt(localStorage.getItem('dreamformer_published_count') || '0');
        publishedCount++;

        // Store updated count
        localStorage.setItem('dreamformer_published_count', publishedCount);

        // Get celebrated milestones
        const celebratedMilestones = JSON.parse(localStorage.getItem('dreamformer_celebrated_milestones') || '[]');

        // Check for milestones
        const milestones = [1, 5, 10, 25, 50, 100];

        if (milestones.includes(publishedCount) && !celebratedMilestones.includes(publishedCount)) {
            // Mark as celebrated
            celebratedMilestones.push(publishedCount);
            localStorage.setItem('dreamformer_celebrated_milestones', JSON.stringify(celebratedMilestones));

            // Trigger appropriate celebration
            if (publishedCount === 1) {
                DreamformerCelebrations.celebrateFirstCreation();
            } else {
                DreamformerCelebrations.celebrateMilestone(publishedCount);
            }

        }
    }

    closeEditor() {
        if (this.isPageMode) {
            // Go back to page list when in page mode
            window.location.href = aiSiteBuilder.adminUrl + 'admin.php?page=dreamformer&tab=pages';
        } else {
            // Go back to theme library when in theme mode
            if (window.aiThemeLibrary) {
                window.aiThemeLibrary.initializeThemeLibrary();
            }
        }
    }

    /**
     * Cancel current AI request
     */
    cancelCurrentRequest() {

        // Set flag to prevent error message duplication
        this.userCancelled = true;

        // Abort fetch requests (stops browser → WordPress and browser → Vercel)
        if (this.currentAbortController) {
            this.currentAbortController.abort();
            this.currentAbortController = null;
        }

        // Cancel stream reader (stops reading from Vercel)
        if (this.currentReader) {
            this.currentReader.cancel().catch(err => {
            });
            this.currentReader = null;
        }

        // Clean up progress messages
        if (this.currentProgressMsg) {
            this.currentProgressMsg.remove();
            this.currentProgressMsg = null;
        }

        if (this.currentLivePreviewMsg) {
            this.currentLivePreviewMsg.remove();
            this.currentLivePreviewMsg = null;
        }

        // Reset UI state
        this.isProcessing = false;
        this.updateStatus('ready');
        $('#ai-chat-input').prop('disabled', false);

        // Transform stop button back to send button
        const $sendBtn = $('.chat-send-btn');
        $sendBtn
            .removeClass('stop-mode')
            .prop('disabled', false)
            .attr('title', 'Send message')
            .html('<i data-lucide="arrow-right" class="icon-primary"></i>');

        // Re-initialize Lucide for the new icon
        if (typeof lucide !== 'undefined' && $sendBtn.length) {
            lucide.createIcons({ root: $sendBtn[0] });
        }

        $('.preview-overlay').hide();

        // Show cancellation message
        this.addChatMessage('system', '<i data-lucide="square" class="icon-inline icon-sm icon-danger"></i> Request cancelled by user');

    }

    clearChat() {
        // Confirm with user
        if (!confirm('Are you sure you want to clear the conversation?\n\nThis will:\n• Clear all chat messages\n• Clear conversation history (AI will forget previous context)\n• Reset element selection\n\nThis cannot be undone.')) {
            return;
        }

        // Clear conversation history
        this.conversationHistory = [];

        // Clear current context (element selection)
        this.currentContext = {};

        // Clear visual chat messages
        $('#chat-messages').empty();

        // Show welcome message again
        $('#chat-messages').html(this.renderChatHistory());

        // Initialize icons in welcome message
        if (typeof lucide !== 'undefined') {
            lucide.createIcons({ root: document.getElementById('chat-messages') });
        }

        // Show success feedback (icon will be initialized by addChatMessage)
        this.addChatMessage('ai', 'Conversation cleared! Starting fresh. <i data-lucide="party-popper" class="icon-inline icon-sm icon-success animate-pop"></i>');
    }

    toggleSidebar() {
        const $sidebar = $('.editor-sidebar');
        const $btn = $('#toggle-sidebar-btn');
        const $icon = $btn.find('[data-lucide]');

        // Check current state
        const isCurrentlyHidden = $sidebar.hasClass('sidebar-hidden');

        if (isCurrentlyHidden) {
            // Show: restore saved width or default 400px
            const savedWidth = localStorage.getItem('dreamformer_sidebar_width') || 400;
            $sidebar.css('width', savedWidth + 'px');
            $sidebar.removeClass('sidebar-hidden');
            $icon.attr('data-lucide', 'eye');
            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
            $btn.attr('title', 'Hide sidebar');
        } else {
            // Hide: set width to 0 (just like resize does)
            $sidebar.css('width', '0');
            $sidebar.addClass('sidebar-hidden');
            $icon.attr('data-lucide', 'eye-off');
            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
            $btn.attr('title', 'Show sidebar');
        }

        // Save state to localStorage
        localStorage.setItem('dreamformer_sidebar_visible', !$sidebar.hasClass('sidebar-hidden'));
    }

    restoreSidebarState() {
        const isVisible = localStorage.getItem('dreamformer_sidebar_visible');

        // If user previously hid sidebar, hide it on load
        if (isVisible === 'false') {
            const $sidebar = $('.editor-sidebar');
            const $btn = $('#toggle-sidebar-btn');
            const $icon = $btn.find('[data-lucide]');

            // Set width to 0 directly (same as toggle)
            $sidebar.css('width', '0');
            $sidebar.addClass('sidebar-hidden');
            $icon.attr('data-lucide', 'eye-off');
            if (typeof lucide !== 'undefined') lucide.createIcons({ root: $icon[0].parentElement });
            $btn.attr('title', 'Show sidebar');
        }
    }

    initSidebarResize() {
        const $sidebar = $('.editor-sidebar');
        const $handle = $('.sidebar-resize-handle');
        const $iframe = $('#theme-preview-iframe');

        // Load saved width from localStorage
        const savedWidth = localStorage.getItem('dreamformer_sidebar_width');
        if (savedWidth) {
            $sidebar.css('width', savedWidth + 'px');
        }

        let isResizing = false;
        let startX = 0;
        let startWidth = 0;
        let pendingWidth = null;
        let rafId = null;

        // Min and max constraints
        const MIN_WIDTH = 300;
        const MAX_WIDTH = 800;

        $handle.on('mousedown', (e) => {
            // Don't start resize on mobile
            if (window.innerWidth <= 960) return;

            isResizing = true;
            startX = e.pageX;
            startWidth = $sidebar.outerWidth();

            // Disable pointer events on iframe to prevent it from capturing mouse
            $iframe.css('pointer-events', 'none');

            // Add resizing class for visual feedback
            $sidebar.addClass('resizing');
            $('body').css('cursor', 'ew-resize');
            $('body').css('user-select', 'none');

            e.preventDefault();
        });

        $(document).on('mousemove', (e) => {
            if (!isResizing) return;

            // Calculate new width (inverted because handle is on left)
            const deltaX = startX - e.pageX;
            let newWidth = startWidth + deltaX;

            // Apply constraints
            newWidth = Math.max(MIN_WIDTH, Math.min(MAX_WIDTH, newWidth));

            // Throttle updates with requestAnimationFrame for smooth 60fps
            pendingWidth = newWidth;
            if (!rafId) {
                rafId = requestAnimationFrame(() => {
                    if (pendingWidth !== null) {
                        $sidebar.css('width', pendingWidth + 'px');
                    }
                    rafId = null;
                });
            }
        });

        $(document).on('mouseup', () => {
            if (!isResizing) return;

            isResizing = false;

            // Re-enable pointer events on iframe
            $iframe.css('pointer-events', 'auto');

            $sidebar.removeClass('resizing');
            $('body').css('cursor', '');
            $('body').css('user-select', '');

            // Save to localStorage
            const finalWidth = $sidebar.outerWidth();
            localStorage.setItem('dreamformer_sidebar_width', finalWidth);
        });
    }

    updateStatus(status) {
        const statusText = {
            'ready': '✨ Ready for magic',
            'processing': '🎨 Creating something amazing...',
            'error': '⚠️ Oops, hit a snag',
            'saved': '✅ Beautiful! Changes saved',
            'draft': '🟡 Draft - Changes not published',
            'published': '🟢 Published',
            'publishing': '⚙️ Publishing changes...'
        };

        $('.editor-status').text(statusText[status] || '✨ Ready for magic');
        $('.editor-status').attr('data-status', status);

        // Update button text for page mode
        if (this.isPageMode) {
            const buttonText = (status === 'draft' || status === 'publishing') ? 'Publish Changes' : 'Save Page';
            $('#save-theme-btn').text(buttonText);
        }
    }
    
    formatTime(timestamp) {
        const date = new Date(timestamp);
        return date.toLocaleString();
    }
    
    escapeHtml(text) {
        if (!text || typeof text !== 'string') {
            return '';
        }
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, m => map[m]);
    }

    /**
     * Parse basic markdown to HTML
     * Input is already HTML-escaped, so we just need to convert markdown syntax
     */
    parseMarkdown(text) {
        if (!text || typeof text !== 'string') {
            return '';
        }

        let html = text;

        // Bold: **text** or __text__
        html = html.replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>');
        html = html.replace(/__(.+?)__/g, '<strong>$1</strong>');

        // Italic: *text* or _text_
        html = html.replace(/\*(.+?)\*/g, '<em>$1</em>');
        html = html.replace(/_(.+?)_/g, '<em>$1</em>');

        // Code: `code`
        html = html.replace(/`(.+?)`/g, '<code>$1</code>');

        // Links: [text](url)
        html = html.replace(/\[(.+?)\]\((.+?)\)/g, '<a href="$2" target="_blank">$1</a>');

        // Line breaks
        html = html.replace(/\n/g, '<br>');

        return html;
    }

    showError(message) {
        let errorDiv = $('#ai-error-notification');
        if (errorDiv.length === 0) {
            errorDiv = $('<div id="ai-error-notification" class="notice notice-error is-dismissible"></div>');
            $('.ai-theme-editor').prepend(errorDiv);
        }
        
        errorDiv.html(`<p>${message}</p>`).show();
        
        setTimeout(() => {
            errorDiv.fadeOut();
        }, 5000);
    }
    
    showSuccess(message) {
        let successDiv = $('#ai-success-notification');
        if (successDiv.length === 0) {
            successDiv = $('<div id="ai-success-notification" class="notice notice-success is-dismissible"></div>');
            $('.ai-theme-editor').prepend(successDiv);
        }

        successDiv.html(`<p>${message}</p>`).show();

        setTimeout(() => {
            successDiv.fadeOut();
        }, 7000);
    }

    showWarning(message) {
        let warningDiv = $('#ai-warning-notification');
        if (warningDiv.length === 0) {
            warningDiv = $('<div id="ai-warning-notification" class="notice notice-warning is-dismissible"></div>');
            $('.ai-theme-editor').prepend(warningDiv);
        }

        warningDiv.html(`<p>${message}</p>`).show();

        setTimeout(() => {
            warningDiv.fadeOut();
        }, 8000);
    }
    
    /**
     * Initialize tabbed interface functionality
     */
    initializeTabs() {
        const self = this;

        // Tab switching - clean and simple
        $('.tab-button').on('click', function(e) {
            e.preventDefault();

            const targetTab = $(this).data('tab');

            // Switch tabs
            $('.tab-button').removeClass('active');
            $('.editor-tab-content').hide();

            $(this).addClass('active');
            $(`#${targetTab}-tab`).show();

            // Tab-specific initialization
            if (targetTab === 'chat') {
                $('#ai-chat-input').focus();
            }
        });
    }

    
    /**
     * Switch to a specific tab programmatically
     */
    switchToTab(tabName) {
        $(`.tab-button[data-tab="${tabName}"]`).click();
    }

    /**
     * Setup image upload functionality
     */
    setupImageUpload() {
        // Bind upload button click
        $(document).on('click', '.chat-upload-btn', () => {
            $('#chat-image-upload').click();
        });
        
        // Bind media library button click
        $(document).on('click', '.chat-media-btn', () => {
            this.openMediaLibrary();
        });
        
        // Handle file selection
        $(document).on('change', '#chat-image-upload', (e) => {
            const file = e.target.files[0];
            if (file) {
                this.handleImageSelect(file);
            }
        });
        
        // Setup drag & drop
        this.setupDragDrop();
        
        // Setup clipboard paste
        this.setupClipboardPaste();
    }
    
    /**
     * Handle image file selection
     */
    async handleImageSelect(file) {
        // Validate file type (images and videos)
        const isImage = file.type.startsWith('image/');
        const isVideo = file.type.startsWith('video/');

        if (!isImage && !isVideo) {
            this.showUploadStatus('Please select an image or video file', 'error');
            return;
        }

        // Check file size (5MB for images, 50MB for videos)
        const maxSize = isVideo ? 50 * 1024 * 1024 : 5 * 1024 * 1024;
        const sizeLabel = isVideo ? '50MB' : '5MB';

        if (file.size > maxSize) {
            this.showUploadStatus(`File size must be less than ${sizeLabel}`, 'error');
            return;
        }
        
        // Show uploading status
        this.showUploadStatus('✨ Adding your image to the library...', 'loading');
        
        try {
            // Upload to WordPress Media Library
            const media = await this.uploadToWordPressMedia(file);
            
            // Create visual media reference instead of ugly URL

            // Store media data for AI processing
            this.pendingImage = {
                url: media.source_url,
                id: media.id,
                filename: media.title?.rendered || media.source_url.split('/').pop(),
                alt: media.alt_text || '',
                type: isVideo ? 'video' : 'image',
                mimeType: file.type
            };

            // Show media preview in chat
            this.showImagePreview(this.pendingImage);

            // Don't add anything to chat input - let user type naturally

            // Show success
            const mediaType = isVideo ? 'Video' : 'Image';
            this.showUploadStatus(`<i data-lucide="check-circle-2" class="icon-inline icon-sm icon-success animate-bounce-in"></i> ${mediaType} uploaded successfully!`, 'success');
            
            // Focus chat input
            $('#ai-chat-input').focus();
            
            // Clear file input
            $('#chat-image-upload').val('');
            
            // Hide status after 3 seconds
            setTimeout(() => this.hideUploadStatus(), 3000);
            
        } catch (error) {
            this.showUploadStatus('Failed to upload image. Please try again.', 'error');
            setTimeout(() => this.hideUploadStatus(), 5000);
        }
    }
    
    /**
     * Upload image to WordPress Media Library
     */
    async uploadToWordPressMedia(file) {
        const formData = new FormData();
        formData.append('file', file);
        
        const response = await fetch(`${aiSiteBuilder.restUrl}wp/v2/media`, {
            method: 'POST',
            headers: {
                'X-WP-Nonce': aiSiteBuilder.nonce
            },
            body: formData
        });
        
        if (!response.ok) {
            throw new Error(`Upload failed: ${response.statusText}`);
        }
        
        return response.json();
    }
    
    /**
     * Open WordPress Media Library modal
     */
    openMediaLibrary() {
        // Check if wp.media is available
        if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
            this.showUploadStatus('Media Library not available', 'error');
            return;
        }

        // Create media frame (allow both images and videos)
        const frame = wp.media({
            title: 'Select or Upload Media',
            button: {
                text: 'Use This Media'
            },
            multiple: false,
            library: {
                type: ['image', 'video']
            }
        });

        // Handle selection
        frame.on('select', () => {
            const attachment = frame.state().get('selection').first().toJSON();

            // Detect if this is a video or image
            const isVideo = attachment.type === 'video' || (attachment.mime && attachment.mime.startsWith('video/'));

            // Store media data for AI processing
            this.pendingImage = {
                url: attachment.url,
                id: attachment.id,
                filename: attachment.filename || attachment.url.split('/').pop(),
                alt: attachment.alt || '',
                type: isVideo ? 'video' : 'image',
                mimeType: attachment.mime || ''
            };

            // Show media preview in chat
            this.showImagePreview(this.pendingImage);

            // Don't add anything to chat input - let user type naturally
            $('#ai-chat-input').focus();

            // Show success
            const mediaType = isVideo ? 'Video' : 'Image';
            this.showUploadStatus('<i data-lucide="check-circle-2" class="icon-inline icon-sm icon-success animate-bounce-in"></i> ' + mediaType + ' selected from Media Library', 'success');
            setTimeout(() => this.hideUploadStatus(), 3000);

            // Focus chat input
            $('#ai-chat-input').focus();
        });

        // Open the modal
        frame.open();
    }
    
    /**
     * Setup drag and drop functionality
     */
    setupDragDrop() {
        const chatContainer = $('.chat-container');
        
        // Prevent default drag behaviors
        $(document).on('dragenter dragover', (e) => {
            e.preventDefault();
            e.stopPropagation();
        });
        
        // Handle drag enter on chat container
        chatContainer.on('dragenter', (e) => {
            e.preventDefault();
            e.stopPropagation();
            chatContainer.addClass('drag-over');
        });
        
        // Handle drag leave
        chatContainer.on('dragleave', (e) => {
            e.preventDefault();
            e.stopPropagation();
            // Only remove class if leaving the container
            if (e.target === chatContainer[0]) {
                chatContainer.removeClass('drag-over');
            }
        });
        
        // Handle drop
        chatContainer.on('drop', (e) => {
            e.preventDefault();
            e.stopPropagation();
            chatContainer.removeClass('drag-over');
            
            const files = e.originalEvent.dataTransfer.files;
            if (files.length > 0) {
                // Handle first image or video file
                for (let file of files) {
                    if (file.type.startsWith('image/') || file.type.startsWith('video/')) {
                        this.handleImageSelect(file);
                        break;
                    }
                }
            }
        });
    }
    
    /**
     * Setup clipboard paste for images and videos
     */
    setupClipboardPaste() {
        $('#ai-chat-input').on('paste', (e) => {
            const clipboardData = e.originalEvent.clipboardData;
            if (!clipboardData) return;

            const items = clipboardData.items;
            for (let item of items) {
                if (item.type.startsWith('image/') || item.type.startsWith('video/')) {
                    e.preventDefault();
                    const file = item.getAsFile();
                    if (file) {
                        this.handleImageSelect(file);
                    }
                    break;
                }
            }
        });
    }
    
    /**
     * Show upload status message
     */
    showUploadStatus(message, type = 'info') {
        const statusEl = $('#chat-upload-status');
        statusEl.removeClass('error success loading info')
                 .addClass(type)
                 .html(message)
                 .show();
    }
    
    /**
     * Hide upload status message
     */
    hideUploadStatus() {
        $('#chat-upload-status').fadeOut();
    }
    
    /**
     * Show image/video preview above chat input
     */
    showImagePreview(mediaData) {
        // Remove any existing preview
        $('.chat-image-preview').remove();

        // Create media element based on type
        const isVideo = mediaData.type === 'video';
        const mediaElement = isVideo
            ? `<video src="${mediaData.url}" controls style="width: 100%; height: 150px; object-fit: cover;"></video>`
            : `<img src="${mediaData.url}" alt="${mediaData.alt || mediaData.filename}">`;

        const icon = isVideo ? '<i data-lucide="video" class="icon-inline icon-sm icon-primary"></i>' : '<i data-lucide="image" class="icon-inline icon-sm icon-primary"></i>';
        const removeTitle = isVideo ? 'Remove video' : 'Remove image';

        // Create preview element
        const previewHtml = `
            <div class="chat-image-preview">
                <div class="image-preview-card">
                    ${mediaElement}
                    <div class="image-preview-info">
                        <span class="image-icon">${icon}</span>
                        <span class="image-name">${mediaData.filename}</span>
                        <button type="button" class="remove-image-btn" title="${removeTitle}">✕</button>
                    </div>
                </div>
            </div>
        `;

        // Insert before chat input
        $('.chat-input-form').prepend(previewHtml);

        // Bind remove button
        $('.remove-image-btn').on('click', () => {
            this.removeImagePreview();
        });
    }
    
    /**
     * Remove image preview
     */
    removeImagePreview() {
        $('.chat-image-preview').fadeOut(200, function() {
            $(this).remove();
        });
        this.pendingImage = null;
    }

    /**
     * Initialize Lucide icons - called manually when content is added
     * No MutationObserver to avoid performance issues during streaming
     */
    initializeLucideIcons() {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    /**
     * Initialize postMessage listener for iframe communication
     * Receives messages from injected selection handler in isolated iframe
     */
    initPostMessageListener() {
        // Prevent adding duplicate listeners on editor reopen
        if (this.postMessageListenerAdded) {
            return;
        }
        this.postMessageListenerAdded = true;

        window.addEventListener('message', (e) => {
            // Safety guard: validate message structure
            if (!e.data || typeof e.data !== 'object' || !e.data.type) {
                return;
            }

            // Handle messages from preview iframe
            switch (e.data.type) {
                case 'element-selected':
                    this.handleElementSelected(e.data);
                    break;

                case 'element-deselected':
                    this.handleElementDeselected();
                    break;

                case 'handler-loaded':
                    this.handlerLoaded = true;
                    break;

                case 'dreamformer_heartbeat':
                    // Heartbeat received - iframe is alive
                    this.lastHeartbeat = Date.now();
                    break;
            }
        });

        // Check for handler loading timeout (silently - no console noise in production)
        setTimeout(() => {
            if (!this.handlerLoaded) {
                // Handler didn't load, but we already showed a UI warning above
                // No need for console spam
            }
        }, 3000);
    }

    /**
     * Handle element selection from iframe
     */
    handleElementSelected(data) {
        try {

            // Update current context
            this.currentContext = {
                selected_element: data.selector,
                element_type: data.element_type,
                element_classes: data.element_classes,
                parent_section: data.parent_section,
                section_index: data.section_index,
                computed_styles: data.computed_styles,
                html: data.html,
                text: data.text,
                css_rules: [],
                css_loading: true
            };

            // Fetch CSS rules from server
            this.fetchCSSRulesForElement(data.selector);

            // Update chat input placeholder with context (use native DOM to avoid timing issues)
            const bgColor = data.computed_styles.background || 'transparent';
            const suggestion = `Selected: ${data.element_type} (${bgColor})`;
            const chatInput = document.getElementById('ai-chat-input');
            if (chatInput) {
                chatInput.placeholder = suggestion;
            } else {
                console.warn('[Editor] Chat input element not found in DOM');
            }

            // Show save snippet button
            $('#save-snippet-btn').fadeIn(200);

            // Update manual editor if available
            if (window.manualPropertyControls) {
                // Manual editor will update when CSS rules are fetched
            }
        } catch (error) {
            console.error('[Editor] Error handling element selection:', error);
        }
    }

    /**
     * Handle element deselection from iframe
     */
    handleElementDeselected() {
        try {

            // Clear context
            this.currentContext = {};

            // Reset chat input placeholder (use native DOM for consistency)
            const chatInput = document.getElementById('ai-chat-input');
            if (chatInput) {
                chatInput.placeholder = 'Ask me to edit your page...';
            }

            // Hide save snippet button
            $('#save-snippet-btn').fadeOut(200);
        } catch (error) {
            console.error('[Editor] Error handling element deselection:', error);
        }
    }
}

// Initialize and expose globally
$(document).ready(function() {
    window.aiThemeEditor = new AIThemeEditor();

    // Check if we should auto-open a page editor
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('edit') === 'page' && urlParams.get('id')) {
        const pageId = urlParams.get('id');
        window.aiThemeEditor.openEditor(pageId, 'page');
    }
});

})(jQuery);