/**
 * Setup Wizard - First-run with feature overview
 * Handles 3-step flow: Features → Theme → Success
 */

(function($) {
	'use strict';

	$(document).ready(function() {

		// === STEP NAVIGATION ===

		function showStep(stepNum) {
			$('.setup-step').hide();
			$('.setup-step[data-step="' + stepNum + '"]').fadeIn(300);

			// Reinitialize Lucide icons
			if (typeof lucide !== 'undefined') {
				lucide.createIcons();
			}
		}

		// Step 0 → Step 1 (Start Setup button)
		$('#start-setup').on('click', function() {
			showStep(1);
		});

		// Skip welcome → Go directly to theme selection
		$('#skip-welcome').on('click', function() {
			showStep(1);
		});

		// Step 1 → Step 0 (Back button)
		$('#back-to-overview').on('click', function() {
			showStep(0);
		});

		// === THEME SELECTION ===

		// Theme card selection visual feedback
		$('input[name="base_theme"]').on('change', function() {
			$('.theme-choice-card').removeClass('selected');
			$(this).closest('.theme-choice-card').addClass('selected');
		});

		// Mark first theme as selected by default
		$('input[name="base_theme"]:checked').closest('.theme-choice-card').addClass('selected');

		// === STEP 1 → STEP 2: Complete Setup ===

		$('#complete-setup').on('click', async function() {
			const $btn = $(this);
			const $icon = $btn.find('i');
			const $text = $btn.find('span');
			const selectedTheme = $('input[name="base_theme"]:checked').val();

			// Disable button and show loading
			$btn.prop('disabled', true);
			$icon.attr('data-lucide', 'loader-2').addClass('animate-spin');
			$text.text(selectedTheme === 'keep_current' ? 'Completing setup...' : 'Activating theme...');
			if (typeof lucide !== 'undefined') {
				lucide.createIcons();
			}

			try {
				const response = await fetch(
					window.dreamformerSetup.restUrl + 'ai-builder/v1/setup/activate-theme',
					{
						method: 'POST',
						headers: {
							'Content-Type': 'application/json',
							'X-WP-Nonce': window.dreamformerSetup.nonce
						},
						body: JSON.stringify({ theme: selectedTheme })
					}
				);

				const result = await response.json();

				if (response.ok && result.success) {
					// Show success step
					showStep(2);
				} else {
					// Show error
					const message = result.message || result.data?.message || 'Setup failed';
					alert(message);

					// Reset button
					$icon.attr('data-lucide', 'arrow-right').removeClass('animate-spin');
					$text.text('Complete Setup');
					$btn.prop('disabled', false);
					if (typeof lucide !== 'undefined') {
						lucide.createIcons();
					}
				}

			} catch (error) {
				console.error('Setup error:', error);
				alert('Setup failed. Please try again.');

				// Reset button
				$icon.attr('data-lucide', 'arrow-right').removeClass('animate-spin');
				$text.text('Complete Setup');
				$btn.prop('disabled', false);
				if (typeof lucide !== 'undefined') {
					lucide.createIcons();
				}
			}
		});

		// === STEP 2: SUCCESS OPTIONS ===

		// Show Quick Start → Reload with Quick Start box visible
		$('#show-quick-start').on('click', function() {
			window.location.reload();
		});

		// Skip Quick Start → Reload with Quick Start dismissed
		$('#skip-quick-start').on('click', async function() {
			// Mark Quick Start as dismissed
			try {
				await fetch(
					window.dreamformerSetup.restUrl + 'ai-builder/v1/setup/dismiss-quick-start',
					{
						method: 'POST',
						headers: {
							'Content-Type': 'application/json',
							'X-WP-Nonce': window.dreamformerSetup.nonce
						}
					}
				);
			} catch (error) {
				console.error('Failed to dismiss quick start:', error);
			}

			// Reload dashboard
			window.location.reload();
		});

	});

})(jQuery);
