/**
 * Page Generation Streaming UI
 * Handles real-time page generation with SSE streaming
 */

(function($) {
    'use strict';

    window.PageGenerationStreamer = class PageGenerationStreamer {
        constructor() {
            this.eventSource = null;
            this.modal = null;
            this.progressBar = null;
            this.stageText = null;
            this.stageIcon = null;
            this.currentProgress = 0;
            this.abortController = null;
        }

        /**
         * Start streaming page generation
         */
        async start(prompt, pageType, businessInfo, referencePageId, attachment = null) {
            try {
                // Show modal
                this.showModal();

                // Step 1: Call WordPress to prepare stream
                this.updateStage('🔄 Preparing generation...', 5);

                const requestBody = {
                    prompt: prompt,
                    page_type: pageType || 'custom',
                    business_info: businessInfo || {},
                    reference_page_id: referencePageId || null
                };

                // Add attachment if provided
                if (attachment) {
                    requestBody.attachment = attachment;
                }

                // Stream directly from WordPress (which proxies to Vercel with API key)
                this.updateStage('🧠 Analyzing your request...', 10);

                this.abortController = new AbortController();

                const response = await fetch(aiSiteBuilder.restUrl + 'ai-builder/v1/pages/generate-stream', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'text/event-stream',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    signal: this.abortController.signal,
                    body: JSON.stringify(requestBody)
                });

                if (!response.ok) {
                    throw new Error(`Streaming error: HTTP ${response.status}`);
                }

                // Step 3: Process SSE stream
                const reader = response.body.getReader();
                const decoder = new TextDecoder();
                let buffer = '';
                let result = null;

                while (true) {
                    const {done, value} = await reader.read();
                    if (done) break;

                    buffer += decoder.decode(value, {stream: true});
                    const events = buffer.split('\n\n');
                    buffer = events.pop();

                    for (const eventData of events) {
                        if (!eventData.trim()) continue;

                        const lines = eventData.split('\n');
                        let eventType = 'message';
                        let data = '';

                        for (const line of lines) {
                            if (line.startsWith('event:')) {
                                eventType = line.substring(6).trim();
                            } else if (line.startsWith('data:')) {
                                data = line.substring(5).trim();
                            }
                        }

                        if (eventType === 'stage') {
                            // Update progress stage
                            const progress = this.getProgressFromStage(data);
                            this.updateStage(data, progress);
                        } else if (eventType === 'content') {
                            // Live code streaming
                            this.appendLiveContent(data);
                        } else if (eventType === 'complete') {
                            // Generation complete
                            result = JSON.parse(data);
                            this.updateStage('✅ Page created successfully!', 100);
                        } else if (eventType === 'error') {
                            // Error occurred
                            throw new Error(data);
                        }
                    }
                }

                // Step 4: Create the page
                if (!result) {
                    throw new Error('No result received from stream');
                }

                await this.createPage(result, prompt, pageType, businessInfo);

            } catch (error) {
                console.error('Streaming generation error:', error);
                if (error.name === 'AbortError') {
                    this.showError('Generation cancelled');
                } else {
                    this.showError(error.message || 'Generation failed');
                }
            }
        }

        /**
         * Start polling-based generation (alternative to SSE streaming)
         */
        async startPolling(prompt, pageType, businessInfo, referencePageId, attachment = null) {
            try {
                // Show modal
                this.showModal();

                // Step 1: Create polling job
                this.updateStage('🔄 Preparing generation...', 5);

                const requestBody = {
                    prompt: prompt,
                    page_type: pageType || 'custom',
                    business_info: businessInfo || {},
                    reference_page_id: referencePageId || null
                };

                // Add attachment if provided
                if (attachment) {
                    requestBody.attachment = attachment;
                }

                const createResponse = await fetch(aiSiteBuilder.restUrl + 'ai-builder/v1/pages/generate-polling', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    body: JSON.stringify(requestBody)
                });

                const createData = await createResponse.json();

                if (!createData.success) {
                    throw new Error(createData.message || 'Failed to start generation');
                }

                const jobId = createData.job_id;

                // Step 2: Poll for status
                this.updateStage('🧠 Analyzing your request...', 10);

                let lastContent = '';
                let pollCount = 0;
                const maxPolls = 180; // 6 minutes max (180 * 2 seconds)

                const pollInterval = setInterval(async () => {
                    try {
                        pollCount++;

                        // Safety: Stop polling after max attempts
                        if (pollCount > maxPolls) {
                            clearInterval(pollInterval);
                            throw new Error('Generation timeout - please try again');
                        }

                        // Poll WordPress proxy (fixes vercelBaseUrl undefined issue)
                        const pollUrl = aiSiteBuilder.restUrl + 'ai-builder/v1/pages/poll-status/' + jobId;
                        const pollResponse = await fetch(pollUrl, {
                            method: 'GET',
                            headers: {
                                'Accept': 'application/json',
                                'X-WP-Nonce': aiSiteBuilder.nonce
                            }
                        });

                        if (!pollResponse.ok) {
                            throw new Error(`Poll failed: HTTP ${pollResponse.status}`);
                        }

                        const statusData = await pollResponse.json();

                        // Update UI
                        this.updateStage(statusData.stage, statusData.progress);

                        // Handle status
                        if (statusData.status === 'processing') {
                            // Animate new content appearing
                            if (statusData.content_so_far && statusData.content_so_far !== lastContent) {
                                const newContent = statusData.content_so_far.substring(lastContent.length);
                                this.appendLiveContent(newContent);
                                lastContent = statusData.content_so_far;
                            }
                        } else if (statusData.status === 'completed') {
                            // Stop polling
                            clearInterval(pollInterval);

                            // Create page from result
                            await this.createPage(statusData.result, prompt, pageType, businessInfo);

                        } else if (statusData.status === 'failed') {
                            // Stop polling
                            clearInterval(pollInterval);

                            // Show error
                            const errorMessage = statusData.error?.message || 'Generation failed';
                            this.showError(errorMessage);
                        }

                    } catch (error) {
                        console.error('Poll error:', error);
                        // Continue polling on transient errors
                    }
                }, 2000); // Poll every 2 seconds

            } catch (error) {
                console.error('Polling generation error:', error);
                if (error.name === 'AbortError') {
                    this.showError('Generation cancelled');
                } else {
                    this.showError(error.message || 'Generation failed');
                }
            }
        }

        /**
         * Create WordPress page from streaming result
         */
        async createPage(result, prompt, pageType, businessInfo) {
            try {
                const createResponse = await fetch(aiSiteBuilder.restUrl + 'ai-builder/v1/pages/create-from-stream', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    body: JSON.stringify({
                        result: result,
                        prompt: prompt,
                        page_type: pageType || 'custom',
                        business_info: businessInfo || {}
                    })
                });

                const createData = await createResponse.json();

                if (!createData.success) {
                    throw new Error(createData.message || 'Failed to create page');
                }

                // Show success
                this.showSuccess(createData.page_id, createData.dreamformer_url, createData.preview_url);

            } catch (error) {
                console.error('Page creation error:', error);
                this.showError(error.message || 'Failed to create page');
            }
        }

        /**
         * Get progress percentage from stage text
         */
        getProgressFromStage(stageText) {
            const progressMap = {
                '🔄': 5,   // Preparing
                '🧠': 15,  // Thinking
                '🔍': 20,  // Understanding
                '📝': 30,  // Writing metadata
                '✍️': 50,  // Creating HTML structure
                '🎨': 70,  // Styling with CSS
                '⚡': 90,  // Adding JavaScript
                '✨': 92,  // Polishing (old, fallback)
                '🔧': 97,  // Finalizing
                '✅': 100  // Complete
            };

            const emoji = stageText.charAt(0);
            return progressMap[emoji] || this.currentProgress + 5;
        }

        /**
         * Update stage and progress
         */
        updateStage(stageText, targetProgress) {
            // Update stage text
            if (this.stageText) {
                this.stageText.text(stageText);
            }

            // Animate progress bar
            if (targetProgress !== undefined) {
                this.animateProgress(targetProgress);
            }

            // Add pulse effect
            if (this.stageIcon) {
                this.stageIcon.addClass('pulse-once');
                setTimeout(() => {
                    this.stageIcon.removeClass('pulse-once');
                }, 300);
            }
        }

        /**
         * Animate progress bar to target
         */
        animateProgress(targetProgress) {
            if (targetProgress <= this.currentProgress) {
                return;
            }

            const step = () => {
                if (this.currentProgress >= targetProgress) {
                    return;
                }
                this.currentProgress++;
                if (this.progressBar) {
                    this.progressBar.css('width', this.currentProgress + '%');
                }
                $('.progress-percentage').text(this.currentProgress + '%');
                setTimeout(step, 20);
            };
            step();
        }

        /**
         * Append live content chunk to preview
         */
        appendLiveContent(chunk) {
            if (!this.liveCodePreview) {
                this.liveCodePreview = this.modal.find('.live-code-preview');
            }

            // Unescape newlines and append
            const unescaped = chunk.replace(/\\n/g, '\n').replace(/\\r/g, '\r');
            this.liveCodePreview.append(unescaped);

            // Auto-scroll to bottom
            this.liveCodePreview.scrollTop(this.liveCodePreview[0].scrollHeight);
        }

        /**
         * Show modal overlay
         */
        showModal() {
            // Create fullscreen Hostinger-style modal
            const modalHtml = `
                <div id="page-generation-modal" class="page-gen-overlay">
                    <!-- Top Status Bar -->
                    <div class="page-gen-topbar">
                        <div class="topbar-left">
                            <div class="logo-section">
                                <i data-lucide="sparkles" class="logo-icon"></i>
                                <span class="logo-text">Dreamformer</span>
                            </div>
                            <div class="stage-indicator">
                                <i data-lucide="loader-2" class="stage-icon animate-spin"></i>
                                <span class="stage-text">Initializing...</span>
                            </div>
                        </div>
                        <div class="topbar-right">
                            <div class="progress-info">
                                <span class="progress-percentage">0%</span>
                            </div>
                            <button class="page-gen-cancel" title="Cancel generation">
                                <i data-lucide="x"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Progress Bar (below topbar) -->
                    <div class="progress-container">
                        <div class="progress-bar"></div>
                    </div>

                    <!-- Main Content Area - Fullscreen Code Preview -->
                    <div class="page-gen-main">
                        <div class="code-preview-fullscreen">
                            <div class="code-preview-header">
                                <i data-lucide="code-2" class="icon-inline"></i>
                                <span>Generating Your Page</span>
                                <div class="pulse-indicator"></div>
                            </div>
                            <pre class="live-code-preview"></pre>
                        </div>
                    </div>

                    <!-- Success State (hidden initially, fullscreen) -->
                    <div class="page-gen-success" style="display: none;">
                        <button class="success-close-btn" title="Close">
                            <i data-lucide="x"></i>
                        </button>
                        <div class="success-content">
                            <div class="success-icon-wrapper">
                                <div class="success-icon-bg"></div>
                                <i data-lucide="check-circle" class="success-icon"></i>
                            </div>
                            <h3>Page Created Successfully!</h3>
                            <p class="success-subtitle">Your AI-generated page is ready to edit and publish</p>
                            <div class="success-actions">
                                <a href="#" class="btn btn-primary edit-page-btn">
                                    <i data-lucide="wand-2"></i>
                                    <span>Edit with AI</span>
                                </a>
                                <a href="#" class="btn btn-secondary view-page-btn" target="_blank">
                                    <i data-lucide="external-link"></i>
                                    <span>View Page</span>
                                </a>
                            </div>
                        </div>
                    </div>

                    <!-- Error State (hidden initially, fullscreen) -->
                    <div class="page-gen-error" style="display: none;">
                        <div class="error-content">
                            <div class="error-icon-wrapper">
                                <div class="error-icon-bg"></div>
                                <i data-lucide="alert-circle" class="error-icon"></i>
                            </div>
                            <h3>Generation Failed</h3>
                            <p class="error-message"></p>
                            <button class="btn btn-primary close-modal-btn">
                                <i data-lucide="arrow-left"></i>
                                <span>Back to Dashboard</span>
                            </button>
                        </div>
                    </div>
                </div>
            `;

            // Add to page
            $('body').append(modalHtml);

            // Initialize Lucide icons
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }

            // Store references
            this.modal = $('#page-generation-modal');
            this.progressBar = this.modal.find('.progress-bar');
            this.stageText = this.modal.find('.stage-text');
            this.stageIcon = this.modal.find('.stage-icon');

            // Wire up cancel button
            this.modal.find('.page-gen-cancel').on('click', () => {
                this.cancel();
            });

            this.modal.find('.close-modal-btn').on('click', () => {
                this.close();
            });

            this.modal.find('.success-close-btn').on('click', () => {
                this.close();
            });

            // Show with animation
            setTimeout(() => {
                this.modal.addClass('active');
            }, 10);
        }

        /**
         * Show success state
         */
        showSuccess(pageId, editUrl, previewUrl) {
            // Hide main content area and topbar
            this.modal.find('.page-gen-main').fadeOut(300);
            this.modal.find('.page-gen-topbar').fadeOut(300);
            this.modal.find('.progress-container').fadeOut(300);

            // Show success
            this.modal.find('.page-gen-success').fadeIn(300);

            // Wire up buttons
            this.modal.find('.edit-page-btn').attr('href', editUrl);
            this.modal.find('.view-page-btn').attr('href', previewUrl);

            // Trigger confetti
            if (window.DreamformerCelebrations) {
                setTimeout(() => {
                    window.DreamformerCelebrations.confetti({
                        count: 100,
                        duration: 3000
                    });
                }, 200);
            }

            // Reinitialize Lucide icons
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }
        }

        /**
         * Show error state
         */
        showError(message) {
            // Check if this is an insufficient credits error
            if (message && message.toLowerCase().includes('insufficient credits')) {
                // Close this modal and show upgrade modal
                this.close();

                if (typeof showDreamformerUpgradeModal === 'function') {
                    showDreamformerUpgradeModal({
                        title: 'Out of Credits',
                        message: message + ' Upgrade your plan to get more credits for AI page generation.',
                        features: [
                            {
                                icon: 'zap',
                                label: 'More Monthly Credits',
                                description: 'Generate more pages and edits each month'
                            },
                            {
                                icon: 'sparkles',
                                label: 'Page Generation',
                                description: 'Create complete AI-powered pages'
                            },
                            {
                                icon: 'palette',
                                label: 'Theme Editing',
                                description: 'Customize your theme with AI assistance'
                            }
                        ],
                        upgradeUrl: aiSiteBuilder.pricingUrl
                    });
                    return;
                }
            }

            // Hide main content area and topbar
            this.modal.find('.page-gen-main').fadeOut(300);
            this.modal.find('.page-gen-topbar').fadeOut(300);
            this.modal.find('.progress-container').fadeOut(300);

            // Show error
            const errorDiv = this.modal.find('.page-gen-error');
            errorDiv.find('.error-message').text(message);
            errorDiv.fadeIn(300);
        }

        /**
         * Cancel generation
         */
        cancel() {
            if (this.abortController) {
                this.abortController.abort();
            }
            this.close();
        }

        /**
         * Close modal
         */
        close() {
            if (this.modal) {
                this.modal.removeClass('active');
                setTimeout(() => {
                    this.modal.remove();
                }, 300);
            }
        }
    };

})(jQuery);
