/**
 * Manual CSS Property Controls for Dreamformer
 *
 * Provides interactive controls for editing CSS properties:
 * - Color picker for color properties
 * - Size control with unit selector
 * - Spacing control for margins/padding
 */

(function($) {
    'use strict';

    // Debug mode - set to true to enable console logging
    const DEBUG = false;

    /**
     * Debug logging helper
     * Logs only when DEBUG is enabled or window.dreamformerDebug is true
     */
    function log(...args) {
        if (DEBUG || (typeof window.dreamformerDebug !== 'undefined' && window.dreamformerDebug === true)) {
            log('', ...args);
        }
    }

class ManualPropertyControls {
    constructor(entityId, isPageMode = false) {
        this.entityId = entityId;
        this.isPageMode = isPageMode;
        this.currentRules = [];
        this.bindEvents();
    }

    /**
     * Display manual controls for CSS rules and text content
     * NEW LAYOUT: Compact header with state toggle, grouped property cards
     */
    displayControlsForRules(cssRules, textContent = [], responseData = null) {
        this.currentRules = cssRules;
        this.currentTextContent = textContent;
        this.currentResponseData = responseData;
        this.currentState = 'base'; // Track current state view

        const content = $('#manual-controls-content');
        content.empty();

        if ((!cssRules || cssRules.length === 0) && (!textContent || textContent.length === 0)) {
            content.html('<p class="no-controls">No editable properties found</p>');
            return;
        }

        // Find base selector and hover states
        let baseSelector = null;
        let highestSpecificity = 0;
        let hoverSelectors = [];

        if (cssRules && cssRules.length > 0) {
            cssRules.forEach(rule => {
                const isPseudo = rule.selector.includes(':');
                if (!isPseudo) {
                    const specificity = this.calculateSpecificity(rule.selector);
                    if (specificity > highestSpecificity) {
                        highestSpecificity = specificity;
                        baseSelector = rule.selector;
                    }
                }
            });

            if (baseSelector) {
                cssRules.forEach(rule => {
                    if (rule.selector.startsWith(baseSelector + ':')) {
                        hoverSelectors.push(rule.selector);
                    }
                });
            }
        }

        // Store for state switching
        this.baseSelector = baseSelector;
        this.hoverSelectors = hoverSelectors;
        this.selectorGroups = this.groupBySelector(cssRules);

        // Build new compact layout
        let html = '';

        // 1. Compact header with state toggle
        const hasHover = hoverSelectors.length > 0;
        html += `
            <div class="me-header">
                <div class="me-selector-info">
                    <span class="me-selector-badge">${this.formatSelectorName(baseSelector || 'element')}</span>
                    <code class="me-selector-code">${baseSelector || ''}</code>
                </div>
                ${hasHover ? `
                <div class="me-state-toggle">
                    <button class="me-state-btn active" data-state="base">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="3"/></svg>
                        Base
                    </button>
                    <button class="me-state-btn" data-state="hover">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2a3 3 0 0 0-3 3v7a3 3 0 0 0 6 0V5a3 3 0 0 0-3-3Z"/><path d="M19 10v2a7 7 0 0 1-14 0v-2"/></svg>
                        Hover
                    </button>
                </div>
                ` : ''}
            </div>
        `;

        // 2. Text content section (compact)
        if (textContent && textContent.length > 0) {
            const editableItems = textContent.filter(item => item.editable);
            if (editableItems.length > 0) {
                html += this.createCompactTextSection(editableItems);
            }
        }

        // 3. Property cards container (will be populated by state)
        html += `<div class="me-properties-container" id="me-properties-container"></div>`;

        // 4. Quick actions
        if (baseSelector) {
            html += this.createQuickActionsSection(baseSelector);
        }

        content.html(html);

        // Render properties for current state
        this.renderPropertiesForState('base');

        // Initialize controls
        this.initializeControls();
        this.initStateToggle();

        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    /**
     * Render properties for a specific state (base or hover)
     */
    renderPropertiesForState(state) {
        this.currentState = state;
        const container = $('#me-properties-container');
        container.empty();

        const selector = state === 'base' ? this.baseSelector : this.hoverSelectors[0];
        if (!selector || !this.selectorGroups[selector]) {
            container.html('<p class="me-no-props">No properties for this state</p>');
            return;
        }

        const allProps = this.selectorGroups[selector].properties || [];
        const props = this.filterEssentialProperties(allProps, state === 'hover');

        // Group properties by type
        const colorProps = props.filter(p => this.getPropertyType(p.property) === 'color');
        const sizeProps = props.filter(p => ['size', 'spacing'].includes(this.getPropertyType(p.property)));
        const otherProps = props.filter(p => !['color', 'size', 'spacing'].includes(this.getPropertyType(p.property)));

        let html = '<div class="me-cards-grid">';

        // Colors card
        if (colorProps.length > 0) {
            html += this.createPropertyCard('Colors', '🎨', colorProps, selector, 'colors');
        }

        // Sizes card
        if (sizeProps.length > 0) {
            html += this.createPropertyCard('Sizes', '📐', sizeProps, selector, 'sizes');
        }

        html += '</div>';

        // Other properties (full width)
        if (otherProps.length > 0) {
            html += this.createPropertyCard('Other', '⚙️', otherProps, selector, 'other');
        }

        // Add property button
        html += this.createCompactAddProperty(selector);

        container.html(html);

        // Reinitialize controls for new content
        this.storeOriginalValues();
        this.initializePropertyControls();

        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    /**
     * Create a property card for grouped properties
     */
    createPropertyCard(title, icon, properties, selector, cardType) {
        let html = `
            <div class="me-card me-card-${cardType}">
                <div class="me-card-header">
                    <span class="me-card-icon">${icon}</span>
                    <span class="me-card-title">${title}</span>
                </div>
                <div class="me-card-body">
        `;

        properties.forEach(prop => {
            html += this.createCompactPropertyControl(prop, selector, cardType);
        });

        html += `
                </div>
            </div>
        `;

        return html;
    }

    /**
     * Create compact property control (inline layout)
     */
    createCompactPropertyControl(prop, selector, cardType) {
        const propName = this.formatPropertyName(prop.property);
        const controlId = `prop-${selector}-${prop.property}`.replace(/[^a-zA-Z0-9-]/g, '-');
        const propType = this.getPropertyType(prop.property);

        let controlHtml = '';

        if (propType === 'color') {
            const colorValue = this.extractColorValue(prop.value);
            controlHtml = `
                <div class="me-prop-row me-prop-color" data-selector="${selector}" data-property="${prop.property}">
                    <label class="me-prop-label">${propName}</label>
                    <div class="me-prop-input-group">
                        <input type="color" class="me-color-picker" value="${colorValue}">
                        <input type="text" class="me-color-text" value="${prop.value}" data-original="${prop.value}">
                        <button class="me-apply-btn" title="Apply (Enter)">✓</button>
                    </div>
                </div>
            `;
        } else if (propType === 'size' || propType === 'spacing') {
            const sizeInfo = this.parseSizeValue(prop.value);
            controlHtml = `
                <div class="me-prop-row me-prop-size" data-selector="${selector}" data-property="${prop.property}">
                    <label class="me-prop-label">${propName}</label>
                    <div class="me-prop-input-group">
                        <input type="number" class="me-size-value" value="${sizeInfo.value}" min="0" step="1" data-original="${prop.value}">
                        <select class="me-size-unit">
                            <option value="px" ${sizeInfo.unit === 'px' ? 'selected' : ''}>px</option>
                            <option value="em" ${sizeInfo.unit === 'em' ? 'selected' : ''}>em</option>
                            <option value="rem" ${sizeInfo.unit === 'rem' ? 'selected' : ''}>rem</option>
                            <option value="%" ${sizeInfo.unit === '%' ? 'selected' : ''}>%</option>
                        </select>
                        <button class="me-apply-btn" title="Apply (Enter)">✓</button>
                    </div>
                </div>
            `;
        } else {
            controlHtml = `
                <div class="me-prop-row me-prop-text" data-selector="${selector}" data-property="${prop.property}">
                    <label class="me-prop-label">${propName}</label>
                    <div class="me-prop-input-group">
                        <input type="text" class="me-text-value" value="${prop.value}" data-original="${prop.value}">
                        <button class="me-apply-btn" title="Apply (Enter)">✓</button>
                    </div>
                </div>
            `;
        }

        return controlHtml;
    }

    /**
     * Create compact text section
     */
    createCompactTextSection(textItems) {
        let html = `
            <div class="me-text-section">
                <div class="me-section-header">
                    <span class="me-section-icon">✏️</span>
                    <span class="me-section-title">Text Content</span>
                </div>
        `;

        textItems.forEach(item => {
            const truncated = item.content.length > 50 ? item.content.substring(0, 50) + '...' : item.content;
            html += `
                <div class="me-text-row" data-selector="${item.selector}" data-match-index="${item.match_index}" data-element-type="${item.element_type}">
                    <span class="me-text-badge">${this.getElementDisplayName(item.element_type)}</span>
                    <div class="me-text-preview">${truncated}</div>
                    <button class="me-text-edit-btn" title="Edit text">✏️</button>
                </div>
                <div class="me-text-editor" style="display: none;">
                    <textarea class="me-text-input" rows="2">${item.content}</textarea>
                    <div class="me-text-actions">
                        <button class="me-text-save">Save</button>
                        <button class="me-text-cancel">Cancel</button>
                    </div>
                </div>
            `;
        });

        html += '</div>';
        return html;
    }

    /**
     * Create quick actions section
     */
    createQuickActionsSection(selector) {
        return `
            <div class="me-quick-actions">
                <div class="me-section-header">
                    <span class="me-section-icon">⚡</span>
                    <span class="me-section-title">Quick Actions</span>
                </div>
                <div class="me-actions-grid">
                    <button class="me-action-btn" data-action="center" data-selector="${selector}" title="Center content">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="21" x2="3" y1="6" y2="6"/><line x1="17" x2="7" y1="12" y2="12"/><line x1="19" x2="5" y1="18" y2="18"/></svg>
                        Center
                    </button>
                    <button class="me-action-btn" data-action="shadow" data-selector="${selector}" title="Add shadow">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect width="18" height="18" x="3" y="3" rx="2"/></svg>
                        Shadow
                    </button>
                    <button class="me-action-btn" data-action="rounded" data-selector="${selector}" title="Round corners">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect width="18" height="18" x="3" y="3" rx="5"/></svg>
                        Rounded
                    </button>
                    <button class="me-action-btn" data-action="bold" data-selector="${selector}" title="Make bold">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M6 12h9a4 4 0 0 1 0 8H7a1 1 0 0 1-1-1V5a1 1 0 0 1 1-1h7a4 4 0 0 1 0 8"/></svg>
                        Bold
                    </button>
                </div>
            </div>
        `;
    }

    /**
     * Create compact add property section
     */
    createCompactAddProperty(selector) {
        const commonProperties = [
            { name: 'color', label: 'Text Color', default: '#000000' },
            { name: 'background-color', label: 'Background', default: '#ffffff' },
            { name: 'font-size', label: 'Font Size', default: '16px' },
            { name: 'padding', label: 'Padding', default: '10px' },
            { name: 'margin', label: 'Margin', default: '0' },
            { name: 'border-radius', label: 'Radius', default: '0' }
        ];

        let options = commonProperties.map(p =>
            `<option value="${p.name}" data-default="${p.default}">${p.label}</option>`
        ).join('');

        return `
            <div class="me-add-property" data-selector="${selector}">
                <select class="me-add-select">
                    <option value="">+ Add property...</option>
                    ${options}
                </select>
            </div>
        `;
    }

    /**
     * Initialize state toggle handlers
     */
    initStateToggle() {
        $('.me-state-btn').off('click').on('click', (e) => {
            const btn = $(e.currentTarget);
            const state = btn.data('state');

            $('.me-state-btn').removeClass('active');
            btn.addClass('active');

            this.renderPropertiesForState(state);
        });
    }

    /**
     * Initialize property controls in new layout
     */
    initializePropertyControls() {
        // Color picker sync
        $('.me-color-picker').off('input').on('input', (e) => {
            const picker = $(e.target);
            picker.siblings('.me-color-text').val(picker.val());
            this.triggerLivePreview(picker.closest('.me-prop-row'));
        });

        $('.me-color-text').off('input').on('input', (e) => {
            const text = $(e.target);
            const colorValue = this.extractColorValue(text.val());
            if (colorValue) {
                text.siblings('.me-color-picker').val(colorValue);
            }
            this.triggerLivePreview(text.closest('.me-prop-row'));
        });

        // Size input changes
        $('.me-size-value, .me-size-unit').off('input change').on('input change', (e) => {
            this.triggerLivePreview($(e.target).closest('.me-prop-row'));
        });

        // Text input changes
        $('.me-text-value').off('input').on('input', (e) => {
            this.triggerLivePreview($(e.target).closest('.me-prop-row'));
        });

        // Apply button clicks
        $('.me-apply-btn').off('click').on('click', (e) => {
            const row = $(e.target).closest('.me-prop-row');
            this.applyPropertyFromRow(row);
        });

        // Keyboard shortcuts
        $('.me-prop-row input, .me-prop-row select').off('keydown').on('keydown', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                this.applyPropertyFromRow($(e.target).closest('.me-prop-row'));
            } else if (e.key === 'Escape') {
                e.preventDefault();
                this.resetPropertyRow($(e.target).closest('.me-prop-row'));
            }
        });

        // Text edit toggle
        $('.me-text-edit-btn').off('click').on('click', (e) => {
            const row = $(e.target).closest('.me-text-row');
            row.next('.me-text-editor').slideToggle(200);
        });

        // Text save/cancel
        $('.me-text-save').off('click').on('click', (e) => {
            const editor = $(e.target).closest('.me-text-editor');
            const row = editor.prev('.me-text-row');
            this.applyTextFromEditor(row, editor);
        });

        $('.me-text-cancel').off('click').on('click', (e) => {
            $(e.target).closest('.me-text-editor').slideUp(200);
        });

        // Quick actions
        $('.me-action-btn').off('click').on('click', (e) => {
            const btn = $(e.target).closest('.me-action-btn');
            const action = btn.data('action');
            const selector = btn.data('selector');
            this.applyQuickAction(action, selector);
        });

        // Add property select
        $('.me-add-select').off('change').on('change', (e) => {
            const select = $(e.target);
            const property = select.val();
            if (property) {
                const defaultValue = select.find('option:selected').data('default');
                const selector = select.closest('.me-add-property').data('selector');
                this.addNewProperty(selector, property, defaultValue);
                select.val('');
            }
        });
    }

    /**
     * Trigger live preview for a property row
     */
    triggerLivePreview(row) {
        const selector = row.data('selector');
        const property = row.data('property');
        const value = this.getValueFromRow(row);

        row.addClass('me-modified');
        this.updatePreviewStyles(selector, property, value);
    }

    /**
     * Get current value from a property row
     */
    getValueFromRow(row) {
        if (row.hasClass('me-prop-color')) {
            return row.find('.me-color-text').val();
        } else if (row.hasClass('me-prop-size')) {
            return row.find('.me-size-value').val() + row.find('.me-size-unit').val();
        } else {
            return row.find('.me-text-value').val();
        }
    }

    /**
     * Apply property from row
     */
    async applyPropertyFromRow(row) {
        const selector = row.data('selector');
        const property = row.data('property');
        const value = this.getValueFromRow(row);

        const btn = row.find('.me-apply-btn');
        btn.text('...').prop('disabled', true);

        try {
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({ selector, property, value })
            });

            const result = await response.json();

            if (result.success) {
                btn.text('✓').addClass('success');
                row.removeClass('me-modified');
                row.find('[data-original]').attr('data-original', value);

                setTimeout(() => {
                    btn.text('✓').removeClass('success').prop('disabled', false);
                }, 1000);

                // Update preview iframe immediately via postMessage
                this.updatePreviewStyles(selector, property, value);

                // Glow refresh button to indicate changes
                this.glowRefreshButton();

                // Update version control buttons
                if (window.aiThemeEditor?.updateVersionControlButtons) {
                    window.aiThemeEditor.updateVersionControlButtons();
                }

                // Refresh preview with scroll position preserved
                if (window.aiThemeEditor?.forcePreviewRefresh) {
                    const scrollPos = await this.saveIframeScrollPosition();
                    window.aiThemeEditor.forcePreviewRefresh();
                    this.restoreIframeScrollPosition(scrollPos);
                }
            } else {
                throw new Error(result.message);
            }
        } catch (error) {
            console.error('Property update failed:', error);
            btn.text('✗').addClass('error');
            setTimeout(() => btn.text('✓').removeClass('error').prop('disabled', false), 2000);
        }
    }

    /**
     * Reset property row to original value
     */
    resetPropertyRow(row) {
        const original = row.find('[data-original]').attr('data-original');
        if (!original) return;

        if (row.hasClass('me-prop-color')) {
            row.find('.me-color-text').val(original);
            row.find('.me-color-picker').val(this.extractColorValue(original));
        } else if (row.hasClass('me-prop-size')) {
            const parsed = this.parseSizeValue(original);
            row.find('.me-size-value').val(parsed.value);
            row.find('.me-size-unit').val(parsed.unit);
        } else {
            row.find('.me-text-value').val(original);
        }

        row.removeClass('me-modified');
        const selector = row.data('selector');
        const property = row.data('property');
        this.updatePreviewStyles(selector, property, original);
    }

    /**
     * Apply text content from editor
     */
    async applyTextFromEditor(row, editor) {
        const selector = row.data('selector');
        const matchIndex = row.data('match-index');
        const elementType = row.data('element-type');
        const value = editor.find('.me-text-input').val();

        try {
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    selector,
                    property: 'text-content',
                    value,
                    match_index: matchIndex,
                    element_type: elementType
                })
            });

            const result = await response.json();

            if (result.success) {
                row.find('.me-text-preview').text(value.length > 50 ? value.substring(0, 50) + '...' : value);
                editor.slideUp(200);

                if (window.aiThemeEditor?.forcePreviewRefresh) {
                    window.aiThemeEditor.forcePreviewRefresh();
                }
            }
        } catch (error) {
            console.error('Text update failed:', error);
        }
    }

    /**
     * Apply quick action
     */
    async applyQuickAction(action, selector) {
        const actionStyles = {
            center: { property: 'text-align', value: 'center' },
            shadow: { property: 'box-shadow', value: '0 4px 12px rgba(0,0,0,0.15)' },
            rounded: { property: 'border-radius', value: '12px' },
            bold: { property: 'font-weight', value: '700' }
        };

        const style = actionStyles[action];
        if (!style) return;

        // Show loading state on button
        const btn = $(`.me-action-btn[data-action="${action}"]`);
        btn.addClass('loading');

        try {
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    selector,
                    property: style.property,
                    value: style.value
                })
            });

            const result = await response.json();

            if (result.success) {
                // Update preview immediately
                this.updatePreviewStyles(selector, style.property, style.value);

                // Glow refresh button
                this.glowRefreshButton();

                // Update version control buttons
                if (window.aiThemeEditor?.updateVersionControlButtons) {
                    window.aiThemeEditor.updateVersionControlButtons();
                }

                // Refresh preview with scroll position preserved
                if (window.aiThemeEditor?.forcePreviewRefresh) {
                    const scrollPos = await this.saveIframeScrollPosition();
                    window.aiThemeEditor.forcePreviewRefresh();
                    this.restoreIframeScrollPosition(scrollPos);
                }

                // Re-render to show new property
                this.renderPropertiesForState(this.currentState);

                btn.removeClass('loading').addClass('success');
                setTimeout(() => btn.removeClass('success'), 1000);
            }
        } catch (error) {
            console.error('Quick action failed:', error);
            btn.removeClass('loading').addClass('error');
            setTimeout(() => btn.removeClass('error'), 2000);
        }
    }

    /**
     * Add new property
     */
    async addNewProperty(selector, property, value) {
        try {
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({ selector, property, value })
            });

            const result = await response.json();

            if (result.success) {
                // Update preview immediately
                this.updatePreviewStyles(selector, property, value);

                // Glow refresh button
                this.glowRefreshButton();

                // Update version control buttons
                if (window.aiThemeEditor?.updateVersionControlButtons) {
                    window.aiThemeEditor.updateVersionControlButtons();
                }

                // Refresh preview with scroll position preserved
                if (window.aiThemeEditor?.forcePreviewRefresh) {
                    const scrollPos = await this.saveIframeScrollPosition();
                    window.aiThemeEditor.forcePreviewRefresh();
                    this.restoreIframeScrollPosition(scrollPos);
                }

                // Re-render to show new property in cards
                this.renderPropertiesForState(this.currentState);
            }
        } catch (error) {
            console.error('Add property failed:', error);
        }
    }

    /**
     * Get property type for grouping
     */
    getPropertyType(property) {
        const colorProps = ['color', 'background', 'background-color', 'border-color', 'outline-color'];
        const sizeProps = ['font-size', 'width', 'height', 'max-width', 'min-width', 'border-radius'];
        const spacingProps = ['padding', 'margin', 'padding-top', 'padding-bottom', 'padding-left', 'padding-right',
                             'margin-top', 'margin-bottom', 'margin-left', 'margin-right', 'gap'];

        if (colorProps.includes(property)) return 'color';
        if (sizeProps.includes(property)) return 'size';
        if (spacingProps.includes(property)) return 'spacing';
        return 'text';
    }

    /**
     * Create text content editing group - PROMINENT UI
     */
    createTextContentGroup(textContent) {
        // Filter for editable items only
        const editableItems = textContent.filter(item => item.editable);

        // Don't create wrapper if no editable items
        if (editableItems.length === 0) {
            return '';
        }

        let groupHtml = `
            <div class="text-content-section">
                <div class="text-content-header">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 3a2.85 2.83 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5Z"></path><path d="m15 5 4 4"></path></svg>
                    <h4>Edit Text</h4>
                </div>
                <div class="text-content-controls">
        `;

        editableItems.forEach(textItem => {
            groupHtml += this.createTextContentControl(textItem);
        });

        groupHtml += '</div></div>';
        return groupHtml;
    }

    /**
     * Create individual text content control - SIMPLIFIED
     */
    createTextContentControl(textItem) {
        const controlId = `text-control-${textItem.selector}-${textItem.match_index}`.replace(/[^a-zA-Z0-9-]/g, '-');
        const elementLabel = this.getElementDisplayName(textItem.element_type);

        return `
            <div class="text-content-control-simple" data-selector="${textItem.selector}" data-match-index="${textItem.match_index}" data-element-type="${textItem.element_type}">
                <label class="text-content-label" for="${controlId}">
                    <span class="element-type-badge">${elementLabel}</span>
                    <code class="text-selector">${textItem.selector}</code>
                </label>
                <div class="text-edit-wrapper">
                    <textarea id="${controlId}" class="text-content-input-simple" rows="3" placeholder="Type your text here...">${textItem.content}</textarea>
                    <button type="button" class="apply-text-btn-simple" data-control-id="${controlId}">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>
                        Update Text
                    </button>
                </div>
            </div>
        `;
    }

    /**
     * Get display name for element type
     */
    getElementDisplayName(elementType) {
        const displayNames = {
            h1: 'Heading 1',
            h2: 'Heading 2',
            h3: 'Heading 3',
            h4: 'Heading 4',
            h5: 'Heading 5',
            h6: 'Heading 6',
            p: 'Paragraph',
            button: 'Button',
            a: 'Link',
            span: 'Text',
            div: 'Text Block',
            li: 'List Item'
        };
        return displayNames[elementType] || 'Text';
    }

    /**
     * Group CSS properties by type for better organization
     */
    groupPropertiesByType(cssRules) {
        const groups = {
            colors: [],
            typography: [],
            spacing: [],
            layout: []
        };

        cssRules.forEach(rule => {
            if (rule.properties) {
                Object.keys(rule.properties).forEach(propName => {
                    const propData = rule.properties[propName];
                    const groupName = this.getPropertyGroup(propName, propData.type);

                    groups[groupName].push({
                        selector: rule.selector,
                        file: rule.file,
                        property: propName,
                        value: propData.value,
                        type: propData.type,
                        parsed: propData.parsed_value,
                        important: propData.important
                    });
                });
            }
        });

        // Remove empty groups
        Object.keys(groups).forEach(key => {
            if (groups[key].length === 0) {
                delete groups[key];
            }
        });

        return groups;
    }

    /**
     * NEW: Group CSS rules by selector for better UX
     * Groups all properties for each selector together
     */
    groupBySelector(cssRules) {
        const selectorGroups = {};

        cssRules.forEach(rule => {
            const selector = rule.selector;

            if (!selectorGroups[selector]) {
                selectorGroups[selector] = {
                    selector: selector,
                    file: rule.file,
                    properties: []
                };
            }

            // Add all properties for this selector
            if (rule.properties) {
                Object.keys(rule.properties).forEach(propName => {
                    const propData = rule.properties[propName];
                    selectorGroups[selector].properties.push({
                        property: propName,
                        value: propData.value,
                        type: propData.type,
                        parsed: propData.parsed_value,
                        important: propData.important
                    });
                });
            }
        });

        return selectorGroups;
    }
    
    /**
     * Determine which group a property belongs to
     */
    getPropertyGroup(propertyName, propertyType) {
        if (propertyType === 'color' || propertyName.includes('color')) {
            return 'colors';
        }
        
        if (propertyType === 'typography' || 
            ['font-size', 'font-weight', 'font-family', 'line-height'].includes(propertyName)) {
            return 'typography';
        }
        
        if (propertyType === 'spacing' || 
            propertyName.includes('margin') || propertyName.includes('padding')) {
            return 'spacing';
        }
        
        return 'layout';
    }
    
    /**
     * Create HTML for a property group
     */
    createPropertyGroup(groupName, properties) {
        const groupTitle = this.getGroupTitle(groupName);
        let groupHtml = `
            <div class="property-group" data-group="${groupName}">
                <h4 class="property-group-title">${groupTitle}</h4>
                <div class="property-controls">
        `;
        
        properties.forEach(prop => {
            groupHtml += this.createPropertyControl(prop);
        });
        
        groupHtml += '</div></div>';
        return groupHtml;
    }
    
    /**
     * Get display title for property group
     */
    getGroupTitle(groupName) {
        const titles = {
            colors: '<i data-lucide="palette" class="icon-inline icon-sm icon-primary"></i> Colors',
            typography: '<i data-lucide="type" class="icon-inline icon-sm icon-info"></i> Typography',
            spacing: '<i data-lucide="spacing-horizontal" class="icon-inline icon-sm icon-secondary"></i> Spacing',
            layout: '<i data-lucide="layout" class="icon-inline icon-sm icon-success"></i> Layout'
        };
        return titles[groupName] || groupName;
    }
    
    /**
     * Create control for individual property
     */
    createPropertyControl(prop) {
        const controlId = `control-${prop.selector}-${prop.property}`.replace(/[^a-zA-Z0-9-]/g, '-');
        
        let controlHtml = `
            <div class="property-control" data-selector="${prop.selector}" data-property="${prop.property}" data-file="${prop.file}">
                <label class="property-label" for="${controlId}">
                    ${this.formatPropertyName(prop.property)}
                    <span class="property-selector">${prop.selector}</span>
                </label>
        `;
        
        // Create appropriate input based on property type
        switch (prop.type) {
            case 'color':
                controlHtml += this.createColorControl(controlId, prop);
                break;
            case 'size':
                controlHtml += this.createSizeControl(controlId, prop);
                break;
            case 'spacing':
                controlHtml += this.createSpacingControl(controlId, prop);
                break;
            default:
                controlHtml += this.createTextControl(controlId, prop);
        }
        
        controlHtml += '</div>';
        return controlHtml;
    }
    
    /**
     * Create color picker control
     */
    createColorControl(controlId, prop) {
        const colorValue = this.extractColorValue(prop.value);
        return `
            <div class="color-control">
                <input type="color" id="${controlId}" value="${colorValue}" class="color-picker" title="Pick a color">
                <input type="text" value="${prop.value}" class="color-text" placeholder="Enter color value" data-original="${prop.value}">
                <div class="control-actions">
                    <button type="button" class="reset-property-btn" title="Reset (Esc)">↺</button>
                    <button type="button" class="apply-property-btn" data-property-type="color" title="Apply (Enter)">Apply</button>
                </div>
                <span class="original-value" title="Original value">was: ${prop.value}</span>
            </div>
        `;
    }

    /**
     * Create size control with units
     */
    createSizeControl(controlId, prop) {
        const sizeInfo = this.parseSizeValue(prop.value);
        return `
            <div class="size-control">
                <input type="number" id="${controlId}" value="${sizeInfo.value}" class="size-value" min="0" step="0.1" data-original="${prop.value}">
                <select class="size-unit">
                    <option value="px" ${sizeInfo.unit === 'px' ? 'selected' : ''}>px</option>
                    <option value="em" ${sizeInfo.unit === 'em' ? 'selected' : ''}>em</option>
                    <option value="rem" ${sizeInfo.unit === 'rem' ? 'selected' : ''}>rem</option>
                    <option value="%" ${sizeInfo.unit === '%' ? 'selected' : ''}>%</option>
                    <option value="vw" ${sizeInfo.unit === 'vw' ? 'selected' : ''}>vw</option>
                    <option value="vh" ${sizeInfo.unit === 'vh' ? 'selected' : ''}>vh</option>
                </select>
                <div class="control-actions">
                    <button type="button" class="reset-property-btn" title="Reset (Esc)">↺</button>
                    <button type="button" class="apply-property-btn" data-property-type="size" title="Apply (Enter)">Apply</button>
                </div>
                <span class="original-value" title="Original value">was: ${prop.value}</span>
            </div>
        `;
    }

    /**
     * Create spacing control
     */
    createSpacingControl(controlId, prop) {
        return `
            <div class="spacing-control">
                <input type="text" id="${controlId}" value="${prop.value}" class="spacing-value" placeholder="e.g., 10px 5px" data-original="${prop.value}">
                <div class="spacing-presets">
                    <button type="button" class="preset-btn" data-value="0">None</button>
                    <button type="button" class="preset-btn" data-value="5px">XS</button>
                    <button type="button" class="preset-btn" data-value="10px">S</button>
                    <button type="button" class="preset-btn" data-value="20px">M</button>
                    <button type="button" class="preset-btn" data-value="30px">L</button>
                </div>
                <div class="control-actions">
                    <button type="button" class="reset-property-btn" title="Reset (Esc)">↺</button>
                    <button type="button" class="apply-property-btn" data-property-type="spacing" title="Apply (Enter)">Apply</button>
                </div>
                <span class="original-value" title="Original value">was: ${prop.value}</span>
            </div>
        `;
    }

    /**
     * Create text control for other properties
     */
    createTextControl(controlId, prop) {
        return `
            <div class="text-control">
                <input type="text" id="${controlId}" value="${prop.value}" class="text-value" data-original="${prop.value}">
                <div class="control-actions">
                    <button type="button" class="reset-property-btn" title="Reset (Esc)">↺</button>
                    <button type="button" class="apply-property-btn" data-property-type="text" title="Apply (Enter)">Apply</button>
                </div>
                <span class="original-value" title="Original value">was: ${prop.value}</span>
            </div>
        `;
    }
    
    /**
     * Initialize control interactions
     */
    initializeControls() {
        // Store original values for reset functionality
        this.storeOriginalValues();

        // Apply button clicks for CSS properties
        $('.apply-property-btn').off('click').on('click', (e) => {
            const btn = $(e.target);
            const control = btn.closest('.property-control');
            const propertyType = btn.data('property-type');

            log('Apply button clicked:', {
                isDisabled: btn.prop('disabled'),
                buttonText: btn.text(),
                selector: control.data('selector'),
                property: control.data('property')
            });

            this.applyPropertyChange(control, propertyType);
        });

        // Reset button clicks
        $('.reset-property-btn').off('click').on('click', (e) => {
            e.preventDefault();
            const btn = $(e.target).closest('.reset-property-btn');
            const control = btn.closest('.property-control');
            this.resetToOriginal(control);
        });

        // KEYBOARD SHORTCUTS: Enter to apply, Escape to reset
        $('.property-control input, .property-control select').off('keydown').on('keydown', (e) => {
            const control = $(e.target).closest('.property-control');

            if (e.key === 'Enter') {
                e.preventDefault();
                const propertyType = control.find('.apply-property-btn').data('property-type');
                this.applyPropertyChange(control, propertyType);
            } else if (e.key === 'Escape') {
                e.preventDefault();
                this.resetToOriginal(control);
                $(e.target).blur();
            }
        });

        // LIVE PREVIEW: Debounced preview while editing
        let livePreviewTimeout = null;
        $('.property-control input, .property-control select').off('input change').on('input change', (e) => {
            const control = $(e.target).closest('.property-control');
            const selector = control.data('selector');
            const property = control.data('property');

            // Get current value based on control type
            const value = this.getControlValue(control);

            // Mark as modified
            control.addClass('modified');

            // Debounced live preview (doesn't save, just shows in iframe)
            clearTimeout(livePreviewTimeout);
            livePreviewTimeout = setTimeout(() => {
                this.updatePreviewStyles(selector, property, value);
            }, 150);
        });

        // Apply button clicks for text content
        $('.apply-text-btn, .apply-text-btn-simple').off('click').on('click', (e) => {
            const btn = $(e.target).closest('button');
            const control = btn.closest('.text-content-control, .text-content-control-simple');

            this.applyTextContentChange(control);
        });

        // Text content: Enter (with Ctrl/Cmd) to apply, Escape to reset
        $('.text-content-input, .text-content-input-simple').off('keydown').on('keydown', (e) => {
            const control = $(e.target).closest('.text-content-control, .text-content-control-simple');

            if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) {
                e.preventDefault();
                this.applyTextContentChange(control);
            } else if (e.key === 'Escape') {
                e.preventDefault();
                this.resetTextToOriginal(control);
                $(e.target).blur();
            }
        });

        // Preset button clicks for spacing
        $('.preset-btn').off('click').on('click', (e) => {
            const btn = $(e.target);
            const value = btn.data('value');
            const control = btn.closest('.property-control');
            const input = btn.closest('.spacing-control').find('.spacing-value');
            input.val(value);

            // Trigger live preview
            const selector = control.data('selector');
            const property = control.data('property');
            this.updatePreviewStyles(selector, property, value);
            control.addClass('modified');
        });

        // Color picker changes - with live preview
        $('.color-picker').off('input change').on('input change', (e) => {
            const colorPicker = $(e.target);
            const textInput = colorPicker.siblings('.color-text');
            textInput.val(colorPicker.val());

            // Live preview
            const control = colorPicker.closest('.property-control');
            const selector = control.data('selector');
            const property = control.data('property');
            this.updatePreviewStyles(selector, property, colorPicker.val());
            control.addClass('modified');
        });

        // Color text changes
        $('.color-text').off('input').on('input', (e) => {
            const textInput = $(e.target);
            const value = textInput.val();
            const colorValue = this.extractColorValue(value);
            if (colorValue) {
                textInput.siblings('.color-picker').val(colorValue);
            }
        });

        // NEW: Collapsible selector groups
        $('.selector-header').off('click').on('click', (e) => {
            const header = $(e.currentTarget);
            const group = header.closest('.selector-group');
            const properties = group.find('.selector-properties').first();
            const icon = header.find('.collapse-icon');

            if (group.hasClass('expanded')) {
                // Collapse
                group.removeClass('expanded').addClass('collapsed');
                properties.slideUp(200);
                icon.removeClass('rotated');
            } else {
                // Expand
                group.removeClass('collapsed').addClass('expanded');
                properties.slideDown(200);
                icon.addClass('rotated');

                // Re-init Lucide icons for newly visible elements
                if (typeof lucide !== 'undefined') {
                    lucide.createIcons();
                }
            }
        });

        // NEW: Toggle advanced properties
        $('.toggle-advanced-btn').off('click').on('click', (e) => {
            e.stopPropagation(); // Don't trigger selector header click
            const btn = $(e.currentTarget);
            const toggle = btn.closest('.advanced-properties-toggle');
            const advanced = toggle.next('.advanced-properties');
            const icon = btn.find('.collapse-icon');

            if (toggle.hasClass('collapsed')) {
                // Show advanced
                toggle.removeClass('collapsed').addClass('expanded');
                advanced.slideDown(200);
                icon.addClass('rotated');
                btn.html('<i data-lucide="chevron-down" class="collapse-icon rotated"></i> Hide Advanced');
                if (typeof lucide !== 'undefined') {
                    lucide.createIcons();
                }
            } else {
                // Hide advanced
                toggle.removeClass('expanded').addClass('collapsed');
                advanced.slideUp(200);
                icon.removeClass('rotated');
                const count = advanced.find('.property-control').length;
                btn.html('<i data-lucide="chevron-right" class="collapse-icon"></i> Show Advanced (' + count + ' more)');
                if (typeof lucide !== 'undefined') {
                    lucide.createIcons();
                }
            }
        });

        // Initialize Add Property handlers
        this.initAddPropertyHandlers();
    }

    /**
     * Apply property change to CSS
     */
    async applyPropertyChange(control, propertyType) {
        const selector = control.data('selector');
        const property = control.data('property');
        const file = control.data('file');
        
        let value = '';
        
        // Get value based on control type
        switch (propertyType) {
            case 'color':
                value = control.find('.color-text').val();
                break;
            case 'size':
                const sizeVal = control.find('.size-value').val();
                const sizeUnit = control.find('.size-unit').val();
                value = sizeVal + sizeUnit;
                break;
            case 'spacing':
                value = control.find('.spacing-value').val();
                break;
            default:
                value = control.find('.text-value').val();
        }
        
        if (!value) {
            alert('Please enter a value');
            return;
        }
        
        // Show loading state
        const applyBtn = control.find('.apply-property-btn');
        const originalText = applyBtn.text();
        applyBtn.text('Applying...').prop('disabled', true);
        
        try {
            // Call REST API to update property - use correct endpoint based on mode
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    selector: selector,
                    property: property,
                    value: value,
                    file: file
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Success feedback
                applyBtn.text('✓ Applied').addClass('success');

                // Update preview iframe if available
                this.updatePreviewStyles(selector, property, value);

                // Make refresh button glow to indicate changes need preview refresh
                this.glowRefreshButton();

                // Update version control buttons (same as AI editor does)
                if (window.aiThemeEditor && typeof window.aiThemeEditor.updateVersionControlButtons === 'function') {
                    window.aiThemeEditor.updateVersionControlButtons();
                }

                // Refresh preview with scroll position preserved
                if (window.aiThemeEditor && typeof window.aiThemeEditor.forcePreviewRefresh === 'function') {
                    // Save scroll position before refresh
                    const scrollPos = await this.saveIframeScrollPosition();
                    log('Saved scroll position before CSS refresh:', scrollPos);

                    // Refresh the iframe (ensure changes are visible)
                    window.aiThemeEditor.forcePreviewRefresh();

                    // Restore scroll position (method handles timing internally via handler-loaded message)
                    this.restoreIframeScrollPosition(scrollPos);
                }

                // Reset button after delay
                setTimeout(() => {
                    applyBtn.text(originalText).removeClass('success').prop('disabled', false);
                }, 2000);
                
            } else {
                throw new Error(result.message || 'Update failed');
            }
            
        } catch (error) {
            console.error('Property update failed:', error);
            applyBtn.text('✗ Failed').addClass('error');
            
            // Reset button after delay
            setTimeout(() => {
                applyBtn.text(originalText).removeClass('error').prop('disabled', false);
            }, 3000);
            
            alert('Failed to update property: ' + error.message);
        }
    }

    /**
     * Apply text content change
     */
    async applyTextContentChange(control) {
        const selector = control.data('selector');
        const matchIndex = control.data('match-index');
        const elementType = control.data('element-type');
        const newText = control.find('.text-content-input, .text-content-input-simple').val();

        if (!newText.trim()) {
            alert('Please enter some text content');
            return;
        }

        // Show loading state
        const applyBtn = control.find('.apply-text-btn, .apply-text-btn-simple');
        const originalText = applyBtn.html();
        applyBtn.html('<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="spin"><path d="M21 12a9 9 0 1 1-6.219-8.56"></path></svg> Updating...').prop('disabled', true);

        try {
            // Call REST API to update text content
            const endpoint = this.isPageMode ? 'pages' : 'themes';
            const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': aiSiteBuilder.nonce
                },
                body: JSON.stringify({
                    selector: selector,
                    property: 'text-content',
                    value: newText,
                    match_index: matchIndex,
                    element_type: elementType
                })
            });

            const result = await response.json();

            if (result.success) {
                applyBtn.html('<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg> Updated!').addClass('success');

                // Make refresh button glow to indicate changes need preview refresh
                this.glowRefreshButton();

                // Update version control buttons (same as AI editor does)
                if (window.aiThemeEditor && typeof window.aiThemeEditor.updateVersionControlButtons === 'function') {
                    window.aiThemeEditor.updateVersionControlButtons();
                }

                // Text changes require full refresh, but preserve scroll position
                if (window.aiThemeEditor && typeof window.aiThemeEditor.forcePreviewRefresh === 'function') {
                    // Save scroll position before refresh
                    const scrollPos = await this.saveIframeScrollPosition();
                    log('Saved scroll position before text update refresh:', scrollPos);

                    // Refresh the iframe
                    window.aiThemeEditor.forcePreviewRefresh();

                    // Restore scroll position (method handles timing internally via handler-loaded message)
                    this.restoreIframeScrollPosition(scrollPos);
                }

                // Reset button after delay
                setTimeout(() => {
                    applyBtn.html(originalText).removeClass('success').prop('disabled', false);
                }, 2000);

            } else {
                throw new Error(result.message || 'Update failed');
            }

        } catch (error) {
            console.error('Text update failed:', error);
            applyBtn.html('<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="15" x2="9" y1="9" y2="15"></line><line x1="9" x2="15" y1="9" y2="15"></line></svg> Failed').addClass('error');

            // Reset button after delay
            setTimeout(() => {
                applyBtn.html(originalText).removeClass('error').prop('disabled', false);
            }, 3000);

            alert('Failed to update text: ' + error.message);
        }
    }

    /**
     * Update preview iframe styles for immediate feedback
     */
    updatePreviewStyles(selector, property, value) {
        try {
            const iframe = document.getElementById('theme-preview-iframe');
            if (!iframe) {
                console.warn('[Manual Editor] Preview iframe not found');
                return;
            }

            if (!iframe.contentWindow) {
                console.warn('[Manual Editor] Iframe not loaded yet');
                return;
            }

            // Send style update to iframe via postMessage
            // Injected handler will apply the style
            iframe.contentWindow.postMessage({
                type: 'update-manual-styles',
                selector: selector,
                property: property,
                value: value
            }, '*');

            log('Style update sent via postMessage:', {
                selector, property, value
            });
        } catch (error) {
            console.error('[Manual Editor] Failed to send style update:', error);
            // Don't alert user for preview update failures - they can refresh manually
        }
    }

    /**
     * Save iframe scroll position before refresh
     * @returns {Promise<Object>} Scroll position {x, y}
     */
    async saveIframeScrollPosition() {
        try {
            const iframe = document.getElementById('theme-preview-iframe');
            if (!iframe || !iframe.contentWindow) {
                return { x: 0, y: 0 };
            }

            // Request scroll position from iframe via postMessage
            return new Promise((resolve) => {
                const messageHandler = (e) => {
                    if (e.data && e.data.type === 'scroll-position-response') {
                        window.removeEventListener('message', messageHandler);
                        resolve(e.data.position || { x: 0, y: 0 });
                    }
                };

                window.addEventListener('message', messageHandler);

                // Request scroll position from iframe
                iframe.contentWindow.postMessage({ type: 'get-scroll-position' }, '*');

                // Timeout fallback (100ms)
                setTimeout(() => {
                    window.removeEventListener('message', messageHandler);
                    resolve({ x: 0, y: 0 });
                }, 100);
            });
        } catch (error) {
            console.error('[Manual Editor] Failed to save scroll position:', error);
            return { x: 0, y: 0 };
        }
    }

    /**
     * Restore iframe scroll position after refresh
     * @param {Object} position Scroll position {x, y}
     */
    restoreIframeScrollPosition(position) {
        try {
            const iframe = document.getElementById('theme-preview-iframe');
            if (!iframe || !iframe.contentWindow) {
                return;
            }

            // Listen for handler-loaded message from iframe (indicates it's ready)
            const messageHandler = (e) => {
                if (e.data && e.data.type === 'handler-loaded') {
                    window.removeEventListener('message', messageHandler);

                    // Handler is loaded, now restore scroll
                    setTimeout(() => {
                        iframe.contentWindow.postMessage({
                            type: 'restore-scroll-position',
                            position: position
                        }, '*');
                        log('Scroll position restore requested:', position);
                    }, 200); // Small delay for DOM settling
                }
            };

            window.addEventListener('message', messageHandler);

            // Fallback timeout (if handler-loaded message never comes)
            setTimeout(() => {
                window.removeEventListener('message', messageHandler);
                // Try restore anyway
                iframe.contentWindow.postMessage({
                    type: 'restore-scroll-position',
                    position: position
                }, '*');
                log('Scroll position restore (fallback):', position);
            }, 1000);
        } catch (error) {
            console.error('[Manual Editor] Failed to restore scroll position:', error);
        }
    }

    /**
     * Bind global events
     */
    bindEvents() {
        // Listen for context updates to show/hide manual controls
        $(document).on('contextUpdated', (e, cssRules) => {
            if (cssRules && cssRules.length > 0) {
                this.displayControlsForRules(cssRules);
            }
        });
    }

    /**
     * Store original values for all controls (for reset functionality)
     */
    storeOriginalValues() {
        this.originalValues = {};

        $('.property-control').each((i, el) => {
            const control = $(el);
            const key = `${control.data('selector')}-${control.data('property')}`;
            this.originalValues[key] = this.getControlValue(control);
        });

        // Store original text content
        this.originalTextValues = {};
        $('.text-content-control, .text-content-control-simple').each((i, el) => {
            const control = $(el);
            const key = `${control.data('selector')}-${control.data('match-index')}`;
            this.originalTextValues[key] = control.find('.text-content-input, .text-content-input-simple').val();
        });
    }

    /**
     * Get current value from a property control
     */
    getControlValue(control) {
        // Color control
        const colorText = control.find('.color-text');
        if (colorText.length) return colorText.val();

        // Size control
        const sizeValue = control.find('.size-value');
        if (sizeValue.length) {
            const unit = control.find('.size-unit').val();
            return sizeValue.val() + unit;
        }

        // Spacing control
        const spacingValue = control.find('.spacing-value');
        if (spacingValue.length) return spacingValue.val();

        // Text control
        const textValue = control.find('.text-value');
        if (textValue.length) return textValue.val();

        return '';
    }

    /**
     * Set value on a property control
     */
    setControlValue(control, value) {
        // Color control
        const colorText = control.find('.color-text');
        if (colorText.length) {
            colorText.val(value);
            const colorValue = this.extractColorValue(value);
            control.find('.color-picker').val(colorValue);
            return;
        }

        // Size control
        const sizeValue = control.find('.size-value');
        if (sizeValue.length) {
            const parsed = this.parseSizeValue(value);
            sizeValue.val(parsed.value);
            control.find('.size-unit').val(parsed.unit);
            return;
        }

        // Spacing control
        const spacingValue = control.find('.spacing-value');
        if (spacingValue.length) {
            spacingValue.val(value);
            return;
        }

        // Text control
        const textValue = control.find('.text-value');
        if (textValue.length) {
            textValue.val(value);
            return;
        }
    }

    /**
     * Reset a property control to its original value
     */
    resetToOriginal(control) {
        const key = `${control.data('selector')}-${control.data('property')}`;
        const originalValue = this.originalValues?.[key];

        if (originalValue !== undefined) {
            this.setControlValue(control, originalValue);
            control.removeClass('modified');

            // Reset live preview
            const selector = control.data('selector');
            const property = control.data('property');
            this.updatePreviewStyles(selector, property, originalValue);

            // Visual feedback
            control.addClass('reset-flash');
            setTimeout(() => control.removeClass('reset-flash'), 300);
        }
    }

    /**
     * Reset text content to original value
     */
    resetTextToOriginal(control) {
        const key = `${control.data('selector')}-${control.data('match-index')}`;
        const originalValue = this.originalTextValues?.[key];

        if (originalValue !== undefined) {
            control.find('.text-content-input, .text-content-input-simple').val(originalValue);

            // Visual feedback
            control.addClass('reset-flash');
            setTimeout(() => control.removeClass('reset-flash'), 300);
        }
    }

    /**
     * Helper: Extract color value for color picker
     */
    extractColorValue(cssValue) {
        // Handle hex colors
        const hexMatch = cssValue.match(/#[0-9a-fA-F]{6}/);
        if (hexMatch) return hexMatch[0];
        
        // Handle rgb/rgba - convert to hex for color picker
        const rgbMatch = cssValue.match(/rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)/);
        if (rgbMatch) {
            const r = parseInt(rgbMatch[1]);
            const g = parseInt(rgbMatch[2]); 
            const b = parseInt(rgbMatch[3]);
            return '#' + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
        }
        
        // Default fallback
        return '#000000';
    }
    
    /**
     * Helper: Parse size value into number and unit
     */
    parseSizeValue(cssValue) {
        const match = cssValue.match(/^([0-9.]+)(.*)$/);
        if (match) {
            return {
                value: parseFloat(match[1]),
                unit: match[2] || 'px'
            };
        }
        return { value: 0, unit: 'px' };
    }

    /**
     * Format selector into human-readable name
     */
    formatSelectorName(selector) {
        // Remove pseudo-classes for base name
        const baseName = selector.split(':')[0];
        const pseudoClass = selector.includes(':') ? selector.split(':')[1] : null;

        // Convert .minimal-btn--secondary to "Secondary Button"
        let readable = baseName
            .replace(/^\./, '')  // Remove leading dot
            .replace(/--/g, ' ')  // Convert -- to space
            .replace(/-/g, ' ')   // Convert - to space
            .replace(/\b\w/g, l => l.toUpperCase());  // Title case

        // Add state if present
        if (pseudoClass) {
            readable += ` (${pseudoClass.charAt(0).toUpperCase() + pseudoClass.slice(1)})`;
        }

        return readable;
    }

    /**
     * Detect if property should use color picker
     */
    shouldUseColorPicker(prop) {
        // Check if property type is already marked as color
        if (prop.type === 'color') return true;

        // Check property name
        if (prop.property.includes('color')) return true;
        if (prop.property === 'background') return true;
        if (prop.property === 'background-color') return true;

        // Check value format
        if (/^#[0-9a-f]{3,6}$/i.test(prop.value)) return true;
        if (/^rgba?\(/.test(prop.value)) return true;
        if (/^hsla?\(/.test(prop.value)) return true;

        return false;
    }

    /**
     * Categorize properties into common vs advanced
     */
    isCommonProperty(propName) {
        const commonProps = [
            'color', 'background', 'background-color',
            'font-size', 'font-weight', 'font-family',
            'padding', 'margin',
            'width', 'height', 'display',
            'border', 'border-radius', 'border-color'
        ];
        return commonProps.includes(propName);
    }

    /**
     * Filter to only essential properties for simplified manual editor
     */
    filterEssentialProperties(properties, isPseudoState = false) {
        // For hover/focus/active states: only show colors
        if (isPseudoState) {
            const hoverEssentials = ['color', 'background', 'background-color'];
            return properties.filter(prop => hoverEssentials.includes(prop.property));
        }

        // For base state: show all essential properties
        const essentialProps = [
            'color',
            'background',
            'background-color',
            'border-color',
            'font-size',
            'padding',
            'margin',
            'border-radius'
        ];

        return properties.filter(prop => essentialProps.includes(prop.property));
    }

    /**
     * Calculate CSS specificity for a selector
     */
    calculateSpecificity(selector) {
        // Remove pseudo-elements for calculation
        const cleanSelector = selector.split('::')[0];

        // Count IDs (worth 100)
        const ids = (cleanSelector.match(/#/g) || []).length;

        // Count classes, attributes, pseudo-classes (worth 10)
        const classes = (cleanSelector.match(/\./g) || []).length;
        const attributes = (cleanSelector.match(/\[/g) || []).length;
        const pseudoClasses = (cleanSelector.match(/:/g) || []).length;

        // Count elements (worth 1)
        const elements = (cleanSelector.match(/\b[a-z]+\b/g) || []).length;

        return (ids * 100) + ((classes + attributes + pseudoClasses) * 10) + elements;
    }

    /**
     * Create collapsible selector group
     */
    createSelectorGroup(selector, selectorData) {
        const readableName = this.formatSelectorName(selector);
        const isBaseSelector = !selector.includes(':');
        const isExpanded = isBaseSelector; // Base selectors expanded by default

        // Separate common and advanced properties
        const commonProps = [];
        const advancedProps = [];

        selectorData.properties.forEach(prop => {
            if (this.isCommonProperty(prop.property)) {
                commonProps.push(prop);
            } else {
                advancedProps.push(prop);
            }
        });

        let html = `
            <div class="selector-group ${isExpanded ? 'expanded' : 'collapsed'}" data-selector="${selector}">
                <div class="selector-header">
                    <i data-lucide="chevron-right" class="collapse-icon ${isExpanded ? 'rotated' : ''}"></i>
                    <strong class="selector-name">${readableName}</strong>
                    <code class="selector-code">${selector}</code>
                    <span class="property-count">${selectorData.properties.length}</span>
                </div>
                <div class="selector-properties" style="${isExpanded ? '' : 'display:none'}">
        `;

        // Add common properties
        if (commonProps.length > 0) {
            commonProps.forEach(prop => {
                html += this.createPropertyControlImproved(prop, selector, selectorData.file);
            });
        }

        // Add advanced properties (collapsed by default)
        if (advancedProps.length > 0) {
            html += `
                <div class="advanced-properties-toggle collapsed">
                    <button type="button" class="toggle-advanced-btn">
                        <i data-lucide="chevron-right" class="collapse-icon"></i>
                        Show Advanced (${advancedProps.length} more)
                    </button>
                </div>
                <div class="advanced-properties" style="display:none">
            `;

            advancedProps.forEach(prop => {
                html += this.createPropertyControlImproved(prop, selector, selectorData.file);
            });

            html += `</div>`;
        }

        html += `
                </div>
            </div>
        `;

        return html;
    }

    /**
     * Create SIMPLIFIED selector group (no collapsible, no advanced, essential only)
     */
    createSimplifiedSelectorGroup(selector, selectorData, stateType = 'base') {
        const readableName = this.formatSelectorName(selector);
        const stateClass = stateType === 'hover' ? 'state-hover' : 'state-base';

        // State label and icon
        let stateLabel = '';
        if (stateType === 'hover') {
            stateLabel = '<span class="state-badge hover-badge"><svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2a3 3 0 0 0-3 3v7a3 3 0 0 0 6 0V5a3 3 0 0 0-3-3Z"></path><path d="M19 10v2a7 7 0 0 1-14 0v-2"></path><line x1="12" x2="12" y1="19" y2="22"></line></svg> Hover State</span>';
        } else {
            stateLabel = '<span class="state-badge base-badge"><svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><circle cx="12" cy="12" r="3"></circle></svg> Base State</span>';
        }

        let html = `
            <div class="selector-group-simple ${stateClass}" data-selector="${selector}">
                <div class="selector-header-simple">
                    ${stateLabel}
                    <strong class="selector-name">${readableName}</strong>
                    <code class="selector-code">${selector}</code>
                </div>
                <div class="selector-properties-simple">
        `;

        // Add only the filtered essential properties
        selectorData.properties.forEach(prop => {
            html += this.createPropertyControlImproved(prop, selector, selectorData.file);
        });

        html += `
                </div>
            </div>
        `;

        return html;
    }

    /**
     * Create property control with improved type detection
     */
    createPropertyControlImproved(prop, selector, file) {
        const controlId = `control-${selector}-${prop.property}`.replace(/[^a-zA-Z0-9-]/g, '-');

        let controlHtml = `
            <div class="property-control" data-selector="${selector}" data-property="${prop.property}" data-file="${file}">
                <label class="property-label" for="${controlId}">
                    ${this.formatPropertyName(prop.property)}
                </label>
        `;

        // Improved type detection
        if (this.shouldUseColorPicker(prop)) {
            controlHtml += this.createColorControl(controlId, prop);
        } else if (prop.type === 'size') {
            controlHtml += this.createSizeControl(controlId, prop);
        } else if (prop.type === 'spacing') {
            controlHtml += this.createSpacingControl(controlId, prop);
        } else {
            controlHtml += this.createTextControl(controlId, prop);
        }

        controlHtml += '</div>';
        return controlHtml;
    }

    /**
     * Create element context header
     */
    createSelectedElementHeader(firstSelector) {
        const readableName = this.formatSelectorName(firstSelector);

        return `
            <div class="selected-element-header">
                <div class="element-badge">
                    <i data-lucide="sparkles" class="icon-inline icon-sm"></i>
                    <span>Currently Editing</span>
                </div>
                <h4 class="element-name">${readableName}</h4>
                <code class="element-selector">${firstSelector}</code>
            </div>
        `;
    }

    /**
     * Make refresh button glow to indicate changes need preview refresh
     */
    glowRefreshButton() {
        const refreshBtn = $('#refresh-preview-btn');
        if (refreshBtn.length > 0) {
            // Add glow class and update title
            refreshBtn.addClass('needs-refresh glow-pulse');
            refreshBtn.attr('title', 'Click to refresh preview - Changes applied!');

            // Add visual indicator (change icon or add badge)
            if (!refreshBtn.find('.change-indicator').length) {
                refreshBtn.append('<span class="change-indicator">!</span>');
            }
        }
    }

    /**
     * Remove glow from refresh button (called when refreshed)
     */
    clearRefreshButtonGlow() {
        const refreshBtn = $('#refresh-preview-btn');
        if (refreshBtn.length > 0) {
            refreshBtn.removeClass('needs-refresh glow-pulse');
            refreshBtn.attr('title', 'Refresh preview');
            refreshBtn.find('.change-indicator').remove();
        }
    }


    /**
     * Helper: Format property name for display
     */
    formatPropertyName(propName) {
        return propName
            .replace(/-/g, ' ')
            .replace(/\b\w/g, l => l.toUpperCase());
    }

    /**
     * Create "Add New Property" button and dropdown
     */
    createAddPropertyButton(selector) {
        const commonProperties = [
            { name: 'color', label: 'Text Color', type: 'color', default: '#000000' },
            { name: 'background-color', label: 'Background Color', type: 'color', default: '#ffffff' },
            { name: 'font-size', label: 'Font Size', type: 'size', default: '16px' },
            { name: 'font-weight', label: 'Font Weight', type: 'text', default: '400' },
            { name: 'padding', label: 'Padding', type: 'spacing', default: '10px' },
            { name: 'margin', label: 'Margin', type: 'spacing', default: '0' },
            { name: 'border-radius', label: 'Border Radius', type: 'size', default: '0' },
            { name: 'border', label: 'Border', type: 'text', default: '1px solid #ccc' },
            { name: 'opacity', label: 'Opacity', type: 'text', default: '1' },
            { name: 'box-shadow', label: 'Box Shadow', type: 'text', default: 'none' }
        ];

        let optionsHtml = commonProperties.map(prop =>
            `<option value="${prop.name}" data-type="${prop.type}" data-default="${prop.default}">${prop.label}</option>`
        ).join('');

        return `
            <div class="add-property-section" data-selector="${selector}">
                <div class="add-property-header">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="12" x2="12" y1="8" y2="16"></line><line x1="8" x2="16" y1="12" y2="12"></line></svg>
                    <span>Add New Property</span>
                </div>
                <div class="add-property-form" style="display: none;">
                    <select class="new-property-select">
                        <option value="">Select property...</option>
                        ${optionsHtml}
                    </select>
                    <div class="new-property-value-wrapper" style="display: none;">
                        <input type="text" class="new-property-value" placeholder="Value...">
                        <button type="button" class="add-property-confirm-btn">Add</button>
                        <button type="button" class="add-property-cancel-btn">✕</button>
                    </div>
                </div>
            </div>
        `;
    }

    /**
     * Initialize Add Property event handlers (called in initializeControls)
     */
    initAddPropertyHandlers() {
        // Toggle add property form
        $('.add-property-header').off('click').on('click', (e) => {
            const section = $(e.currentTarget).closest('.add-property-section');
            const form = section.find('.add-property-form');
            form.slideToggle(200);
        });

        // Property select change
        $('.new-property-select').off('change').on('change', (e) => {
            const select = $(e.target);
            const wrapper = select.siblings('.new-property-value-wrapper');
            const valueInput = wrapper.find('.new-property-value');

            if (select.val()) {
                const selectedOption = select.find('option:selected');
                const defaultValue = selectedOption.data('default');
                valueInput.val(defaultValue);
                wrapper.show();
                valueInput.focus();
            } else {
                wrapper.hide();
            }
        });

        // Add property confirm
        $('.add-property-confirm-btn').off('click').on('click', async (e) => {
            const section = $(e.target).closest('.add-property-section');
            const selector = section.data('selector');
            const select = section.find('.new-property-select');
            const property = select.val();
            const value = section.find('.new-property-value').val();

            if (!property || !value) {
                alert('Please select a property and enter a value');
                return;
            }

            // Get property type for control creation
            const selectedOption = select.find('option:selected');
            const propertyType = selectedOption.data('type');

            // Apply the new property via REST API
            const btn = $(e.target);
            btn.text('Adding...').prop('disabled', true);

            try {
                const endpoint = this.isPageMode ? 'pages' : 'themes';
                const response = await fetch(`${aiSiteBuilder.restUrl}ai-builder/v1/${endpoint}/${this.entityId}/update-property`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    body: JSON.stringify({
                        selector: selector,
                        property: property,
                        value: value
                    })
                });

                const result = await response.json();

                if (result.success) {
                    btn.text('✓ Added!').addClass('success');

                    // Update preview
                    this.updatePreviewStyles(selector, property, value);

                    // Refresh to show new control
                    if (window.aiThemeEditor && typeof window.aiThemeEditor.forcePreviewRefresh === 'function') {
                        window.aiThemeEditor.forcePreviewRefresh();
                    }

                    // Reset form
                    setTimeout(() => {
                        select.val('');
                        section.find('.new-property-value-wrapper').hide();
                        btn.text('Add').removeClass('success').prop('disabled', false);
                    }, 1500);

                } else {
                    throw new Error(result.message || 'Failed to add property');
                }
            } catch (error) {
                console.error('Add property failed:', error);
                btn.text('Failed').addClass('error');
                setTimeout(() => {
                    btn.text('Add').removeClass('error').prop('disabled', false);
                }, 2000);
                alert('Failed to add property: ' + error.message);
            }
        });

        // Cancel button
        $('.add-property-cancel-btn').off('click').on('click', (e) => {
            const section = $(e.target).closest('.add-property-section');
            section.find('.new-property-select').val('');
            section.find('.new-property-value-wrapper').hide();
        });

        // Enter key in value input
        $('.new-property-value').off('keydown').on('keydown', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                $(e.target).closest('.add-property-section').find('.add-property-confirm-btn').click();
            } else if (e.key === 'Escape') {
                e.preventDefault();
                $(e.target).closest('.add-property-section').find('.add-property-cancel-btn').click();
            }
        });
    }
}

// Make ManualPropertyControls available globally
window.ManualPropertyControls = ManualPropertyControls;

// Global instance will be initialized when theme editor loads
window.manualPropertyControls = null;

})(jQuery);