/**
 * Dreamformer - Force Hide Freemius Pricing Cards
 *
 * NUCLEAR SOLUTION: Aggressively hides all individual plan cards
 * and only shows the dropdown selector on the Freemius pricing page.
 *
 * This uses multiple techniques to ensure cards are hidden:
 * 1. Inline style injection
 * 2. DOM attribute modification
 * 3. MutationObserver to catch dynamically added elements
 * 4. Continuous polling for the first 10 seconds
 * 5. Multiple event listeners to catch all loading stages
 */

(function() {
    'use strict';

    // Configuration
    const CONFIG = {
        // All possible selectors for plan cards (cast wide net)
        // IMPORTANT: Only target the CARDS (li.fs-package), NOT the dropdown container
        CARD_SELECTORS: [
            '.fs-packages li.fs-package',
            'li.fs-package',
            'ul.fs-packages > li',
            '.fs-section--packages ul.fs-packages > li'
        ],

        // Container selectors to watch
        CONTAINER_SELECTORS: [
            '#fs_pricing_wrapper',
            '#fs_pricing_app',
            '.fs-section--packages',
            '.fs-packages'
        ],

        // How long to aggressively poll (milliseconds)
        AGGRESSIVE_POLL_DURATION: 10000, // 10 seconds
        POLL_INTERVAL: 50, // Every 50ms

        // Debug mode (set to false in production)
        DEBUG: false
    };

    // Statistics
    let stats = {
        hideAttempts: 0,
        cardsFound: 0,
        lastHideTime: null
    };

    /**
     * Log debug messages
     */
    function log(...args) {
        if (CONFIG.DEBUG) {
            console.log('[DF Hide Cards]', ...args);
        }
    }

    /**
     * NUCLEAR OPTION: Hide a single card element with extreme prejudice
     */
    function nukeCard(card) {
        if (!card) return false;

        // Skip if already nuked
        if (card.hasAttribute('data-df-nuked')) {
            return false;
        }

        // Mark as nuked
        card.setAttribute('data-df-nuked', 'true');

        // Method 1: Inline styles (highest priority)
        card.style.cssText = `
            display: none !important;
            visibility: hidden !important;
            opacity: 0 !important;
            height: 0 !important;
            max-height: 0 !important;
            min-height: 0 !important;
            width: 0 !important;
            max-width: 0 !important;
            min-width: 0 !important;
            margin: 0 !important;
            padding: 0 !important;
            border: none !important;
            overflow: hidden !important;
            position: absolute !important;
            left: -9999px !important;
            top: -9999px !important;
            clip: rect(0, 0, 0, 0) !important;
            pointer-events: none !important;
            user-select: none !important;
        `;

        // Method 2: HTML attributes
        card.setAttribute('hidden', 'hidden');
        card.setAttribute('aria-hidden', 'true');
        card.setAttribute('data-df-hidden', 'true');

        // Method 3: CSS classes
        card.classList.add('df-force-hidden', 'df-nuked');

        // Method 4: Remove from accessibility tree
        card.setAttribute('role', 'presentation');
        card.setAttribute('tabindex', '-1');

        // Method 5: For extra safety, remove all child content
        // (Commented out - too aggressive, but available if needed)
        // card.innerHTML = '';

        stats.cardsFound++;
        log('Nuked card:', card);

        return true;
    }

    /**
     * Find and nuke all plan cards
     */
    function hideAllPlanCards() {
        stats.hideAttempts++;
        stats.lastHideTime = Date.now();

        let cardsHidden = 0;

        // Try each selector
        CONFIG.CARD_SELECTORS.forEach(selector => {
            try {
                const cards = document.querySelectorAll(selector);
                cards.forEach(card => {
                    if (nukeCard(card)) {
                        cardsHidden++;
                    }
                });
            } catch (e) {
                log('Error with selector', selector, e);
            }
        });

        if (cardsHidden > 0) {
            log(`Hide attempt #${stats.hideAttempts}: Hidden ${cardsHidden} cards (${stats.cardsFound} total)`);
        }

        return cardsHidden;
    }

    /**
     * Setup MutationObserver to watch for dynamically added cards
     */
    function setupMutationObserver() {
        // Watch each container
        CONFIG.CONTAINER_SELECTORS.forEach(selector => {
            const container = document.querySelector(selector);
            if (!container) return;

            const observer = new MutationObserver((mutations) => {
                let needsHide = false;

                mutations.forEach(mutation => {
                    // Check if new nodes were added
                    if (mutation.addedNodes.length > 0) {
                        mutation.addedNodes.forEach(node => {
                            // Check if it's an element node
                            if (node.nodeType === 1) {
                                // Check if it matches our card selectors
                                CONFIG.CARD_SELECTORS.forEach(cardSelector => {
                                    if (node.matches && node.matches(cardSelector)) {
                                        needsHide = true;
                                    }
                                    // Also check children
                                    if (node.querySelectorAll) {
                                        const childCards = node.querySelectorAll(cardSelector);
                                        if (childCards.length > 0) {
                                            needsHide = true;
                                        }
                                    }
                                });
                            }
                        });
                    }
                });

                if (needsHide) {
                    log('MutationObserver detected new cards, hiding...');
                    hideAllPlanCards();
                }
            });

            observer.observe(container, {
                childList: true,
                subtree: true,
                attributes: false, // Don't watch attributes to avoid infinite loops
                characterData: false
            });

            log('MutationObserver setup on', selector);
        });

        // Also observe the entire body as fallback
        const bodyObserver = new MutationObserver(() => {
            // Debounce: only run if we haven't hidden anything in the last 100ms
            if (Date.now() - stats.lastHideTime > 100) {
                hideAllPlanCards();
            }
        });

        bodyObserver.observe(document.body, {
            childList: true,
            subtree: true
        });

        log('Body MutationObserver setup');
    }

    /**
     * Aggressive polling for the first few seconds
     */
    function startAggressivePolling() {
        let pollCount = 0;
        const maxPolls = CONFIG.AGGRESSIVE_POLL_DURATION / CONFIG.POLL_INTERVAL;

        const interval = setInterval(() => {
            hideAllPlanCards();
            pollCount++;

            if (pollCount >= maxPolls) {
                clearInterval(interval);
                log('Aggressive polling complete:', stats);
            }
        }, CONFIG.POLL_INTERVAL);

        log('Started aggressive polling:', maxPolls, 'attempts over', CONFIG.AGGRESSIVE_POLL_DURATION, 'ms');
    }

    /**
     * Initialize the card hiding system
     */
    function init() {
        log('Initializing card hiding system...');

        // Immediate first attempt
        hideAllPlanCards();

        // Setup MutationObserver
        setupMutationObserver();

        // Start aggressive polling
        startAggressivePolling();

        // Also hide on common events
        window.addEventListener('load', hideAllPlanCards);
        document.addEventListener('DOMContentLoaded', hideAllPlanCards);

        // Hide when React/Freemius might be ready
        setTimeout(hideAllPlanCards, 100);
        setTimeout(hideAllPlanCards, 300);
        setTimeout(hideAllPlanCards, 500);
        setTimeout(hideAllPlanCards, 1000);
        setTimeout(hideAllPlanCards, 2000);
        setTimeout(hideAllPlanCards, 3000);

        log('Card hiding system initialized');
    }

    // Run immediately if DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Expose stats for debugging
    if (CONFIG.DEBUG) {
        window.dfHideCardsStats = stats;
    }

})();
