/**
 * Dreamformer Pricing Page - Particle Animation
 * Creates an animated particle network background
 */

(function() {
  'use strict';

  // Wait for DOM to be ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initPricingParticles);
  } else {
    initPricingParticles();
  }

  function initPricingParticles() {
    const pricingWrapper = document.getElementById('fs_pricing_wrapper');
    if (!pricingWrapper) return;

    // Create floating orbs
    for (let i = 1; i <= 7; i++) {
      const orb = document.createElement('div');
      orb.className = `df-pricing-orb df-pricing-orb-${i}`;
      pricingWrapper.insertBefore(orb, pricingWrapper.firstChild);
    }

    // Create canvas element for particle network
    const canvas = document.createElement('canvas');
    canvas.className = 'df-pricing-particles';
    canvas.id = 'pricingParticles';

    // Insert at the beginning of the wrapper
    pricingWrapper.insertBefore(canvas, pricingWrapper.firstChild);

    const ctx = canvas.getContext('2d');

    // Set canvas size
    function resizeCanvas() {
      canvas.width = window.innerWidth;
      canvas.height = Math.max(document.documentElement.scrollHeight, window.innerHeight);
    }
    resizeCanvas();

    const particles = [];
    const particleCount = 80;

    // Particle class
    class Particle {
      constructor() {
        this.x = Math.random() * canvas.width;
        this.y = Math.random() * canvas.height;
        this.size = Math.random() * 2.5 + 0.5;
        this.speedX = Math.random() * 1 - 0.5;
        this.speedY = Math.random() * 1 - 0.5;
        this.opacity = Math.random() * 0.6 + 0.2;
      }

      update() {
        this.x += this.speedX;
        this.y += this.speedY;

        // Bounce off edges
        if (this.x > canvas.width || this.x < 0) this.speedX *= -1;
        if (this.y > canvas.height || this.y < 0) this.speedY *= -1;
      }

      draw() {
        ctx.fillStyle = `rgba(139, 92, 246, ${this.opacity})`;
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
        ctx.fill();
      }
    }

    // Initialize particles
    for (let i = 0; i < particleCount; i++) {
      particles.push(new Particle());
    }

    // Connect nearby particles with lines
    function connectParticles() {
      for (let i = 0; i < particles.length; i++) {
        for (let j = i + 1; j < particles.length; j++) {
          const dx = particles[i].x - particles[j].x;
          const dy = particles[i].y - particles[j].y;
          const distance = Math.sqrt(dx * dx + dy * dy);

          if (distance < 150) {
            ctx.strokeStyle = `rgba(139, 92, 246, ${0.2 * (1 - distance / 150)})`;
            ctx.lineWidth = 1;
            ctx.beginPath();
            ctx.moveTo(particles[i].x, particles[i].y);
            ctx.lineTo(particles[j].x, particles[j].y);
            ctx.stroke();
          }
        }
      }
    }

    // Animation loop
    function animate() {
      ctx.clearRect(0, 0, canvas.width, canvas.height);

      particles.forEach(particle => {
        particle.update();
        particle.draw();
      });

      connectParticles();
      requestAnimationFrame(animate);
    }

    // Start animation
    animate();

    // Handle window resize
    let resizeTimeout;
    window.addEventListener('resize', () => {
      clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(() => {
        resizeCanvas();
      }, 250);
    });

    // Update canvas height on scroll (for long pages)
    let scrollTimeout;
    window.addEventListener('scroll', () => {
      clearTimeout(scrollTimeout);
      scrollTimeout = setTimeout(() => {
        const newHeight = Math.max(document.documentElement.scrollHeight, window.innerHeight);
        if (Math.abs(canvas.height - newHeight) > 100) {
          canvas.height = newHeight;
        }
      }, 250);
    });
  }
})();
