/**
 * =====================================================
 * PAGE LIST VIEW - Grid/Card Display ONLY
 * =====================================================
 *
 * This file handles the page list/grid interface at:
 * - admin.php?page=dreamformer&tab=pages
 *
 * Responsibilities:
 * - Display page cards in grid layout
 * - Page creation modals (template selection)
 * - Page actions: duplicate, delete
 * - Preview thumbnails in iframes (sandbox, no editing)
 * - Product styling modals
 *
 * What it DOES NOT do:
 * - Actual page editing (handled by theme-editor.js)
 * - AI chat interface (handled by theme-editor.js)
 * - Element selection (handled by theme-editor.js)
 *
 * When user clicks "Edit with AI" button:
 * - Redirects to: admin.php?page=dreamformer&edit=page&id=334
 * - theme-editor.js takes over as the unified editor
 * =====================================================
 */

(function($) {
    'use strict';

    class DreamformerPages {
        constructor() {
            this.selectedAttachment = null; // {filename, mime_type, size, content (base64)}
            this.init();
        }

        init() {
            this.bindEvents();
            this.loadPages();
            this.loadReferencePages(); // Load reference pages on init
            this.initAutoExpand(); // Auto-expand textarea
            this.initTypewriterAnimation(); // Typewriter demo
            this.setupAttachmentUpload(); // File attachment handling
        }

        bindEvents() {
            // Template chip clicks - auto-fill prompt
            $(document).on('click', '[data-action="select-template"]', (e) => {
                e.preventDefault();
                const type = $(e.currentTarget).data('type');
                const prompt = this.getTemplatePrompt(type);
                const $input = $('#page-prompt-input');
                // Mark as user-typed so focus handler doesn't clear the value
                $input.data('user-typed', true).removeClass('demo-active');
                $input.val(prompt).focus();
            });

            // Generate from prompt button
            $(document).on('click', '[data-action="generate-from-prompt"]', async (e) => {
                e.preventDefault();
                const prompt = $('#page-prompt-input').val().trim();

                if (!prompt) {
                    this.showMessage('error', 'Please describe the page you want to create.');
                    $('#page-prompt-input').focus();
                    return;
                }

                // Get reference page ID from dropdown
                const referencePageId = $('#reference-page-select').val();

                // ALWAYS use polling (fixes Hostinger HTTP/2 timeout issues)
                const streamer = new PageGenerationStreamer();
                await streamer.startPolling(
                    prompt,
                    'custom',
                    {}, // No business info needed
                    referencePageId ? parseInt(referencePageId, 10) : null,
                    this.selectedAttachment // Pass attachment if selected
                );

                // Clear the prompt and attachment after generation starts
                $('#page-prompt-input').val('');
                $('#reference-page-select').val('');
                this.removeAttachment();

                // Refresh pages list
                this.loadPages();
            });

            // Enter key in textarea (Ctrl+Enter or Cmd+Enter to generate)
            $(document).on('keydown', '#page-prompt-input', (e) => {
                if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
                    e.preventDefault();
                    $('[data-action="generate-from-prompt"]').click();
                }
            });

            // Attach file button
            $(document).on('click', '[data-action="attach-file"]', (e) => {
                e.preventDefault();
                $('#page-attachment-input').click();
            });

            // Remove attachment button
            $(document).on('click', '[data-action="remove-attachment"]', (e) => {
                e.preventDefault();
                this.removeAttachment();
            });

            // Style Products button
            $(document).on('click', '[data-action="style-products"]', (e) => {
                this.openStyleProductsModal();
            });

            // Manage Product Styles button
            $(document).on('click', '[data-action="manage-product-styles"]', (e) => {
                this.openManageStylesModal();
            });

            // Refresh pages
            $(document).on('click', '[data-action="refresh-pages"]', () => {
                this.loadPages();
            });

            // Modal events
            $(document).on('click', '.modal-close, .btn-secondary', () => {
                this.closeModal();
            });

            $(document).on('click', '.dreamformer-modal-overlay', (e) => {
                if (e.target === e.currentTarget) {
                    this.closeModal();
                }
            });

            // Form submission
            $(document).on('submit', '#page-generation-form', (e) => {
                e.preventDefault();
                this.generatePage();
            });

            // Product style form submission
            $(document).on('submit', '#product-style-form', (e) => {
                e.preventDefault();
                this.generateProductStyles();
            });

            // Edit page buttons
            $(document).on('click', '.edit-page-btn', (e) => {
                const pageId = $(e.currentTarget).data('page-id');
                this.editPage(pageId);
            });

            // Page action buttons (clone, delete)
            $(document).on('click', '.page-duplicate-btn', (e) => {
                const pageId = $(e.currentTarget).data('page-id');
                this.duplicatePage(pageId);
            });

            $(document).on('click', '.page-delete-btn', (e) => {
                const pageId = $(e.currentTarget).data('page-id');
                this.deletePage(pageId);
            });

            // Export button
            $(document).on('click', '.page-export-btn', (e) => {
                const pageId = $(e.currentTarget).data('page-id');
                this.exportPage(pageId);
            });

            // Import button (in toolbar - will be added)
            $(document).on('click', '[data-action="import-page"]', () => {
                this.openImportModal();
            });

            // Import modal delegated event handlers
            $(document).on('click', '#browse-file-btn', () => {
                $('#import-file-input').click();
            });

            $(document).on('click', '.import-drop-zone', (e) => {
                if (e.target.id !== 'browse-file-btn') {
                    $('#import-file-input').click();
                }
            });

            $(document).on('dragover', '.import-drop-zone', (e) => {
                e.preventDefault();
                $('.import-drop-zone').css('background', '#e3f2fd');
            });

            $(document).on('dragleave', '.import-drop-zone', () => {
                $('.import-drop-zone').css('background', '#f9f9f9');
            });

            $(document).on('drop', '.import-drop-zone', (e) => {
                e.preventDefault();
                $('.import-drop-zone').css('background', '#f9f9f9');
                const files = e.originalEvent.dataTransfer.files;
                if (files.length > 0) {
                    this.handleImportFileSelect(files[0]);
                }
            });

            $(document).on('change', '#import-file-input', (e) => {
                if (e.target.files.length > 0) {
                    this.handleImportFileSelect(e.target.files[0]);
                }
            });

            $(document).on('click', '#import-page-btn', async (e) => {
                this.performImport();
            });
        }

        async openGenerateModal(type = 'custom') {
            const typeLabels = {
                landing: 'Landing Page',
                about: 'About Page',
                contact: 'Contact Page',
                blog: 'Blog Page',
                services: 'Services Page',
                app: 'Interactive App',
                cart: 'Shopping Cart',
                shop: 'Shop Page',
                checkout: 'Checkout Page',
                custom: 'Custom Page'
            };

            const typePlaceholders = {
                landing: 'Create a modern landing page for my digital marketing agency with hero section, services overview, testimonials, and contact form...',
                about: 'Create an about page for my company that tells our story, introduces the team, and showcases our values...',
                contact: 'Create a contact page with contact form, business hours, location map, and multiple ways to reach us...',
                blog: 'Create a blog page with article listings, featured posts, categories, and search functionality...',
                services: 'Create a services page showcasing our offerings with pricing, features, testimonials, and call-to-action buttons...',
                app: 'Create an interactive app like to-do list, calculator, timer, converter, game, quiz, or any tool with JavaScript functionality...',
                cart: 'Create a shopping cart page with product display, quantity controls, totals, and checkout functionality...',
                shop: 'Create a shop page to display all products with categories, filters, and promotional banners...',
                checkout: 'Create a secure checkout page with billing/shipping forms, payment methods, and trust indicators...',
                custom: 'Describe your page in detail - what it should include, the style, target audience, etc...'
            };

            const modal = `
                <div class="dreamformer-modal-overlay">
                    <div class="dreamformer-modal">
                        <div class="modal-header">
                            <h3>Generate ${typeLabels[type]}</h3>
                            <button class="modal-close">&times;</button>
                        </div>
                        <div class="modal-body">
                            <form id="page-generation-form">
                                <input type="hidden" name="page_type" value="${type}">

                                <div class="form-group">
                                    <label for="page-prompt">Describe your page *</label>
                                    <textarea
                                        id="page-prompt"
                                        name="prompt"
                                        placeholder="${typePlaceholders[type]}"
                                        required
                                    ></textarea>
                                </div>

                                <div class="form-group-row">
                                    <div class="form-group">
                                        <label for="business-name">Business Name</label>
                                        <input
                                            type="text"
                                            id="business-name"
                                            name="business_name"
                                            placeholder="Your business name"
                                        >
                                    </div>
                                    <div class="form-group">
                                        <label for="business-industry">Industry</label>
                                        <input
                                            type="text"
                                            id="business-industry"
                                            name="business_industry"
                                            placeholder="e.g., Digital Marketing, Restaurant"
                                        >
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label for="reference-page"><i data-lucide="palette" class="icon-inline icon-sm icon-primary"></i> Use style from existing page</label>
                                    <select id="reference-page" name="reference_page_id">
                                        <option value="">No reference - fresh style</option>
                                    </select>
                                    <small>Keep the same visual style but create new content for this page type</small>
                                </div>

                                <div class="modal-actions">
                                    <button type="button" class="btn-secondary">Cancel</button>
                                    <button type="submit" class="btn-primary">
                                        <span class="btn-text">Generate Page</span>
                                        <span class="btn-loading" style="display: none;">✨ Creating magic...</span>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            `;

            $('body').append(modal);
            $('#page-prompt').focus();

            // Load reference pages for dropdown
            await this.loadReferencePages();
        }

        /**
         * Get template-specific prompt text
         */
        getTemplatePrompt(type) {
            const prompts = {
                landing: 'Create a modern landing page with a hero section, key features, testimonials, and a strong call-to-action. Use engaging copy and clean design.',
                about: 'Create an about page that tells our company story, showcases our team, highlights our mission and values, and builds trust with visitors.',
                contact: 'Create a contact page with a contact form, business hours, location map, phone/email details, and social media links.',
                blog: 'Create a blog page with recent posts, categories, search functionality, and an engaging layout for article browsing.',
                services: 'Create a services page showcasing our offerings with pricing, features, testimonials, and call-to-action buttons.',
                app: 'Create an interactive app page with JavaScript functionality - like a calculator, converter, timer, to-do list, or similar tool.'
            };

            return prompts[type] || '';
        }

        /**
         * Auto-expand textarea as user types
         */
        initAutoExpand() {
            const $textarea = $('#page-prompt-input');
            if (!$textarea.length) return;

            const minHeight = 200; // Match CSS min-height
            const maxHeight = 500; // Max height before scrolling

            const autoExpand = () => {
                // Reset height to get accurate scrollHeight
                $textarea.css('height', minHeight + 'px');

                const scrollHeight = $textarea[0].scrollHeight;
                const newHeight = Math.min(Math.max(scrollHeight, minHeight), maxHeight);

                $textarea.css('height', newHeight + 'px');
            };

            // Auto-expand on input - but NOT during demo typing
            $textarea.on('input', function() {
                // Only auto-expand when user is actually typing, not during demo
                if (!$(this).hasClass('demo-active')) {
                    autoExpand();
                }
            });

            // Initial sizing
            autoExpand();
        }

        /**
         * Typewriter animation for educational demo - ENHANCED
         * Features: Fast speed, variable timing, human-like pauses, blinking cursor
         */
        initTypewriterAnimation() {
            const $textarea = $('#page-prompt-input');
            if (!$textarea.length) return;

            const examples = [
                'Create a stunning landing page with animated hero section, feature cards, testimonials, and a powerful call-to-action...',
                'Build a professional portfolio showcasing my projects with image galleries, case studies, tech stack badges, and contact form...',
                'Design an engaging about page with company timeline, team member profiles, mission statement, and photo gallery...',
                'Make a modern contact page with interactive map, business hours, contact form with validation, and social media links...',
                'Generate a blog page with post grid, category filters, search bar, tags, and reading time estimates...',
                'Create a services page with pricing tables, feature comparisons, testimonials, FAQ accordion, and booking button...',
                'Build an e-commerce product page with image zoom, size selector, add to cart, reviews, and related products...',
                'Design a restaurant website with menu categories, reservation system, photo gallery, and location map...',
                'Make a SaaS landing page with feature highlights, pricing tiers, customer logos, and free trial signup...',
                'Create an educational course page with curriculum sections, instructor bios, student testimonials, and enrollment form...'
            ];

            let currentExample = 0;
            let currentChar = 0;
            let baseTypingSpeed = 25;        // Fast & snappy
            let pauseAfterComplete = 1000;   // Pause before clearing
            let isUserInteracting = false;
            let typingTimeout = null;

            // Add active typing class to enhance glow
            const setTypingState = (isActive) => {
                const $container = $textarea.closest('.page-prompt-container');
                if (isActive) {
                    $container.addClass('is-typing');
                } else {
                    $container.removeClass('is-typing');
                }
            };

            // Update textarea value with cursor if needed
            const updateTextarea = (text, includeCursor = false) => {
                const displayText = includeCursor ? text + '|' : text;
                $textarea.val(displayText);
                // DON'T manipulate height during demo - it causes layout thrashing
                // The CSS min-height/max-height will handle it
                // $textarea.css('height', 'auto').css('height', $textarea[0].scrollHeight + 'px');

                // Add demo-active class to make text look like placeholder
                if (displayText) {
                    $textarea.addClass('demo-active');
                } else {
                    $textarea.removeClass('demo-active');
                }
            };

            // Calculate delay with variable speed and human-like pauses
            const getTypingDelay = (char) => {
                // Add random variation (20-40ms range)
                const variation = Math.random() * 20 + 10;

                // Check for punctuation pauses
                if (char === '.') return 250 + variation; // Longer pause at periods
                if (char === ',') return 150 + variation; // Medium pause at commas
                if (char === '!') return 250 + variation; // Longer pause at exclamation
                if (char === '?') return 250 + variation; // Longer pause at questions
                if (char === ' ') return baseTypingSpeed + (variation / 2); // Slight variation on spaces

                return baseTypingSpeed + variation;
            };

            const type = () => {
                // Stop if user is interacting
                if ($textarea.is(':focus') || isUserInteracting) {
                    clearTimeout(typingTimeout);
                    setTypingState(false);
                    return;
                }

                const currentText = examples[currentExample];

                // Typing forward
                if (currentChar < currentText.length) {
                    const nextChar = currentText[currentChar];
                    const newText = currentText.substring(0, currentChar + 1);
                    updateTextarea(newText, true); // Show cursor while typing
                    currentChar++;

                    // Use variable delay based on character
                    const delay = getTypingDelay(nextChar);
                    typingTimeout = setTimeout(type, delay);
                } else {
                    // Finished typing, pause then instantly clear and start next
                    updateTextarea(currentText, true);
                    setTypingState(false);
                    typingTimeout = setTimeout(() => {
                        // Instantly clear and move to next example
                        currentExample = (currentExample + 1) % examples.length;
                        currentChar = 0;
                        updateTextarea('', true);

                        // Brief pause then start typing next example
                        setTimeout(() => {
                            setTypingState(true);
                            type();
                        }, 200);
                    }, pauseAfterComplete);
                }
            };

            // Start typewriter after a short delay
            setTimeout(() => {
                if ($textarea.val() === '' && !$textarea.is(':focus')) {
                    setTypingState(true);
                    type();
                }
            }, 1000);

            // Stop animation when user focuses
            $textarea.on('focus', () => {
                isUserInteracting = true;
                clearTimeout(typingTimeout);
                setTypingState(false);

                const currentValue = $textarea.val();
                if (currentValue !== '' && !$textarea.data('user-typed')) {
                    // If it's demo text, remove cursor and clear
                    $textarea.val(currentValue.replace(/\|$/, ''));
                    $textarea.val('');
                }

                // Remove demo styling when user focuses
                $textarea.removeClass('demo-active');
            });

            // Mark as user-typed when user actually types
            $textarea.on('input', function() {
                if ($(this).is(':focus')) {
                    $(this).data('user-typed', true);
                    isUserInteracting = true;
                }
            });

            // Resume animation when user blurs and hasn't typed
            $textarea.on('blur', () => {
                setTimeout(() => {
                    if ($textarea.val() === '' || $textarea.val() === '|') {
                        $textarea.data('user-typed', false);
                        isUserInteracting = false;
                        currentChar = 0;
                        setTypingState(true);
                        type();
                    }
                }, 500);
            });
        }

        /**
         * Setup file attachment handling
         */
        setupAttachmentUpload() {
            const $fileInput = $('#page-attachment-input');

            $fileInput.on('change', (e) => {
                const file = e.target.files[0];
                if (file) {
                    this.handleAttachmentSelect(file);
                }
            });
        }

        /**
         * Handle file selection - validate and convert to base64
         */
        async handleAttachmentSelect(file) {
            // Validate file type - Accept images, PDFs, and text-based documents
            const allowedTypes = [
                'application/pdf',
                'image/png',
                'image/jpeg',
                'image/jpg',
                'text/plain',
                'text/markdown',
                'text/csv',
                'text/html',
                'text/xml',
                'application/json',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
            ];

            // Also allow files by extension if mime type doesn't match (browser inconsistencies)
            const allowedExtensions = ['.pdf', '.png', '.jpg', '.jpeg', '.md', '.txt', '.doc', '.docx', '.csv', '.json', '.xml', '.html'];
            const fileExtension = file.name.toLowerCase().substring(file.name.lastIndexOf('.'));

            if (!allowedTypes.includes(file.type) && !allowedExtensions.includes(fileExtension)) {
                this.showMessage('error', 'Invalid file type. Allowed: PDF, images (PNG/JPG), text files (MD, TXT, DOC, DOCX, CSV, JSON, XML, HTML).');
                $('#page-attachment-input').val(''); // Clear input
                return;
            }

            // Validate file size (5MB = 5242880 bytes)
            const maxSize = 5 * 1024 * 1024;
            if (file.size > maxSize) {
                this.showMessage('error', 'File too large. Maximum file size is 5MB.');
                $('#page-attachment-input').val(''); // Clear input
                return;
            }

            try {
                // Read file as base64
                const base64Content = await this.readFileAsBase64(file);

                // Store attachment data
                this.selectedAttachment = {
                    filename: file.name,
                    mime_type: file.type,
                    size: file.size,
                    content: base64Content // Without data:image/... prefix
                };

                // Show preview
                this.showAttachmentPreview(file, base64Content);

            } catch (error) {
                console.error('[Attachment] Error reading file:', error);
                this.showMessage('error', 'Failed to read file. Please try again.');
                $('#page-attachment-input').val('');
            }
        }

        /**
         * Read file as base64 using FileReader
         */
        readFileAsBase64(file) {
            return new Promise((resolve, reject) => {
                const reader = new FileReader();

                reader.onload = (e) => {
                    const result = e.target.result;
                    // Remove data URL prefix (data:image/png;base64,...)
                    const base64 = result.split(',')[1];
                    resolve(base64);
                };

                reader.onerror = () => {
                    reject(new Error('Failed to read file'));
                };

                reader.readAsDataURL(file);
            });
        }

        /**
         * Show attachment preview with thumbnail
         */
        showAttachmentPreview(file, base64Content) {
            const $preview = $('#attachment-preview');
            const $thumbnail = $preview.find('.attachment-thumbnail');
            const $icon = $preview.find('.attachment-icon');
            const $filename = $preview.find('.attachment-filename');
            const $size = $preview.find('.attachment-size');

            // Set filename and size
            $filename.text(file.name);
            $size.text(this.formatFileSize(file.size));

            // Show appropriate preview
            if (file.type.startsWith('image/')) {
                // Show thumbnail for images
                $thumbnail.attr('src', 'data:' + file.type + ';base64,' + base64Content);
                $thumbnail.show();
                $icon.hide();
            } else {
                // Show icon for PDFs
                $thumbnail.hide();
                $icon.show();
            }

            // Show preview container
            $preview.fadeIn(200);

            // Re-render Lucide icons
            if (window.lucide) {
                lucide.createIcons();
            }
        }

        /**
         * Remove attachment
         */
        removeAttachment() {
            this.selectedAttachment = null;
            $('#page-attachment-input').val('');
            $('#attachment-preview').fadeOut(200);
        }

        /**
         * Format file size for display
         */
        formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
        }

        async loadReferencePages() {
            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/pages',
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    }
                });

                if (response.success && response.pages.length > 0) {
                    const defaultOption = '<option value="">No reference - fresh style</option>';
                    const pageOptions = response.pages
                        .filter(page => page.status === 'publish') // Only published pages
                        .map(page => {
                            const truncatedTitle = page.title.length > 30
                                ? page.title.substring(0, 30) + '...'
                                : page.title;
                            const pageType = page.page_type ? ` (${page.page_type})` : '';
                            return `<option value="${page.id}">${truncatedTitle}${pageType}</option>`;
                        })
                        .join('');

                    // Populate both the main UI dropdown and modal dropdown (if exists)
                    $('#reference-page-select').html(defaultOption + pageOptions);
                    $('#reference-page').html(defaultOption + pageOptions);
                }
            } catch (error) {
                console.error('Failed to load reference pages:', error);
                // Keep default option if loading fails
            }
        }

        closeModal() {
            $('.dreamformer-modal-overlay').remove();
        }

        async generatePage() {
            const $form = $('#page-generation-form');
            const $submitBtn = $form.find('.btn-primary');
            const formData = this.getFormData($form);

            // Validate required fields
            if (!formData.prompt || formData.prompt.trim() === '') {
                this.showMessage('error', 'Please describe your page before generating.');
                return;
            }

            // Debug: Log the form data being sent

            // Close the form modal
            this.closeModal();

            // ALWAYS use polling (fixes Hostinger HTTP/2 timeout issues)
            const streamer = new PageGenerationStreamer();
            await streamer.startPolling(
                formData.prompt,
                formData.page_type,
                formData.business_info,
                formData.reference_page_id
            );

            // Refresh the grid after successful generation
            this.loadPages();
        }

        getFormData($form) {
            const referencePageId = $form.find('[name="reference_page_id"]').val();

            return {
                prompt: $form.find('[name="prompt"]').val(),
                page_type: $form.find('[name="page_type"]').val(),
                reference_page_id: referencePageId ? parseInt(referencePageId, 10) : null,
                business_info: {
                    name: $form.find('[name="business_name"]').val(),
                    industry: $form.find('[name="business_industry"]').val()
                }
            };
        }

        async loadPages() {
            const $grid = $('#ai-pages-grid');
            $grid.html('<p class="loading-text">Loading your pages...</p>');

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/pages',
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    }
                });

                if (response.success) {
                    this.renderPages(response.pages);
                } else {
                    $grid.html('<p class="loading-text">Failed to load pages.</p>');
                }

            } catch (error) {
                console.error('Failed to load pages:', error);
                $grid.html('<p class="loading-text">Failed to load pages.</p>');
            }
        }

        renderPages(pages) {
            const $grid = $('#ai-pages-grid');

            if (pages.length === 0) {
                $grid.html(`
                    <div style="text-align: center; padding: 60px 20px; color: #6b7280;">
                        <h4>No pages yet</h4>
                        <p>Create your first AI-generated page using the buttons above.</p>
                    </div>
                `);
                return;
            }

            const pageCards = pages.map(page => this.renderPageCard(page)).join('');
            $grid.html(pageCards);
        }

        renderPageCard(page) {
            const statusClass = page.status === 'publish' ? 'published' : 'draft';
            const statusText = page.status === 'publish' ? 'Published' : 'Draft';
            const date = new Date(page.date).toLocaleDateString();

            // Truncate long titles
            const title = page.title.length > 50 ? page.title.substring(0, 50) + '...' : page.title;

            // Fix edit URL to use main dashboard instead of old submenu
            const editUrl = `${aiSiteBuilder.adminUrl}admin.php?page=dreamformer&edit=page&id=${page.id}`;

            return `
                <div class="page-card">
                    <div class="theme-preview-wrapper">
                        <div class="page-preview">
                            <iframe src="${page.preview_url}" sandbox="allow-same-origin" loading="lazy"></iframe>
                        </div>
                        <div class="theme-overlay">
                            <a href="${page.preview_url}" target="_blank" class="button theme-preview-btn">
                                <span class="dashicons dashicons-visibility"></span> Preview
                            </a>
                        </div>
                    </div>

                    <div class="theme-info">
                        <div class="theme-header">
                            <h3 class="theme-title" title="${page.title}">${title}</h3>
                        </div>

                        <div class="theme-meta">
                            <span class="page-status status-${page.status}">${statusText}</span>
                            <span class="business-type">${page.page_type || 'Custom'} Page</span>
                            <span class="theme-date">Created ${date}</span>
                        </div>

                        <div class="theme-actions">
                            <a href="${editUrl}" class="button button-primary theme-edit-btn">
                                <span class="dashicons dashicons-edit"></span> Edit with AI
                            </a>
                            <a href="${page.preview_url}" target="_blank" class="button theme-preview-btn">
                                <span class="dashicons dashicons-visibility"></span> Preview
                            </a>
                            <button class="button page-duplicate-btn" data-page-id="${page.id}">
                                <span class="dashicons dashicons-admin-page"></span> Clone
                            </button>
                            <button class="button page-export-btn" data-page-id="${page.id}">
                                <span class="dashicons dashicons-download"></span> Export
                            </button>
                            <button class="button page-delete-btn" data-page-id="${page.id}">
                                <span class="dashicons dashicons-trash"></span> Delete
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }

        editPage(pageId) {
            // Redirect to the unified editor (existing functionality)
            window.location.href = `${aiSiteBuilder.adminUrl}admin.php?page=dreamformer&edit=page&id=${pageId}`;
        }

        async duplicatePage(pageId) {
            if (!confirm('Are you sure you want to clone this page?')) {
                return;
            }

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + `ai-builder/v1/pages/${pageId}/duplicate`,
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    }
                });

                if (response.success) {
                    this.showMessage('success', 'Page cloned successfully!');
                    this.loadPages(); // Refresh the grid
                } else {
                    this.showMessage('error', 'Failed to clone page: ' + (response.error || 'Unknown error'));
                }
            } catch (error) {
                console.error('Page duplication error:', error);
                this.showMessage('error', 'Failed to clone page. Please try again.');
            }
        }

        async deletePage(pageId) {
            if (!confirm('Are you sure you want to delete this page? This action cannot be undone.')) {
                return;
            }

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + `ai-builder/v1/pages/${pageId}`,
                    method: 'DELETE',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    }
                });

                if (response.success) {
                    this.showMessage('success', 'Page deleted successfully!');
                    this.loadPages(); // Refresh the grid
                } else {
                    this.showMessage('error', 'Failed to delete page: ' + (response.error || 'Unknown error'));
                }
            } catch (error) {
                console.error('Page deletion error:', error);
                this.showMessage('error', 'Failed to delete page. Please try again.');
            }
        }

        showMessage(type, message) {
            const $messageDiv = $('#pages-message');
            $messageDiv
                .removeClass('success error')
                .addClass(type + ' show')
                .html(message);

            // Auto-hide after 5 seconds
            setTimeout(() => {
                $messageDiv.removeClass('show');
            }, 5000);
        }

        /**
         * Open product styling modal
         */
        async openStyleProductsModal() {
            const modal = `
                <div class="dreamformer-modal-overlay">
                    <div class="dreamformer-modal">
                        <div class="modal-header">
                            <h3>Style WooCommerce Product Pages</h3>
                            <button class="modal-close">&times;</button>
                        </div>
                        <div class="modal-body">
                            <form id="product-style-form">
                                <div class="form-group">
                                    <label for="style-prompt">Describe your product page style *</label>
                                    <textarea
                                        id="style-prompt"
                                        name="prompt"
                                        placeholder="Create modern, minimalist product pages with large images, clean typography, prominent add-to-cart buttons, and a luxury feel..."
                                        required
                                    ></textarea>
                                </div>

                                <div class="form-group-row">
                                    <div class="form-group">
                                        <label for="brand-colors">Brand Colors</label>
                                        <input
                                            type="text"
                                            id="brand-colors"
                                            name="brand_colors"
                                            placeholder="#2563eb, #111827"
                                        >
                                        <small>Enter hex colors separated by commas</small>
                                    </div>
                                    <div class="form-group">
                                        <label for="style-mood">Style Mood</label>
                                        <select id="style-mood" name="style_mood">
                                            <option value="">Choose a style...</option>
                                            <option value="modern">Modern & Clean</option>
                                            <option value="luxury">Luxury & Elegant</option>
                                            <option value="playful">Fun & Playful</option>
                                            <option value="minimal">Minimal & Simple</option>
                                            <option value="bold">Bold & Dynamic</option>
                                            <option value="professional">Professional & Corporate</option>
                                        </select>
                                    </div>
                                </div>

                                <div class="info-box" style="background: #f0f9ff; padding: 16px; border-radius: 8px; margin: 20px 0; border-left: 4px solid #2563eb;">
                                    <p style="margin: 0; color: #1e40af;">
                                        <strong><i data-lucide="sparkles" class="icon-inline icon-sm icon-ai"></i> This will style ALL product pages</strong><br>
                                        Your styling will automatically apply to all current and future WooCommerce products.
                                        This includes product images, titles, prices, descriptions, add-to-cart buttons, and reviews.
                                    </p>
                                </div>

                                <div class="modal-actions">
                                    <button type="button" class="btn-secondary">Cancel</button>
                                    <button type="submit" class="btn-primary">
                                        <span class="btn-text">Generate Product Styles</span>
                                        <span class="btn-loading" style="display: none;">✨ Crafting beautiful styles...</span>
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            `;

            $('body').append(modal);
            $('#style-prompt').focus();
        }

        /**
         * Generate product page styles
         */
        async generateProductStyles() {
            const $form = $('#product-style-form');
            const $submitBtn = $form.find('.btn-primary');

            // Get form data
            const prompt = $form.find('[name="prompt"]').val().trim();
            const brandColorsInput = $form.find('[name="brand_colors"]').val().trim();
            const styleMood = $form.find('[name="style_mood"]').val();

            // Validate required fields
            if (!prompt) {
                this.showMessage('error', 'Please describe your desired product page style.');
                return;
            }

            // Parse brand colors
            const brandColors = brandColorsInput
                ? brandColorsInput.split(',').map(c => c.trim()).filter(c => c.length > 0)
                : [];

            const formData = {
                prompt: prompt,
                brand_colors: brandColors,
                style_mood: styleMood,
                business_info: {
                    name: '',
                    industry: ''
                }
            };


            // Update button state
            $submitBtn.prop('disabled', true);
            $submitBtn.find('.btn-text').hide();
            $submitBtn.find('.btn-loading').show();

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/woocommerce/style-products',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce,
                        'Content-Type': 'application/json'
                    },
                    data: JSON.stringify(formData)
                });

                if (response.success) {
                    let successMessage = 'Product pages styled successfully! ';
                    if (response.sample_product_url) {
                        successMessage += `<a href="${response.sample_product_url}" target="_blank">View Sample Product</a>`;
                    }

                    this.showMessage('success', successMessage);
                    this.closeModal();
                } else {
                    throw new Error(response.error || 'Style generation failed');
                }

            } catch (error) {
                console.error('Product styling error:', error);

                let errorMessage = 'Failed to generate product styles. ';
                if (error.responseJSON && error.responseJSON.message) {
                    errorMessage += error.responseJSON.message;
                } else {
                    errorMessage += 'Please try again.';
                }

                this.showMessage('error', errorMessage);
            }

            // Reset button state
            $submitBtn.prop('disabled', false);
            $submitBtn.find('.btn-text').show();
            $submitBtn.find('.btn-loading').hide();
        }

        /**
         * Reset product page styles to default
         */
        async resetProductStyles() {
            if (!confirm('Are you sure you want to reset all product page styles? This will remove all AI-generated styling and cannot be undone.')) {
                return;
            }

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/woocommerce/reset-styles',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce,
                        'Content-Type': 'application/json'
                    }
                });

                if (response.success) {
                    this.showMessage('success', 'Product styles have been reset to defaults. Please refresh your product pages to see the changes.');
                } else {
                    throw new Error(response.error || 'Reset failed');
                }

            } catch (error) {
                console.error('Product styles reset error:', error);

                let errorMessage = 'Failed to reset product styles. ';
                if (error.responseJSON && error.responseJSON.message) {
                    errorMessage += error.responseJSON.message;
                } else {
                    errorMessage += 'Please try again.';
                }

                this.showMessage('error', errorMessage);
            }
        }

        /**
         * Open manage product styles modal
         */
        async openManageStylesModal() {
            try {
                // Get current styles from the backend
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/woocommerce/get-styles',
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    }
                });

                // Create the manage styles modal
                const modal = `
                    <div class="dreamformer-modal-overlay">
                        <div class="dreamformer-modal" style="max-width: 900px; max-height: 90vh; overflow-y: auto; margin: 20px auto;">
                            <div class="modal-header">
                                <h3>Manage WooCommerce Product Styles</h3>
                                <button class="modal-close">&times;</button>
                            </div>
                            <div class="modal-body" style="max-height: none; overflow: visible;">
                                <div class="info-box" style="background: #f0f9ff; padding: 16px; border-radius: 8px; margin-bottom: 20px; border-left: 4px solid #2563eb;">
                                    <p style="margin: 0; color: #1e40af;">
                                        <strong><i data-lucide="calendar" class="icon-inline icon-sm icon-info"></i> Generated:</strong> ${response.generated_at || 'Unknown'}<br>
                                        <strong><i data-lucide="palette" class="icon-inline icon-sm icon-primary"></i> Style Mood:</strong> ${response.style_mood || 'Not specified'}<br>
                                        <strong><i data-lucide="bot" class="icon-inline icon-sm icon-ai"></i> AI Model:</strong> ${response.ai_model || 'Unknown'}<br>
                                        <strong><i data-lucide="file-text" class="icon-inline icon-sm icon-secondary"></i> Original Prompt:</strong> ${response.prompt || 'Not available'}
                                    </p>
                                </div>

                                <div class="form-group">
                                    <label for="css-editor" style="font-weight: 600; margin-bottom: 8px; display: block;">
                                        CSS Code <span style="color: #6b7280;">(Edit directly or generate new styles)</span>
                                    </label>
                                    <textarea
                                        id="css-editor"
                                        style="width: 100%; height: 300px; font-family: 'Courier New', monospace; font-size: 13px; line-height: 1.4; border: 1px solid #d1d5db; border-radius: 6px; padding: 12px; resize: vertical; box-sizing: border-box;"
                                        placeholder="/* No styles found. Generate some first using the 'Generate New' button! */"
                                    >${response.css || ''}</textarea>
                                    <small style="color: #6b7280; margin-top: 8px; display: block;">
                                        <i data-lucide="lightbulb" class="icon-inline icon-sm icon-warning"></i> Tip: You can manually edit the CSS above or generate completely new styles.
                                    </small>
                                </div>

                                <div class="modal-actions" style="gap: 10px; display: flex; flex-wrap: wrap; margin-top: 20px; padding-top: 20px; border-top: 1px solid #e5e7eb; position: sticky; bottom: 0; background: white; z-index: 10;">
                                    <button type="button" class="btn-primary" id="save-css-btn" style="background: #059669;">
                                        <i data-lucide="save" class="icon-inline icon-sm"></i> Save Changes
                                    </button>
                                    <button type="button" class="btn-secondary" id="copy-css-btn">
                                        <i data-lucide="clipboard" class="icon-inline icon-sm"></i> Copy CSS
                                    </button>
                                    <button type="button" class="btn-secondary" id="generate-new-btn" style="background: #7c3aed; color: white;">
                                        <i data-lucide="sparkles" class="icon-inline icon-sm"></i> Generate New
                                    </button>
                                    <button type="button" class="btn-danger" id="delete-css-btn">
                                        <i data-lucide="trash-2" class="icon-inline icon-sm"></i> Delete All
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                `;

                $('body').append(modal);

                // Bind event handlers for the modal actions
                $('#copy-css-btn').on('click', () => this.copyCSSToClipboard());
                $('#save-css-btn').on('click', () => this.saveEditedStyles());
                $('#generate-new-btn').on('click', () => {
                    this.closeModal();
                    this.openStyleProductsModal(); // Reuse existing generation modal
                });
                $('#delete-css-btn').on('click', () => {
                    this.closeModal();
                    this.resetProductStyles(); // Reuse existing reset functionality
                });

            } catch (error) {
                console.error('Failed to load current styles:', error);

                if (error.status === 404) {
                    // No styles found - show helpful message and redirect to generation
                    if (confirm('No product styles found. Would you like to generate some now?')) {
                        this.openStyleProductsModal();
                    }
                } else {
                    // Other error
                    let errorMessage = 'Failed to load current styles. ';
                    if (error.responseJSON && (error.responseJSON.error || error.responseJSON.message)) {
                        errorMessage += (error.responseJSON.error || error.responseJSON.message);
                    } else {
                        errorMessage += 'Please try again.';
                    }
                    this.showMessage('error', errorMessage);
                }
            }
        }

        /**
         * Copy CSS to clipboard
         */
        copyCSSToClipboard() {
            const css = $('#css-editor').val();
            if (css) {
                navigator.clipboard.writeText(css).then(() => {
                    this.showMessage('success', 'CSS copied to clipboard!');
                }).catch(() => {
                    // Fallback for older browsers
                    $('#css-editor').select();
                    document.execCommand('copy');
                    this.showMessage('success', 'CSS copied to clipboard!');
                });
            } else {
                this.showMessage('error', 'No CSS to copy.');
            }
        }

        /**
         * Save edited CSS styles
         */
        async saveEditedStyles() {
            const css = $('#css-editor').val().trim();

            if (!css) {
                this.showMessage('error', 'CSS cannot be empty.');
                return;
            }

            const $saveBtn = $('#save-css-btn');
            const originalText = $saveBtn.text();

            // Update button state
            $saveBtn.prop('disabled', true).html('<i data-lucide="loader-2" class="icon-inline icon-sm animate-spin"></i> Saving...');

            try {
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/woocommerce/update-styles',
                    method: 'PUT',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce,
                        'Content-Type': 'application/json'
                    },
                    data: JSON.stringify({ css: css })
                });

                if (response.success) {
                    this.showMessage('success', 'Styles updated successfully! Changes will be visible on your product pages.');
                    this.closeModal();
                } else {
                    throw new Error(response.error || 'Save failed');
                }

            } catch (error) {
                console.error('Save styles error:', error);

                let errorMessage = 'Failed to save styles. ';
                if (error.responseJSON && error.responseJSON.error) {
                    errorMessage += error.responseJSON.error;
                } else {
                    errorMessage += 'Please check your CSS and try again.';
                }

                this.showMessage('error', errorMessage);
            }

            // Reset button state
            $saveBtn.prop('disabled', false).text(originalText);
        }

        async exportPage(pageId) {
            try {
                // Call export endpoint
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + `ai-builder/v1/pages/${pageId}/export`,
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce
                    },
                    dataType: 'json'  // Ensure jQuery parses as JSON
                });

                // Extract clean data (avoid jQuery wrapper circular references)
                const exportData = typeof response === 'object' && response !== null ? response : JSON.parse(response);

                // Create download
                const dataStr = JSON.stringify(exportData, null, 2);
                const dataBlob = new Blob([dataStr], {type: 'application/json'});
                const url = URL.createObjectURL(dataBlob);
                const link = document.createElement('a');
                const filename = `dreamformer-page-${pageId}-${new Date().toISOString().split('T')[0]}.json`;
                link.href = url;
                link.download = filename;
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
                URL.revokeObjectURL(url);

                this.showMessage('success', 'Page exported successfully!');
            } catch (error) {
                console.error('Export error:', error);
                this.showMessage('error', 'Failed to export page: ' + (error.responseJSON?.error || 'Unknown error'));
            }
        }

        openImportModal() {
            const modalHtml = `
                <div class="dreamformer-modal-overlay">
                    <div class="dreamformer-modal" style="max-width: 600px;">
                        <div class="modal-header">
                            <h3><span class="dashicons dashicons-upload"></span> Import Dreamformer Page</h3>
                            <button class="modal-close">&times;</button>
                        </div>
                        <div class="modal-body">
                            <p>Upload a Dreamformer page export file (.json) to import it to this site.</p>
                            <div class="import-drop-zone" style="border: 2px dashed #ddd; padding: 40px; text-align: center; margin: 20px 0; border-radius: 8px; background: #f9f9f9;">
                                <span class="dashicons dashicons-cloud-upload" style="font-size: 48px; color: #666; display: block; margin-bottom: 10px;"></span>
                                <p style="margin: 10px 0;"><strong>Drop file here or click to browse</strong></p>
                                <input type="file" id="import-file-input" accept=".json" style="display: none;" />
                                <button class="button button-primary" id="browse-file-btn">Choose File</button>
                                <p class="description" style="margin-top: 10px;">Selected file: <span id="selected-filename">None</span></p>
                            </div>
                            <div id="import-warnings" style="display: none; background: #fff3cd; border-left: 4px solid #ffc107; padding: 12px; margin: 15px 0;">
                                <strong>⚠️ Warnings:</strong>
                                <ul id="warnings-list" style="margin: 5px 0 0 20px;"></ul>
                            </div>
                            <div class="modal-actions">
                                <button class="button button-secondary modal-close">Cancel</button>
                                <button class="button button-primary" id="import-page-btn" disabled>Import Page</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            // Remove any existing modals to prevent duplicates
            $('.dreamformer-modal-overlay').remove();

            $('body').append(modalHtml);

            // Show modal
            const $modal = $('.dreamformer-modal-overlay');
            setTimeout(() => {
                $modal.addClass('active');
            }, 10);
        }

        handleImportFileSelect(file) {
            if (!file.name.endsWith('.json')) {
                this.showMessage('error', 'Please select a valid JSON file');
                return;
            }

            this.selectedImportFile = file;
            $('#selected-filename').text(file.name);
            $('#import-page-btn').prop('disabled', false);
        }

        async performImport() {
            if (!this.selectedImportFile) return;

            const $importBtn = $('#import-page-btn');
            const originalText = $importBtn.text();
            $importBtn.prop('disabled', true).html('<i data-lucide="loader-2" class="icon-inline icon-sm animate-spin"></i> Importing...');

            try {
                // Read file
                const fileContent = await new Promise((resolve, reject) => {
                    const reader = new FileReader();
                    reader.onload = (e) => resolve(e.target.result);
                    reader.onerror = reject;
                    reader.readAsText(this.selectedImportFile);
                });

                // Parse JSON
                const data = JSON.parse(fileContent);

                // Validate
                if (!data.format_version || !data.page) {
                    throw new Error('Invalid export file format');
                }

                // Import via API
                const response = await $.ajax({
                    url: aiSiteBuilder.restUrl + 'ai-builder/v1/pages/import',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': aiSiteBuilder.nonce,
                        'Content-Type': 'application/json'
                    },
                    data: JSON.stringify({ data })
                });

                if (response.success) {
                    // Show warnings if any
                    if (response.warnings && response.warnings.length > 0) {
                        const warningsList = response.warnings.map(w => `<li>${w}</li>`).join('');
                        $('#warnings-list').html(warningsList);
                        $('#import-warnings').show();

                        // Wait a moment for user to see warnings
                        await new Promise(resolve => setTimeout(resolve, 3000));
                    }

                    this.showMessage('success', 'Page imported successfully! Opening in editor...');
                    this.closeModal();
                    this.loadPages(); // Refresh grid

                    // Redirect to editor after short delay
                    setTimeout(() => {
                        window.location.href = response.dreamformer_url;
                    }, 1500);
                } else {
                    throw new Error(response.error || 'Import failed');
                }

            } catch (error) {
                console.error('Import error:', error);
                let errorMsg = 'Import failed: ';
                if (error.responseJSON && error.responseJSON.error) {
                    errorMsg += error.responseJSON.error;
                } else if (error.message) {
                    errorMsg += error.message;
                } else {
                    errorMsg += 'Unknown error';
                }
                this.showMessage('error', errorMsg);
                $importBtn.prop('disabled', false).text(originalText);
            }
        }
    }

    // Initialize when document ready
    $(document).ready(() => {
        new DreamformerPages();
    });

})(jQuery);