/**
 * Dreamformer Code Editor
 *
 * Monaco-based code editor for manual editing of AI-generated pages
 * Integrates with existing draft/publish workflow
 *
 * @since 3.3.0
 */

(function($) {
    'use strict';

    // Debug mode - set to true to enable console logging
    const DEBUG = false;

    /**
     * Debug logging helper
     * Logs only when DEBUG is enabled or window.dreamformerDebug is true
     */
    function log(...args) {
        if (DEBUG || (typeof window.dreamformerDebug !== 'undefined' && window.dreamformerDebug === true)) {
            log('', ...args);
        }
    }

    // Make class available globally so theme-editor.js can initialize it
    window.DreamformerCodeEditor = class DreamformerCodeEditor {
        constructor(pageId) {
            this.pageId = pageId;
            this.editor = null;
            this.currentFile = 'index.html';
            this.files = {};
            this.isDirty = false;
            this.isCodeEditorVisible = false;
            this.monacoLoaded = false;
            this.autoSaveInterval = null;

            this.init();
        }

        async init() {
            log('Initializing for page', this.pageId);

            // Setup toggle button listener
            this.setupToggleButton();

            // Setup keyboard shortcuts (global)
            this.setupGlobalShortcuts();

            // Setup help tooltip
            this.setupHelpTooltip();
        }

        setupToggleButton() {
            log('setupToggleButton() called');

            const toggleBtn = $('#toggle-code-editor-btn');
            log('Button search result:', toggleBtn);
            log('Button exists:', toggleBtn.length > 0);
            log('Button HTML:', toggleBtn.length > 0 ? toggleBtn[0].outerHTML : 'N/A');

            if (!toggleBtn.length) {
                console.error('[Code Editor] Toggle button NOT FOUND in DOM!');
                log('Available buttons:', $('button').map(function() { return this.id; }).get());
                return;
            }

            log('Attaching click handler to button...');

            toggleBtn.on('click', async (e) => {
                log('BUTTON CLICKED!', e);
                e.preventDefault();

                log('Current state - isCodeEditorVisible:', this.isCodeEditorVisible);

                if (!this.isCodeEditorVisible) {
                    log('Calling showCodeEditor()...');
                    await this.showCodeEditor();
                } else {
                    log('Calling showPreview()...');
                    this.showPreview();
                }
            });

            // Test if handler was attached
            const events = $._data(toggleBtn[0], 'events');
            log('Events attached to button:', events);
            log('Toggle button initialized successfully ✓');
        }

        async showCodeEditor() {
            log('Showing editor');

            // Load Monaco if not already loaded
            if (!this.monacoLoaded) {
                const loadSuccess = await this.loadMonaco();
                if (!loadSuccess) {
                    alert('Failed to load code editor. Please refresh the page.');
                    return;
                }
            }

            // Load files if not already loaded
            if (Object.keys(this.files).length === 0) {
                const loadSuccess = await this.loadFiles();
                if (!loadSuccess) {
                    return; // Error already shown
                }
            }

            // Create editor if not already created
            if (!this.editor) {
                this.createEditor();
            }

            // Toggle visibility
            $('#preview-container').hide();
            $('#code-editor-container').css('display', 'flex'); // Use flex, not block
            $('#toggle-code-editor-btn').addClass('active');

            this.isCodeEditorVisible = true;

            // Layout Monaco (required after visibility change)
            if (this.editor) {
                // Force layout multiple times to ensure proper sizing
                setTimeout(() => {
                    this.editor.layout();
                    log('Monaco layout triggered');
                }, 50);

                setTimeout(() => {
                    this.editor.layout();
                    log('Monaco layout re-triggered');
                }, 200);
            }

            // Start auto-save
            this.startAutoSave();

            log('Editor shown');
        }

        showPreview() {
            log('Showing preview');

            // Save current file before switching
            if (this.editor && this.isDirty) {
                this.files[this.currentFile] = this.editor.getValue();
            }

            // Toggle visibility
            $('#code-editor-container').hide();
            $('#preview-container').show();
            $('#toggle-code-editor-btn').removeClass('active').removeClass('has-changes');

            this.isCodeEditorVisible = false;

            // Stop auto-save
            this.stopAutoSave();

            // Refresh preview to show any code changes
            const iframe = $('#theme-preview-iframe');
            if (iframe.length) {
                const currentSrc = iframe.attr('src');
                // Add timestamp to force reload
                const separator = currentSrc.includes('?') ? '&' : '?';
                iframe.attr('src', currentSrc.split('&_refresh=')[0] + separator + '_refresh=' + Date.now());
            }

            log('Preview shown and refreshed');
        }

        async loadMonaco() {
            log('Loading Monaco Editor');

            return new Promise((resolve) => {
                const script = document.createElement('script');
                script.src = 'https://cdnjs.cloudflare.com/ajax/libs/monaco-editor/0.45.0/min/vs/loader.min.js';

                script.onload = () => {
                    require.config({
                        paths: {
                            vs: 'https://cdnjs.cloudflare.com/ajax/libs/monaco-editor/0.45.0/min/vs'
                        }
                    });

                    require(['vs/editor/editor.main'], () => {
                        log('Monaco Editor loaded successfully');
                        this.monacoLoaded = true;
                        resolve(true);
                    });
                };

                script.onerror = () => {
                    console.error('[Code Editor] Failed to load Monaco Editor');
                    resolve(false);
                };

                document.head.appendChild(script);
            });
        }

        async loadFiles() {
            log('Loading files from server');
            this.updateStatus('Loading files...');

            try {
                const response = await fetch(
                    `/wp-json/ai-builder/v1/pages/${this.pageId}/code-files`,
                    {
                        headers: {
                            'X-WP-Nonce': aiSiteBuilder.nonce
                        }
                    }
                );

                const data = await response.json();

                // Check for premium_required error (403 Forbidden)
                if (!response.ok) {
                    if (data.code === 'premium_required' || data.code === 'no_subscription') {
                        // Show styled upgrade modal
                        showDreamformerUpgradeModal({
                            title: 'Code Editor - Pro Feature',
                            message: data.message || 'Manual code editing requires a Pro plan. Upgrade to unlock full control over your AI-generated pages.',
                            features: [
                                {
                                    icon: 'sparkles',
                                    label: 'Manual Code Editing',
                                    description: 'Edit HTML, CSS, and JavaScript directly'
                                },
                                {
                                    icon: 'code',
                                    label: 'Monaco Editor',
                                    description: 'Professional VS Code editing experience with syntax highlighting'
                                },
                                {
                                    icon: 'save',
                                    label: 'Auto-save & Version Control',
                                    description: 'Automatic saving every 2 minutes, never lose your changes'
                                },
                                {
                                    icon: 'palette',
                                    label: 'Multi-file Support',
                                    description: 'Switch between HTML, CSS, and JavaScript files seamlessly'
                                }
                            ],
                            upgradeUrl: data.upgrade_url || aiSiteBuilder.pricingUrl
                        });

                        this.updateStatus('Premium required');
                        return false;
                    }
                    throw new Error(data.message || 'Failed to load files');
                }

                if (data.success) {
                    this.files = data.files;
                    log('Files loaded:', Object.keys(this.files));
                    this.updateStatus('Ready');
                    return true;
                } else {
                    throw new Error(data.message || 'Failed to load files');
                }
            } catch (error) {
                console.error('[Code Editor] Error loading files:', error);
                alert('Failed to load page files: ' + error.message);
                this.updateStatus('Load failed');
                return false;
            }
        }

        createEditor() {
            log('Creating Monaco editor instance');

            const container = document.getElementById('monaco-editor-container');
            if (!container) {
                console.error('[Code Editor] Container not found');
                return;
            }

            this.editor = monaco.editor.create(container, {
                value: this.files[this.currentFile] || '',
                language: this.getLanguage(this.currentFile),
                theme: 'vs-dark',
                automaticLayout: false, // We'll handle this manually
                minimap: { enabled: true },
                fontSize: 14,
                lineNumbers: 'on',
                wordWrap: 'on',
                formatOnPaste: true,
                formatOnType: true,
                scrollBeyondLastLine: false,
                renderWhitespace: 'selection',
                folding: true,
                bracketPairColorization: { enabled: true }
            });

            // Setup event listeners
            this.setupEditorEvents();
            this.setupTabSwitching();

            log('Editor created');
        }

        setupEditorEvents() {
            // Track changes
            this.editor.onDidChangeModelContent(() => {
                this.isDirty = true;
                this.updateStatus('Modified - Not Saved');

                // Add visual indicator on toggle button
                $('#toggle-code-editor-btn').addClass('has-changes');
            });

            // Track cursor position
            this.editor.onDidChangeCursorPosition((e) => {
                $('#editor-position').text(`Ln ${e.position.lineNumber}, Col ${e.position.column}`);
            });

            // Keyboard shortcuts
            this.editor.addCommand(monaco.KeyMod.CtrlCmd | monaco.KeyCode.KeyS, () => {
                this.saveFiles();
            });

            // Warn before leaving with unsaved changes
            $(window).on('beforeunload', (e) => {
                if (this.isDirty) {
                    const message = 'You have unsaved changes. Are you sure you want to leave?';
                    e.returnValue = message;
                    return message;
                }
            });
        }

        setupTabSwitching() {
            $('.code-tab').on('click', (e) => {
                const filename = $(e.currentTarget).data('file');
                this.switchFile(filename);
            });
        }

        setupGlobalShortcuts() {
            // Cmd/Ctrl + E to toggle code editor
            $(document).on('keydown', (e) => {
                if ((e.metaKey || e.ctrlKey) && e.key === 'e') {
                    e.preventDefault();
                    $('#toggle-code-editor-btn').trigger('click');
                }
            });
        }

        setupHelpTooltip() {
            const $helpBtn = $('#code-editor-help-btn');
            const $tooltip = $('#code-editor-help-tooltip');

            log('setupHelpTooltip() - Button found:', $helpBtn.length > 0);
            log('setupHelpTooltip() - Tooltip found:', $tooltip.length > 0);

            if (!$helpBtn.length || !$tooltip.length) {
                log('Help tooltip elements not found - skipping setup');
                return; // Elements not found
            }

            log('Setting up help tooltip interactions');

            // Toggle tooltip on click
            $helpBtn.on('click', (e) => {
                e.stopPropagation();
                $tooltip.toggleClass('show');
            });

            // Close tooltip when clicking outside
            $(document).on('click', (e) => {
                if (!$(e.target).closest('.code-editor-help').length) {
                    $tooltip.removeClass('show');
                }
            });

            // Close tooltip on ESC
            $(document).on('keydown', (e) => {
                if (e.key === 'Escape' && $tooltip.hasClass('show')) {
                    $tooltip.removeClass('show');
                }
            });
        }

        switchFile(filename) {
            if (filename === this.currentFile) {
                return; // Already on this file
            }

            log('Switching to file:', filename);

            // Save current file content
            this.files[this.currentFile] = this.editor.getValue();

            // Switch to new file
            this.currentFile = filename;

            // Update editor
            const model = this.editor.getModel();
            monaco.editor.setModelLanguage(model, this.getLanguage(filename));
            this.editor.setValue(this.files[filename] || '');

            // Update active tab
            $('.code-tab').removeClass('active');
            $(`.code-tab[data-file="${filename}"]`).addClass('active');

            this.updateStatus('Ready');
        }

        async saveFiles() {
            log('Saving files');

            // Get current editor content
            this.files[this.currentFile] = this.editor.getValue();

            this.updateStatus('Saving...');

            try {
                const response = await fetch(
                    `/wp-json/ai-builder/v1/pages/${this.pageId}/code-files`,
                    {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-WP-Nonce': aiSiteBuilder.nonce
                        },
                        body: JSON.stringify({ files: this.files })
                    }
                );

                const data = await response.json();

                // Check for premium_required error (403 Forbidden)
                if (!response.ok) {
                    if (data.code === 'premium_required' || data.code === 'no_subscription') {
                        alert('🔒 Code Editor - Premium Feature\n\n' + data.message + '\n\nYour changes have not been saved. Please upgrade to Pro plan.');
                        this.updateStatus('Save blocked - Premium required');
                        return;
                    }
                    throw new Error(data.message || 'Save failed');
                }

                if (data.success) {
                    this.isDirty = false;
                    this.updateStatus('Saved');

                    // Remove change indicator
                    $('#toggle-code-editor-btn').removeClass('has-changes');

                    // Refresh preview in background (if not visible)
                    this.refreshPreviewInBackground();

                    log('Files saved successfully');

                    // Show success briefly
                    setTimeout(() => {
                        if (!this.isDirty) {
                            this.updateStatus('Ready');
                        }
                    }, 2000);
                } else {
                    throw new Error(data.message || 'Save failed');
                }
            } catch (error) {
                console.error('[Code Editor] Save error:', error);
                this.updateStatus('Save failed: ' + error.message);
                alert('Failed to save files: ' + error.message);
            }
        }

        refreshPreviewInBackground() {
            // Refresh preview iframe in background so it's ready when user toggles
            const iframe = $('#theme-preview-iframe');
            if (iframe.length) {
                const currentSrc = iframe.attr('src');
                const separator = currentSrc.includes('?') ? '&' : '?';
                iframe.attr('src', currentSrc.split('&_refresh=')[0] + separator + '_refresh=' + Date.now());
                log('Preview refreshed in background');
            }
        }

        startAutoSave() {
            // Auto-save every 2 minutes
            if (this.autoSaveInterval) {
                clearInterval(this.autoSaveInterval);
            }

            this.autoSaveInterval = setInterval(() => {
                if (this.isDirty) {
                    log('Auto-save triggered');
                    this.saveFiles();
                }
            }, 120000); // 2 minutes

            log('Auto-save started (2 min interval)');
        }

        stopAutoSave() {
            if (this.autoSaveInterval) {
                clearInterval(this.autoSaveInterval);
                this.autoSaveInterval = null;
                log('Auto-save stopped');
            }
        }

        getLanguage(filename) {
            const ext = filename.split('.').pop().toLowerCase();
            const langMap = {
                'html': 'html',
                'css': 'css',
                'js': 'javascript',
                'json': 'json'
            };
            return langMap[ext] || 'plaintext';
        }

        updateStatus(text) {
            const statusEl = $('#editor-status');
            if (statusEl.length) {
                statusEl.text(text);
            }
        }
    }; // End of DreamformerCodeEditor class

    /**
     * Show premium upgrade modal
     *
     * @param {Object} options - Modal options
     * @param {string} options.title - Modal title
     * @param {string} options.message - Main message
     * @param {Array} options.features - Array of feature objects {icon, label, description}
     * @param {string} options.upgradeUrl - URL to pricing page
     * @param {Function} options.onUpgrade - Callback when upgrade clicked
     * @param {Function} options.onCancel - Callback when cancelled
     */
    window.showDreamformerUpgradeModal = function(options) {
        const defaults = {
            title: 'Upgrade to Pro',
            message: 'This feature requires a Pro plan.',
            features: [
                {
                    icon: 'sparkles',
                    label: 'Manual Code Editing',
                    description: 'Edit HTML, CSS, and JavaScript directly'
                },
                {
                    icon: 'code',
                    label: 'Monaco Editor',
                    description: 'Full VS Code editing experience'
                },
                {
                    icon: 'save',
                    label: 'Auto-save & Version Control',
                    description: 'Never lose your changes'
                }
            ],
            upgradeUrl: aiSiteBuilder.pricingUrl,
            onUpgrade: null,
            onCancel: null
        };

        const settings = Object.assign({}, defaults, options);

        // Create modal HTML with Lucide icons
        const featuresHtml = settings.features.map(f => `
            <div class="dreamformer-upgrade-modal__feature">
                <i data-lucide="${f.icon}" class="dreamformer-upgrade-modal__feature-icon"></i>
                <div class="dreamformer-upgrade-modal__feature-text">
                    <span class="dreamformer-upgrade-modal__feature-label">${f.label}</span>
                    <span class="dreamformer-upgrade-modal__feature-desc">${f.description}</span>
                </div>
            </div>
        `).join('');

        const modalHtml = `
            <div class="dreamformer-upgrade-modal" id="dreamformer-upgrade-modal">
                <div class="dreamformer-upgrade-modal__content">
                    <button class="dreamformer-upgrade-modal__close" id="modal-close">×</button>
                    <div class="dreamformer-upgrade-modal__header">
                        <i data-lucide="lock" class="dreamformer-upgrade-modal__icon"></i>
                        <h2 class="dreamformer-upgrade-modal__title">${settings.title}</h2>
                        <p class="dreamformer-upgrade-modal__subtitle">Premium Feature</p>
                    </div>
                    <div class="dreamformer-upgrade-modal__body">
                        <p class="dreamformer-upgrade-modal__message">${settings.message}</p>
                        <div class="dreamformer-upgrade-modal__features">
                            ${featuresHtml}
                        </div>
                        <div class="dreamformer-upgrade-modal__actions">
                            <button class="dreamformer-upgrade-modal__button dreamformer-upgrade-modal__button--secondary" id="modal-cancel">
                                Maybe Later
                            </button>
                            <button class="dreamformer-upgrade-modal__button dreamformer-upgrade-modal__button--primary" id="modal-upgrade">
                                Upgrade to Pro →
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if any
        $('#dreamformer-upgrade-modal').remove();

        // Add modal to body
        $('body').append(modalHtml);

        const $modal = $('#dreamformer-upgrade-modal');

        // Initialize Lucide icons in modal
        if (typeof lucide !== 'undefined' && lucide.createIcons) {
            // Lucide scans the entire document for data-lucide attributes
            lucide.createIcons();
        }

        // Show modal with animation
        setTimeout(() => {
            $modal.addClass('show');
        }, 10);

        // Handle upgrade button
        $('#modal-upgrade').on('click', function() {
            $modal.removeClass('show');
            setTimeout(() => {
                $modal.remove();
            }, 300);

            if (settings.onUpgrade) {
                settings.onUpgrade();
            } else {
                window.location.href = settings.upgradeUrl;
            }
        });

        // Handle cancel button
        $('#modal-cancel, #modal-close').on('click', function() {
            $modal.removeClass('show');
            setTimeout(() => {
                $modal.remove();
            }, 300);

            if (settings.onCancel) {
                settings.onCancel();
            }
        });

        // Close on backdrop click
        $modal.on('click', function(e) {
            if (e.target === this) {
                $('#modal-cancel').trigger('click');
            }
        });

        // Close on ESC key
        $(document).on('keydown.upgradeModal', function(e) {
            if (e.key === 'Escape') {
                $('#modal-cancel').trigger('click');
                $(document).off('keydown.upgradeModal');
            }
        });
    };

    // NOTE: Initialization is handled by theme-editor.js after HTML is rendered
    // This ensures the toggle button exists before we try to attach event listeners

})(jQuery);
